/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_debug_console.h"
#include "board.h"

#include "fsl_wdog.h"

#include "pin_mux.h"
#include "clock_config.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define DEMO_WDOG_BASE       WDOG1
#define DEMO_WDOG_IRQHandler WDOG1_IRQHandler

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

static volatile int irq_counter = 0;
volatile uint32_t g_systickCounter = 0;
volatile uint32_t g_msTicks = 0;

/*******************************************************************************
 * Code
 ******************************************************************************/
void DEMO_WDOG_IRQHandler(void)
{
    WDOG_ClearInterruptStatus(DEMO_WDOG_BASE, kWDOG_InterruptFlag);
    /* User code. User can do urgent case before timeout reset.
     * IE. user can backup the ram data or ram log to flash.
     * the period is set by config.interruptTimeValue, user need to
     * check the period between interrupt and timeout.
     */
    irq_counter++;
}

/*!
 * @brief Main function
 */
int main(void)
{
	int timeMs = 0;
	bool firstIrq = true;
    uint16_t resetFlag = 0U;
    wdog_config_t config;
    BOARD_ConfigMPU();
    BOARD_InitWDOGPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();
    SystemCoreClockUpdate();

    PRINTF("\r\n******** System Start ********\r\n");
    PRINTF("System reset by:");

    resetFlag = WDOG_GetStatusFlags(DEMO_WDOG_BASE);

    switch (resetFlag & (kWDOG_PowerOnResetFlag | kWDOG_TimeoutResetFlag | kWDOG_SoftwareResetFlag))
    {
        case kWDOG_PowerOnResetFlag:
            PRINTF(" Power On Reset!\r\n");
            break;
        case kWDOG_TimeoutResetFlag:
            PRINTF(" Time Out Reset!\r\n");
            break;
        case kWDOG_SoftwareResetFlag:
            PRINTF(" Software Reset!\r\n");
            break;
        default:
            PRINTF(" Error status!\r\n");
            break;
    }

	PRINTF("\r\n- Test the WDOG refresh function by using interrupt.\r\n");
	/*
	 * wdogConfig->enableWdog = true;
	 * wdogConfig->workMode.enableWait = true;
	 * wdogConfig->workMode.enableStop = false;
	 * wdogConfig->workMode.enableDebug = false;
	 * wdogConfig->enableInterrupt = false;
	 * wdogConfig->enablePowerdown = false;
	 * wdogConfig->resetExtension = false;
	 * wdogConfig->timeoutValue = 0xFFU;
	 * wdogConfig->interruptTimeValue = 0x04u;
	 */
	WDOG_GetDefaultConfig(&config);
	config.timeoutValue       = 0xFU; /* Timeout value is (0xF+1)/2 = 8 sec. */
	config.enableInterrupt    = true;
	config.interruptTimeValue = 0x4U; /* Interrupt occurred (0x4)/2 = 2 sec before WDOG timeout. */
	WDOG_Init(DEMO_WDOG_BASE, &config);

	PRINTF("--- wdog Init done, timeout=%ds, irq=%ds before ---\r\n\r\n", (config.timeoutValue+1)/2, config.interruptTimeValue/2);

	PRINTF("   Feeds the WDOG every second for the first 5 seconds\r\n");
	PRINTF("   and no interrupts should be seen during that period.\r\n");
	PRINTF("   Then stops feeding the WDOG and an interrupt should come 6s later.\r\n");
	PRINTF("   Will then feed once to reset the WDOG.\r\n");
	PRINTF("   After 6 seconds the interrupt will occur again and this time it will\r\n");
	PRINTF("   be ignored so the board will reset 2 seconds later.\r\n\r\n");
    while (1)
    {
    	if ((timeMs % 1000)==0) {
    		PRINTF("%3ds\r\n", timeMs/1000);

    		if (timeMs < 5000) {
                WDOG_Refresh(DEMO_WDOG_BASE);
                PRINTF("WDOG has be refreshed!\r\n");
    		} else if (timeMs == 5000) {
                WDOG_Refresh(DEMO_WDOG_BASE);
                PRINTF("WDOG has be refreshed! Should see interrupt in 6 seconds\r\n");
    		}
    	}
    	if (irq_counter > 0) {
    		PRINTF("GOT WARNING INTERRUPT!\r\n");
    		if (firstIrq) {
    			firstIrq = false;
                WDOG_Refresh(DEMO_WDOG_BASE);
                PRINTF("WDOG has be refreshed! Should see interrupt in 6 seconds\r\n");
    		} else {
    			PRINTF("Ignoring it. Should reset in 2 seconds\r\n");
    		}
    		irq_counter = 0;
    	}

        /* Delay. */
        SDK_DelayAtLeastUs(100*1000, SystemCoreClock);
        timeMs += 100;
    }
}
