/*
 * Copyright 2021-2023 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "wifi_bt_config.h"
#include "pin_mux.h"
#include "fsl_gpio.h"
#include "pca6416.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#ifdef WIFI_BT_USE_USD_INTERFACE
  /* The iMXRT1062 Developers Kit does not have support for the uSD connector
     for Wi-Fi so make sure to prevent selection of a module that use it. */
  #error "uSD interface is not supported on this board"
#endif

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
extern void BOARD_SD_Pin_Config(uint32_t freq);
extern uint32_t BOARD_USDHC1ClockConfiguration(void);

/*******************************************************************************
 * Variables
 ******************************************************************************/
/*!brief sdmmc dma buffer */
AT_NONCACHEABLE_SECTION_ALIGN(static uint32_t s_sdmmcHostDmaBuffer[BOARD_SDMMC_HOST_DMA_DESCRIPTOR_BUFFER_SIZE],
                              SDMMCHOST_DMA_DESCRIPTOR_BUFFER_ALIGN_SIZE);
#if defined SDMMCHOST_ENABLE_CACHE_LINE_ALIGN_TRANSFER && SDMMCHOST_ENABLE_CACHE_LINE_ALIGN_TRANSFER
/* two cache line length for sdmmc host driver maintain unalign transfer */
SDK_ALIGN(static uint8_t s_sdmmcCacheLineAlignBuffer[BOARD_SDMMC_DATA_BUFFER_ALIGN_SIZE * 2U],
          BOARD_SDMMC_DATA_BUFFER_ALIGN_SIZE);
#endif

static sd_io_voltage_t s_ioVoltage = {
    .type = BOARD_SDMMC_SD_IO_VOLTAGE_CONTROL_TYPE,
    .func = NULL,
};
static sdmmchost_t s_host;
static sdio_card_int_t s_sdioInt;

/*******************************************************************************
 * Code
 ******************************************************************************/
void BOARD_WIFI_BT_Enable(bool enable)
{
    if (enable)
    {
        /* Enable module */
#ifdef WIFI_BT_USE_M2_INTERFACE

        // Control WL_REG_ON.
        // Connected to both pins 23 and 56 on the M.2 connector.
        PCA6416_SetPins(PCA_WL_REG_ON_3V3);

        // Control BT_REG_ON.
        // Connected to pin 54 on the M.2 connector
        PCA6416_SetPins(PCA_BT_REG_ON_3V3);

#elif defined(WIFI_BT_USE_USD_INTERFACE)

        /* Enable power supply for SD */
        BOARD_SDMMC_SD_POWER_RESET_GPIO_BASE->GDIR &= ~(1UL << BOARD_SDMMC_SD_POWER_RESET_GPIO_PIN);

#endif /* WIFI_BT_USE_M2_INTERFACE */

        vTaskDelay(pdMS_TO_TICKS(100));
    }
    else
    {
        /* Disable module */
#ifdef WIFI_BT_USE_M2_INTERFACE

        // Control WL_REG_ON.
        // Connected to both pins 23 and 56 on the M.2 connector.
        PCA6416_ClearPins(PCA_WL_REG_ON_3V3);

        // Control BT_REG_ON.
        // Connected to pin 54 on the M.2 connector
        PCA6416_ClearPins(PCA_BT_REG_ON_3V3);

#elif defined(WIFI_BT_USE_USD_INTERFACE)

        /* Disable power supply for SD */
        BOARD_SDMMC_SD_POWER_RESET_GPIO_BASE->GDIR |= (1UL << BOARD_SDMMC_SD_POWER_RESET_GPIO_PIN);

#endif /* WIFI_BT_USE_M2_INTERFACE */

        vTaskDelay(pdMS_TO_TICKS(100));
    }
}

void BOARD_WIFI_BT_Config(void *card, sdio_int_t cardInt)
{
    assert(card);

    s_host.dmaDesBuffer         = s_sdmmcHostDmaBuffer;
    s_host.dmaDesBufferWordsNum = BOARD_SDMMC_HOST_DMA_DESCRIPTOR_BUFFER_SIZE;
    s_host.enableCacheControl   = BOARD_SDMMC_HOST_CACHE_CONTROL;
#if defined SDMMCHOST_ENABLE_CACHE_LINE_ALIGN_TRANSFER && SDMMCHOST_ENABLE_CACHE_LINE_ALIGN_TRANSFER
    s_host.cacheAlignBuffer     = s_sdmmcCacheLineAlignBuffer;
    s_host.cacheAlignBufferSize = BOARD_SDMMC_DATA_BUFFER_ALIGN_SIZE * 2U;
#endif

    ((sdio_card_t *)card)->host                                = &s_host;
    ((sdio_card_t *)card)->host->hostController.base           = BOARD_SDMMC_SDIO_HOST_BASEADDR;
    ((sdio_card_t *)card)->host->hostController.sourceClock_Hz = BOARD_USDHC1ClockConfiguration();

    ((sdio_card_t *)card)->usrParam.cd         = NULL;
    ((sdio_card_t *)card)->usrParam.pwr        = NULL;
    ((sdio_card_t *)card)->usrParam.ioStrength = BOARD_SD_Pin_Config;
    ((sdio_card_t *)card)->usrParam.ioVoltage  = &s_ioVoltage;
    ((sdio_card_t *)card)->usrParam.maxFreq    = BOARD_SDMMC_SD_HOST_SUPPORT_SDR104_FREQ;
    if (cardInt != NULL)
    {
        s_sdioInt.cardInterrupt                 = cardInt;
        ((sdio_card_t *)card)->usrParam.sdioInt = &s_sdioInt;
    }

    NVIC_SetPriority(BOARD_SDMMC_SDIO_HOST_IRQ, BOARD_SDMMC_SDIO_HOST_IRQ_PRIORITY);

#if !defined(COEX_APP_SUPPORT) || (defined(COEX_APP_SUPPORT) && !defined(CONFIG_WIFI_IND_DNLD))
    BOARD_WIFI_BT_Enable(false);
#endif
}
