/*
 * Copyright 2018-2019 NXP
 * All rights reserved.
 *
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_common.h"
#include "fsl_device_registers.h"
#include "fsl_adapter_timer.h"
#include "fsl_gpt.h"

typedef struct _hal_timer_handle_struct_t
{
    uint32_t timeout;
    uint32_t timerClock_Hz;
    hal_timer_callback_t callback;
    void *callbackParam;
    uint8_t instance;
} hal_timer_handle_struct_t;

/*******************************************************************************
 * Variables
 ******************************************************************************/

static GPT_Type *const s_GptBase[] = GPT_BASE_PTRS;

static hal_timer_handle_t s_timerHandle[sizeof(s_GptBase) / sizeof(GPT_Type *)];
/************************************************************************************
*************************************************************************************
* Private prototypes
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/
static void HAL_TimerInterruptHandle(uint8_t instance)
{
    hal_timer_handle_struct_t *halTimerState = (hal_timer_handle_struct_t *)s_timerHandle[instance];

    GPT_ClearStatusFlags(s_GptBase[halTimerState->instance], kGPT_OutputCompare1Flag);
    if (halTimerState->callback != NULL)
    {
        halTimerState->callback(halTimerState->callbackParam);
    }
}

void GPT0_IRQHandler(void);
void GPT0_IRQHandler(void)
{
    HAL_TimerInterruptHandle(0);
    SDK_ISR_EXIT_BARRIER;
}

void GPT1_IRQHandler(void);
void GPT1_IRQHandler(void)
{
    HAL_TimerInterruptHandle(1);
    SDK_ISR_EXIT_BARRIER;
}
void GPT2_IRQHandler(void);
void GPT2_IRQHandler(void)
{
    HAL_TimerInterruptHandle(2);
    SDK_ISR_EXIT_BARRIER;
}
/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/
hal_timer_status_t HAL_TimerInit(hal_timer_handle_t halTimerHandle, hal_timer_config_t *halTimerConfig)
{
    IRQn_Type instanceIrq[] = GPT_IRQS;
    IRQn_Type irqId;
    hal_timer_handle_struct_t *halTimerState = halTimerHandle;
    /* Structure of initialize GPT */
    gpt_config_t gptConfig;

    assert(sizeof(hal_timer_handle_struct_t) == HAL_TIMER_HANDLE_SIZE);
    assert(halTimerConfig);
    assert(halTimerHandle);
    assert(halTimerConfig->instance < (sizeof(s_GptBase) / sizeof(GPT_Type *)));

    halTimerState->timeout       = halTimerConfig->timeout;
    halTimerState->instance      = halTimerConfig->instance;
    halTimerState->timerClock_Hz = halTimerConfig->srcClock_Hz;
    irqId                        = instanceIrq[halTimerState->instance];
    GPT_GetDefaultConfig(&gptConfig);

    /* Initialize GPT module */
    GPT_Init(s_GptBase[halTimerState->instance], &gptConfig);
    /* Set both GPT modules to 1 second duration */
    GPT_SetOutputCompareValue(s_GptBase[halTimerState->instance], kGPT_OutputCompare_Channel1,
                              (uint32_t)USEC_TO_COUNT(halTimerState->timeout, halTimerState->timerClock_Hz));
    /* Enable GPT Output Compare1 interrupt */
    GPT_EnableInterrupts(s_GptBase[halTimerState->instance], (uint32_t)kGPT_OutputCompare1InterruptEnable);

    s_timerHandle[halTimerState->instance] = halTimerHandle;
    NVIC_SetPriority((IRQn_Type)irqId, HAL_TIMER_ISR_PRIORITY);
    (void)EnableIRQ(irqId);
    return kStatus_HAL_TimerSuccess;
}

/*************************************************************************************/
void HAL_TimerDeinit(hal_timer_handle_t halTimerHandle)
{
    assert(halTimerHandle);
    hal_timer_handle_struct_t *halTimerState = halTimerHandle;
    s_timerHandle[halTimerState->instance]   = NULL;
    GPT_Deinit(s_GptBase[halTimerState->instance]);
}

/*************************************************************************************/
void HAL_TimerEnable(hal_timer_handle_t halTimerHandle)
{
    assert(halTimerHandle);
    hal_timer_handle_struct_t *halTimerState = halTimerHandle;
    GPT_StartTimer(s_GptBase[halTimerState->instance]);
}

/*************************************************************************************/
void HAL_TimerDisable(hal_timer_handle_t halTimerHandle)
{
    assert(halTimerHandle);
    hal_timer_handle_struct_t *halTimerState = halTimerHandle;
    GPT_StopTimer(s_GptBase[halTimerState->instance]);
}

/*************************************************************************************/
void HAL_TimerInstallCallback(hal_timer_handle_t halTimerHandle, hal_timer_callback_t callback, void *callbackParam)
{
    assert(halTimerHandle);
    hal_timer_handle_struct_t *halTimerState = halTimerHandle;
    halTimerState->callback                  = callback;
    halTimerState->callbackParam             = callbackParam;
}

uint32_t HAL_TimerGetMaxTimeout(hal_timer_handle_t halTimerHandle)
{
    uint32_t reserveCount;
    assert(halTimerHandle);
    hal_timer_handle_struct_t *halTimerState = halTimerHandle;
    reserveCount                             = (uint32_t)MSEC_TO_COUNT((4), (halTimerState->timerClock_Hz));
    if ((reserveCount < MSEC_TO_COUNT((1), (halTimerState->timerClock_Hz))) ||
        (reserveCount > (GPT_CNT_COUNT_MASK - 0xfU)))
    {
        /* make sure 1ms timeout doesn't overflow */
        assert(USEC_TO_COUNT(1000, (halTimerState->timerClock_Hz)) < GPT_CNT_COUNT_MASK);
        return 1000;
    }

    return (uint32_t)COUNT_TO_USEC(((uint64_t)GPT_CNT_COUNT_MASK - (uint64_t)reserveCount),
                                   halTimerState->timerClock_Hz);
}
/* return micro us */
uint32_t HAL_TimerGetCurrentTimerCount(hal_timer_handle_t halTimerHandle)
{
    assert(halTimerHandle);
    hal_timer_handle_struct_t *halTimerState = halTimerHandle;
    return (uint32_t)COUNT_TO_USEC((uint64_t)GPT_GetCurrentTimerCount(s_GptBase[halTimerState->instance]),
                                   halTimerState->timerClock_Hz);
}

hal_timer_status_t HAL_TimerUpdateTimeout(hal_timer_handle_t halTimerHandle, uint32_t timeout)
{
    uint32_t tickCount;
    assert(halTimerHandle);
    hal_timer_handle_struct_t *halTimerState = halTimerHandle;
    halTimerState->timeout                   = timeout;
    tickCount = (uint32_t)USEC_TO_COUNT(halTimerState->timeout, halTimerState->timerClock_Hz);

    if ((tickCount < 1U) || (tickCount > (GPT_CNT_COUNT_MASK - 0xfU)))
    {
        return kStatus_HAL_TimerOutOfRanger;
    }
    GPT_SetOutputCompareValue(s_GptBase[halTimerState->instance], kGPT_OutputCompare_Channel1, tickCount);
    return kStatus_HAL_TimerSuccess;
}

void HAL_TimerExitLowpower(hal_timer_handle_t halTimerHandle)
{
    assert(halTimerHandle);
}

void HAL_TimerEnterLowpower(hal_timer_handle_t halTimerHandle)
{
    assert(halTimerHandle);
}
