/*******************************************************************************
*
* E M B E D D E D   W I Z A R D   P R O J E C T
*
*                                                Copyright (c) TARA Systems GmbH
*                                    written by Paul Banach and Manfred Schweyer
*
********************************************************************************
*
* This software and related documentation ("Software") are intellectual
* property owned by TARA Systems and are copyright of TARA Systems.
* Any modification, copying, reproduction or redistribution of the Software in
* whole or in part by any means not in accordance with the End-User License
* Agreement for Embedded Wizard is expressly prohibited. The removal of this
* preamble is expressly prohibited.
*
********************************************************************************
*
* DESCRIPTION:
*   The header 'ewextbmp' declares the platform dependent representation of a
*   bitmap resource. This includes data structures to store the attributes of
*   a bitmap resource and a set of macros to cover these structures.
*
*   This header file provides the interface between the platform independent
*   Graphics Engine and the platform specific representation of resources as
*   they will be generated by the Embedded Wizard resource converters of the
*   particular target system.
*
*******************************************************************************/

#ifndef EWEXTBMP_H
#define EWEXTBMP_H


#ifdef __cplusplus
  extern "C"
  {
#endif


/* The following is an identification number for bitmap resources. */
#define EW_MAGIC_NO_BITMAP      0x626D7064
#define EW_MAGIC_NO_BITMAP_R90  0x626D7065
#define EW_MAGIC_NO_BITMAP_R180 0x626D7066
#define EW_MAGIC_NO_BITMAP_R270 0x626D7067


/* The macro EW_BITMAP_PIXEL_SECTION_NAME is used to determine the section where
   the linker should locate the memory areas containing bitmap pixel data. */
#if defined EW_BITMAP_PIXEL_SECTION_NAME && !defined EW_BITMAP_PIXEL_PRAGMA
  #define EW_STRINGIZE( aArg )      EW_STRINGIZE_ARG( aArg )
  #define EW_STRINGIZE_ARG( aArg )  #aArg

  #if defined __ICCARM__
    #define EW_BITMAP_PIXEL_PRAGMA                                             \
      _Pragma(EW_STRINGIZE(location=EW_STRINGIZE( EW_BITMAP_PIXEL_SECTION_NAME )))
  #elif defined __CC_ARM
    #define EW_BITMAP_PIXEL_PRAGMA                                             \
      __attribute__((section ( EW_STRINGIZE( EW_BITMAP_PIXEL_SECTION_NAME ))))
  #elif defined __GNUC__
    #define EW_BITMAP_PIXEL_PRAGMA                                             \
      __attribute__((section ( EW_STRINGIZE( EW_BITMAP_PIXEL_SECTION_NAME ))))
  #endif
#endif

#ifndef EW_BITMAP_PIXEL_PRAGMA
  #define EW_BITMAP_PIXEL_PRAGMA
#endif



/*******************************************************************************
* TYPE:
*   XBmpFrameRes
*
* DESCRIPTION:
*   The structure XBmpFrameRes describes the attributes of a single frame as it
*   will be stored within a bitmap resource.
*
* ELEMENTS:
*   OpqX, OpqY,
*   OpqWidth,
*   OpqHeight - Optional full-opaque area of the frame relative to the top-left
*     corner of the bitmap resource.
*   Pixel1,
*   Pixel2    - Offset within the compressed pixel data stream of the bitmap.
*
*******************************************************************************/
typedef struct
{
  int               OpqX;
  int               OpqY;
  int               OpqWidth;
  int               OpqHeight;
  unsigned int      Pixel1;
  unsigned int      Pixel2;
} XBmpFrameRes;


/*******************************************************************************
* TYPE:
*   XBmpRes
*
* DESCRIPTION:
*   The structure XBmpRes describes the attributes of a single bitmap resource
*   as it will be stored in the code memory.
*
* ELEMENTS:
*   MagicNo     - Unique ID of this resource type. It exists for verification
*     purpose only.
*   Format      - Unique color format of the bitmap resource. This can be one
*     of EW_DRIVER_VARIANT_XXX macros.
*   FrameWidth,
*   FrameHeight - Size in pixel of a single frame of the bitmap resource.
*   FrameDelay  - Delay in milliseconds for animated bitmap resources. If == 0,
*     no animation will take place.
*   NoOfFrames  - Number of real frames, this bitmap resources consists of. At
*     least one frame has to exists.
*   Mapping        - A table with indices to map between virtual frames numbers
*     and the numbers of really existing frames. This is the case if the bitmap
*     contains dupplicates of frames. The number of entries within the table
*     corresponds to the number of virtual frames. The table has to be finished
*     by 0xFFFF.
*   Frames      - Pointer to an array containing the attributes of the included
*     frames.
*   Pixel1,
*   Pixel2      - Pointer to the pixel data of the bitmap. Depending on the used
*     configuration, the data stored there can be compressed or not.
*   Clut        - Pointer to an array containing color information for bitmap
*     resources stored in the Index8 format. The first value in the array gets 
*     the number of following color entries. The color entries are stored with
*     32-bit per value. Every value is stored in the format corresponding to the
*     of pixel in the Native bitmap.
*   Compressed  - Determines whether data stored in Pixel1/Pixel2 is compressed
*     or not. Compressed pixel data has always to be copied to a surface memory
*     before being used. Uncompressed data can (depending on the target system)
*     be accessed directly when the bitmap is e.g. composed on the screen, etc.
*   Name        - The name of the bitmap resource. It is used for debugging
*     only.
*
*******************************************************************************/
typedef struct XBmpRes
{
  unsigned int          MagicNo;
  int                   Format;
  int                   FrameWidth;
  int                   FrameHeight;
  int                   FrameDelay;
  int                   NoOfFrames;
  const unsigned short* Mapping;
  const XBmpFrameRes*   Frames;
  const void*           Pixel1;
  const void*           Pixel2;
  const unsigned int*   Clut;
  int                   Compressed;
  const char*           Name;
} XBmpRes;


/*******************************************************************************
* MACRO:
*   EW_DECLARE_BITMAP_RES
*   EW_DEFINE_BITMAP_RES
*   EW_BITMAP_FRAMES
*   EW_BITMAP_FRAME
*   EW_BITMAP_PIXEL
*   EW_BITMAP_PIXEL_R90
*   EW_BITMAP_PIXEL_R180
*   EW_BITMAP_PIXEL_R270
*   EW_BITMAP_PIXEL_U8
*   EW_BITMAP_PIXEL_U8_R90
*   EW_BITMAP_PIXEL_U8_R180
*   EW_BITMAP_PIXEL_U8_R270
*   EW_BITMAP_PIXEL_U16
*   EW_BITMAP_PIXEL_U16_R90
*   EW_BITMAP_PIXEL_U16_R180
*   EW_BITMAP_PIXEL_U16_R270
*   EW_BITMAP_PIXEL_U32
*   EW_BITMAP_PIXEL_U32_R90
*   EW_BITMAP_PIXEL_U32_R180
*   EW_BITMAP_PIXEL_U32_R270
*   EW_BITMAP_PIXEL1_U8
*   EW_BITMAP_PIXEL1_U8_R90
*   EW_BITMAP_PIXEL1_U8_R180
*   EW_BITMAP_PIXEL1_U8_R270
*   EW_BITMAP_PIXEL1_U16
*   EW_BITMAP_PIXEL1_U16_R90
*   EW_BITMAP_PIXEL1_U16_R180
*   EW_BITMAP_PIXEL1_U16_R270
*   EW_BITMAP_PIXEL1_U32
*   EW_BITMAP_PIXEL1_U32_R90
*   EW_BITMAP_PIXEL1_U32_R180
*   EW_BITMAP_PIXEL1_U32_R270
*   EW_BITMAP_PIXEL2_U8
*   EW_BITMAP_PIXEL2_U8_R90
*   EW_BITMAP_PIXEL2_U8_R180
*   EW_BITMAP_PIXEL2_U8_R270
*   EW_BITMAPS_TABLE
*   EW_BITMAP
*   EW_END_OF_BITMAP_RES
*
* DESCRIPTION:
*   Following macros are used to build up the structure of a bitmap resource in
*   a better readable way. The macros are used by the Embedded Wizard bitmap
*   resource converter.
*
*   Please note the different variants of the EW_BITMAP_PIXEL macros:
*
*   - Macros having RXXX in their name determine whether the pixel data of the
*     bitmap is stored in rotated foramt (rotated by 90, 180, 270 degree). This
*     is essential for target systems with rotated display.
*
*   - Macros having UXX in their name are used to cover uncompressed pixel data.
*     The value XX (8,16 or 32) determine the size in bits for a single pixel in
*     the storage. This is essential for target systems supporting bitmaps from
*     the constant ROM code area.
*
*   - The digit 1 or 2 following the PIXEL word determines the memory plane of
*     the pixel data. If the digit is omitted, the entire pixel information is
*     stored within a single plane.
*
*******************************************************************************/
#define EW_DECLARE_BITMAP_RES( aName )                                         \
  extern const XVariant aName;

#define EW_DEFINE_BITMAP_RES( aName )                                          \
  enum                                                                         \
  {                                                                            \
    _d_##aName

#define EW_BITMAP_FRAMES( aName, aLangId, aFormat, aFrameWidth, aFrameHeight,  \
    aFrameDelay )                                                              \
  };                                                                           \
  enum         { _ff_##aName##aLangId = aFormat      };                        \
  enum         { _fw_##aName##aLangId = aFrameWidth  };                        \
  enum         { _fh_##aName##aLangId = aFrameHeight };                        \
  enum         { _fd_##aName##aLangId = aFrameDelay  };                        \
  static const XBmpFrameRes _f_##aName##aLangId[] =                            \
  {

#define EW_BITMAP_FRAME( aOpqX, aOpqY, aOpqWidth, aOpqHeight, aPixel1,         \
    aPixel2 )                                                                  \
  { aOpqX, aOpqY, aOpqWidth, aOpqHeight, aPixel1, aPixel2 }


#define EW_BITMAP_MAPPING( aName, aLangId )                                    \
  };                                                                           \
  static const unsigned short _fm_##aName##aLangId[] =                         \
  {
  
#define EW_BITMAP_PIXEL( aName, aLangId )                                      \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned int _cp_##aName##aLangId[];                            \
  extern const unsigned int _cl_##aName##aLangId[];                            \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP,                                                        \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _cp_##aName##aLangId,                                                      \
    _cp_##aName##aLangId,                                                      \
    _cl_##aName##aLangId,                                                      \
    1,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned int _cp_##aName##aLangId[] =           \
  {

#define EW_BITMAP_PIXEL_R90( aName, aLangId )                                  \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned int _cp_##aName##aLangId[];                            \
  extern const unsigned int _cl_##aName##aLangId[];                            \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R90,                                                    \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _cp_##aName##aLangId,                                                      \
    _cp_##aName##aLangId,                                                      \
    _cl_##aName##aLangId,                                                      \
    1,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned int _cp_##aName##aLangId[] =           \
  {

#define EW_BITMAP_PIXEL_R180( aName, aLangId )                                 \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned int _cp_##aName##aLangId[];                            \
  extern const unsigned int _cl_##aName##aLangId[];                            \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R180,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _cp_##aName##aLangId,                                                      \
    _cp_##aName##aLangId,                                                      \
    _cl_##aName##aLangId,                                                      \
    1,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned int _cp_##aName##aLangId[] =           \
  {

#define EW_BITMAP_PIXEL_R270( aName, aLangId )                                 \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned int _cp_##aName##aLangId[];                            \
  extern const unsigned int _cl_##aName##aLangId[];                            \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R270,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _cp_##aName##aLangId,                                                      \
    _cp_##aName##aLangId,                                                      \
    _cl_##aName##aLangId,                                                      \
    1,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned int _cp_##aName##aLangId[] =           \
  {

#define EW_BITMAP_PIXEL_U8( aName, aLangId )                                   \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned char _p1_##aName##aLangId[];                           \
  extern const unsigned int  _cl_##aName##aLangId[];                           \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP,                                                        \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    0,                                                                         \
    _cl_##aName##aLangId,                                                      \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned char _p1_##aName##aLangId[] =          \
  {

#define EW_BITMAP_PIXEL_U8_R90( aName, aLangId )                               \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned char _p1_##aName##aLangId[];                           \
  extern const unsigned int  _cl_##aName##aLangId[];                           \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R90,                                                    \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    0,                                                                         \
    _cl_##aName##aLangId,                                                      \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned char _p1_##aName##aLangId[] =          \
  {

#define EW_BITMAP_PIXEL_U8_R180( aName, aLangId )                              \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned char _p1_##aName##aLangId[];                           \
  extern const unsigned int  _cl_##aName##aLangId[];                           \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R180,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    0,                                                                         \
    _cl_##aName##aLangId,                                                      \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned char _p1_##aName##aLangId[] =          \
  {

#define EW_BITMAP_PIXEL_U8_R270( aName, aLangId )                              \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned char _p1_##aName##aLangId[];                           \
  extern const unsigned int  _cl_##aName##aLangId[];                           \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R270,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    0,                                                                         \
    _cl_##aName##aLangId,                                                      \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned char _p1_##aName##aLangId[] =          \
  {

#define EW_BITMAP_PIXEL_U16( aName, aLangId )                                  \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned short _p1_##aName##aLangId[];                          \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP,                                                        \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned short _p1_##aName##aLangId[] =         \
  {

#define EW_BITMAP_PIXEL_U16_R90( aName, aLangId )                              \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned short _p1_##aName##aLangId[];                          \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R90,                                                    \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned short _p1_##aName##aLangId[] =         \
  {

#define EW_BITMAP_PIXEL_U16_R180( aName, aLangId )                             \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned short _p1_##aName##aLangId[];                          \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R180,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned short _p1_##aName##aLangId[] =         \
  {

#define EW_BITMAP_PIXEL_U16_R270( aName, aLangId )                             \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned short _p1_##aName##aLangId[];                          \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R270,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned short _p1_##aName##aLangId[] =         \
  {

#define EW_BITMAP_PIXEL_U32( aName, aLangId )                                  \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned int _p1_##aName##aLangId[];                            \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP,                                                        \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned int _p1_##aName##aLangId[] =           \
  {

#define EW_BITMAP_PIXEL_U32_R90( aName, aLangId )                              \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned int _p1_##aName##aLangId[];                            \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R90,                                                    \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned int _p1_##aName##aLangId[] =           \
  {

#define EW_BITMAP_PIXEL_U32_R180( aName, aLangId )                             \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned int _p1_##aName##aLangId[];                            \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R180,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned int _p1_##aName##aLangId[] =           \
  {

#define EW_BITMAP_PIXEL_U32_R270( aName, aLangId )                             \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned int _p1_##aName##aLangId[];                            \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R270,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned int _p1_##aName##aLangId[] =           \
  {

#define EW_BITMAP_PIXEL1_U8( aName, aLangId )                                  \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned char _p1_##aName##aLangId[];                           \
  extern const unsigned char _p2_##aName##aLangId[];                           \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP,                                                        \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    _p2_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned char _p1_##aName##aLangId[] =          \
  {

#define EW_BITMAP_PIXEL1_U8_R90( aName, aLangId )                              \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned char _p1_##aName##aLangId[];                           \
  extern const unsigned char _p2_##aName##aLangId[];                           \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R90,                                                    \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    _p2_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned char _p1_##aName##aLangId[] =          \
  {

#define EW_BITMAP_PIXEL1_U8_R180( aName, aLangId )                             \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned char _p1_##aName##aLangId[];                           \
  extern const unsigned char _p2_##aName##aLangId[];                           \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R180,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    _p2_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned char _p1_##aName##aLangId[] =          \
  {

#define EW_BITMAP_PIXEL1_U8_R270( aName, aLangId )                             \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned char _p1_##aName##aLangId[];                           \
  extern const unsigned char _p2_##aName##aLangId[];                           \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R270,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    _p2_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned char _p1_##aName##aLangId[] =          \
  {

#define EW_BITMAP_PIXEL1_U16( aName, aLangId )                                 \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned short _p1_##aName##aLangId[];                          \
  extern const unsigned char  _p2_##aName##aLangId[];                          \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP,                                                        \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    _p2_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned short _p1_##aName##aLangId[] =         \
  {

#define EW_BITMAP_PIXEL1_U16_R90( aName, aLangId )                             \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned short _p1_##aName##aLangId[];                          \
  extern const unsigned char  _p2_##aName##aLangId[];                          \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R90,                                                    \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    _p2_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned short _p1_##aName##aLangId[] =         \
  {

#define EW_BITMAP_PIXEL1_U16_R180( aName, aLangId )                            \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned short _p1_##aName##aLangId[];                          \
  extern const unsigned char  _p2_##aName##aLangId[];                          \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R180,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    _p2_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned short _p1_##aName##aLangId[] =         \
  {

#define EW_BITMAP_PIXEL1_U16_R270( aName, aLangId )                            \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned short _p1_##aName##aLangId[];                          \
  extern const unsigned char  _p2_##aName##aLangId[];                          \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R270,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    _p2_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned short _p1_##aName##aLangId[] =         \
  {

#define EW_BITMAP_PIXEL1_U32( aName, aLangId )                                 \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned int  _p1_##aName##aLangId[];                           \
  extern const unsigned char _p2_##aName##aLangId[];                           \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP,                                                        \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    _p2_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned int _p1_##aName##aLangId[] =           \
  {

#define EW_BITMAP_PIXEL1_U32_R90( aName, aLangId )                             \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned int  _p1_##aName##aLangId[];                           \
  extern const unsigned char _p2_##aName##aLangId[];                           \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R90,                                                    \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    _p2_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned int _p1_##aName##aLangId[] =           \
  {

#define EW_BITMAP_PIXEL1_U32_R180( aName, aLangId )                            \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned int  _p1_##aName##aLangId[];                           \
  extern const unsigned char _p2_##aName##aLangId[];                           \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R180,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    _p2_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned int _p1_##aName##aLangId[] =           \
  {

#define EW_BITMAP_PIXEL1_U32_R270( aName, aLangId )                            \
    0xFFFF                                                                     \
  };                                                                           \
  extern const unsigned int  _p1_##aName##aLangId[];                           \
  extern const unsigned char _p2_##aName##aLangId[];                           \
  static const XBmpRes _##aName##aLangId =                                     \
  {                                                                            \
    EW_MAGIC_NO_BITMAP_R270,                                                   \
    _ff_##aName##aLangId,                                                      \
    _fw_##aName##aLangId,                                                      \
    _fh_##aName##aLangId,                                                      \
    _fd_##aName##aLangId,                                                      \
    sizeof( _f_##aName##aLangId ) / sizeof( _f_##aName##aLangId[0] ),          \
    _fm_##aName##aLangId,                                                      \
    _f_##aName##aLangId,                                                       \
    _p1_##aName##aLangId,                                                      \
    _p2_##aName##aLangId,                                                      \
    0,                                                                         \
    0,                                                                         \
    #aName                                                                     \
  };                                                                           \
                                                                               \
  EW_BITMAP_PIXEL_PRAGMA const unsigned int _p1_##aName##aLangId[] =           \
  {

#define EW_BITMAP_PIXEL2_U8( aName, aLangId )                                  \
  };                                                                           \
  EW_BITMAP_PIXEL_PRAGMA const unsigned char _p2_##aName##aLangId[] =          \
  {

#define EW_BITMAP_PIXEL2_U8_R90( aName, aLangId )                              \
  };                                                                           \
  EW_BITMAP_PIXEL_PRAGMA const unsigned char _p2_##aName##aLangId[] =          \
  {

#define EW_BITMAP_PIXEL2_U8_R180( aName, aLangId )                             \
  };                                                                           \
  EW_BITMAP_PIXEL_PRAGMA const unsigned char _p2_##aName##aLangId[] =          \
  {

#define EW_BITMAP_PIXEL2_U8_R270( aName, aLangId )                             \
  };                                                                           \
  EW_BITMAP_PIXEL_PRAGMA const unsigned char _p2_##aName##aLangId[] =          \
  {

#define EW_BITMAP_CLUT( aName, aLangId )                                       \
  };                                                                           \
  const unsigned int _cl_##aName##aLangId[] =                                  \
  {

#define EW_BITMAP_CLUT_EMPTY( aName, aLangId )                                 \
  };                                                                           \
  const unsigned int _cl_##aName##aLangId[] =                                  \
  {                                                                            \
    0

#define EW_BITMAPS_TABLE( aName )                                              \
  };                                                                           \
  static const XResource _##aName[] =                                          \
  {

#define EW_BITMAP( aName, aLangId )                                            \
    { aLangId, &_##aName##aLangId }

#define EW_END_OF_BITMAP_RES( aName )                                          \
  };


#ifdef __cplusplus
  }
#endif

#endif /* EWEXTBMP_H */


/* pba, msy */
