/*
 * Copyright (c) 2020 SixOctets Systems
 * Copyright (c) 2019 Aaron Tsui <aaron.tsui@outlook.com>
 * Copyright 2021-2022 NXP
 *
 * SPDX-License-Identifier: Apache-2.0
 */
#include <porting.h>
#include <stdio.h>
#include <string.h>
#include <stddef.h>
#include <errno/errno.h>
#include <toolchain.h>
#include <porting.h>
#include <fsl_debug_console.h>
#include <sys/byteorder.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/uuid.h>
#include <bluetooth/gatt.h>
#include <bluetooth/services/ias.h>


#define BT_IAS_ALERT_LVL_LEN 1


#if defined(CONFIG_BT_IAS_SEC_AUTH) && (CONFIG_BT_IAS_SEC_AUTH > 0)
#define IAS_ALERT_LEVEL_PERM BT_GATT_PERM_WRITE_AUTHEN
#elif defined(CONFIG_BT_IAS_SEC_ENC)
#define IAS_ALERT_LEVEL_PERM BT_GATT_PERM_WRITE_ENCRYPT
#else
#define IAS_ALERT_LEVEL_PERM BT_GATT_PERM_WRITE
#endif

struct alerting_device {
	enum bt_ias_alert_lvl alert_level;
};

static struct alerting_device devices[CONFIG_BT_MAX_CONN];
static enum bt_ias_alert_lvl curr_lvl;
static struct bt_ias_cb s_bt_ias_cb = {0};
static void set_alert_level(void)
{
	enum bt_ias_alert_lvl alert_level;
        struct bt_ias_cb *cb;

	alert_level = devices[0].alert_level;
	for (int i = 1; i < CONFIG_BT_MAX_CONN; i++) {
		if (alert_level < devices[i].alert_level) {
			alert_level = devices[i].alert_level;
		}
	}

	if (curr_lvl == alert_level) {
		return;
	}

	curr_lvl = alert_level;

	if (alert_level == BT_IAS_ALERT_LVL_HIGH_ALERT) {
                if (s_bt_ias_cb.high_alert) {
                        s_bt_ias_cb.high_alert();
                }
		PRINTF("High alert");
	} else if (alert_level == BT_IAS_ALERT_LVL_MILD_ALERT) {
                    if (s_bt_ias_cb.mild_alert) {
                            s_bt_ias_cb.mild_alert();
                    }
                    PRINTF("Mild alert");
	} else {
                  if (s_bt_ias_cb.no_alert) {
                          s_bt_ias_cb.no_alert();
                  }
                  PRINTF("No alert");
	}
}

static void disconnected(struct bt_conn *conn, uint8_t reason)
{
	devices[bt_conn_index(conn)].alert_level = BT_IAS_ALERT_LVL_NO_ALERT;
	set_alert_level();
}

int bt_ias_local_alert_stop(void)
{
	if (curr_lvl == BT_IAS_ALERT_LVL_NO_ALERT) {
		return -EALREADY;
	}

	for (int idx = 0; idx < CONFIG_BT_MAX_CONN; idx++) {
		devices[idx].alert_level = BT_IAS_ALERT_LVL_NO_ALERT;
	}
	curr_lvl = BT_IAS_ALERT_LVL_NO_ALERT;
	set_alert_level();

	return 0;
}

static ssize_t bt_ias_write_alert_lvl(struct bt_conn *conn, const struct bt_gatt_attr *attr,
				      const void *buf, uint16_t len, uint16_t offset, uint8_t flags)
{
	struct net_buf_simple data;
	enum bt_ias_alert_lvl alert_val;

	if (offset > 0) {
		return BT_GATT_ERR(BT_ATT_ERR_INVALID_OFFSET);
	}

	if (len != BT_IAS_ALERT_LVL_LEN) {
		return BT_GATT_ERR(BT_ATT_ERR_INVALID_ATTRIBUTE_LEN);
	}

	net_buf_simple_init_with_data(&data, (void *)buf, len);
	alert_val = (enum bt_ias_alert_lvl)net_buf_simple_pull_u8(&data);
	devices[bt_conn_index(conn)].alert_level = alert_val;

	if ( alert_val > BT_IAS_ALERT_LVL_HIGH_ALERT) {
		return BT_GATT_ERR(BT_ATT_ERR_VALUE_NOT_ALLOWED);
	}
	set_alert_level();

	return len;
}

#if 0
BT_CONN_CB_DEFINE(conn_callbacks) = {
	.disconnected = disconnected,
};
#else
struct bt_conn_cb conn_callbacks = {
	.disconnected = disconnected,
};
#endif

/* Init/deinit APIs */
int bt_ias_init(struct bt_ias_cb *cb)
{
    memcpy(&s_bt_ias_cb,cb, sizeof(struct bt_ias_cb));
	
	bt_conn_cb_register(&conn_callbacks);

    return 0;
}

int bt_ias_deinit(void)
{
    memset(&s_bt_ias_cb,0x0, sizeof(struct bt_ias_cb));
    return 0;
}

/* Immediate Alert Service Declaration */
BT_GATT_SERVICE_DEFINE(ias_svc,
	BT_GATT_PRIMARY_SERVICE(BT_UUID_IAS),
	BT_GATT_CHARACTERISTIC(BT_UUID_ALERT_LEVEL, BT_GATT_CHRC_WRITE_WITHOUT_RESP,
			       IAS_ALERT_LEVEL_PERM, NULL,
			       bt_ias_write_alert_lvl, NULL));
