/*
 * Copyright 2023 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef __FWK_FSABSTRACTION_H__
#define __FWK_FSABSTRACTION_H__

#include "stdint.h"

/**
 * \brief Provide an easy RAM buffer read and write API to Little FS File System.
 *
 * The File System module provides :
 * - an easy initialization and configuration for LittleFS Filesystem withpossibility of sharing the file system between
 * multiple callers.
 * - easy to use APIs function to easy read, write buffers from RAM to/from filesystem
 *
 * The File system module initializes the LittleFS file system using the mflash driver abstraction provided by
 * fwk_lfs_mflash.[ch]. The LittleFS filesystem can not be shared with other users. The reason for this is that same
 * file cache buffer is used for all file opened. A previous file must be closed before a new file can be opened. This
 * is enforced by taking mutex on file open and release the mutex on file closing.
 *
 * FileSystem configuration is retrieved from peripherals.[ch], these files are typically located in
 *   board folders and are application specific files. These files can be generated by config tool.
 *
 * File System module requires include paths :
 *   - path to middleware\littlefs for lfs.h and lfs_util.h
 *   - paths to components\flash\mflash for mflash_common.h and components\flash\mflash\<device_name> for mflash_drv.h
 *   - path to Flash drivers for mflash_drv.c usage (depends on the device)
 * File System module requires underlayer .c files :
 *   - middleware\littlefs\lfs.c and lfs_util.c
 *    - .\fwk_lfs_mflash.c
 *   - components\flash\mflash\<device_name>\mflash_drv.c
 *   - flash drivers (depends on the device)
 *
 * LFS_THREADSAFE is not required, user shall keep FWK_FSABSTRACTION_THREADSAFE set to 1 instead (default). t
 * This provides a higher level mutex protection at fwk_fsabstraction level instead of lfs level
 */

#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

/**
 * \brief make LittleFS file opening to closing atomic
 *
 * Prevent multiple file opening in LittleFS. Because the same cache buffer is used for all file. Only
 * one file can be opened at a time.
 * When FWK_FSABSTRACTION_THREADSAFE is set, LFS_THREADSAFE can be kept undefined.
 * However, if FWK_FSABSTRACTION_THREADSAFE is not set, APIs below are not re entrant.
 *
 */
#ifndef FWK_FSABSTRACTION_THREADSAFE
#define FWK_FSABSTRACTION_THREADSAFE 1
#endif

/* Flag enabled only for debug purpose: it enables the display of the time to write in flash */
#ifndef FWK_FSABSTRACTION_PROFILING
#define FWK_FSABSTRACTION_PROFILING 0
#endif

/**
 * \brief Initialize and configure the File System using LittleFS
 *
 *   -Get FileSystem Configuration from peripherals.[ch] files (start address, size, etc..)
 *   -Initialize LittleFS
 *   -Mount LittleFS Partition
 *
 * \return int 0 if successfull , 1 if FileSystem is already initialized, negative in case of error
 *         Refer to lfs.h API for error code if negative
 */
int FSA_Init(void);

/**
 * \brief Deinitialize the filesystem, release the allocated ressources
 *
 * The file system is fully deinitialized only when All users have called this
 *
 * \return int 0 if successfull , 1 if FileSystem is still initialized by other requester, negative in case of error
 *         Refer to lfs.h API for error code if negative
 */
int FSA_DeInit(void);

/**
 * \brief Read a buffer from LittleFS file system
 *
 * \param file_name
 * \param buffer
 * \param buf_length
 * \return int  0 if the file does not exist (In this case, the file is created but input buffer is not modified
 *              Number of bytes read if positive
 *              -255 if file system is not mounted
 *              other negative values in case of error: Refer to lfs.h API for error code if negative
 */
int FSA_ReadBufferFromFile(const char *file_name, uint8_t *buffer, uint16_t buf_length);

/**
 * \brief Read a buffer from LittleFS file system at a specific file offset

 * \param file_name
 * \param buffer
 * \param buf_length
 * \param offset , Offset should between 0 and file size
 * \return int  0 if int  0 if the file does not exist (In this case, the file is created but input buffer is not
 modified
 *              Number of bytes read if positive
 *              -255 if file system is not mounted
 *              other negative values in case of error: Refer to lfs.h API for error code if negative
 *  @note If offset value is equal to the file size, no byte will be read and function will return 0
 */
int FSA_ReadBufferFromFileLocation(const char *file_name, uint8_t *buffer, uint16_t buf_length, unsigned int offset);

/**
 * \brief Write a buffer to LittleFS file system
 *
 * This function is protected by a general mutex when @FWK_FSABSTRACTION_THREADSAFE is defined. However, the RAM buffer
 * is not protected so RAM buffer shall not be modified while the buffer is being copied into the File system. For RAM
 * buffer protection, please use API fwk_filesystemExt.h
 *
 * \param file_name
 * \param buffer
 * \param buf_length
 * \return int  0 if successful
 *              Number of bytes written if positive
 *              -255 if file system is not mounted
 *              Other negative values in case of error: Refer to lfs.h API for error code
 */
int FSA_WriteBufferToFile(const char *file_name, const uint8_t *buffer, uint16_t buf_length);

/**
 * \brief Clean/Empty the File content
 *
 *  Delete the file and create an empty file of size 0 byte
 *
 * \param file_name
 * \return int error code from LittleFS
 */
int FSA_DeleteFile(const char *file_name);

/**
 * \brief Return the size of the file
 *
 * \param file_name
 * \return size of the file if exists, otherwise error code from LittleFS
 * \note after call to FSA_DeleteFile(), file size will be 0 byte
 */
int FSA_CheckFileSize(const char *file_name);

/**
 * @brief Force a format of the File System, All files will be wiped out
 *     likely to be used in case of File system failure
 *
 * @return int 0 if successful, Refer to lfs.h API for error code if negative
 */
int FSA_Format(void);

/**
 * @brief FSA initalize underlying flash storage.
 *
 * @return int 0 if successful, Refer to lfs.h API for error code if negative
 */
int FSA_FlashInit(void);

#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif