/*
 * Copyright 2022 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef __FWL_FILESYSTEM_H__
#define __FWL_FILESYSTEM_H__

#include "stdint.h"

/**
 * \brief Provide easy initialization, configuration for LittleFS Filsystem with
 *      possibility of sharing the file system between multiple callers.
 *
 * Provides functions to easy read, write buffers from RAM to filesystem
 *
 * FileSystem configuration is retrieved from peripherals.[ch], these files are typically located in
 *   board folders and can be generated by clock config tool
 *
 */

#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus */

/**
 * \brief local buffer size to be used for writes in file system.
 *
 * When calling \ref FS_WriteBufferToFile(), copy the given buffer to a static local buffer so the data
 *    does not get modified during the write command in the filesystem
 *
 * If this MACRO is set to zero, the copy gets disabled. The user shall make sure the buffer can not be
 *    modified by the write operation
 */
#ifndef FWK_FILESYSTEM_LOCAL_BUF_SIZE
#define FWK_FILESYSTEM_LOCAL_BUF_SIZE 0
#endif

/* Flag enabled only for debug purpose: it enables the display of the time to write in flash */
#ifndef FWK_FILESYSTEM_PROFILING
#define FWK_FILESYSTEM_PROFILING 0
#endif

/**
 * \brief Initialize and configure the File System using LittleFS
 *
 *   -Get FileSystem Configuration from peripherals.[ch] files (start address, size, etc..)
 *   -Initialize LittleFS
 *   -Mount LittleFS Partition
 *
 * \return int 0 if successfull , 1 if FileSystem is already initialized, negative in case of error
 *         Refer to lfs.h API for error code if negative
 */
int FS_Init(void);

/**
 * \brief Deinitialize the filesystem, release the allocated ressources
 *
 * The file system is fully deinitialized only when All users have called this
 *
 * \return int 0 if successfull , 1 if FileSystem is still initialized by other requester, negative in case of error
 *         Refer to lfs.h API for error code if negative
 */
int FS_DeInit(void);

/**
 * \brief Read a buffer from LittleFS file system

 * \param file_name
 * \param buffer
 * \param buf_length
 * \return int  0 if successful
 *              Number of bytes read if positive
 *              -255 if file system is not mounted
 *              other negative values in case of error: Refer to lfs.h API for error code if negative
 *         File is created if it does not exist
 */
int FS_ReadBufferFromFile(const char *file_name, uint8_t *buffer, uint16_t buf_length);

/**
 * \brief Read a buffer from LittleFS file system at a specific file offset

 * \param file_name
 * \param buffer
 * \param buf_length
 * \param offset
 * \return int  0 if successful
 *              Number of bytes read if positive
 *              -255 if file system is not mounted
 *              other negative values in case of error: Refer to lfs.h API for error code if negative
 *         File is created if it does not exist
 *         Offset provided should between 0 and file size
 */
int FS_ReadBufferFromFileLocation(const char *file_name, uint8_t *buffer, uint16_t buf_length, unsigned int offset);

/**
 * \brief Write a buffer to LittleFS file system
 *
 * This function first performs a copy to a local buffer (size given by @ref FWK_FILESYSTEM_LOCAL_BUF_SIZE)
 *      so the data is not modified during the write command
 *
 * \param file_name
 * \param buffer
 * \param buf_length
 * \return int  0 if successful
 *              Number of bytes written if positive
 *              -255 if file system is not mounted
 *              Other negative values in case of error:
 *         File is created if it does not exist
 *         File is deleted if buf_length is 0
 */
int FS_WriteBufferToFile(const char *file_name, const uint8_t *buffer, uint32_t buf_length);

/**
 * \brief Clean/Empty the File content
 *
 *  Delete the file and create an empty file
 *
 * \param file_name
 * \return int error code from LittleFS
 */
int FS_DeleteFile(const char *file_name);

/**
 * \brief Return the size of the file
 *
 * \param file_name
 * \return size of the file if exists, otherwise error code from LittleFS
 */
int FS_CheckFileSize(const char *file_name);

/**
 * \brief Initialize and configure the File System using LittleFS if not already done and returns File System Handle
 *
 * \return pointer to file system handle
 */
void *FS_InitGetHandle(void);

#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif
