/*
 * SHA1-based PRF
 * Copyright (c) 2003-2005, Jouni Malinen <j@w1.fi>
 *
 * This software may be distributed under the terms of the BSD license.
 * See README for more details.
 */

#include "includes.h"

#include "utils/common.h"
#include "sha1.h"
#include "crypto.h"

#ifdef CONFIG_INTERNAL_SHA1

/**
 * sha1_prf - SHA1-based Pseudo-Random Function (PRF) (IEEE 802.11i, 8.5.1.1)
 * @key: Key for PRF
 * @key_len: Length of the key in bytes
 * @label: A unique label for each purpose of the PRF
 * @data: Extra data to bind into the key
 * @data_len: Length of the data
 * @buf: Buffer for the generated pseudo-random key
 * @buf_len: Number of bytes of key to generate
 * Returns: 0 on success, -1 of failure
 *
 * This function is used to derive new, cryptographically separate keys from a
 * given key (e.g., PMK in IEEE 802.11i).
 */
int sha1_prf(const u8 *key, size_t key_len, const char *label, const u8 *data, size_t data_len, u8 *buf, size_t buf_len)
{
    u8 counter = 0;
    size_t pos, plen;
    u8 hash[SHA1_MAC_LEN];
    size_t label_len = os_strlen(label) + 1;
    const unsigned char *addr[3];
    size_t len[3];

    addr[0] = (u8 *)label;
    len[0]  = label_len;
    addr[1] = data;
    len[1]  = data_len;
    addr[2] = &counter;
    len[2]  = 1;

    pos = 0;
    while (pos < buf_len)
    {
        plen = buf_len - pos;
        if (plen >= SHA1_MAC_LEN)
        {
            if (hmac_sha1_vector(key, key_len, 3, addr, len, &buf[pos]))
                return -1;
            pos += SHA1_MAC_LEN;
        }
        else
        {
            if (hmac_sha1_vector(key, key_len, 3, addr, len, hash))
                return -1;
            os_memcpy(&buf[pos], hash, plen);
            break;
        }
        counter++;
    }
    forced_memzero(hash, sizeof(hash));

    return 0;
}
#endif
