/*
 * Copyright 2018 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_flexspi.h"
#include "mflash_drv.h"
#include "pin_mux.h"
#include <stdbool.h>

#define LUT_SEQ_INDEX_READ_ARRAY             0
#define LUT_SEQ_INDEX_BLOCK_ERASE_4K        11
#define LUT_SEQ_INDEX_WRITE_ENABLE           7
#define LUT_SEQ_INDEX_READ_STATUS_REG_BYTE1  1
#define LUT_SEQ_INDEX_WRITE_STAT_CTRL_REG    5
#define LUT_SEQ_INDEX_UNPROTECT_SECTOR       8
#define LUT_SEQ_INDEX_CMD_ENTER_QPI          9
#define LUT_SEQ_INDEX_RETURN_SPI            14
#define LUT_SEQ_INDEX_CHIP_ERASE            13
#define LUT_SEQ_INDEX_BLOCK_ERASE_4K        11
#define LUT_SEQ_INDEX_PAGE_PROG             12

#define QINST_CMD          kFLEXSPI_Command_SDR
#define QINST_DUMMY_DDR    kFLEXSPI_Command_DUMMY_DDR
#define QINST_ADDR_DDR     kFLEXSPI_Command_RADDR_DDR
#define QINST_READ_DDR     kFLEXSPI_Command_READ_DDR
#define QINST_WRITE_DDR    kFLEXSPI_Command_WRITE_DDR

#define EXIP_CMD_WRITE_STATUS_REG_BYTE1  0x01    // Write Status Register Byte 1 
#define EXIP_CMD_PAGE_PROG               0x02    // Byte/Page Program (1 - 256 Bytes)     
#define EXIP_CMD_READ_STATUS_REG_BYTE1   0x05    // Read Status Register Byte 1
#define EXIP_CMD_WRITE_ENABLE            0x06    // Write Enable 
#define EXIP_CMD_READARRAY               0x0B    // Read Array
#define EXIP_CMD_BURST_READ_WRAP         0x0C    // Burst Read with Wrap 
#define EXIP_CMD_BLOCK_ERASE_4K          0x20    // Block Erase (4 Kbytes) 
#define EXIP_CMD_WRITE_STATUS_REG_BYTE2  0x31    // Write Status Register Byte 2
#define EXIP_CMD_PROTECT_SECTOR          0x36    // Protect Sector 
#define EXIP_CMD_UNPROTECT_SECTOR        0x39    // Unprotect Sector 
#define EXIP_CMD_READ_SECT_PROT_REG      0x3C    // Read Sector Protection Registers     
#define EXIP_CMD_CHIP_ERASE              0x60    // Chip Erase    
#define EXIP_CMD_READ_STAT_CTRL_REGS     0x65    // Read Status/Control Registers 
#define EXIP_CMD_WRITE_STAT_CTRL_REGS    0x71    // Write Status/Control Registers 
#define EXIP_CMD_RETURN_SPI              0xFF    // Return to Standard SPI Mode

#define PAD_1  kFLEXSPI_1PAD
#define PAD_4  kFLEXSPI_4PAD
#define PAD_8  kFLEXSPI_8PAD

#define NON_SPI_DUMMY_CYCLES  18

#define MODE_OCTAL_DDR    0x88
#define MODE_OCTAL_SDR    0x08
#define MODE_QUAD_DDR     0x84
#define MODE_QUAD_SDR     0x04

#define CUSTOM_LUT_LENGTH 64
#define FLASH_BUSY_STATUS_POL 1
#define FLASH_BUSY_STATUS_OFFSET 0

#define FLASH_SIZE 0x4000

flexspi_device_config_t deviceconfig = {
    .flexspiRootClk       = 100000000,
    .flashSize            = FLASH_SIZE,
    .CSIntervalUnit       = kFLEXSPI_CsIntervalUnit1SckCycle,
    .CSInterval           = 5,
    .CSHoldTime           = 2,
    .CSSetupTime          = 4,
    .dataValidTime        = 1,
    .columnspace          = 0,
    .enableWordAddress    = 0,
    .AWRSeqIndex          = 0,
    .AWRSeqNumber         = 0,
    .ARDSeqIndex          = LUT_SEQ_INDEX_READ_ARRAY,
    .ARDSeqNumber         = 1,
    .AHBWriteWaitUnit     = kFLEXSPI_AhbWriteWaitUnit2AhbCycle,
    .AHBWriteWaitInterval = 0,
};

const uint32_t customLUT[CUSTOM_LUT_LENGTH] = {
 // OPI DDR mode LUT
 //uint32_t opi_ddr_lut[LUT_SIZE] =

   //Read Array
         [0] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_READARRAY) |
			   (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
         [1] = (QINST_DUMMY_DDR << 10) | (PAD_8 << 8) | (NON_SPI_DUMMY_CYCLES*2+1) |
			   (QINST_READ_DDR << 26) | (PAD_8 << 24) | (128 << 16),
         //[2] = (QINST_JMP_ON_CS << 10) | (PAD_8 << 8) | 0,

	     // Read Status (byte 1)
         [4] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_READ_STATUS_REG_BYTE1) |
			 (QINST_DUMMY_DDR << 26) | (PAD_8 << 24) | (8 << 16),
         [5] = (QINST_READ_DDR << 10) | (PAD_8 << 8) | (1),
                // Write Enable
                [12] = FLEXSPI_LUT_SEQ(QINST_CMD, PAD_1, EXIP_CMD_WRITE_ENABLE, kFLEXSPI_Command_STOP, PAD_1, 0x0),
                // Write Status/Control Registers (this specifc sequence will writes 2 bytes to status/control regs 2-3)
                [56] = FLEXSPI_LUT_SEQ(QINST_CMD, PAD_1, EXIP_CMD_WRITE_STAT_CTRL_REGS, QINST_CMD, PAD_1, 0x02),
                [57] = FLEXSPI_LUT_SEQ(kFLEXSPI_Command_WRITE_SDR, PAD_1, 0x02, kFLEXSPI_Command_STOP, PAD_1, 0x0),

         // Read Status/Control Registers
		 [8] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_READ_STAT_CTRL_REGS) |
		        (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (8 << 16),
         [9] = (QINST_DUMMY_DDR << 10) | (PAD_8 << 8) | (9) |
			    (QINST_READ_DDR << 26) | (PAD_8 << 24) | (2 << 16), // 2 bytes in QPI-DDR to round up to a full clock cycle

         // Write Status Register Byte 2
         [16] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_WRITE_STATUS_REG_BYTE2) |
                (QINST_WRITE_DDR << 26) | (PAD_8 << 24) | (1 << 16),

         // Write Status/Control Registers
         //[20] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_WRITE_STAT_CTRL_REGS) |
		//		(QINST_WRITE_DDR << 26) | (PAD_8 << 24) | (4 << 16),

         // Write Status/Control Registers
         [20] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_WRITE_STAT_CTRL_REGS) |
				(QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (8 << 16),
		 [21] = (QINST_WRITE_DDR << 10) | (PAD_8 << 8) | (1),

         // Burst Read with Wrap
         [24] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_BURST_READ_WRAP) |
			   (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
         [25] = (QINST_DUMMY_DDR << 10) | (PAD_8 << 8) | (45) |
			   (QINST_READ_DDR << 26) | (PAD_8 << 24) | (128 << 16),
         //[26] = (QINST_JMP_ON_CS << 10) | (PAD_8 << 8) | 0,

         // Write Enable     
         [28] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_WRITE_ENABLE),


         // Unprotect Sector
         [32] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_UNPROTECT_SECTOR) |
                (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),


         // Protect Sector
         [36] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_PROTECT_SECTOR) |
                (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
	 
         // Read Sector Protection Registers
		 [40] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_READ_SECT_PROT_REG) |
				(QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
		 [41] = (QINST_DUMMY_DDR << 10) | (PAD_8 << 8) | (9) |
				(QINST_READ_DDR << 26) | (PAD_8 << 24) | (1 << 16),

         // Block Erase 4K
		 [44] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_BLOCK_ERASE_4K) |
				(QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),

         // Page Program
         [48] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_PAGE_PROG) |
				(QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
         [49] = (QINST_WRITE_DDR << 10) | (PAD_8 << 8) | (128), 

         // Chip Erase 
		 [52] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_CHIP_ERASE),
};

static status_t flexspi_nor_wait_bus_busy(FLEXSPI_Type *base)
{
    /* Wait status ready. */
    bool isBusy;
    uint32_t readValue;
    status_t status;
    flexspi_transfer_t flashXfer;

    flashXfer.deviceAddress = 0;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Read;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_READ_STATUS_REG_BYTE1;
    flashXfer.data          = &readValue;
    flashXfer.dataSize      = 1;

    do
    {
        status = FLEXSPI_TransferBlocking(base, &flashXfer);

        if (status != kStatus_Success)
        {
            return status;
        }
        if (FLASH_BUSY_STATUS_POL)
        {
            if (readValue & (1U << FLASH_BUSY_STATUS_OFFSET))
            {
                isBusy = true;
            }
            else
            {
                isBusy = false;
            }
        }
        else
        {
            if (readValue & (1U << FLASH_BUSY_STATUS_OFFSET))
            {
                isBusy = false;
            }
            else
            {
                isBusy = true;
            }
        }

    } while (isBusy);

    return status;
}

static status_t flexspi_nor_write_enable(FLEXSPI_Type *base, uint32_t baseAddr)
{
    flexspi_transfer_t flashXfer;
    status_t status;

    /* Write enable */
    flashXfer.deviceAddress = 0;//baseAddr;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Command;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_WRITE_ENABLE;

    status = FLEXSPI_TransferBlocking(base, &flashXfer);

    return status;
}

static status_t flexspi_nor_flash_sector_erase(FLEXSPI_Type *base, uint32_t address)
{
    status_t status;
    flexspi_transfer_t flashXfer;

    /* Write enable */
    status = flexspi_nor_write_enable(base, 0);

    if (status != kStatus_Success)
    {
        return status;
    }

    flashXfer.deviceAddress = address;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Command;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_BLOCK_ERASE_4K;
    status                  = FLEXSPI_TransferBlocking(base, &flashXfer);

    if (status != kStatus_Success)
    {
        return status;
    }

    status = flexspi_nor_wait_bus_busy(base);

    return status;
}

status_t flexspi_nor_enable_quad_mode(FLEXSPI_Type *base)
{
    flexspi_transfer_t flashXfer;
    status_t status;
    uint32_t writeValue = MODE_OCTAL_DDR;

    /* Write enable */
    status = flexspi_nor_write_enable(base, 0);

    if (status != kStatus_Success)
    {
        return status;
    }

    /* Enable quad mode. */
    flashXfer.deviceAddress = 2;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Write;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_WRITE_STAT_CTRL_REG;
    flashXfer.data          = &writeValue;
    flashXfer.dataSize      = 1;

    status = FLEXSPI_TransferBlocking(base, &flashXfer);
    if (status != kStatus_Success)
    {
        return status;
    }

    status = flexspi_nor_wait_bus_busy(base);

    return status;
}

status_t flexspi_nor_unprotect_all(FLEXSPI_Type *base)
{
    flexspi_transfer_t flashXfer;
    status_t status;
    uint32_t writeValue = 0;

    /* Write enable */
    status = flexspi_nor_write_enable(base, 0);

    if (status != kStatus_Success)
    {
        return status;
    }

    /* write 0 to status/control register 1 - this will unprotect all sectors. */
    flashXfer.deviceAddress = 1;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Write;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_WRITE_STAT_CTRL_REG;
    flashXfer.data          = &writeValue;
    flashXfer.dataSize      = 1;

    status = FLEXSPI_TransferBlocking(base, &flashXfer);
    if (status != kStatus_Success)
    {
        return status;
    }

    status = flexspi_nor_wait_bus_busy(base);

    return status;
}

static status_t flexspi_nor_flash_page_program(FLEXSPI_Type *base, uint32_t address, const uint32_t *src)
{
    status_t status;
    flexspi_transfer_t flashXfer;

    /* Write enable */
    status = flexspi_nor_write_enable(base, address);

    if (status != kStatus_Success)
    {
        return status;
    }

    /* Prepare page program command */
    flashXfer.deviceAddress = address;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Write;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_PAGE_PROG;
    flashXfer.data          = (uint32_t *)src;
    flashXfer.dataSize      = MFLASH_PAGE_SIZE;
    status                  = FLEXSPI_TransferBlocking(base, &flashXfer);

    if (status != kStatus_Success)
    {
        return status;
    }

    status = flexspi_nor_wait_bus_busy(base);

    return status;
}

static status_t flexspi_nor_read_data(FLEXSPI_Type *base, uint32_t startAddress, uint32_t *buffer, uint32_t length)
{
    status_t status;
    flexspi_transfer_t flashXfer;
    uint32_t readAddress = startAddress;

    /* Read page. */
    flashXfer.deviceAddress = readAddress;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Read;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_READ_ARRAY;
    flashXfer.data          = buffer;
    flashXfer.dataSize      = length;

    status = FLEXSPI_TransferBlocking(base, &flashXfer);

    return status;
}

/* Internal - erase single sector */
static int32_t mflash_drv_sector_erase_internal(uint32_t sector_addr)
{
    __asm("cpsid i");

    status_t status;
    status = flexspi_nor_flash_sector_erase(MFLASH_FLEXSPI, sector_addr);

    /* Do software reset. */
    FLEXSPI_SoftwareReset(MFLASH_FLEXSPI);

    __asm("cpsie i");
    /* Flush pipeline to allow pending interrupts take place
     * before starting next loop */
    __ISB();

    if (status != kStatus_Success)
    {
        return -1;
    }

    return 0;
}

/* Calling wrapper for 'mflash_drv_sector_erase_internal'.
 * Erase one sector starting at 'sector_addr' - must be sector aligned.
 */
int32_t mflash_drv_sector_erase(uint32_t sector_addr)
{
    if (false == mflash_drv_is_sector_aligned(sector_addr))
        return -1;

    volatile int32_t result;
    result = mflash_drv_sector_erase_internal(sector_addr);
    return result;
}

/* Internal - write single page */
static int32_t mflash_drv_page_program_internal(uint32_t page_addr, uint32_t *data)
{
    __asm("cpsid i");

    status_t status;
    status = flexspi_nor_flash_page_program(MFLASH_FLEXSPI, page_addr, data);

    /* Do software reset. */
    FLEXSPI_SoftwareReset(MFLASH_FLEXSPI);

    __asm("cpsie i");
    /* Flush pipeline to allow pending interrupts take place
     * before starting next loop */
    __ISB();

    if (status != kStatus_Success)
    {
        return -1;
    }

    return 0;
}

/* Calling wrapper for 'mflash_drv_page_program_internal'.
 * Write 'data' to 'page_addr' - must be page aligned.
 * NOTE: Don't try to store constant data that are located in XIP !!
 */
int32_t mflash_drv_page_program(uint32_t page_addr, uint32_t *data)
{
    if (false == mflash_drv_is_page_aligned(page_addr))
        return -1;

    volatile int32_t result;
    result = mflash_drv_page_program_internal(page_addr, data);
    return result;
}

/* Internal - read data */
static int32_t mflash_drv_read_internal(uint32_t addr, uint32_t *buffer, uint32_t length)
{
    __asm("cpsid i");

    status_t status;
    status = flexspi_nor_read_data(MFLASH_FLEXSPI, addr, buffer, length);

    /* Do software reset. */
    FLEXSPI_SoftwareReset(MFLASH_FLEXSPI);

    __asm("cpsie i");
    /* Flush pipeline to allow pending interrupts take place
     * before starting next loop */
    __ISB();

    if (status != kStatus_Success)
    {
        return -1;
    }

    return 0;
}

/* Calling wrapper for 'mflash_drv_read_internal'. */
int32_t mflash_drv_read(uint32_t addr, uint32_t *buffer, uint32_t length)
{
    /* Check alignment */
    if (((uint32_t)buffer % 4) || (length % 4))
        return -1;

    volatile int32_t result;
    result = mflash_drv_read_internal(addr, buffer, length);
    return result;
}

/* Initialize flash peripheral,
 * cannot be invoked directly, requires calling wrapper in non XIP memory */
static int32_t mflash_drv_init_internal(void)
{
    /* NOTE: Multithread access is not supported for SRAM target.
     *       XIP target MUST be protected by disabling global interrupts
     *       since all ISR (and API that is used inside) is placed at XIP.
     *       It is necessary to place at least "mflash_drv.o", "fsl_flexspi.o" to SRAM */
    /* disable interrupts when running from XIP
     * TODO: store/restore previous PRIMASK on stack to avoid
     * failure in case of nested critical sections !! */
    __asm("cpsid i");

    flexspi_config_t config;
    status_t status;

    /* Get FLEXSPI default settings and configure the flexspi. */
    FLEXSPI_GetDefaultConfig(&config);

    /* Set AHB buffer size for reading data through AHB bus. */
    config.ahbConfig.enableAHBPrefetch   = true;
    config.ahbConfig.enableAHBBufferable = true;
    config.ahbConfig.enableAHBCachable   = true;
    config.rxSampleClock                 = kFLEXSPI_ReadSampleClkExternalInputFromDqsPad;//kFLEXSPI_ReadSampleClkLoopbackFromDqsPad;
	// Need to set the combination-enable option. This options combines 8 data lines
	// from FlexSPI channel A with 4 data lines from FlexSPI channel B to form an
	// 8-line bus for octal. On this SoC this is the only way to enable octal.
	config.enableCombination = true;    
    FLEXSPI_Init(MFLASH_FLEXSPI, &config);

    /* AHB Read Address option bit. This option bit is intend to remove AHB burst start address alignment limitation */
    MFLASH_FLEXSPI->AHBCR |= FLEXSPI_AHBCR_READADDROPT_MASK;

    /* Configure flash settings according to serial flash feature. */
    FLEXSPI_SetFlashConfig(MFLASH_FLEXSPI, &deviceconfig, kFLEXSPI_PortA1);

    /* Update LUT table. */
    FLEXSPI_UpdateLUT(MFLASH_FLEXSPI, 0, customLUT, CUSTOM_LUT_LENGTH);

    /* Enter quad mode. */
    status = flexspi_nor_enable_quad_mode(MFLASH_FLEXSPI);
    if (status != kStatus_Success)
    {
        return -1;
    }

    /* Unprotect all sectors in preparation of future writes. */
    status = flexspi_nor_unprotect_all(MFLASH_FLEXSPI);
    if (status != kStatus_Success)
    {
        return -1;
    }

    __asm("cpsie i");

    return 0;
}

/* API - initialize 'mflash' */
int32_t mflash_drv_init(void)
{
    volatile int32_t result;
    /* Necessary to have double wrapper call in non_xip memory */
    result = mflash_drv_init_internal();

    return result;
}

/* Temporary sector shadow buffer. Use uint32_t type to force 4B alignment and
 * improve copy operation */
static uint32_t g_flashm_sector[MFLASH_SECTOR_SIZE / sizeof(uint32_t)];

/* Internal - write data of 'data_len' to single sector 'sector_addr', starting from 'sect_off' */
static int32_t mflash_drv_sector_update(uint32_t sector_addr, uint32_t sect_off, const uint8_t *data, uint32_t data_len)
{
    int sector_erase_req      = 0;
    uint32_t page_program_map = 0; /* Current implementation is limited to 32 pages per sector */

    /* Address not aligned to sector boundary */
    if (false == mflash_drv_is_sector_aligned(sector_addr))
        return -1;
    /* Offset + length exceeed sector size */
    if (sect_off + data_len > MFLASH_SECTOR_SIZE)
        return -1;

    if (0 != mflash_drv_read(sector_addr, &g_flashm_sector[0], sizeof(g_flashm_sector)))
    {
        return -2;
    }

    /* Diff the data to determine pages to be programed */
    for (uint32_t i = 0; i < data_len; i++)
    {
        uint8_t cur_value = ((uint8_t *)(g_flashm_sector))[sect_off + i];
        uint8_t new_value = data[i];

        if ((cur_value | new_value) != cur_value)
        {
            /* A bit needs to be flipped from 0 to 1, the whole sector has to be erased */
            sector_erase_req = 1;
            break;
        }

        if (cur_value != new_value)
        {
            /* There is a change, the page has to be programmed for sure */
            page_program_map |= 1 << ((sect_off + i) / MFLASH_PAGE_SIZE);
        }
    }

#if !defined(MFLASH_INC_WRITES) || !MFLASH_INC_WRITES
    /* Perform blank check page by page until decission for sector erase is made or we reach last page of the sector */
    for (int page_idx = 0; (0 == sector_erase_req) && page_idx < MFLASH_SECTOR_SIZE / MFLASH_PAGE_SIZE; page_idx++)
    {
        /* Check only pages which need to be programed */
        if (page_program_map & (1 << page_idx))
        {
            int page_word_start = page_idx * (MFLASH_PAGE_SIZE / sizeof(g_flashm_sector[0]));
            int page_word_end   = page_word_start + (MFLASH_PAGE_SIZE / sizeof(g_flashm_sector[0]));

            for (int i = page_word_start; i < page_word_end; i++)
            {
                if (g_flashm_sector[i] != 0xFFFFFFFF)
                {
                    /* Mark sector to be erased and quit */
                    sector_erase_req = 1;
                    break;
                }
            }
        }
    }
#endif

    /* Copy data to be programmed byte by byte to shadow buffer at proper position */
    for (uint32_t i = 0; i < data_len; i++)
    {
        ((uint8_t *)g_flashm_sector)[sect_off + i] = data[i];
    }

    /* If sector is to be erased, update page program map according to non-blank areas in the shadow buffer */
    if (0 != sector_erase_req)
    {
        for (int page_idx = 0; page_idx < MFLASH_SECTOR_SIZE / MFLASH_PAGE_SIZE; page_idx++)
        {
            int page_word_start = page_idx * (MFLASH_PAGE_SIZE / sizeof(g_flashm_sector[0]));
            int page_word_end   = page_word_start + (MFLASH_PAGE_SIZE / sizeof(g_flashm_sector[0]));

            for (int i = page_word_start; i < page_word_end; i++)
            {
                if (g_flashm_sector[i] != 0xFFFFFFFF)
                {
                    /* Mark the page for programming and go for next one */
                    page_program_map |= (1 << page_idx);
                    break;
                }
            }
        }
    }

    /* Erase the sector if required */
    if (0 != sector_erase_req)
    {
        if (0 != mflash_drv_sector_erase(sector_addr))
        {
            return -2;
        }
    }

    /* Program the required pages */
    for (int page_idx = 0; page_idx < MFLASH_SECTOR_SIZE / MFLASH_PAGE_SIZE; page_idx++)
    {
        if (page_program_map & (1 << page_idx))
        {
            if (0 !=
                mflash_drv_page_program(sector_addr + page_idx * MFLASH_PAGE_SIZE,
                                        g_flashm_sector + page_idx * (MFLASH_PAGE_SIZE / sizeof(g_flashm_sector[0]))))
            {
                return -3;
            }
        }
    }

    // mflash_drv_read_mode();
    return 0;
}

/* Write data to flash, cannot be invoked directly, requires calling wrapper in non XIP memory */
int32_t mflash_drv_write_internal(uint32_t addr, const uint8_t *data, uint32_t data_len)
{
    /* Interval <0, sector_size) */
    uint32_t to_write = 0;
    /* Interval (data_len, 0>  */
    uint32_t to_remain = data_len;
    /* Physical address in external FLASH device */

    int32_t result = 0;

    for (
        /* Calculate address of first sector */
        uint32_t sect_a = (addr / MFLASH_SECTOR_SIZE) * MFLASH_SECTOR_SIZE,
                 /* and first sector offset */
        sect_of = addr % MFLASH_SECTOR_SIZE,
                 /* and set first data offset to 0*/
        data_of = 0;
        /* Continue until sector address exceed target adddress + data_length */
        sect_a < addr + data_len;
        /* Move to next sector */
        sect_a += MFLASH_SECTOR_SIZE,
                 /* and move pointer to data */
        data_of += to_write)
    {
        /* If remaining data is exceed 'sector_size', write 'sector_size' length */
        if (to_remain > MFLASH_SECTOR_SIZE - sect_of)
        {
            to_write  = MFLASH_SECTOR_SIZE - sect_of;
            to_remain = to_remain - to_write;
        }
        /* else write remaining data length */
        else
        {
            to_write  = to_remain;
            to_remain = 0;
        }

        /* Write at 'sect_a' sector, starting at 'sect_of' using '&data[data_of]' of length 'to_write' */
        result = mflash_drv_sector_update(sect_a, sect_of, data + data_of, to_write);
        if (0 != result)
            return -1;
        /* Only first sector is allowed to have an offset */
        sect_of = 0;
    }

    return 0;
}

/* Calling wrapper for 'mflash_drv_write_internal'.
 * Write 'data' of 'data_len' to 'any_addr' - which doesn't have to be sector aligned.
 * NOTE: Don't try to store constant data that are located in XIP !!
 */
int32_t mflash_drv_write(uint32_t addr, const uint8_t *data, uint32_t data_len)
{
    volatile int32_t result;
    result = mflash_drv_write_internal(addr, data, data_len);
    return result;
}

/* Returns pointer (AHB address) where requested physical address range of FLASH is available or NULL in case of failure
 */
void *mflash_drv_mmap(uint32_t addr, uint32_t len)
{
    /* take FLEXSPI remapping into account */
    uint32_t remap_offset = IOMUXC_GPR->GPR32 & 0xFFFFF000;
    uint32_t remap_start  = IOMUXC_GPR->GPR30 & 0xFFFFF000;
    uint32_t remap_end    = IOMUXC_GPR->GPR31 & 0xFFFFF000;

    /* calculate the bus address where the requested FLASH region is expected to be available */
    uint32_t bus_addr = addr + FlexSPI_AMBA_BASE;

    if (remap_offset == 0 || (remap_end <= remap_start))
    {
        /* remapping is not active */
        return (void *)bus_addr;
    }

    if ((remap_start >= bus_addr + len) || (remap_end <= bus_addr))
    {
        /* remapping window does not span bus addresses normally assigned for requested range of FLASH */
        return (void *)bus_addr;
    }

    if ((remap_start + remap_offset <= bus_addr) && (remap_end + remap_offset >= bus_addr + len))
    {
        /* remapping window coveres the whole requested range of FLASH, return address adjusted by negative offset */
        return (void *)(bus_addr - remap_offset);
    }

    /* the bus address region normally assigned for requested range of FLASH is partially shadowed by remapping, fail */
    return NULL;
}
