/*
 * Copyright 2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*  Standard C Included Files */
#include <stdio.h>
#include <string.h>
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "fsl_debug_console.h"
#include "fsl_lpi2c.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define EXAMPLE_I2C_MASTER_BASE (LPI2C1_BASE)

/* Select USB1 PLL (480 MHz) as master lpi2c clock source */
#define LPI2C_CLOCK_SOURCE_SELECT (0U)
/* Clock divider for master lpi2c clock source */
#define LPI2C_CLOCK_SOURCE_DIVIDER (5U)
/* Get frequency of lpi2c clock */
#define LPI2C_CLOCK_FREQUENCY ((CLOCK_GetFreq(kCLOCK_Usb1PllClk) / 8) / (LPI2C_CLOCK_SOURCE_DIVIDER + 1U))

#define LPI2C_MASTER_CLOCK_FREQUENCY LPI2C_CLOCK_FREQUENCY


#define EXAMPLE_I2C_MASTER ((LPI2C_Type *)EXAMPLE_I2C_MASTER_BASE)

#define I2C_BAUDRATE               100000U


/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
 * Code
 ******************************************************************************/

static bool probe_addr(LPI2C_Type* bus, uint8_t addr7bit, bool verbose)
{
    uint8_t data = 0;
    status_t res;
    res = LPI2C_MasterStart(EXAMPLE_I2C_MASTER, addr7bit, kLPI2C_Write);
    if (res == kStatus_Success) {
        res = LPI2C_MasterSend(EXAMPLE_I2C_MASTER, &data, 1);
        if (res == kStatus_Success) {
            res = LPI2C_MasterStop(EXAMPLE_I2C_MASTER);
            if (res == kStatus_Success) {
                if (verbose) {
                    PRINTF("0x%x: xx : Found I2C device on 7-bit adress 0x%x\r\n", addr7bit, addr7bit);
                }
                return true;
            } else if (verbose) {
                PRINTF("0x%x: -- : Master Stop FAIL, error=%d\r\n", addr7bit, res);
            }
        } else if (verbose) {
            PRINTF("0x%x: -- : Master Send FAIL, error=%d\r\n", addr7bit, res);
        }
    } else if (verbose) {
        PRINTF("0x%x: -- : Master Started FAIL, error=%d\r\n", addr7bit, res);
    }
    return false;
}

static void probe_bus(LPI2C_Type* bus)
{
    uint8_t start = 0;

    PRINTF("     0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f\r\n");
    for (uint8_t offset = 0; offset < 0x80; offset += 0x10) {
        if (offset == 0) {
            PRINTF("00:         ");
            start = 3;
        } else {
            PRINTF("%x:", offset);
            start = 0;
        }
        for (uint8_t x = start; x <= 0xf; x++) {
            if (probe_addr(bus, offset+x, false)) {
                PRINTF(" XX");
            } else {
                PRINTF(" --");
            }
        }
        PRINTF("\r\n");
    }
}


/*!
 * @brief Main function
 */
int main(void)
{
    BOARD_ConfigMPU();
    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();
    SystemCoreClockUpdate();

    /*Clock setting for LPI2C*/
    CLOCK_SetMux(kCLOCK_Lpi2cMux, LPI2C_CLOCK_SOURCE_SELECT);
    CLOCK_SetDiv(kCLOCK_Lpi2cDiv, LPI2C_CLOCK_SOURCE_DIVIDER);

    PRINTF("\r\nLPI2C probe example.\r\n");

    lpi2c_master_config_t masterConfig;

    /*
     * masterConfig.debugEnable = false;
     * masterConfig.ignoreAck = false;
     * masterConfig.pinConfig = kLPI2C_2PinOpenDrain;
     * masterConfig.baudRate_Hz = 100000U;
     * masterConfig.busIdleTimeout_ns = 0;
     * masterConfig.pinLowTimeout_ns = 0;
     * masterConfig.sdaGlitchFilterWidth_ns = 0;
     * masterConfig.sclGlitchFilterWidth_ns = 0;
     */
    LPI2C_MasterGetDefaultConfig(&masterConfig);

    /* Change the default baudrate configuration */
    masterConfig.baudRate_Hz = I2C_BAUDRATE;

    /* Initialize the LPI2C master peripheral */
    LPI2C_MasterInit(EXAMPLE_I2C_MASTER, &masterConfig, LPI2C_MASTER_CLOCK_FREQUENCY);

    PRINTF("Probing I2C1\r\n");
    probe_bus(EXAMPLE_I2C_MASTER);

    PRINTF("DONE\r\n");
    while (1)
    {
    }
}
