/*
 * Copyright 2018-2021 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdbool.h>

#include "mflash_drv.h"

#include "fsl_flexspi.h"
#include "fsl_cache.h"
#include "pin_mux.h"

#define LUT_SEQ_INDEX_READ_ARRAY             0
#define LUT_SEQ_INDEX_BLOCK_ERASE_4K        11
#define LUT_SEQ_INDEX_WRITE_ENABLE           7
#define LUT_SEQ_INDEX_READ_STATUS_REG_BYTE1  1
#define LUT_SEQ_INDEX_WRITE_STAT_CTRL_REG    5
#define LUT_SEQ_INDEX_UNPROTECT_SECTOR       8
#define LUT_SEQ_INDEX_CMD_ENTER_QPI          9
#define LUT_SEQ_INDEX_RETURN_SPI            14
#define LUT_SEQ_INDEX_CHIP_ERASE            13
#define LUT_SEQ_INDEX_BLOCK_ERASE_4K        11
#define LUT_SEQ_INDEX_PAGE_PROG             12

#define QINST_CMD          kFLEXSPI_Command_SDR
#define QINST_DUMMY_DDR    kFLEXSPI_Command_DUMMY_DDR
#define QINST_ADDR_DDR     kFLEXSPI_Command_RADDR_DDR
#define QINST_READ_DDR     kFLEXSPI_Command_READ_DDR
#define QINST_WRITE_DDR    kFLEXSPI_Command_WRITE_DDR

#define EXIP_CMD_WRITE_STATUS_REG_BYTE1  0x01    // Write Status Register Byte 1 
#define EXIP_CMD_PAGE_PROG               0x02    // Byte/Page Program (1 - 256 Bytes)     
#define EXIP_CMD_READ_STATUS_REG_BYTE1   0x05    // Read Status Register Byte 1
#define EXIP_CMD_WRITE_ENABLE            0x06    // Write Enable 
#define EXIP_CMD_READARRAY               0x0B    // Read Array
#define EXIP_CMD_BURST_READ_WRAP         0x0C    // Burst Read with Wrap 
#define EXIP_CMD_BLOCK_ERASE_4K          0x20    // Block Erase (4 Kbytes) 
#define EXIP_CMD_WRITE_STATUS_REG_BYTE2  0x31    // Write Status Register Byte 2
#define EXIP_CMD_PROTECT_SECTOR          0x36    // Protect Sector 
#define EXIP_CMD_UNPROTECT_SECTOR        0x39    // Unprotect Sector 
#define EXIP_CMD_READ_SECT_PROT_REG      0x3C    // Read Sector Protection Registers     
#define EXIP_CMD_CHIP_ERASE              0x60    // Chip Erase    
#define EXIP_CMD_READ_STAT_CTRL_REGS     0x65    // Read Status/Control Registers 
#define EXIP_CMD_WRITE_STAT_CTRL_REGS    0x71    // Write Status/Control Registers 
#define EXIP_CMD_RETURN_SPI              0xFF    // Return to Standard SPI Mode

#define PAD_1  kFLEXSPI_1PAD
#define PAD_4  kFLEXSPI_4PAD
#define PAD_8  kFLEXSPI_8PAD

#define NON_SPI_DUMMY_CYCLES  18

#define MODE_OCTAL_DDR    0x88
#define MODE_OCTAL_SDR    0x08
#define MODE_QUAD_DDR     0x84
#define MODE_QUAD_SDR     0x04

#define CUSTOM_LUT_LENGTH        64
#define FLASH_BUSY_STATUS_POL    1
#define FLASH_BUSY_STATUS_OFFSET 0

#define FLASH_SIZE 0x4000

#ifndef XIP_EXTERNAL_FLASH
flexspi_device_config_t deviceconfig = {
    .flexspiRootClk       = 100000000,
    .flashSize            = FLASH_SIZE,
    .CSIntervalUnit       = kFLEXSPI_CsIntervalUnit1SckCycle,
    .CSInterval           = 5,
    .CSHoldTime           = 2,
    .CSSetupTime          = 4,
    .dataValidTime        = 1,
    .columnspace          = 0,
    .enableWordAddress    = 0,
    .AWRSeqIndex          = 0,
    .AWRSeqNumber         = 0,
    .ARDSeqIndex          = LUT_SEQ_INDEX_READ_ARRAY,
    .ARDSeqNumber         = 1,
    .AHBWriteWaitUnit     = kFLEXSPI_AhbWriteWaitUnit2AhbCycle,
    .AHBWriteWaitInterval = 0,
};
#endif

static uint32_t customLUT[CUSTOM_LUT_LENGTH] = {
 // OPI DDR mode LUT
 //uint32_t opi_ddr_lut[LUT_SIZE] =

   //Read Array
         [0] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_READARRAY) |
			   (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
         [1] = (QINST_DUMMY_DDR << 10) | (PAD_8 << 8) | (NON_SPI_DUMMY_CYCLES*2+1) |
			   (QINST_READ_DDR << 26) | (PAD_8 << 24) | (128 << 16),
         //[2] = (QINST_JMP_ON_CS << 10) | (PAD_8 << 8) | 0,

	     // Read Status (byte 1)
         [4] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_READ_STATUS_REG_BYTE1) |
			 (QINST_DUMMY_DDR << 26) | (PAD_8 << 24) | (8 << 16),
         [5] = (QINST_READ_DDR << 10) | (PAD_8 << 8) | (1),
                // Write Enable
                [12] = FLEXSPI_LUT_SEQ(QINST_CMD, PAD_1, EXIP_CMD_WRITE_ENABLE, kFLEXSPI_Command_STOP, PAD_1, 0x0),
                // Write Status/Control Registers (this specifc sequence will writes 2 bytes to status/control regs 2-3)
                [56] = FLEXSPI_LUT_SEQ(QINST_CMD, PAD_1, EXIP_CMD_WRITE_STAT_CTRL_REGS, QINST_CMD, PAD_1, 0x02),
                [57] = FLEXSPI_LUT_SEQ(kFLEXSPI_Command_WRITE_SDR, PAD_1, 0x02, kFLEXSPI_Command_STOP, PAD_1, 0x0),

         // Read Status/Control Registers
		 [8] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_READ_STAT_CTRL_REGS) |
		        (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (8 << 16),
         [9] = (QINST_DUMMY_DDR << 10) | (PAD_8 << 8) | (9) |
			    (QINST_READ_DDR << 26) | (PAD_8 << 24) | (2 << 16), // 2 bytes in QPI-DDR to round up to a full clock cycle

         // Write Status Register Byte 2
         [16] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_WRITE_STATUS_REG_BYTE2) |
                (QINST_WRITE_DDR << 26) | (PAD_8 << 24) | (1 << 16),

         // Write Status/Control Registers
         //[20] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_WRITE_STAT_CTRL_REGS) |
		//		(QINST_WRITE_DDR << 26) | (PAD_8 << 24) | (4 << 16),

         // Write Status/Control Registers
         [20] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_WRITE_STAT_CTRL_REGS) |
				(QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (8 << 16),
		 [21] = (QINST_WRITE_DDR << 10) | (PAD_8 << 8) | (1),

         // Burst Read with Wrap
         [24] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_BURST_READ_WRAP) |
			   (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
         [25] = (QINST_DUMMY_DDR << 10) | (PAD_8 << 8) | (45) |
			   (QINST_READ_DDR << 26) | (PAD_8 << 24) | (128 << 16),
         //[26] = (QINST_JMP_ON_CS << 10) | (PAD_8 << 8) | 0,

         // Write Enable     
         [28] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_WRITE_ENABLE),


         // Unprotect Sector
         [32] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_UNPROTECT_SECTOR) |
                (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),


         // Protect Sector
         [36] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_PROTECT_SECTOR) |
                (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
	 
         // Read Sector Protection Registers
		 [40] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_READ_SECT_PROT_REG) |
				(QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
		 [41] = (QINST_DUMMY_DDR << 10) | (PAD_8 << 8) | (9) |
				(QINST_READ_DDR << 26) | (PAD_8 << 24) | (1 << 16),

         // Block Erase 4K
		 [44] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_BLOCK_ERASE_4K) |
				(QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),

         // Page Program
         [48] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_PAGE_PROG) |
				(QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
         [49] = (QINST_WRITE_DDR << 10) | (PAD_8 << 8) | (128), 

         // Chip Erase 
		 [52] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_CHIP_ERASE),
};

static status_t flexspi_nor_wait_bus_busy(FLEXSPI_Type *base)
{
    /* Wait status ready. */
    bool isBusy;
    uint32_t readValue;
    status_t status;
    flexspi_transfer_t flashXfer;

    flashXfer.deviceAddress = 0;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Read;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_READ_STATUS_REG_BYTE1;
    flashXfer.data          = &readValue;
    flashXfer.dataSize      = 1;

    do
    {
        status = FLEXSPI_TransferBlocking(base, &flashXfer);

        if (status != kStatus_Success)
        {
            return status;
        }
        if (FLASH_BUSY_STATUS_POL)
        {
            if (readValue & (1U << FLASH_BUSY_STATUS_OFFSET))
            {
                isBusy = true;
            }
            else
            {
                isBusy = false;
            }
        }
        else
        {
            if (readValue & (1U << FLASH_BUSY_STATUS_OFFSET))
            {
                isBusy = false;
            }
            else
            {
                isBusy = true;
            }
        }

    } while (isBusy);

    return status;
}

static status_t flexspi_nor_write_enable(FLEXSPI_Type *base, uint32_t address)
{
    flexspi_transfer_t flashXfer;
    status_t status;

    /* Write enable */
    flashXfer.deviceAddress = 0;//address;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Command;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_WRITE_ENABLE;

    status = FLEXSPI_TransferBlocking(base, &flashXfer);

    return status;
}

status_t flexspi_nor_enable_quad_mode(FLEXSPI_Type *base)
{
    flexspi_transfer_t flashXfer;
    status_t status;
    uint32_t writeValue = MODE_OCTAL_DDR;

    /* Write enable */
    status = flexspi_nor_write_enable(base, 0);

    if (status != kStatus_Success)
    {
        return status;
    }

    /* Enable quad mode. */
    flashXfer.deviceAddress = 2;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Write;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_WRITE_STAT_CTRL_REG;
    flashXfer.data          = &writeValue;
    flashXfer.dataSize      = 1;

    status = FLEXSPI_TransferBlocking(base, &flashXfer);
    if (status != kStatus_Success)
    {
        return status;
    }

    status = flexspi_nor_wait_bus_busy(base);

    return status;
}

static status_t flexspi_nor_flash_sector_erase(FLEXSPI_Type *base, uint32_t address)
{
    status_t status;
    flexspi_transfer_t flashXfer;

    /* Write enable */
    status = flexspi_nor_write_enable(base, 0);

    if (status != kStatus_Success)
    {
        return status;
    }

    flashXfer.deviceAddress = address;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Command;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_BLOCK_ERASE_4K;
    status                  = FLEXSPI_TransferBlocking(base, &flashXfer);

    if (status != kStatus_Success)
    {
        return status;
    }

    status = flexspi_nor_wait_bus_busy(base);

    return status;
}

status_t flexspi_nor_unprotect_all(FLEXSPI_Type *base)
{
    flexspi_transfer_t flashXfer;
    status_t status;
    uint32_t writeValue = 0;

    /* Write enable */
    status = flexspi_nor_write_enable(base, 0);

    if (status != kStatus_Success)
    {
        return status;
    }

    /* write 0 to status/control register 1 - this will unprotect all sectors. */
    flashXfer.deviceAddress = 1;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Write;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_WRITE_STAT_CTRL_REG;
    flashXfer.data          = &writeValue;
    flashXfer.dataSize      = 1;

    status = FLEXSPI_TransferBlocking(base, &flashXfer);
    if (status != kStatus_Success)
    {
        return status;
    }

    status = flexspi_nor_wait_bus_busy(base);

    return status;
}

static status_t flexspi_nor_flash_page_program(FLEXSPI_Type *base, uint32_t address, const uint32_t *src)
{
    status_t status;
    flexspi_transfer_t flashXfer;

    /* Write enable */
    status = flexspi_nor_write_enable(base, address);

    if (status != kStatus_Success)
    {
        return status;
    }

    /* Prepare page program command */
    flashXfer.deviceAddress = address;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Write;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_PAGE_PROG;
    flashXfer.data          = (uint32_t *)src;
    flashXfer.dataSize      = MFLASH_PAGE_SIZE;
    status                  = FLEXSPI_TransferBlocking(base, &flashXfer);

    if (status != kStatus_Success)
    {
        return status;
    }

    status = flexspi_nor_wait_bus_busy(base);

    return status;
}

static status_t flexspi_nor_read_data(FLEXSPI_Type *base, uint32_t startAddress, uint32_t *buffer, uint32_t length)
{
    status_t status;
    flexspi_transfer_t flashXfer;
    uint32_t readAddress = startAddress;

    /* Read page. */
    flashXfer.deviceAddress = readAddress;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Read;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_READ_ARRAY;
    flashXfer.data          = buffer;
    flashXfer.dataSize      = length;

    status = FLEXSPI_TransferBlocking(base, &flashXfer);

    return status;
}

/* Initialize flash peripheral,
 * cannot be invoked directly, requires calling wrapper in non XIP memory */
static int32_t mflash_drv_init_internal(void)
{
    /* NOTE: Multithread access is not supported for SRAM target.
     *       XIP target MUST be protected by disabling global interrupts
     *       since all ISR (and API that is used inside) is placed at XIP.
     *       It is necessary to place at least "mflash_drv.o", "fsl_flexspi.o" to SRAM */
    /* disable interrupts when running from XIP */
    uint32_t primask = __get_PRIMASK();

    __asm("cpsid i");

    status_t status = kStatus_Success;

#ifndef XIP_EXTERNAL_FLASH
    flexspi_config_t config;

    /* Get FLEXSPI default settings and configure the flexspi. */
    FLEXSPI_GetDefaultConfig(&config);

    /* Set AHB buffer size for reading data through AHB bus. */
    config.ahbConfig.enableAHBPrefetch   = true;
    config.ahbConfig.enableAHBBufferable = true;
    config.ahbConfig.enableAHBCachable   = true;
    config.rxSampleClock                 = kFLEXSPI_ReadSampleClkExternalInputFromDqsPad;//kFLEXSPI_ReadSampleClkLoopbackFromDqsPad;
	// Need to set the combination-enable option. This options combines 8 data lines
	// from FlexSPI channel A with 4 data lines from FlexSPI channel B to form an
	// 8-line bus for octal. On this SoC this is the only way to enable octal.
	config.enableCombination = true;    
    FLEXSPI_Init(MFLASH_FLEXSPI, &config);

    /* AHB Read Address option bit. This option bit is intend to remove AHB burst start address alignment limitation */
    MFLASH_FLEXSPI->AHBCR |= FLEXSPI_AHBCR_READADDROPT_MASK;

    /* Configure flash settings according to serial flash feature. */
    FLEXSPI_SetFlashConfig(MFLASH_FLEXSPI, &deviceconfig, kFLEXSPI_PortA1);
#endif

    /* Update LUT table. */
    FLEXSPI_UpdateLUT(MFLASH_FLEXSPI, 0, customLUT, CUSTOM_LUT_LENGTH);

#ifndef XIP_EXTERNAL_FLASH
    /* Enter quad mode. */
    status = flexspi_nor_enable_quad_mode(MFLASH_FLEXSPI);
#endif

    if (status == kStatus_Success)
    {
        /* Unprotect all sectors in preparation of future writes. */
        status = flexspi_nor_unprotect_all(MFLASH_FLEXSPI);
    }

    if (primask == 0)
    {
        __asm("cpsie i");
    }

    return status;
}

/* API - initialize 'mflash' */
int32_t mflash_drv_init(void)
{
    /* Necessary to have double wrapper call in non_xip memory */
    return mflash_drv_init_internal();
}

/* Internal - erase single sector */
static int32_t mflash_drv_sector_erase_internal(uint32_t sector_addr)
{
    status_t status;
    uint32_t primask = __get_PRIMASK();

    __asm("cpsid i");

    status = flexspi_nor_flash_sector_erase(MFLASH_FLEXSPI, sector_addr);

    /* Do software reset. */
    FLEXSPI_SoftwareReset(MFLASH_FLEXSPI);

    DCACHE_InvalidateByRange(MFLASH_BASE_ADDRESS + sector_addr, MFLASH_SECTOR_SIZE);

    if (primask == 0)
    {
        __asm("cpsie i");
    }

    /* Flush pipeline to allow pending interrupts take place
     * before starting next loop */
    __ISB();

    return status;
}

/* Calling wrapper for 'mflash_drv_sector_erase_internal'.
 * Erase one sector starting at 'sector_addr' - must be sector aligned.
 */
int32_t mflash_drv_sector_erase(uint32_t sector_addr)
{
    if (0 == mflash_drv_is_sector_aligned(sector_addr))
        return kStatus_InvalidArgument;

    return mflash_drv_sector_erase_internal(sector_addr);
}

/* Internal - write single page */
static int32_t mflash_drv_page_program_internal(uint32_t page_addr, uint32_t *data)
{
    uint32_t primask = __get_PRIMASK();

    __asm("cpsid i");

    status_t status;
    status = flexspi_nor_flash_page_program(MFLASH_FLEXSPI, page_addr, data);

    /* Do software reset. */
    FLEXSPI_SoftwareReset(MFLASH_FLEXSPI);

    DCACHE_InvalidateByRange(MFLASH_BASE_ADDRESS + page_addr, MFLASH_PAGE_SIZE);

    if (primask == 0)
    {
        __asm("cpsie i");
    }

    /* Flush pipeline to allow pending interrupts take place
     * before starting next loop */
    __ISB();

    return status;
}

/* Calling wrapper for 'mflash_drv_page_program_internal'.
 * Write 'data' to 'page_addr' - must be page aligned.
 * NOTE: Don't try to store constant data that are located in XIP !!
 */
int32_t mflash_drv_page_program(uint32_t page_addr, uint32_t *data)
{
    if (0 == mflash_drv_is_page_aligned(page_addr))
        return kStatus_InvalidArgument;

    return mflash_drv_page_program_internal(page_addr, data);
}

/* Internal - read data */
static int32_t mflash_drv_read_internal(uint32_t addr, uint32_t *buffer, uint32_t len)
{
    uint32_t primask = __get_PRIMASK();

    __asm("cpsid i");

    status_t status;
    status = flexspi_nor_read_data(MFLASH_FLEXSPI, addr, buffer, len);

    /* Do software reset. */
    FLEXSPI_SoftwareReset(MFLASH_FLEXSPI);

    if (primask == 0)
    {
        __asm("cpsie i");
    }

    /* Flush pipeline to allow pending interrupts take place
     * before starting next loop */
    __ISB();

    return status;
}

/* Calling wrapper for 'mflash_drv_read_internal'. */
int32_t mflash_drv_read(uint32_t addr, uint32_t *buffer, uint32_t len)
{
    /* Check alignment */
    if (((uint32_t)buffer % 4) || (len % 4))
        return kStatus_InvalidArgument;

    return mflash_drv_read_internal(addr, buffer, len);
}

/* Returns pointer (AHB address) to memory area where the specified region of FLASH is mapped, NULL on failure (could
 * not map continuous block) */
void *mflash_drv_phys2log(uint32_t addr, uint32_t len)
{
    /* take FLEXSPI remapping into account */
    uint32_t remap_offset = IOMUXC_GPR->GPR32 & 0xFFFFF000;
    uint32_t remap_start  = IOMUXC_GPR->GPR30 & 0xFFFFF000;
    uint32_t remap_end    = IOMUXC_GPR->GPR31 & 0xFFFFF000;

    /* calculate the bus address where the requested FLASH region is expected to be available */
    uint32_t bus_addr = addr + MFLASH_BASE_ADDRESS;

    if (remap_offset == 0 || (remap_end <= remap_start))
    {
        /* remapping is not active */
        return (void *)bus_addr;
    }

    if ((remap_start >= bus_addr + len) || (remap_end <= bus_addr))
    {
        /* remapping window does not collide with bus addresses normally assigned for requested range of FLASH */
        return (void *)bus_addr;
    }

    if ((remap_start + remap_offset <= bus_addr) && (remap_end + remap_offset >= bus_addr + len))
    {
        /* remapping window coveres the whole requested range of FLASH, return address adjusted by negative offset */
        return (void *)(bus_addr - remap_offset);
    }

    /* the bus address region normally assigned for requested range of FLASH is partially or completely shadowed by
     * remapping, fail */
    return NULL;
}

/* Returns address of physical memory where the area accessible by given pointer is actually stored, UINT32_MAX on
 * failure (could not map as continuous block) */
uint32_t mflash_drv_log2phys(void *ptr, uint32_t len)
{
    /* take FLEXSPI remapping into account */
    uint32_t remap_offset = IOMUXC_GPR->GPR32 & 0xFFFFF000;
    uint32_t remap_start  = IOMUXC_GPR->GPR30 & 0xFFFFF000;
    uint32_t remap_end    = IOMUXC_GPR->GPR31 & 0xFFFFF000;

    /* calculate the bus address where the requested FLASH region is expected to be available */
    uint32_t bus_addr = (uint32_t)ptr;

    if (bus_addr < MFLASH_BASE_ADDRESS)
    {
        /* the pointer points outside of the flash memory area */
        return UINT32_MAX;
    }

    if (remap_offset == 0 || (remap_end <= remap_start))
    {
        /* remapping is not active */
        return (bus_addr - MFLASH_BASE_ADDRESS);
    }

    if ((remap_start >= bus_addr + len) || (remap_end <= bus_addr))
    {
        /* remapping window does not affect the requested memory area */
        return (bus_addr - MFLASH_BASE_ADDRESS);
    }

    if ((remap_start <= bus_addr) && (remap_end >= bus_addr + len))
    {
        /* remapping window coveres the whole address range, return address adjusted by offset */
        return (bus_addr + remap_offset - MFLASH_BASE_ADDRESS);
    }

    /* the bus address region partially collides with the remapping window, hence the range is not mapped to continuous
     * block in the FLASH, fail */
    return UINT32_MAX;
}
