/*
 * Copyright 2020 NXP
 * All rights reserved.
 *
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*!*********************************************************************************
*************************************************************************************
* Include
*************************************************************************************
********************************************************************************** */
#include "fsl_common.h"
#include "fsl_flexspi.h"

#include "fsl_adapter_flash.h"

#ifndef QUAD_MODE_ENABLED
#define QUAD_MODE_ENABLED (0)
#endif
/*****************************************************************************
 *****************************************************************************
 * Private macros
 *****************************************************************************
 *****************************************************************************/

#define LUT_SEQ_INDEX_READ_ARRAY             0
#define LUT_SEQ_INDEX_BLOCK_ERASE_4K        11
#define LUT_SEQ_INDEX_WRITE_ENABLE           7
#define LUT_SEQ_INDEX_READ_STATUS_REG_BYTE1  1
#define LUT_SEQ_INDEX_WRITE_STAT_CTRL_REG    5
#define LUT_SEQ_INDEX_UNPROTECT_SECTOR       8
#define LUT_SEQ_INDEX_CMD_ENTER_QPI          9
#define LUT_SEQ_INDEX_RETURN_SPI            14
#define LUT_SEQ_INDEX_CHIP_ERASE            13
#define LUT_SEQ_INDEX_BLOCK_ERASE_4K        11
#define LUT_SEQ_INDEX_PAGE_PROG             12

#define QINST_CMD          kFLEXSPI_Command_SDR
#define QINST_DUMMY_DDR    kFLEXSPI_Command_DUMMY_DDR
#define QINST_ADDR_DDR     kFLEXSPI_Command_RADDR_DDR
#define QINST_READ_DDR     kFLEXSPI_Command_READ_DDR
#define QINST_WRITE_DDR    kFLEXSPI_Command_WRITE_DDR

#define EXIP_CMD_WRITE_STATUS_REG_BYTE1  0x01    // Write Status Register Byte 1 
#define EXIP_CMD_PAGE_PROG               0x02    // Byte/Page Program (1 - 256 Bytes)     
#define EXIP_CMD_READ_STATUS_REG_BYTE1   0x05    // Read Status Register Byte 1
#define EXIP_CMD_WRITE_ENABLE            0x06    // Write Enable 
#define EXIP_CMD_READARRAY               0x0B    // Read Array
#define EXIP_CMD_BURST_READ_WRAP         0x0C    // Burst Read with Wrap 
#define EXIP_CMD_BLOCK_ERASE_4K          0x20    // Block Erase (4 Kbytes) 
#define EXIP_CMD_WRITE_STATUS_REG_BYTE2  0x31    // Write Status Register Byte 2
#define EXIP_CMD_PROTECT_SECTOR          0x36    // Protect Sector 
#define EXIP_CMD_UNPROTECT_SECTOR        0x39    // Unprotect Sector 
#define EXIP_CMD_READ_SECT_PROT_REG      0x3C    // Read Sector Protection Registers     
#define EXIP_CMD_CHIP_ERASE              0x60    // Chip Erase    
#define EXIP_CMD_READ_STAT_CTRL_REGS     0x65    // Read Status/Control Registers 
#define EXIP_CMD_WRITE_STAT_CTRL_REGS    0x71    // Write Status/Control Registers 
#define EXIP_CMD_RETURN_SPI              0xFF    // Return to Standard SPI Mode

#define PAD_1  kFLEXSPI_1PAD
#define PAD_4  kFLEXSPI_4PAD
#define PAD_8  kFLEXSPI_8PAD

#define NON_SPI_DUMMY_CYCLES  18

#define MODE_OCTAL_DDR    0x88
#define MODE_OCTAL_SDR    0x08
#define MODE_QUAD_DDR     0x84
#define MODE_QUAD_SDR     0x04

#define CUSTOM_LUT_LENGTH        64
//#define FLASH_QUAD_ENABLE        0x40
#define FLASH_BUSY_STATUS_POL    1
#define FLASH_BUSY_STATUS_OFFSET 0

#ifndef FLASH_ADAPTER_SIZE
#define FLASH_ADAPTER_SIZE 0x4000
#endif

#define FLASH_PAGE_SIZE 256U
#define SECTOR_SIZE     0x1000U /* 4K */
#ifdef APP_FLEXSPI_AMBA_BASE
#define FLEXSPI_AMBA_BASE APP_FLEXSPI_AMBA_BASE
#elif defined(FlexSPI1_AMBA_BASE)
#define FLEXSPI_AMBA_BASE FlexSPI1_AMBA_BASE
#elif defined(FlexSPI_AMBA_BASE)
#define FLEXSPI_AMBA_BASE FlexSPI_AMBA_BASE
#else
#error Unsupported flash case
#endif

/*!*********************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
********************************************************************************** */

/*!*********************************************************************************
*************************************************************************************
* Private prototypes
*************************************************************************************
********************************************************************************** */

/*!*********************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
********************************************************************************** */
static FLEXSPI_Type *s_flexspiBase[] = FLEXSPI_BASE_PTRS;

static flexspi_device_config_t s_deviceconfig = {
    .flexspiRootClk       = 100000000,
    .flashSize            = FLASH_ADAPTER_SIZE,
    .CSIntervalUnit       = kFLEXSPI_CsIntervalUnit1SckCycle,
    .CSInterval           = 5,
    .CSHoldTime           = 2,
    .CSSetupTime          = 4,
    .dataValidTime        = 1,
    .columnspace          = 0,
    .enableWordAddress    = (bool)0,
    .AWRSeqIndex          = 0,
    .AWRSeqNumber         = 0,
    .ARDSeqIndex          = LUT_SEQ_INDEX_READ_ARRAY,
    .ARDSeqNumber         = 1,
    .AHBWriteWaitUnit     = kFLEXSPI_AhbWriteWaitUnit2AhbCycle,
    .AHBWriteWaitInterval = 0,
};

static const uint32_t customLUT[CUSTOM_LUT_LENGTH] = {
 // OPI DDR mode LUT
 //uint32_t opi_ddr_lut[LUT_SIZE] =

   //Read Array
         [0] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_READARRAY) |
			   (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
         [1] = (QINST_DUMMY_DDR << 10) | (PAD_8 << 8) | (NON_SPI_DUMMY_CYCLES*2+1) |
			   (QINST_READ_DDR << 26) | (PAD_8 << 24) | (128 << 16),
         //[2] = (QINST_JMP_ON_CS << 10) | (PAD_8 << 8) | 0,

	     // Read Status (byte 1)
         [4] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_READ_STATUS_REG_BYTE1) |
			 (QINST_DUMMY_DDR << 26) | (PAD_8 << 24) | (8 << 16),
         [5] = (QINST_READ_DDR << 10) | (PAD_8 << 8) | (1),
                // Write Enable
                [12] = FLEXSPI_LUT_SEQ(QINST_CMD, PAD_1, EXIP_CMD_WRITE_ENABLE, kFLEXSPI_Command_STOP, PAD_1, 0x0),
                // Write Status/Control Registers (this specifc sequence will writes 2 bytes to status/control regs 2-3)
                [56] = FLEXSPI_LUT_SEQ(QINST_CMD, PAD_1, EXIP_CMD_WRITE_STAT_CTRL_REGS, QINST_CMD, PAD_1, 0x02),
                [57] = FLEXSPI_LUT_SEQ(kFLEXSPI_Command_WRITE_SDR, PAD_1, 0x02, kFLEXSPI_Command_STOP, PAD_1, 0x0),

         // Read Status/Control Registers
		 [8] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_READ_STAT_CTRL_REGS) |
		        (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (8 << 16),
         [9] = (QINST_DUMMY_DDR << 10) | (PAD_8 << 8) | (9) |
			    (QINST_READ_DDR << 26) | (PAD_8 << 24) | (2 << 16), // 2 bytes in QPI-DDR to round up to a full clock cycle

         // Write Status Register Byte 2
         [16] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_WRITE_STATUS_REG_BYTE2) |
                (QINST_WRITE_DDR << 26) | (PAD_8 << 24) | (1 << 16),

         // Write Status/Control Registers
         //[20] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_WRITE_STAT_CTRL_REGS) |
		//		(QINST_WRITE_DDR << 26) | (PAD_8 << 24) | (4 << 16),

         // Write Status/Control Registers
         [20] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_WRITE_STAT_CTRL_REGS) |
				(QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (8 << 16),
		 [21] = (QINST_WRITE_DDR << 10) | (PAD_8 << 8) | (1),

         // Burst Read with Wrap
         [24] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_BURST_READ_WRAP) |
			   (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
         [25] = (QINST_DUMMY_DDR << 10) | (PAD_8 << 8) | (45) |
			   (QINST_READ_DDR << 26) | (PAD_8 << 24) | (128 << 16),
         //[26] = (QINST_JMP_ON_CS << 10) | (PAD_8 << 8) | 0,

         // Write Enable     
         [28] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_WRITE_ENABLE),


         // Unprotect Sector
         [32] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_UNPROTECT_SECTOR) |
                (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),


         // Protect Sector
         [36] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_PROTECT_SECTOR) |
                (QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
	 
         // Read Sector Protection Registers
		 [40] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_READ_SECT_PROT_REG) |
				(QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
		 [41] = (QINST_DUMMY_DDR << 10) | (PAD_8 << 8) | (9) |
				(QINST_READ_DDR << 26) | (PAD_8 << 24) | (1 << 16),

         // Block Erase 4K
		 [44] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_BLOCK_ERASE_4K) |
				(QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),

         // Page Program
         [48] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_PAGE_PROG) |
				(QINST_ADDR_DDR << 26) | (PAD_8 << 24) | (32 << 16),
         [49] = (QINST_WRITE_DDR << 10) | (PAD_8 << 8) | (128), 

         // Chip Erase 
		 [52] = (QINST_CMD << 10) | (PAD_8 << 8) | (EXIP_CMD_CHIP_ERASE),
};

/*****************************************************************************
 *****************************************************************************
 * Private functions
 *****************************************************************************
 *****************************************************************************/

static status_t flexspi_nor_write_enable(FLEXSPI_Type *base, uint32_t baseAddr)
{
    flexspi_transfer_t flashXfer;
    status_t status;

    /* Write enable */
    flashXfer.deviceAddress = baseAddr;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Command;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_WRITE_ENABLE;

    status = FLEXSPI_TransferBlocking(base, &flashXfer);

    return status;
}

static status_t flexspi_nor_wait_bus_busy(FLEXSPI_Type *base)
{
    /* Wait status ready. */
    bool isBusy;
    uint32_t readValue;
    status_t status;
    flexspi_transfer_t flashXfer;
    bool busyStaus = (bool)FLASH_BUSY_STATUS_POL;

    flashXfer.deviceAddress = 0;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Read;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_READ_STATUS_REG_BYTE1;
    flashXfer.data          = &readValue;
    flashXfer.dataSize      = 1;

    do
    {
        status = FLEXSPI_TransferBlocking(base, &flashXfer);

        if (status != kStatus_Success)
        {
            return status;
        }
        if (busyStaus)
        {
            if ((readValue & (1U << FLASH_BUSY_STATUS_OFFSET)) > 0U)
            {
                isBusy = true;
            }
            else
            {
                isBusy = false;
            }
        }
        else
        {
            if ((readValue & (1U << FLASH_BUSY_STATUS_OFFSET)) > 0U)
            {
                isBusy = false;
            }
            else
            {
                isBusy = true;
            }
        }

    } while (isBusy);

    return status;
}

#if defined(QUAD_MODE_ENABLED) && (QUAD_MODE_ENABLED == 1)
static status_t flexspi_nor_enable_quad_mode(FLEXSPI_Type *base)
{
    flexspi_transfer_t flashXfer;
    status_t status;
    uint32_t writeValue = MODE_OCTAL_DDR;

    /* Write enable */
    status = flexspi_nor_write_enable(base, 0);

    if (status != kStatus_Success)
    {
        return status;
    }

    /* Enable quad mode. */
    flashXfer.deviceAddress = 2;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Write;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_WRITE_STAT_CTRL_REG;
    flashXfer.data          = &writeValue;
    flashXfer.dataSize      = 1;

    status = FLEXSPI_TransferBlocking(base, &flashXfer);
    if (status != kStatus_Success)
    {
        return status;
    }

    status = flexspi_nor_wait_bus_busy(base);

    /* Do software reset. */
    FLEXSPI_SoftwareReset(base);

    return status;
}
#endif

static status_t flexspi_nor_unprotect_all(FLEXSPI_Type *base)
{
    flexspi_transfer_t flashXfer;
    status_t status;
    uint32_t writeValue = 0;

    /* Write enable */
    status = flexspi_nor_write_enable(base, 0);

    if (status != kStatus_Success)
    {
        return status;
    }

    /* write 0 to status/control register 1 - this will unprotect all sectors. */
    flashXfer.deviceAddress = 1;
    flashXfer.port          = kFLEXSPI_PortA1;
    flashXfer.cmdType       = kFLEXSPI_Write;
    flashXfer.SeqNumber     = 1;
    flashXfer.seqIndex      = LUT_SEQ_INDEX_WRITE_STAT_CTRL_REG;
    flashXfer.data          = &writeValue;
    flashXfer.dataSize      = 1;

    status = FLEXSPI_TransferBlocking(base, &flashXfer);
    if (status != kStatus_Success)
    {
        return status;
    }

    status = flexspi_nor_wait_bus_busy(base);

    return status;
}

/*!*********************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
********************************************************************************** */

/*!
 * @brief Initializes the global flash properties structure members.
 *
 * This function initializes the Flash module for the other Flash APIs.
 *
 *
 * @retval #kStatus_HAL_Flash_Success API was executed successfully.
 * @retval #kStatus_HAL_Flash_InvalidArgument An invalid argument is provided.
 * @retval #kStatus_HAL_Flash_ExecuteInRamFunctionNotReady Execute-in-RAM function is not available.
 * @retval #kStatus_HAL_Flash_PartitionStatusUpdateFailure Failed to update the partition status.
 */
hal_flash_status_t HAL_FlashInit(void)
{
    FLEXSPI_Type *base;
    flexspi_config_t config;
    uint32_t key;

    (void)memset(&config, 0x0, sizeof(flexspi_config_t));
#if defined(__DCACHE_PRESENT) && (__DCACHE_PRESENT == 1U)
    bool DCacheEnableFlag = false;
    /* Disable D cache. */
    if (SCB_CCR_DC_Msk == (SCB_CCR_DC_Msk & SCB->CCR))
    {
        SCB_DisableDCache();
        DCacheEnableFlag = true;
    }
#endif /* __DCACHE_PRESENT */

    key  = DisableGlobalIRQ();
    base = NULL;
    for (uint8_t i = 0; i < (sizeof(s_flexspiBase) / sizeof(FLEXSPI_Type *)); i++)
    {
        if (NULL != s_flexspiBase[i])
        {
            base = s_flexspiBase[i];
            break;
        }
    }
    if (NULL == base)
    {
        return kStatus_HAL_Flash_Fail;
    }

    /*Get FLEXSPI default settings and configure the flexspi. */
    FLEXSPI_GetDefaultConfig(&config);

    /*Set AHB buffer size for reading data through AHB bus. */
    config.ahbConfig.enableAHBPrefetch    = true;
    config.ahbConfig.enableAHBBufferable  = true;
    config.ahbConfig.enableReadAddressOpt = true;//???
    config.ahbConfig.enableAHBCachable    = true;
    config.rxSampleClock                  = kFLEXSPI_ReadSampleClkExternalInputFromDqsPad;//kFLEXSPI_ReadSampleClkLoopbackFromDqsPad;
	// Need to set the combination-enable option. This options combines 8 data lines
	// from FlexSPI channel A with 4 data lines from FlexSPI channel B to form an
	// 8-line bus for octal. On this SoC this is the only way to enable octal.
	config.enableCombination = true;    
    FLEXSPI_Init(base, &config);

    /* Configure flash settings according to serial flash feature. */
    FLEXSPI_SetFlashConfig(base, &s_deviceconfig, kFLEXSPI_PortA1);

    /* Update LUT table. */
    FLEXSPI_UpdateLUT(base, 0, customLUT, CUSTOM_LUT_LENGTH);

    /* Do software reset. */
    FLEXSPI_SoftwareReset(base);

#if defined(QUAD_MODE_ENABLED) && (QUAD_MODE_ENABLED == 1)
    /* Enter quad mode. */
    (hal_flash_status_t) flexspi_nor_enable_quad_mode(base);
#endif

    /* Unprotect all sectors in preparation of future writes. */
    (hal_flash_status_t) flexspi_nor_unprotect_all(base);

    /* Do software reset. */
    FLEXSPI_SoftwareReset(base);

    EnableGlobalIRQ(key);
#if defined(__DCACHE_PRESENT) && (__DCACHE_PRESENT == 1U)
    if (DCacheEnableFlag)
    {
        /* Enable D cache. */
        SCB_EnableDCache();
    }
#endif /* __DCACHE_PRESENT */
    return kStatus_HAL_Flash_Success;
}

/*!
 * \brief  Verify erase data in Flash
 *
 * @param start           The address of the Flash location
 * @param lengthInBytes   The number of bytes to be checked
 * @param margin          Flash margin value
 * @retval #kStatus_HAL_Flash_Success API was executed successfully.
 */
hal_flash_status_t HAL_FlashVerifyErase(uint32_t start, uint32_t lengthInBytes, hal_flash_margin_value_t margin)
{
    hal_flash_status_t state;
    uint32_t address;
    uint32_t value;

    (void)margin;

    if (start < FLEXSPI_AMBA_BASE)
    {
        return kStatus_HAL_Flash_Fail;
    }

    address = start;

    while (address < (start + lengthInBytes))
    {
        value = 0U;
        if ((address + sizeof(uint32_t)) < (start + lengthInBytes))
        {
            state = HAL_FlashRead(address, sizeof(uint32_t), (uint8_t *)&value);
            if (0xffffffffUL != value)
            {
                return kStatus_HAL_Flash_Fail;
            }
            address += sizeof(uint32_t);
        }
        else
        {
            state = HAL_FlashRead(address, 1, (uint8_t *)&value);
            if (0xffU != value)
            {
                return kStatus_HAL_Flash_Fail;
            }
            address++;
        }

        if (kStatus_HAL_Flash_Success != state)
        {
            return kStatus_HAL_Flash_Fail;
        }
    }

    return kStatus_HAL_Flash_Success;
}

/*!
 * \brief  Write alligned data to FLASH
 *
 * @param dest            The address of the Flash location
 * @param size            The number of bytes to be programed
 * @param pData           Pointer to the data to be programmed to Flash
 *
 * @retval #kStatus_HAL_Flash_Success API was executed successfully.
 *
 */
hal_flash_status_t HAL_FlashProgram(uint32_t dest, uint32_t size, uint8_t *pData)
{
    FLEXSPI_Type *base;
    uint32_t address;
    uint32_t writeLength;
    status_t status = (status_t)kStatus_HAL_Flash_Error;
    flexspi_transfer_t flashXfer;
    uint32_t key;

#if defined(__ICACHE_PRESENT) && (__ICACHE_PRESENT == 1U)
    bool ICacheEnableFlag = false;
    /* Disable I cache. */
    if (SCB_CCR_IC_Msk == (SCB_CCR_IC_Msk & SCB->CCR))
    {
        SCB_DisableICache();
        ICacheEnableFlag = true;
    }
#endif /* __ICACHE_PRESENT */

#if defined(__DCACHE_PRESENT) && (__DCACHE_PRESENT == 1U)
    bool DCacheEnableFlag = false;
    /* Disable D cache. */
    if (SCB_CCR_DC_Msk == (SCB_CCR_DC_Msk & SCB->CCR))
    {
        SCB_DisableDCache();
        DCacheEnableFlag = true;
    }
#endif /* __DCACHE_PRESENT */

    if (dest < FLEXSPI_AMBA_BASE)
    {
        return kStatus_HAL_Flash_Fail;
    }

    dest = dest - FLEXSPI_AMBA_BASE;

    base = NULL;
    for (uint8_t i = 0; i < (sizeof(s_flexspiBase) / sizeof(FLEXSPI_Type *)); i++)
    {
        if (NULL != s_flexspiBase[i])
        {
            base = s_flexspiBase[i];
            break;
        }
    }
    if (NULL == base)
    {
        return kStatus_HAL_Flash_Fail;
    }

    address = dest;
    while (address < (dest + size))
    {
        if (0U != (address & ((FLASH_PAGE_SIZE - 1U))))
        {
            writeLength = address - (address & (~(FLASH_PAGE_SIZE - 1U)));
            writeLength = FLASH_PAGE_SIZE - writeLength;

            if ((dest + size - address) < writeLength)
            {
                writeLength = (dest + size - address);
            }
        }
        else
        {
            writeLength = dest + size - address;
        }
        if (writeLength > FLASH_PAGE_SIZE)
        {
            writeLength = FLASH_PAGE_SIZE;
        }

        key = DisableGlobalIRQ();
        /* Write enable */
        status = flexspi_nor_write_enable(base, 0);

        if (status != kStatus_Success)
        {
            EnableGlobalIRQ(key);
            break;
        }

        flashXfer.deviceAddress = address;
        flashXfer.port          = kFLEXSPI_PortA1;
        flashXfer.cmdType       = kFLEXSPI_Write;
        flashXfer.SeqNumber     = 1;
        flashXfer.seqIndex      = LUT_SEQ_INDEX_PAGE_PROG;
        flashXfer.data          = (uint32_t *)((void *)(&pData[address - dest]));
        flashXfer.dataSize      = writeLength;

        status = FLEXSPI_TransferBlocking(base, &flashXfer);

        if (status != kStatus_Success)
        {
            EnableGlobalIRQ(key);
            break;
        }

        status = flexspi_nor_wait_bus_busy(base);
        if (status != kStatus_Success)
        {
            EnableGlobalIRQ(key);
            break;
        }
        address += writeLength;

        FLEXSPI_SoftwareReset(base);
        SDK_DelayAtLeastUs(2U, SDK_DEVICE_MAXIMUM_CPU_CLOCK_FREQUENCY);
        EnableGlobalIRQ(key);
    }
    __DSB();
    __DSB();
#if defined(__DCACHE_PRESENT) && (__DCACHE_PRESENT == 1U)
    if (DCacheEnableFlag)
    {
        /* Enable D cache. */
        SCB_EnableDCache();
    }
#endif /* __DCACHE_PRESENT */
#if defined(__ICACHE_PRESENT) && (__ICACHE_PRESENT == 1U)
    if (ICacheEnableFlag)
    {
        /* Enable I cache. */
        SCB_EnableICache();
    }
#endif /* __ICACHE_PRESENT */
    return (hal_flash_status_t)status;
}

/*!
 * \brief  Write data to FLASH
 *
 * @param dest        The address of the Flash location
 * @param size        The number of bytes to be programed
 * @param pData       Pointer to the data to be programmed to Flash
 *
 * @retval #kStatus_HAL_Flash_Success API was executed successfully.
 *
 */
hal_flash_status_t HAL_FlashProgramUnaligned(uint32_t dest, uint32_t size, uint8_t *pData)
{
    return HAL_FlashProgram(dest, size, pData);
}

/*!
 * \brief  Erase to 0xFF one or more FLASH sectors.
 *
 * @param dest            The start address of the first sector to be erased
 * @param size            The amount of flash to be erased (multiple of sector size)
 *
 * @retval #kStatus_HAL_Flash_Success API was executed successfully.
 *
 */
hal_flash_status_t HAL_FlashEraseSector(uint32_t dest, uint32_t size)
{
    FLEXSPI_Type *base;
    uint32_t address;
    status_t status = (status_t)kStatus_HAL_Flash_Error;
    flexspi_transfer_t flashXfer;
    uint32_t key;

    if (dest < FLEXSPI_AMBA_BASE)
    {
        return kStatus_HAL_Flash_Fail;
    }
#if defined(__ICACHE_PRESENT) && (__ICACHE_PRESENT == 1U)
    bool ICacheEnableFlag = false;
    /* Disable I cache. */
    if (SCB_CCR_IC_Msk == (SCB_CCR_IC_Msk & SCB->CCR))
    {
        SCB_DisableICache();
        ICacheEnableFlag = true;
    }
#endif /* __ICACHE_PRESENT */
    dest = dest - FLEXSPI_AMBA_BASE;
    base = NULL;
    for (uint8_t i = 0; i < (sizeof(s_flexspiBase) / sizeof(FLEXSPI_Type *)); i++)
    {
        if (NULL != s_flexspiBase[i])
        {
            base = s_flexspiBase[i];
            break;
        }
    }
    if (NULL == base)
    {
        return kStatus_HAL_Flash_Fail;
    }

    address = dest;
    while (address < (dest + size))
    {
        /* Write enable */
        status = flexspi_nor_write_enable(base, 0);

        if (status != kStatus_Success)
        {
            break;
        }

        flashXfer.deviceAddress = address;
        flashXfer.port          = kFLEXSPI_PortA1;
        flashXfer.cmdType       = kFLEXSPI_Command;
        flashXfer.SeqNumber     = 1;
        flashXfer.seqIndex      = LUT_SEQ_INDEX_BLOCK_ERASE_4K;

        key    = DisableGlobalIRQ();
        status = FLEXSPI_TransferBlocking(base, &flashXfer);

        if (status != kStatus_Success)
        {
            EnableGlobalIRQ(key);
            break;
        }

        status = flexspi_nor_wait_bus_busy(base);

        address += SECTOR_SIZE;

        /* Do software reset. */
        FLEXSPI_SoftwareReset(base);
        SDK_DelayAtLeastUs(2U, SDK_DEVICE_MAXIMUM_CPU_CLOCK_FREQUENCY);
        EnableGlobalIRQ(key);
    }
    if (status != kStatus_Success)
    {
        key = DisableGlobalIRQ();
        /* Do software reset. */
        FLEXSPI_SoftwareReset(base);
        SDK_DelayAtLeastUs(2U, SDK_DEVICE_MAXIMUM_CPU_CLOCK_FREQUENCY);
        EnableGlobalIRQ(key);
    }
#if defined(__ICACHE_PRESENT) && (__ICACHE_PRESENT == 1U)
    if (ICacheEnableFlag)
    {
        /* Enable I cache. */
        SCB_EnableICache();
    }
#endif /* __ICACHE_PRESENT */

    return (hal_flash_status_t)status;
}

/*!
 * \brief  Read data from FLASH
 *
 *
 * @param scr             The address of the Flash location to be read
 * @param size            The number of bytes to be read
 * @param pData           Pointer to the data to be read from Flash
 *
 * @retval #kStatus_HAL_Flash_Success API was executed successfully.
 *
 */
hal_flash_status_t HAL_FlashRead(uint32_t src, uint32_t size, uint8_t *pData)
{
#if defined(__DCACHE_PRESENT) && (__DCACHE_PRESENT == 1U)
    bool DCacheEnableFlag = false;
    /* Disable D cache. */
    if (SCB_CCR_DC_Msk == (SCB_CCR_DC_Msk & SCB->CCR))
    {
        SCB_DisableDCache();
        DCacheEnableFlag = true;
    }
#endif /* __DCACHE_PRESENT */
    (void)memcpy(pData, (uint8_t *)src, size);
#if defined(__DCACHE_PRESENT) && (__DCACHE_PRESENT == 1U)
    if (DCacheEnableFlag)
    {
        /* Enable D cache. */
        SCB_EnableDCache();
    }
#endif /* __DCACHE_PRESENT */
    return kStatus_HAL_Flash_Success;
}

/*!
 * @brief Returns the desired hal flash property.
 *
 * @param Property        The desired property from the list of properties in
 *                        enum hal_flash_property_tag_t
 * @param value           A pointer to the value returned for the desired flash property.
 *
 * @retval #kStatus_HAL_Flash_Success API was executed successfully.
 * @retval #kStatus_HAL_Flash_InvalidArgument An invalid argument is provided.
 * @retval #kStatus_HAL_Flash_Success Flash currently not support.
 */
hal_flash_status_t HAL_FlashGetProperty(hal_flash_property_tag_t property, uint32_t *value)
{
    return kStatus_HAL_Flash_Success;
}

/*!
 * @brief Set the desired hal flash property.
 *
 * @param Property        The desired property from the list of properties in
 *                        enum hal_flash_property_tag_t
 * @param value           The value would be set to the desired flash property.
 *
 * @retval #kStatus_HAL_Flash_Success API was executed successfully.
 * @retval #kStatus_HAL_Flash_InvalidArgument An invalid argument is provided.
 * @retval #kStatus_HAL_Flash_Success Flash currently not support.
 */
hal_flash_status_t HAL_FlashSetProperty(hal_flash_property_tag_t property, uint32_t value)
{
    return kStatus_HAL_Flash_Success;
}

/*!
 * @brief Returns the security state via the pointer passed into the function.
 *
 * This function retrieves the current flash security status, including the
 * security enabling state and the backdoor key enabling state.
 *
 * @param state           A pointer to the value returned for the current security status
 *
 * @retval #kStatus_HAL_Flash_Success API was executed successfully.
 * @retval #kStatus_HAL_Flash_InvalidArgument An invalid argument is provided.
 * @retval #kStatus_HAL_Flash_Success Flash currently not support.
 */
hal_flash_status_t HAL_FlashGetSecurityState(hal_flash_security_state_t *state)
{
    return kStatus_HAL_Flash_Success;
}
