/**************************************************************************/
/*                                                                        */
/*       Copyright (c) Microsoft Corporation. All rights reserved.        */
/*                                                                        */
/*       This software is licensed under the Microsoft Software License   */
/*       Terms for Microsoft Azure RTOS. Full text of the license can be  */
/*       found in the LICENSE file at https://aka.ms/AzureRTOS_EULA       */
/*       and in the root directory of this software.                      */
/*                                                                        */
/**************************************************************************/


/**************************************************************************/
/**************************************************************************/
/**                                                                       */
/** NetX Component                                                        */
/**                                                                       */
/**   Lightweight M2M Protocol (LWM2M)                                    */
/**                                                                       */
/**************************************************************************/
/**************************************************************************/


#define NX_LWM2M_CLIENT_SOURCE_CODE

#include "nx_lwm2m_client.h"

#ifndef TX_DISABLE_ERROR_CHECKING
#define TX_DISABLE_ERROR_CHECKING
#endif

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_create                            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function creates a LWM2M Client.                               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    ip_ptr                                Pointer to IP instance        */
/*    packet_pool_ptr                       Pointer to packet pool        */
/*    name_ptr                              Endpoint name of LWM2M client */
/*    name_length                           Length of endpoint name       */
/*    msisdn_ptr                            Optional MSISDN number        */
/*    msisdn_length                         Length of MSISDN number       */
/*    binding_mode                          Supported binding and modes   */
/*    stack_ptr                             Client thread's stack pointer */
/*    stack_size                            Client thread's stack size    */
/*    priority                              Priority of client thread     */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_create                                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_create(NX_LWM2M_CLIENT *client_ptr, NX_IP *ip_ptr, NX_PACKET_POOL *packet_pool_ptr,
                              const CHAR *name_ptr, UINT name_length, const CHAR *msisdn_ptr, UINT msisdn_length,
                              UCHAR binding_modes, VOID *stack_ptr, ULONG stack_size, UINT priority)
{
UINT status;

    /* Check for invalid input pointers */
    if ((client_ptr == NX_NULL) || (ip_ptr == NX_NULL) || (packet_pool_ptr == NX_NULL) || (name_ptr == NX_NULL) || (stack_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Check for a memory size error.  */
    if (stack_size < TX_MINIMUM_STACK)
    {
        return(NX_SIZE_ERROR);
    }

    /* Check the priority specified.  */
    if (priority >= TX_MAX_PRIORITIES)
    {
        return(NX_OPTION_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_create(client_ptr, ip_ptr, packet_pool_ptr, name_ptr, name_length, msisdn_ptr, msisdn_length, binding_modes, stack_ptr, stack_size, priority);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_delete                            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function deletes a LWM2M Client.                               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_delete                                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_delete(NX_LWM2M_CLIENT *client_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if (client_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_delete(client_ptr);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_device_callback_set               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function installs the application callback for implementing    */
/*    operations on the LWM2M Device Object resources not handled by the  */
/*    LWM2M Client.                                                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    operation_callback                    The operation callback        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_device_callback_set                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_device_callback_set(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_CLIENT_DEVICE_OPERATION_CALLBACK operation_callback)
{
UINT status;

    /* Check for invalid input pointers */
    if ((client_ptr == NX_NULL) || (operation_callback == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_device_callback_set(client_ptr, operation_callback);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*   _nxe_lwm2m_client_device_error_push                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function adds a new instance to the error code resource of     */
/*    the Device Object.                                                  */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    code                                  The new error code            */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_device_error_push                                  */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_device_error_push(NX_LWM2M_CLIENT *client_ptr, UCHAR code)
{
UINT status;

    /* Check for invalid input pointers */
    if (client_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_device_error_push(client_ptr, code);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_device_error_reset                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function deletes all error code resource instances from the    */
/*    Device Object.                                                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_device_error_reset                                 */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_device_error_reset(NX_LWM2M_CLIENT *client_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if (client_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_device_error_reset(client_ptr);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_device_resource_changed           PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function signals the LWM2M Client that a resource of the       */
/*    Device Object has changed.                                          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    resource_ptr                          The resource ID and value     */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nxe_lwm2m_client_device_resource_changed                           */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_device_resource_changed(NX_LWM2M_CLIENT *client_ptr, const NX_LWM2M_CLIENT_RESOURCE *resource_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((client_ptr == NX_NULL) || (resource_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_device_resource_changed(client_ptr, resource_ptr);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_firmware_create                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions initializes the Firmware Update Object.              */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    firmware_ptr                          Pointer to object data        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    protocols                             The supported protocols       */
/*    package_callback                      The package callback          */
/*    package_uri_callback                  The package URI callback      */
/*    update_callback                       The update callback           */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_create                                    */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_firmware_create(NX_LWM2M_CLIENT_FIRMWARE *firmware_ptr, NX_LWM2M_CLIENT *client_ptr, UINT protocols, NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_CALLBACK package_callback, NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_URI_CALLBACK package_uri_callback, NX_LWM2M_CLIENT_FIRMWARE_UPDATE_CALLBACK update_callback)
{
UINT status;

    /* Check for invalid input pointers */
    if ((firmware_ptr == NX_NULL) || (client_ptr == NX_NULL) ||
        ((package_callback == NX_NULL) && (package_uri_callback == NX_NULL)) || (update_callback == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_firmware_create(firmware_ptr, client_ptr, protocols, package_callback, package_uri_callback, update_callback);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_firmware_package_info_set         PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the package information resources of the         */
/*    Firmware Update Object.                                             */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    firmware_ptr                          Pointer to object data        */
/*    name                                  The name of firmware package  */
/*    name_length                           The length of name            */
/*    version                               The version of firmware       */
/*                                          package                       */
/*    version_length                        The length of version         */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_package_info_set                          */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_firmware_package_info_set(NX_LWM2M_CLIENT_FIRMWARE *firmware_ptr, const CHAR *name, UINT name_length, const CHAR *version, UINT version_length)
{
UINT status;

    /* Check for invalid input pointers */
    if ((firmware_ptr == NX_NULL) || (name == NX_NULL) || (version == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_firmware_package_info_set(firmware_ptr, name,  name_length, version,version_length);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_firmware_result_set               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions sets the Update Result resource of the Firmware      */
/*    Update Object.                                                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    firmware_ptr                          Pointer to object data        */
/*    result                                The Update Result             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_result_set                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_firmware_result_set(NX_LWM2M_CLIENT_FIRMWARE *firmware_ptr, UCHAR result)
{
UINT status;

    /* Check for invalid input pointers */
    if (firmware_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_firmware_result_set(firmware_ptr, result);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_firmware_state_set                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions sets the State of the Firmware Update Object.        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    firmware_ptr                          Pointer to object data        */
/*    state                                 The state of the Firmware     */
/*                                          Object                        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_state_set                                 */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_firmware_state_set(NX_LWM2M_CLIENT_FIRMWARE *firmware_ptr, UCHAR state)
{
UINT status;

    /* Check for invalid input pointers */
    if (firmware_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_firmware_state_set(firmware_ptr, state);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_lock                              PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function locks the LWM2M Client to prevent concurent access.   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_lock                                               */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_lock(NX_LWM2M_CLIENT *client_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if (client_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_lock(client_ptr);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*   _nxe_lwm2m_client_object_add                         PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function adds a new Object implementation to the LWM2M Client. */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_ptr                            Pointer to the Object         */
/*    object_id                             The Object ID                 */
/*    object_operation                      Object operation function     */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_add                                         */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_add(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_ID object_id, 
                                  NX_LWM2M_CLIENT_OBJECT_OPERATION_CALLBACK object_operation)
{
UINT status;

    /* Check for invalid input pointers */
    if ((client_ptr == NX_NULL) || (object_ptr == NX_NULL) || (object_operation == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Objects 0-3 are already implemented by the LWM2M Client */
    if (object_id <= NX_LWM2M_CLIENT_DEVICE_OBJECT_ID)
    {

        /* Return duplicated ID error */
        return(NX_LWM2M_CLIENT_ALREADY_EXIST);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_add(client_ptr, object_ptr, object_id, object_operation);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_object_create                     PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function creates a new Object Instance.                        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id_ptr                       Pointer to the Instance ID    */
/*    num_values                            Number of values to set       */
/*    values                                Pointer to the values         */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_create                                      */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_create(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID *instance_id_ptr, UINT num_values, const NX_LWM2M_CLIENT_RESOURCE *values)
{
UINT status;

    /* Check for invalid input pointers */
    if ((client_ptr == NX_NULL) || ((num_values != 0) && (values == NX_NULL)))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_create(client_ptr, object_id, instance_id_ptr, num_values, values);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_object_delete                     PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function deletes an Object Instance.                           */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id                           The Instance ID               */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_delete                                      */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_delete(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id)
{
UINT status;

    /* Check for invalid input pointers */
    if (client_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_delete(client_ptr, object_id, instance_id);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_object_discover                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the list of resources implemented by an       */
/*    Object.                                                             */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id                           The Instance ID               */
/*    num_resources                         Pointer to the number of      */
/*                                          resources                     */
/*    resources                             Pointer to the resource info  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_discover                                    */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_discover(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id, UINT *num_resources, NX_LWM2M_CLIENT_RESOURCE *resources)
{
UINT status;

    /* Check for invalid input pointers */
    if ((client_ptr == NX_NULL) || (num_resources == NX_NULL) || (resources == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_discover(client_ptr, object_id, instance_id, num_resources, resources);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_object_execute                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function performs the execute operation on an Object resource. */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id                           The Instance ID               */
/*    resource_id                           The Resource ID               */
/*    args_ptr                              Pointer to the arguments      */
/*    args_length                           Length of arguments           */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_execute                                     */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_execute(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id, NX_LWM2M_ID resource_id, const CHAR *args_ptr, UINT args_length)
{
UINT status;

    /* Check for invalid input pointers */
    if ((client_ptr == NX_NULL) || ((args_length != 0) && (args_ptr == NX_NULL)))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_execute(client_ptr, object_id, instance_id, resource_id, args_ptr, args_length);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_object_instance_add               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function add a new Object Instance to the Instance list.       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    instance_ptr                          Pointer to the Object Instance*/
/*    instance_id_ptr                       Pointer to the Instance ID    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_instance_add                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_instance_add(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_ID *instance_id_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((object_ptr == NX_NULL) || (instance_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_instance_add(object_ptr, instance_ptr, instance_id_ptr);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_object_instance_next_get          PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the next Instance of an Object.               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id_ptr                       Pointer to the Instance ID    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nxe_lwm2m_client_object_instance_next_get                          */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_instance_next_get(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID *instance_id_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((client_ptr == NX_NULL) || (instance_id_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_instance_next_get(client_ptr, object_id, instance_id_ptr);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_object_instance_remove            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function removes an Object Instance from Instance list.        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to the Object         */
/*    instance_ptr                          Pointer to the Instance       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_instance_remove                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_instance_remove(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((object_ptr == NX_NULL) || (instance_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_instance_remove(object_ptr, instance_ptr);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_object_next_get                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the next Object of the LWM2M Client.          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id_ptr                         Pointer to the Object ID      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_next_get                                    */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_next_get(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID *object_id_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((client_ptr == NX_NULL) || (object_id_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_next_get(client_ptr, object_id_ptr);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_object_read                       PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function reads resources from an Object Instance.              */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id                           The Instance ID               */
/*    num_values                            The number of resources       */
/*    values                                Pointer to the resources      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_read                                        */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_read(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id, UINT num_values, NX_LWM2M_CLIENT_RESOURCE *values)
{
UINT status;

    /* Check for invalid input pointers */
    if ((client_ptr == NX_NULL) || ((num_values != 0) && (values == NX_NULL)))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_read(client_ptr, object_id, instance_id, num_values, values);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*   _nxe_lwm2m_client_object_remove                      PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function remove an Object from the LWM2M Client Object list.   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_ptr                            Pointer to Object             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_remove                                      */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_remove(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_CLIENT_OBJECT *object_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((client_ptr == NX_NULL) || (object_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Objects 0-3 are implemented internally, can't be removed by user */
    if (object_ptr -> object_id <= NX_LWM2M_CLIENT_DEVICE_OBJECT_ID)
    {

        /* Return duplicated ID error */
        return(NX_LWM2M_CLIENT_FORBIDDEN);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_remove(client_ptr, object_ptr);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_object_resource_changed           PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function signals the LWM2M Client that an object resource.     */
/*    value has changed                                                   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to the Object         */
/*    instance_ptr                          Pointer to the Object Instance*/
/*    resource_ptr                          The resource ID and value     */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_resource_changed                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_resource_changed(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, const NX_LWM2M_CLIENT_RESOURCE *resource_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((object_ptr == NX_NULL) || (instance_ptr == NX_NULL) || (resource_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_resource_changed(object_ptr, instance_ptr, resource_ptr);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_object_write                      PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function writes resources to an Object Instance.               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id                           The Instance ID               */
/*    num_values                            The number of resources       */
/*    values                                Pointer to the resources      */
/*    write_op                              The type of write operation   */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_write                                       */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_object_write(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id, UINT num_values, const NX_LWM2M_CLIENT_RESOURCE *values, UINT write_op)
{
UINT status;

    /* Check for invalid input pointers */
    if ((client_ptr == NX_NULL) || ((num_values != 0) && (values == NX_NULL)))
    {
        return(NX_PTR_ERROR);
    }

    /* Check for invalid write operation */
    if (write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_UPDATE &&
        write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_REPLACE_INSTANCE &&
        write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_REPLACE_RESOURCE &&
        write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP)
    {
        return(NX_OPTION_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_object_write(client_ptr, object_id, instance_id, num_values, values, write_op);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_boolean_get              PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a boolean resource.              */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     bool_ptr                             On return, the boolean value  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_boolean_get Get boolean resource          */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_boolean_get(NX_LWM2M_CLIENT_RESOURCE *value, NX_LWM2M_BOOL *bool_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((value == NX_NULL) || (bool_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_boolean_get(value, bool_ptr);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_boolean_set              PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of a boolean resource.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     bool_data                            The boolean value             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_boolean_set                               */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_boolean_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_BOOL bool_data)
{
UINT status;

    /* Check for invalid input pointers */
    if (resource_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_boolean_set(resource_ptr, bool_data);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_dim_get                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function gets the dim of multiple resource.                    */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     dim                                  Dim of the multiple resource  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_dim_get                                   */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_dim_get(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, UCHAR *dim)
{
UINT status;

    /* Check for invalid input pointers */
    if ((resource_ptr == NX_NULL) || (dim == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_dim_get(resource_ptr, dim);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_dim_set                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the dim of multiple resource.                    */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     dim                                  Dim of the multiple resource  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_dim_set                                   */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_dim_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, UCHAR dim)
{
UINT status;

    /* Check for invalid input pointers */
    if (resource_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_dim_set(resource_ptr, dim);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_float32_get              PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a 32-bit floating point resource.*/
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     float32_ptr                          On return, the floating point */
/*                                          value                         */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_float32_get Get 32-bit floating point     */
/*                                          resource                      */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_float32_get(NX_LWM2M_CLIENT_RESOURCE *value, NX_LWM2M_FLOAT32 *float32_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((value == NX_NULL) || (float32_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_float32_get(value, float32_ptr);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_float32_set              PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of a 32-bit floating point resource.   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     float32_data                         The floating data             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_float32_set                               */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_float32_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_FLOAT32 float32_data)
{
UINT status;

    /* Check for invalid input pointers */
    if (resource_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_float32_set(resource_ptr, float32_data);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_float64_get              PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a 64-bit floating point resource.*/
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     float64_ptr                          On return, the floating point */
/*                                          value                         */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_float64_get Get 64-bit floating point     */
/*                                          resource                      */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_float64_get(NX_LWM2M_CLIENT_RESOURCE *value, NX_LWM2M_FLOAT64 *float64_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((value == NX_NULL) || (float64_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_float64_get(value, float64_ptr);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_float64_set              PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of a 64-bit floating point resource.   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     float64_data                         The floating point value      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_float64_set                               */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_float64_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_FLOAT64 float64_data)
{
UINT status;

    /* Check for invalid input pointers */
    if (resource_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_float64_set(resource_ptr, float64_data);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_info_get                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function gets the information of resource.                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     resource_id                          ID of the resource            */
/*     operation                            Orperation of the resource    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_info_get                                  */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_info_get(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_ID *resource_id, ULONG *operation)
{
UINT status;

    /* Check for invalid input pointers */
    if ((resource_ptr == NX_NULL) || (resource_id == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_info_get(resource_ptr, resource_id, operation);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_info_set                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the information of resource.                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     resource_id                          ID of the resource            */
/*     operation                            Orperation of the resource    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_info_set                                  */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_info_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_ID resource_id, ULONG operation)
{
UINT status;

    /* Check for invalid input pointers */
    if (resource_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_info_set(resource_ptr, resource_id, operation);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_instances_get            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function gets the multiple resource instances.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     resource_instances                   Pointer to the instances      */
/*     count                                Count of the instances        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_instances_get                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_instances_get(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_CLIENT_RESOURCE *resource_instances, UINT *count)
{
UINT status;

    /* Check for invalid input pointers */
    if ((resource_ptr == NX_NULL) || (resource_instances == NX_NULL) || (count == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_instances_get(resource_ptr, resource_instances, count);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_instances_set            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets multiple resource instances.                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     resource_instances                   Pointer to the instances      */
/*     count                                Count of the instances        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_instances_set                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_instances_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_CLIENT_RESOURCE *resource_instances, UINT count)
{
UINT status;

    /* Check for invalid input pointers */
    if ((resource_ptr == NX_NULL) || (resource_instances == NX_NULL) || (count == 0))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_instances_set(resource_ptr, resource_instances, count);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_integer32_get            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a 32-bit integer resource.       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     int32_ptr                            On return, the integer value  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer32_get Get 32-bit integer resource */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_integer32_get(NX_LWM2M_CLIENT_RESOURCE *value, NX_LWM2M_INT32 *int32_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((value == NX_NULL) || (int32_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_integer32_get(value, int32_ptr);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_integer32_set            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of a 32-bit integer resource.          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     int32_data                           The integer value             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer32_set                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_integer32_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_INT32 int32_data)
{
UINT status;

    /* Check for invalid input pointers */
    if (resource_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_integer32_set(resource_ptr, int32_data);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_integer64_get            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a 64-bit integer resource.       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     int64_ptr                            On return, the integer value  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer64_get Get 64-bit integer resource */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_integer64_get(NX_LWM2M_CLIENT_RESOURCE *value, NX_LWM2M_INT64 *int64_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((value == NX_NULL) || (int64_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_integer64_get(value, int64_ptr);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_integer64_set            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of a 64-bit integer resource.          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     int64_data                           The integer value             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer64_set                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_integer64_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_INT64 int64_data)
{
UINT status;

    /* Check for invalid input pointers */
    if (resource_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_integer64_set(resource_ptr, int64_data);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_objlnk_get               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of an Object Link resource.         */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     object_id                            On return, the object ID      */
/*     instance_id                          On return, the instance ID    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_objlnk_get  Get Object Link resource      */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_objlnk_get(NX_LWM2M_CLIENT_RESOURCE *value, NX_LWM2M_ID *object_id, NX_LWM2M_ID *instance_id)
{
UINT status;

    /* Check for invalid input pointers */
    if ((value == NX_NULL) || (object_id == NX_NULL) || (instance_id == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_objlnk_get(value, object_id, instance_id);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_objlnk_set               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of an Object Link resource.            */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     object_id                            The object ID                 */
/*     instance_id                          The instance ID               */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_objlnk_set                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_objlnk_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr,  NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id)
{
UINT status;

    /* Check for invalid input pointers */
    if (resource_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_objlnk_set(resource_ptr, object_id, instance_id);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_opaque_get               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of an opaque resource.              */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     opaque_ptr                           On return, a pointer to the   */
/*                                          opaque data                   */
/*     opaque_length                        On return, the length of the  */
/*                                          opaque data                   */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_opaque_get  Get opaque resource           */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_opaque_get(NX_LWM2M_CLIENT_RESOURCE *value, const VOID **opaque_ptr, UINT *opaque_length)
{
UINT status;

    /* Check for invalid input pointers */
    if ((value == NX_NULL) || (opaque_ptr == NX_NULL) || (opaque_length == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_opaque_get(value, opaque_ptr, opaque_length);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_opaque_set               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of an opaque resource.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     opaque_ptr                           Apointer to the opaque data   */
/*     opaque_length                        The length of the opaque data */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_opaque_set                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_opaque_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, VOID *opaque_ptr, UINT opaque_length)
{
UINT status;

    /* Check for invalid input pointers */
    if ((resource_ptr == NX_NULL) || (opaque_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_opaque_set(resource_ptr, opaque_ptr, opaque_length);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_string_get               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a string resource.               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     string_ptr                           On return, a pointer to the   */
/*                                          string                        */
/*     string_length                        On return, the length of the  */
/*                                          string                        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_string_get  Get string resource           */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_string_get(NX_LWM2M_CLIENT_RESOURCE *value, const CHAR **string_ptr, UINT *string_length)
{
UINT status;

    /* Check for invalid input pointers */
    if ((value == NX_NULL) || (string_ptr == NX_NULL) || (string_length == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_string_get(value, string_ptr, string_length);

    /* Return completion status */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_resource_string_set               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of a string resource.                  */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     Resource_ptr                         Pointer to the resource       */
/*     string_ptr                           A pointer to the string       */
/*     string_length                        Length of the string          */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_string_set                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_resource_string_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, CHAR *string_ptr, UINT string_length)
{
UINT status;

    /* Check for invalid input pointers */
    if ((resource_ptr == NX_NULL) || (string_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_resource_string_set(resource_ptr, string_ptr, string_length);

    /* Return completion status */
    return(status);
}

#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_session_bootstrap                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function starts a session with a Bootstrap server.             */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    security_id                           The Security Instance ID of   */
/*                                          the bootstrap server          */
/*    ip_address                            The IP address of the server  */
/*    port                                  The UDP port of the server    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nxe_lwm2m_client_session_bootstrap                                 */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_session_bootstrap(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_ID security_id, const NXD_ADDRESS *ip_address, UINT port)
{
UINT status;

    /* Check for invalid input pointers */
    if ((session_ptr == NX_NULL) || (ip_address == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_session_bootstrap(session_ptr, security_id, ip_address, port);

    /* Return completion status */
    return(status);
}

#ifdef NX_SECURE_ENABLE_DTLS

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_session_bootstrap_dtls            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function starts a session with a Bootstrap server using a      */
/*    secure channel.                                                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    security_id                           The Security Instance ID of   */
/*                                          the bootstrap server          */
/*    ip_address                            The IP address of the server  */
/*    port                                  The UDP port of the server    */
/*    dtls_session_ptr                      The pointer to the DTLS       */
/*                                          session                       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_bootstrap_dtls                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_session_bootstrap_dtls(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_ID security_id, const NXD_ADDRESS *ip_address, UINT port, NX_SECURE_DTLS_SESSION *dtls_session_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((session_ptr == NX_NULL) || (ip_address == NX_NULL) || (dtls_session_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_session_bootstrap_dtls(session_ptr, security_id, ip_address, port, dtls_session_ptr);

    /* Return completion status */
    return(status);
}
#endif  /* NX_SECURE_ENABLE_DTLS */
#endif /* NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES */


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_session_create                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions creates a LWM2M Client Session.                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    client_ptr                            Pointer to LWM2M Client       */
/*    state_callback                        The session state callback    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_create                                     */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_session_create(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_CLIENT_SESSION_STATE_CALLBACK state_callback)
{
UINT status;

    /* Check for invalid input pointers */
    if ((session_ptr == NX_NULL) || (client_ptr == NX_NULL) || (state_callback == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_session_create(session_ptr, client_ptr, state_callback);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_session_delete                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function deletes a LWM2M Client Session.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_delete                                     */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_session_delete(NX_LWM2M_CLIENT_SESSION *session_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if (session_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_session_delete(session_ptr);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_session_deregister                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function deregisters the Client from a LWM2M Server.           */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_deregister                                 */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_session_deregister(NX_LWM2M_CLIENT_SESSION *session_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if (session_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_session_deregister(session_ptr);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_session_error_get                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the error code of a LWM2M Session.            */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                The error code                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_error_get                                  */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_session_error_get(NX_LWM2M_CLIENT_SESSION *session_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if (session_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_session_error_get(session_ptr);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_session_register                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function registers the Client to a LWM2M server.               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    server_id                             The Short Server ID of the    */
/*                                          LWM2M server                  */
/*    ip_address                            The IP address of the server  */
/*    port                                  The UDP port of the server    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_register                                   */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_session_register(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_ID server_id, const NXD_ADDRESS *ip_address, UINT port)
{
UINT status;

    /* Check for invalid input pointers */
    if ((session_ptr == NX_NULL) || (ip_address == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_session_register(session_ptr, server_id, ip_address, port);

    /* Return completion status */
    return(status);
}

#ifdef NX_SECURE_ENABLE_DTLS

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_session_register_dtls             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function registers the Client to a LWM2M server using a        */
/*    secure channel.                                                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    server_id                             The Short Server ID of the    */
/*                                          LWM2M server                  */
/*    ip_address                            The IP address of the server  */
/*    port                                  The UDP port of the server    */
/*    dtls_session_ptr                      The pointer to the DTLS       */
/*                                          session                       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_register_dtls                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_session_register_dtls(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_ID server_id, const NXD_ADDRESS *ip_address, UINT port, NX_SECURE_DTLS_SESSION *dtls_session_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if ((session_ptr == NX_NULL) || (ip_address == NX_NULL) || (dtls_session_ptr == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_session_register_dtls(session_ptr, server_id, ip_address, port, dtls_session_ptr);

    /* Return completion status */
    return(status);
}

#endif /* NX_SECURE_ENABLE_DTLS */

#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_session_register_info_get         PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function gets the server and security information for the next */
/*    registration step.                                                  */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*     security_instance_id                 Security instance ID          */
/*     server_id                            Server ID                     */
/*     server_uri                           Server URI                    */
/*     server_uri_len                       Length of server URI          */
/*     security_mode                        Security Mode                 */
/*     pub_key_or_id                        Public key or ID              */
/*     pub_key_or_id_len                    Length of public key or ID    */
/*     server_pub_key                       Server public key             */
/*     server_pub_key_len                   Length of server public key   */
/*     secret_key                           Secret key                    */
/*     secret_key_len                       Length of secret key          */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_register_info_get                          */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_session_register_info_get(NX_LWM2M_CLIENT_SESSION *session_ptr, UINT security_instance_id, NX_LWM2M_ID *server_id, CHAR **server_uri, UINT *server_uri_len, UCHAR *security_mode,
                                                 UCHAR **pub_key_or_id, UINT *pub_key_or_id_len, UCHAR **server_pub_key, UINT *server_pub_key_len, UCHAR **secret_key, UINT *secret_key_len)
{
UINT status;

    /* Check for invalid input pointers */
    if ((session_ptr == NX_NULL) || (server_id == NX_NULL) || (server_uri == NX_NULL) || (server_uri_len == NX_NULL) || (security_mode == NX_NULL))
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_session_register_info_get(session_ptr, security_instance_id, server_id, server_uri, server_uri_len, security_mode,
                                                        pub_key_or_id, pub_key_or_id_len, server_pub_key, server_pub_key_len, secret_key, secret_key_len);

    /* Return completion status */
    return(status);
}
#endif /* NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES */


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_session_update                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function updates the Client to a LWM2M Server.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_update                                     */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_session_update(NX_LWM2M_CLIENT_SESSION *session_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if (session_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_session_update(session_ptr);

    /* Return completion status */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nxe_lwm2m_client_unlock                            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function unlocks the LWM2M Client.                             */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_unlock                                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nxe_lwm2m_client_unlock(NX_LWM2M_CLIENT *client_ptr)
{
UINT status;

    /* Check for invalid input pointers */
    if (client_ptr == NX_NULL)
    {
        return(NX_PTR_ERROR);
    }

    /* Call actual function */
    status = _nx_lwm2m_client_unlock(client_ptr);

    /* Return completion status */
    return(status);
}



#if NX_LWM2M_CLIENT_MAX_ACCESS_CONTROL_INSTANCES


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_read                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the read operation.                         */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    object_instance_ptr                   Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_operation                           */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_access_control_read(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *object_instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT resource_count)
{
UINT i, j;
NX_LWM2M_CLIENT_ACCESS_CONTROL_INSTANCE *instance_ptr = (NX_LWM2M_CLIENT_ACCESS_CONTROL_INSTANCE *) object_instance_ptr;
NX_LWM2M_CLIENT_RESOURCE *resource_instances = object_ptr -> object_client_ptr -> nx_lwm2m_client_mulitple_temp_resources;

    NX_PARAMETER_NOT_USED(object_ptr);

    for (i = 0; i < resource_count; i++)
    {
        switch (resource[i].resource_id)
        {

        case NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_ID_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = instance_ptr -> access_control_instance_object_id;
            break;

        case NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_INSTANCE_ID_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = instance_ptr -> access_control_instance_object_instance_id;
            break;

        case NX_LWM2M_CLIENT_ACCESS_CONTROL_ACL_ID:

            if (instance_ptr -> access_control_instance_acl_count == 0)
            {

                /* No ACL defined */
                return(NX_LWM2M_CLIENT_NOT_FOUND);
            }

            for (j = 0; j < instance_ptr -> access_control_instance_acl_count; j++)
            {

                _nx_lwm2m_client_resource_info_set(&resource_instances[j], instance_ptr -> access_control_instance_acl[j].acl_server_id, NX_NULL);
                _nx_lwm2m_client_resource_integer32_set(&resource_instances[j], instance_ptr -> access_control_instance_acl[j].acl_flags);
            }

            _nx_lwm2m_client_resource_instances_set(&resource[i], resource_instances, instance_ptr -> access_control_instance_acl_count);
            break;

        case NX_LWM2M_CLIENT_ACCESS_CONTROL_OWNER_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = instance_ptr -> access_control_instance_owner;
            break;

        /* Unknown resource */
        default:

            return(NX_LWM2M_CLIENT_NOT_FOUND);
        }
    }

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_discover            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the discover operation.                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    object_instance_ptr                   Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    num_resources_ptr                     Pointer to number of resource */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_operation                           */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_access_control_discover(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *object_instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resources, UINT *num_resources_ptr)
{

NX_LWM2M_CLIENT_ACCESS_CONTROL_INSTANCE *instance_ptr = (NX_LWM2M_CLIENT_ACCESS_CONTROL_INSTANCE *) object_instance_ptr;
UINT resource_count;

    NX_PARAMETER_NOT_USED(object_ptr);

    if (instance_ptr -> access_control_instance_acl_count != 0)
    {
        resource_count = 4;
    }
    else
    {
        resource_count = 3;
    }
    if (*num_resources_ptr < resource_count)
    {
        return(NX_LWM2M_CLIENT_BUFFER_TOO_SMALL);
    }

    resources[0].resource_id        = NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_ID_ID;
    resources[0].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ_WRITE;
    resources[1].resource_id        = NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_INSTANCE_ID_ID;
    resources[1].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ_WRITE;
    if (instance_ptr -> access_control_instance_acl_count != 0)
    {
        resources[2].resource_id        = NX_LWM2M_CLIENT_ACCESS_CONTROL_ACL_ID;
        resources[2].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ_WRITE;
        resources[2].resource_type      = NX_LWM2M_CLIENT_RESOURCE_MULTIPLE;
        resources[2].resource_dim       = (UCHAR)instance_ptr -> access_control_instance_acl_count;
    }
    resources[resource_count-1].resource_id        = NX_LWM2M_CLIENT_ACCESS_CONTROL_OWNER_ID;
    resources[resource_count-1].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ_WRITE;

    *num_resources_ptr = resource_count;

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_write               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the write operation.                        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    object_instance_ptr                   Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    write_op                              Type of write operation       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer32_get                             */
/*    _nx_lwm2m_client_resource_instances_get                             */
/*    _nx_lwm2m_client_resource_info_get                                  */
/*    memcpy                                                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_operation                           */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_access_control_write(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *object_instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT resource_count, UINT write_op)
{
NX_LWM2M_CLIENT_ACCESS_CONTROL_INSTANCE *instance_ptr = (NX_LWM2M_CLIENT_ACCESS_CONTROL_INSTANCE *) object_instance_ptr;
UINT i, j;
UINT status;
ULONG flags;
NX_LWM2M_INT32 object_id;
NX_LWM2M_INT32 instance_id;
NX_LWM2M_INT32 owner;
UINT acl_count;
NX_LWM2M_CLIENT_RESOURCE resource_instance[NX_LWM2M_CLIENT_MAX_ACCESS_CONTROL_ACLS];
NX_LWM2M_CLIENT_ACL acl[NX_LWM2M_CLIENT_MAX_ACCESS_CONTROL_ACLS];
NX_LWM2M_INT32 acl_flags;

    NX_PARAMETER_NOT_USED(object_ptr);

    /* XXX replace not supported? */
    if (write_op == NX_LWM2M_CLIENT_OBJECT_WRITE_REPLACE_INSTANCE)
    {
        return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);
    }

    /* Check validity of values before updating instance */
    flags = 0;

    for (i = 0; i < resource_count; i++)
    {
        switch (resource[i].resource_id)
        {

        case NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_ID_ID:

            /* Resource is read-only, can only be set during boostrap or instance creation */
            if (write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP && write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_CREATE)
            {
                return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);
            }
            status = _nx_lwm2m_client_resource_integer32_get(&resource[i], &object_id);
            if (status != NX_SUCCESS)
            {
                return(status);
            }
            /* Object ID must be in range 1-65534 */
            if (object_id < 1 || object_id > 65534)
            {
                return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
            }
            flags |= 1<<NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_ID_ID;
            break;

        case NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_INSTANCE_ID_ID:

            /* Resource is read-only, can only be set during boostrap or instance creation */
            if (write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP && write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_CREATE)
            {
                return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);
            }
            status = _nx_lwm2m_client_resource_integer32_get(&resource[i], &instance_id);
            if (status != NX_SUCCESS)
            {
                return(status);
            }

            /* Instance ID must be in range 0-65535 */
            if (instance_id < 0 || instance_id > 65535)
            {
                return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
            }
            flags |= 1<<NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_INSTANCE_ID_ID;
            break;

        case NX_LWM2M_CLIENT_ACCESS_CONTROL_ACL_ID:

            /* Get all the resource instances */
            acl_count = NX_LWM2M_CLIENT_MAX_ACCESS_CONTROL_ACLS;
            status = _nx_lwm2m_client_resource_instances_get(&resource[i], resource_instance, &acl_count);
            if (status)
            {
                return(status);
            }

            /* Get the value of each resource instance */
            for (j = 0; j < acl_count; j++)
            {
                _nx_lwm2m_client_resource_info_get(&resource_instance[i], &acl[j].acl_server_id, NX_NULL);
                _nx_lwm2m_client_resource_integer32_get(&resource_instance[i], &acl_flags);

                if (acl_flags < 0 || acl_flags > 65535)
                {

                    /* Flag must fit in 16-bit */
                    return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
                }
                acl[j].acl_flags = (USHORT) acl_flags;
            }

            /* Check we have enought space for update */
            if (instance_ptr -> access_control_instance_acl_count != 0 && write_op == NX_LWM2M_CLIENT_OBJECT_WRITE_UPDATE)
            {
                UINT acl_total;
                acl_total = instance_ptr -> access_control_instance_acl_count;
                for (j=0; j<acl_count; j++)
                {
                    UINT k;
                    for (k=0; k<instance_ptr -> access_control_instance_acl_count; k++)
                    {
                        if (instance_ptr -> access_control_instance_acl[k].acl_server_id == acl[j].acl_server_id)
                        {

                            /* Instance already exists */
                            break;
                        }
                    }
                    if (k >= instance_ptr -> access_control_instance_acl_count)
                    {

                        /* New instance */
                        acl_total++;
                    }
                }
                if (acl_total > NX_LWM2M_CLIENT_MAX_ACCESS_CONTROL_ACLS)
                {

                    /* Too many instances */
                    return(NX_LWM2M_CLIENT_NO_MEMORY);
                }
            }
            flags |= 1<<NX_LWM2M_CLIENT_ACCESS_CONTROL_ACL_ID;
            break;

        case NX_LWM2M_CLIENT_ACCESS_CONTROL_OWNER_ID:

            status = _nx_lwm2m_client_resource_integer32_get(&resource[i], &owner);
            if (status != NX_SUCCESS)
            {
                return(status);
            }

            /* Server ID must be in range 0-65535 */
            if (owner < 0 || owner > 65535)
            {
                return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
            }
            flags |= 1<<NX_LWM2M_CLIENT_ACCESS_CONTROL_OWNER_ID;
            break;

        /* Unknown resource */
        default:

            if (write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP)
            {
                return(NX_LWM2M_CLIENT_NOT_FOUND);
            }

            /* Ignore unknown resources during bootstrap */
            break;
        }
    }

    /* Update the instance */
    if (flags & 1<<NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_ID_ID)
    {
        instance_ptr -> access_control_instance_object_id = (NX_LWM2M_ID) object_id;
    }
    if (flags & 1<<NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_INSTANCE_ID_ID)
    {
        instance_ptr -> access_control_instance_object_instance_id = (NX_LWM2M_ID) instance_id;
    }
    if (flags & 1<<NX_LWM2M_CLIENT_ACCESS_CONTROL_ACL_ID)
    {
        if (instance_ptr -> access_control_instance_acl_count != 0 && write_op == NX_LWM2M_CLIENT_OBJECT_WRITE_UPDATE)
        {
            UINT current_count;

            /* Update existing resource */
            current_count = instance_ptr -> access_control_instance_acl_count;
            for (j=0; j<acl_count; j++)
            {
                UINT k;
                for (k=0; k<current_count; k++)
                {
                    if (instance_ptr -> access_control_instance_acl[k].acl_server_id == acl[j].acl_server_id)
                    {

                        /* Instance already exists */
                        break;
                    }
                }
                if (k < current_count)
                {

                    /* Update existing instance */
                    instance_ptr -> access_control_instance_acl[k].acl_flags = acl[j].acl_flags;
                }
                else
                {

                    /* Add new instance */
                    instance_ptr -> access_control_instance_acl[instance_ptr -> access_control_instance_acl_count] = acl[j];
                    instance_ptr -> access_control_instance_acl_count++;
                }
            }
        }
        else
        {

            /* Replace resource */
            instance_ptr -> access_control_instance_acl_count = (USHORT)acl_count;
            if (acl_count != 0)
            {
                memcpy(instance_ptr -> access_control_instance_acl, acl, acl_count*sizeof(acl[0])); /* Use case of memcpy is verified. */
            }
        }
    }
    if (flags & 1<<NX_LWM2M_CLIENT_ACCESS_CONTROL_OWNER_ID)
    {
        instance_ptr -> access_control_instance_owner = (NX_LWM2M_ID) owner;
    }

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_execute             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the execute operation.                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource_id                           The ID of resource            */
/*    args_ptr                              Pointer to arguments          */
/*    args_length                           The length of arguments       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_operation                           */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_access_control_execute(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_ID resource_id, const CHAR *args_ptr, UINT args_length)
{

    NX_PARAMETER_NOT_USED(object_ptr);
    NX_PARAMETER_NOT_USED(instance_ptr);
    NX_PARAMETER_NOT_USED(args_ptr);
    NX_PARAMETER_NOT_USED(args_length);

    switch (resource_id)
    {

    case NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_ID_ID:
    case NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_INSTANCE_ID_ID:
    case NX_LWM2M_CLIENT_ACCESS_CONTROL_ACL_ID:
    case NX_LWM2M_CLIENT_ACCESS_CONTROL_OWNER_ID:

        /* Non-executable resources */
        return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);

    default:

        /* Unkown resource */
        return(NX_LWM2M_CLIENT_NOT_FOUND);
    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_create              PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the create operation.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_id                           Pointer to object instance ID */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    bootstrap                             Flag of bootstrap step        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_write                               */
/*    _nx_lwm2m_client_object_instance_add                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_operation                           */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_access_control_create(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_ID *instance_id, NX_LWM2M_CLIENT_RESOURCE *resource, UINT resource_count, NX_LWM2M_BOOL bootstrap)
{
    NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT *access_ptr = (NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT *) object_ptr;
    NX_LWM2M_CLIENT_ACCESS_CONTROL_INSTANCE *instance_ptr;
    UINT status;

    /* Allocate a new instance */
    instance_ptr = (NX_LWM2M_CLIENT_ACCESS_CONTROL_INSTANCE *) access_ptr -> access_control_object_instances_free_list;
    if (instance_ptr == NX_NULL)
    {
        return(NX_LWM2M_CLIENT_NO_MEMORY);
    }

    /* Initialize resources with default values */
    instance_ptr -> access_control_instance_object_id            = NX_LWM2M_CLIENT_RESERVED_ID;
    instance_ptr -> access_control_instance_object_instance_id   = NX_LWM2M_CLIENT_RESERVED_ID;
    instance_ptr -> access_control_instance_owner                = NX_LWM2M_CLIENT_RESERVED_ID;
    instance_ptr -> access_control_instance_acl_count            = 0;

    /* Write the provided resources */
    status = _nx_lwm2m_client_access_control_write((NX_LWM2M_CLIENT_OBJECT *) access_ptr, (NX_LWM2M_CLIENT_OBJECT_INSTANCE *) instance_ptr, resource, resource_count, bootstrap ? NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP : NX_LWM2M_CLIENT_OBJECT_WRITE_CREATE);
    if (status != NX_SUCCESS)
    {
        return(status);
    }

    /* Update free instance list */
    access_ptr -> access_control_object_instances_free_list = instance_ptr -> access_control_instance.object_instance_next;

    /* Add new instance to the instance list */
    status = _nx_lwm2m_client_object_instance_add(object_ptr, &(instance_ptr -> access_control_instance), instance_id);
    if (status != NX_SUCCESS)
    {

        /* Revert the free instance list */
        access_ptr -> access_control_object_instances_free_list = &(instance_ptr -> access_control_instance);
        return(status);
    }

    /* Return success */
    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_delete              PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the delete operation.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_instance_remove                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_operation                           */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_access_control_delete(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr)
{
    NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT *access_ptr = (NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT *) object_ptr;
    UINT status;

    /* Remove the instance from instance list */
    status = _nx_lwm2m_client_object_instance_remove(object_ptr, instance_ptr);
    if (status)
    {
        return(status);
    }

    /* Release the instance */
    instance_ptr -> object_instance_next = access_ptr -> access_control_object_instances_free_list;
    access_ptr -> access_control_object_instances_free_list = instance_ptr;

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_operation           PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the object operation.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    operation                             The operation type            */
/*    object_ptr                            Pointer to object             */
/*    object_instance_ptr                   Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    args_ptr                              Pointer to arguments          */
/*    args_length                           The length of arguments       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_read                                */
/*    _nx_lwm2m_client_access_control_discover                            */
/*    _nx_lwm2m_client_access_control_write                               */
/*    _nx_lwm2m_client_access_control_execute                             */
/*    _nx_lwm2m_client_access_control_write                               */
/*    _nx_lwm2m_client_access_control_delete                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_access_control_operation(UINT operation, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *object_instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT *resource_count, VOID *args_ptr, UINT args_length)
{
UINT *write_op;
NX_LWM2M_CLIENT_OPERATION_ARGUMENT *arguments;

    switch (operation)
    {
    case NX_LWM2M_CLIENT_OBJECT_READ:

        /* Call read function */
        return(_nx_lwm2m_client_access_control_read(object_ptr, object_instance_ptr, resource, *resource_count));
    case NX_LWM2M_CLIENT_OBJECT_DISCOVER:

        /* Call discover function */
        return(_nx_lwm2m_client_access_control_discover(object_ptr, object_instance_ptr, resource, resource_count));
    case NX_LWM2M_CLIENT_OBJECT_WRITE:

        /* Get the type of write operation */
        write_op = (UINT *)args_ptr;

        /* Call write function */
        return(_nx_lwm2m_client_access_control_write(object_ptr, object_instance_ptr, resource, *resource_count, *write_op));
    case NX_LWM2M_CLIENT_OBJECT_EXECUTE:

        /* Call execute function */
        return(_nx_lwm2m_client_access_control_execute(object_ptr, object_instance_ptr, resource -> resource_id, args_ptr, args_length));
    case NX_LWM2M_CLIENT_OBJECT_CREATE:

        /*  Get arguments */
        arguments = (NX_LWM2M_CLIENT_OPERATION_ARGUMENT *)args_ptr;

        /* Call create function */
        return(_nx_lwm2m_client_access_control_create(object_ptr, &(arguments -> instance_id), resource, *resource_count, arguments -> bootstrap));
    case NX_LWM2M_CLIENT_OBJECT_DELETE:

        /* Call delete function */
        return(_nx_lwm2m_client_access_control_delete(object_ptr, object_instance_ptr));
    default:

        /*Unsupported operation */
        return(NX_LWM2M_CLIENT_NOT_SUPPORTED);

    }
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_access_control_object_create       PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function initializes the Access Control Object.                */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_add                                         */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_create                                             */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_access_control_object_create(NX_LWM2M_CLIENT *client_ptr)
{
int i;
NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT *access_ptr = &(client_ptr -> nx_lwm2m_client_access_control);

    /* Add Access Control Object to the LWM2M Client */
    _nx_lwm2m_client_object_add(client_ptr, &(access_ptr -> access_control_object), NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_ID, _nx_lwm2m_client_access_control_operation);

    /* Initialize the list of free instances */
    access_ptr -> access_control_object_instances_free_list = (NX_LWM2M_CLIENT_OBJECT_INSTANCE *) &access_ptr -> access_control_object_instances[0];
    for (i = 0; i < NX_LWM2M_CLIENT_MAX_ACCESS_CONTROL_INSTANCES - 1; i++)
    {
        access_ptr -> access_control_object_instances[i].access_control_instance.object_instance_next = (NX_LWM2M_CLIENT_OBJECT_INSTANCE *) &access_ptr -> access_control_object_instances[i + 1];
    }
    access_ptr -> access_control_object_instances[NX_LWM2M_CLIENT_MAX_ACCESS_CONTROL_INSTANCES - 1].access_control_instance.object_instance_next = NX_NULL;
}

#endif  /* NX_LWM2M_CLIENT_MAX_ACCESS_CONTROL_INSTANCES */


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_binding_mode_string_get            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the string corresponding to the given         */
/*    binding mode.                                                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    binding_mode                          The binding mode              */
/*    length                                The length of binding mode    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to binding mode string                                      */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_device_read                                        */
/*    _nx_lwm2m_client_server_read                                        */
/*    _nx_lwm2m_client_session_send_request                               */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
const CHAR *_nx_lwm2m_client_binding_mode_string_get(UCHAR binding_mode, UINT *length)
{
    switch (binding_mode)
    {

    case NX_LWM2M_CLIENT_BINDING_U:

        *length = 1;
        return("U");

    case NX_LWM2M_CLIENT_BINDING_UQ:

        *length = 2;
        return("UQ");

    case NX_LWM2M_CLIENT_BINDING_S:

        *length = 1;
        return("S");

    case NX_LWM2M_CLIENT_BINDING_SQ:

        *length = 2;
        return("SQ");

    case NX_LWM2M_CLIENT_BINDING_US:

        *length = 2;
        return("US");

    case NX_LWM2M_CLIENT_BINDING_UQS:

        *length = 3;
        return("UQS");

    case NX_LWM2M_CLIENT_BINDING_USQ:

        *length = 3;
        return("USQ");

    case NX_LWM2M_CLIENT_BINDING_UQSQ:

        *length = 4;
        return("UQSQ");

    default:

        /* Invalid mode */
        *length = 1;
        return("X");
    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_binding_mode_string_parse          PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function parses a string representing a binding mode.          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    string_ptr                          The binding mode string         */
/*    string_length                       The length of the string        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    The binding mode                                                    */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_server_write                                       */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UCHAR _nx_lwm2m_client_binding_mode_string_parse(const CHAR *string_ptr, UINT string_length)
{
const CHAR *string_ptr_max;
UCHAR binding;
CHAR c;

    if (string_length == 0)
    {
        return(0);
    }

    string_ptr_max = string_ptr + string_length;
    c = *string_ptr++;
    if (c == 'U')
    {
        if (string_ptr == string_ptr_max)
        {
            return(NX_LWM2M_CLIENT_BINDING_U);
        }
        c = *string_ptr++;
        if (c == 'Q')
        {
            if (string_ptr == string_ptr_max)
            {
                return(NX_LWM2M_CLIENT_BINDING_UQ);
            }
            binding = NX_LWM2M_CLIENT_BINDING_UQ;
            c = *string_ptr++;
        }
        else
        {
            binding = NX_LWM2M_CLIENT_BINDING_U;
        }
    }
    else
    {
        binding = 0;
    }
    if (c != 'S')
    {
        return(0);
    }
    if (string_ptr == string_ptr_max)
    {
        return(binding | NX_LWM2M_CLIENT_BINDING_S);
    }
    c = *string_ptr++;
    if (c != 'Q' || string_ptr != string_ptr_max)
    {
        return(0);
    }
    return(binding | NX_LWM2M_CLIENT_BINDING_SQ);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_coap_header_parse                  PORTABLE C      */
/*                                                           6.1.10       */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function parses the header of a CoAP message.                  */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    ptr                                   Pointer to source buffer      */
/*    ptr_max                               Pointer to the end of buffer  */
/*    type_ptr                              On return, the type of message*/
/*    code_ptr                              On return, the code value     */
/*    id_ptr                                On return, the message ID     */
/*    token_ptr_ptr                         On return, the pointer to the */
/*                                          token                         */
/*    token_length_ptr                      On return, the token length   */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the end of the CoAP header, NULL if the header is invalid*/
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_receive                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*  01-31-2022     Yuxin Zhou               Modified comment(s), supported*/
/*                                            token and processing        */
/*                                            confirmable response,       */
/*                                            resulting in version 6.1.10 */
/*                                                                        */
/**************************************************************************/
const VOID *_nx_lwm2m_client_coap_header_parse(const VOID *ptr, const VOID *ptr_max, UCHAR *type_ptr, UCHAR *code_ptr, USHORT *id_ptr, const VOID **token_ptr_ptr, UINT *token_length_ptr)
{
const UCHAR *byte_ptr;
UCHAR type;
UCHAR code;
USHORT id;
const VOID *token_ptr;
UINT token_length;

    /* Pointer to header */
    byte_ptr = (const UCHAR *) ptr;

    /* Check for minimum length */
    if ((const UCHAR *) ptr_max - byte_ptr < NX_LWM2M_CLIENT_COAP_HEADER_LEN)
    {
        return(NX_NULL);
    }

    /* Get first byte */
    type = *byte_ptr++;

    /* Check CoAP version */
    if ((type & NX_LWM2M_CLIENT_COAP_VERSION_MASK) != NX_LWM2M_CLIENT_COAP_VERSION_1)
    {
        return(NX_NULL);
    }

    /* Get token length */
    token_length = type & NX_LWM2M_CLIENT_COAP_TOKEN_LEN_MASK;
    if (token_length > NX_LWM2M_CLIENT_COAP_TOKEN_MAXLEN)
    {
        return(NX_NULL);
    }

    /* Get message type */
    type &= NX_LWM2M_CLIENT_COAP_TYPE_MASK;

    /* Get code byte */
    code = *byte_ptr++;

    /* Check for valid code class if not an empty message */
    if (code != 0)
    {
        UCHAR code_class = code & NX_LWM2M_CLIENT_COAP_CLASS_MASK;
        if (code_class != NX_LWM2M_CLIENT_COAP_CLASS_REQUEST &&
            code_class != NX_LWM2M_CLIENT_COAP_CLASS_SUCCESS &&
            code_class != NX_LWM2M_CLIENT_COAP_CLASS_CLIENT_ERROR &&
            code_class != NX_LWM2M_CLIENT_COAP_CLASS_SERVER_ERROR)
        {
            return(NX_NULL);
        }
    }
    else
    {

        /* Ignore empty message */
        return(NX_NULL);
    }

    /* Get message ID */
    id = *byte_ptr++;
    id = (USHORT)(id << 8);
    id |= (USHORT)(*byte_ptr++);

    /* Check token length */
    token_ptr = byte_ptr;
    byte_ptr += token_length;
    if (byte_ptr > (const UCHAR *) ptr_max)
    {
        return(NX_NULL);
    }

    /* Return header information */
    *type_ptr = type;
    *code_ptr = code;
    *id_ptr = id;
    *token_ptr_ptr = token_ptr;
    *token_length_ptr = token_length;
    return(byte_ptr);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_coap_option_parse                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function parse a CoAP option header.                           */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    ptr                                   Pointer to source buffer      */
/*    ptr_max                               Pointer to the end of buffer  */
/*    option_ptr                            On return, the option number  */
/*    value_ptr_ptr                         On return, pointer to the     */
/*                                          option value                  */
/*    value_length_ptr                      On return, length of the value*/
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the next option, NULL if the option is invalid           */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_receive                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
const VOID *_nx_lwm2m_client_coap_option_parse(const VOID *ptr, const VOID *ptr_max, UINT *option_ptr, const VOID **value_ptr_ptr, UINT *value_length_ptr)
{
const UCHAR *byte_ptr;
UCHAR option;
UINT delta;
UINT length;
const VOID *value_ptr;

    if (ptr == ptr_max)
    {

        /* End of message */
        *option_ptr = NX_LWM2M_CLIENT_COAP_OPTION_NONE;
        return(ptr);
    }

    /* Read first byte */
    byte_ptr = (const UCHAR *) ptr;
    option = *byte_ptr++;

    if (option == NX_LWM2M_CLIENT_COAP_PAYLOAD_MARKER)
    {

        /* Begining of payload */
        *option_ptr = NX_LWM2M_CLIENT_COAP_OPTION_NONE;
        return(byte_ptr);
    }

    /* Get option delta */
    delta = option >> 4;
    if (delta == 15)
    {

        /* Invalid delta */
        return(NX_NULL);
    }
    if (delta == 13)
    {

        /* 1-byte delta */
        if (byte_ptr == (const UCHAR *) ptr_max)
        {
            return(NX_NULL);
        }
        delta = *byte_ptr++ + 13u;
    }
    else if (delta == 14)
    {

        /* 2-byte delta */
        if (byte_ptr + 2 > (const UCHAR *) ptr_max)
        {
            return(NX_NULL);
        }
        delta = *byte_ptr++;
        delta <<= 8;
        delta |= *byte_ptr++;
        delta += 269;
    }

    /* Get final option number */
    delta += *option_ptr;
    if (delta == NX_LWM2M_CLIENT_COAP_OPTION_NONE)
    {

        /* Invalid option */
        return(NX_NULL);
    }

    /* Get length */
    length = option & 0x0f;
    if (length == 15)
    {

        /* Invalid length */
        return(NX_NULL);
    }
    if (length == 13)
    {

        /* 1-byte length */
        if (byte_ptr == (const UCHAR *) ptr_max)
        {
            return(NX_NULL);
        }
        length = *byte_ptr++ + 13u;
    }
    else if (length == 14)
    {

        /* 2-byte length */
        if (byte_ptr + 2 > (const UCHAR *) ptr_max)
        {
            return(NX_NULL);
        }
        length = *byte_ptr++;
        length <<= 8;
        length |= *byte_ptr++;
        length += 269;
    }

    /* Pointer to option value */
    value_ptr = byte_ptr;

    /* Advance pointer to next option */
    byte_ptr += length;
    if (byte_ptr > (const UCHAR *) ptr_max)
    {
        return(NX_NULL);
    }

    /* Return option information */
    *option_ptr = delta;
    *value_ptr_ptr = value_ptr;
    *value_length_ptr = length;
    return(byte_ptr);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_coap_option_header_add             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions writes a CoAP option header.                         */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    ptr                                   Pointer to destination buffer */
/*    last_option                           Pointer to the value of the   */
/*                                          last option, updated on return*/
/*    option                                The option to write           */
/*    length                                The length of the option value*/
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the end of the option header (start of value)            */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_send_request                               */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID *_nx_lwm2m_client_coap_option_header_add(VOID *ptr, VOID *ptr_max, UINT *last_option, UINT option, UINT length)
{
UCHAR *byte_ptr;
UCHAR code;
UINT delta;

    if (ptr >= ptr_max)
    {
        return(NX_NULL);
    }

    /* Option delta */
    delta = option - *last_option;
    if (delta >= 269)
    {
        code = 14 << 4;     /* 2-byte option delta */
    }
    else if (delta >= 13)
    {
        code = 13 << 4;     /* 1-byte option delta */
    }
    else
    {
        code = (UCHAR) (delta << 4);
    }

    /* Option length */
    if (length >= 269)
    {
        code |= 14;         /* 2-byte option length */
    }
    else if (length >= 13)
    {
        code |= 13;         /* 1-byte option delta */
    }
    else {
        code |= (UCHAR) length;
    }

    /* Write option header */
    byte_ptr = (UCHAR *) ptr;
    *byte_ptr++ = code;

    /* Write Extended option delta */
    if (delta >= 269)
    {
        if ((byte_ptr + 2) > (UCHAR *)ptr_max)
        {
            return(NX_NULL);
        }
        delta -= 269;
        *byte_ptr++ = (UCHAR) (delta >> 8);
        *byte_ptr++ = (UCHAR)  delta;
    }
    else if (delta >= 13)
    {
        if ((byte_ptr + 1) > (UCHAR *)ptr_max)
        {
            return(NX_NULL);
        }
        delta -= 13;
        *byte_ptr++ = (UCHAR) delta;
    }

    /* Write extended option length */
    if (length >= 269)
    {
        if ((byte_ptr + 2) > (UCHAR *)ptr_max)
        {
            return(NX_NULL);
        }
        length -= 269;
        *byte_ptr++ = (UCHAR) (length >> 8);
        *byte_ptr++ = (UCHAR)  length;
    }
    else if (length >= 13)
    {
        if ((byte_ptr + 1) > (UCHAR *)ptr_max)
        {
            return(NX_NULL);
        }
        length -= 13;
        *byte_ptr++ = (UCHAR) length;
    }

    /* Return updated last option value and pointer */
    *last_option = option;
    return(byte_ptr);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_coap_option_uint_add               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions writes a CoAP option with an integer value.          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    ptr                                   Pointer to destination buffer */
/*    last_option                           Pointer to the value of the   */
/*                                          last option, updated on return*/
/*    option                                The option to write           */
/*    value                                 The option value              */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the end of the option                                    */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_coap_option_header_add                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_send_response                              */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID *_nx_lwm2m_client_coap_option_uint_add(VOID *ptr, VOID *ptr_max, UINT *last_option_ptr, UINT option, ULONG value)
{
UCHAR *byte_ptr;
ULONG tmp;
UINT option_length;

    /* Byte pointer to destination buffer */
    byte_ptr = ptr;

    /* Compute length of value */
    option_length = 0;
    tmp = value;
    while (tmp != 0)
    {
        option_length++;
        tmp >>= 8;
    }

    /* Write option header */
    byte_ptr = _nx_lwm2m_client_coap_option_header_add(byte_ptr, ptr_max, last_option_ptr, option, option_length);
    if ((byte_ptr == NX_NULL) || ((byte_ptr + option_length) > (UCHAR *)ptr_max))
    {
        return(NX_NULL);
    }

    /* Write value */
    if (option_length > 3)
    {
        *byte_ptr++ = (UCHAR) (value >> 24);
    }
    if (option_length > 2)
    {
        *byte_ptr++ = (UCHAR) (value >> 16);
    }
    if (option_length > 1)
    {
        *byte_ptr++ = (UCHAR) (value >> 8);
    }
    if (option_length > 0)
    {
        *byte_ptr++ = (UCHAR) value;
    }

    return(byte_ptr);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_write_attribute                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function writes a single attribute.                            */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    ptr                                   Pointer to source buffer      */
/*    ptr_max                               Pointer to the end of buffer  */
/*    name                                  The name of attribute         */
/*    name_length                           The length of attribute's name*/
/*    is_int32                              If it is 32 bit integer       */
/*    value_ptr                             The pointer of value          */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the next attribute, NULL if the attribute is invalid     */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_strconv_format_int32                               */
/*    _nx_lwm2m_client_strconv_format_notify_number                       */
/*    memcpy                                                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_corelink_attributes_add                            */
/*    _nx_lwm2m_client_session_receive                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static CHAR *_nx_lwm2m_client_write_attribute(CHAR *ptr, CHAR *ptr_max, const CHAR *name, UINT name_length, NX_LWM2M_BOOL is_int32, const VOID *value_ptr)
{
UINT len;

    /* Write attribute name */
    if (ptr_max - ptr < (int) (name_length + 2))
    {
        return(NX_NULL);
    }
    *ptr++ = ';';
    memcpy(ptr, name, name_length); /* Use case of memcpy is verified. */
    ptr += name_length;
    *ptr++ = '=';

    /* Write value */
    if (is_int32)
    {
        len = _nx_lwm2m_client_strconv_format_int32(ptr, (UINT)(ptr_max - ptr), *((const NX_LWM2M_INT32 *) value_ptr));
    }
    else
    {
        len = _nx_lwm2m_client_strconv_format_notify_number(ptr, (UINT)(ptr_max - ptr), *((const NX_LWM2M_CLIENT_NOTIFY_NUMBER *) value_ptr));
    }
    if (len == 0)
    {
        return(NX_NULL);
    }
    ptr += len;

    return(ptr);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_corelink_attributes_add            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function writes the attributes of an object, instance or       */
/*    resource using the CoRE Link Format                                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    ptr                                   Pointer to destination buffer */
/*    ptr_max                               Pointer to the end of buffer  */
/*    server_id                             The optional server ID        */
/*    resource_ptr                          Optional resource information */
/*    attr_flags                            The notify flags indicating   */
/*                                          which attributes are set      */
/*    attr_pmin                             The pmin attribute            */
/*    attr_pmax                             The pmax attribute            */
/*    attr_gt                               The gt attribute              */
/*    attr_lt                               The lt attribute              */
/*    attr_stp                              The stp attribute             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the end of the written attributes, NULL if the           */
/*    destination buffer is too small                                     */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_write_attribute                                    */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_send_response                              */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
CHAR *_nx_lwm2m_client_corelink_attributes_add(CHAR *ptr, CHAR *ptr_max, NX_LWM2M_ID server_id, const NX_LWM2M_CLIENT_RESOURCE *resource_ptr, UINT attr_flags, NX_LWM2M_INT32 attr_pmin, NX_LWM2M_INT32 attr_pmax, NX_LWM2M_CLIENT_NOTIFY_NUMBER attr_gt, NX_LWM2M_CLIENT_NOTIFY_NUMBER attr_lt, NX_LWM2M_CLIENT_NOTIFY_NUMBER attr_stp)
{
NX_LWM2M_INT32 value_as_int32;

    /* Write short server ID */
    if (server_id != 0)
    {

        value_as_int32 = server_id;
        ptr = _nx_lwm2m_client_write_attribute(ptr, ptr_max, "ssid", 4, NX_TRUE, &value_as_int32);
        if (ptr == NX_NULL)
        {
            return(NX_NULL);
        }
    }

    /* Write dimension of multiple resource */
    if ((resource_ptr != NX_NULL) && (resource_ptr -> resource_type == NX_LWM2M_CLIENT_RESOURCE_MULTIPLE))
    {
        value_as_int32 = resource_ptr -> resource_dim;
        ptr = _nx_lwm2m_client_write_attribute(ptr, ptr_max, "dim", 3, NX_TRUE, &value_as_int32);
        if (ptr == NX_NULL)
        {
            return(NX_NULL);
        }
    }

    /* Write notification attributes */
    if (attr_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_PMIN)
    {
        ptr = _nx_lwm2m_client_write_attribute(ptr, ptr_max, "pmin", 4, NX_TRUE, &attr_pmin);
        if (ptr == NX_NULL)
        {
            return(NX_NULL);
        }
    }
    if (attr_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_PMAX)
    {
        ptr = _nx_lwm2m_client_write_attribute(ptr, ptr_max, "pmax", 4, NX_TRUE, &attr_pmax);
        if (ptr == NX_NULL)
        {
            return(NX_NULL);
        }
    }
    if (attr_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_GT)
    {
        ptr = _nx_lwm2m_client_write_attribute(ptr, ptr_max, "gt", 2, NX_FALSE, &attr_gt);
        if (ptr == NX_NULL)
        {
            return(NX_NULL);
        }
    }
    if (attr_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_LT)
    {
        ptr = _nx_lwm2m_client_write_attribute(ptr, ptr_max, "lt", 2, NX_FALSE, &attr_lt);
        if (ptr == NX_NULL)
        {
            return(NX_NULL);
        }
    }
    if (attr_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_STP)
    {
        ptr = _nx_lwm2m_client_write_attribute(ptr, ptr_max, "st", 2, NX_FALSE, &attr_stp);
        if (ptr == NX_NULL)
        {
            return(NX_NULL);
        }
    }

    return(ptr);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_corelink_path_add                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function writes the path of an object, instance or             */
/*    resource using the CoRE Link Format.                                */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    ptr                                   Pointer to destination buffer */
/*    ptr_max                               Pointer to the end of buffer  */
/*    object_id                             The ID of the object          */
/*    instance_id                           The optional instance ID      */
/*    resource_id                           The optional resource ID      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the end of the written path, NULL if the                 */
/*    destination buffer is too small                                     */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_strconv_format_id                                  */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_send_request                               */
/*    _nx_lwm2m_client_session_send_response                              */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
CHAR *_nx_lwm2m_client_corelink_path_add(CHAR *ptr, CHAR *ptr_max, NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id, NX_LWM2M_ID resource_id)
{
UINT len;

    if (ptr_max - ptr < 2)
    {
        return(NX_NULL);
    }
    *ptr++ = '<';
    *ptr++ = '/';

    len = _nx_lwm2m_client_strconv_format_id(ptr, (UINT)(ptr_max - ptr), object_id);
    if (len == 0)
    {
        return(NX_NULL);
    }
    ptr += len;

    if (instance_id != NX_LWM2M_CLIENT_RESERVED_ID)
    {
        if (ptr >= ptr_max)
        {
            return(NX_NULL);
        }
        *ptr++ = '/';

        len = _nx_lwm2m_client_strconv_format_id(ptr, (UINT)(ptr_max - ptr), instance_id);
        if (len == 0)
        {
            return(NX_NULL);
        }
        ptr += len;

        if (resource_id != NX_LWM2M_CLIENT_RESERVED_ID)
        {
            if (ptr >= ptr_max)
            {
                return(NX_NULL);
            }
            *ptr++ = '/';

            len = _nx_lwm2m_client_strconv_format_id(ptr, (UINT)(ptr_max - ptr), resource_id);
            if (len == 0)
            {
                return(NX_NULL);
            }
            ptr += len;
        }
    }

    if (ptr >= ptr_max)
    {
        return(NX_NULL);
    }
    *ptr++ = '>';

    return(ptr);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    coap_message_notify                                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes CoAP socket receive notification.           */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    socket_ptr                            Pointer to the CoAP socket    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_event_flags_set                                                  */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    UDP stack                                                           */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static VOID coap_message_notify(NX_UDP_SOCKET *socket_ptr)
{

    /* Get pointer to LWM2M Client */
    NX_LWM2M_CLIENT *client_ptr = ((NX_LWM2M_CLIENT_SOCKET *) socket_ptr) -> client_ptr;

    /* Set CoAP Message event flag */
    tx_event_flags_set(&client_ptr -> nx_lwm2m_client_event_flags, NX_LWM2M_CLIENT_EVENT_COAP_MESSAGE, TX_OR);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_create                             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function creates a LWM2M Client.                               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    ip_ptr                                Pointer to IP instance        */
/*    packet_pool_ptr                       Pointer to packet pool        */
/*    name_ptr                              Name of LWM2M client          */
/*    name_length                           Length of endpoint name       */
/*    msisdn_ptr                            Optional MSISDN number        */
/*    msisdn_length                         Length of MSISDN number       */
/*    binding_mode                          Supported binding and modes   */
/*    stack_ptr                             Client thread's stack pointer */
/*    stack_size                            Client thread's stack size    */
/*    priority                              Priority of client thread     */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_create                                                     */
/*    tx_mutex_delete                                                     */
/*    tx_event_flags_create                                               */
/*    tx_event_flags_delete                                               */
/*    nx_udp_socket_create                                                */
/*    nx_udp_socket_delete                                                */
/*    nx_udp_socket_bind                                                  */
/*    nx_udp_socket_ubind                                                 */
/*    nx_udp_socket_receive_notify                                        */
/*    _nx_lwm2m_client_security_object_create                             */
/*    _nx_lwm2m_client_server_object_create                               */
/*    _nx_lwm2m_client_access_control_object_create                       */
/*    _nx_lwm2m_client_device_object_create                               */
/*    memset                                                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_create(NX_LWM2M_CLIENT *client_ptr, NX_IP *ip_ptr, NX_PACKET_POOL *packet_pool_ptr,
                             const CHAR *name_ptr, UINT name_length, const CHAR *msisdn_ptr, UINT msisdn_length,
                             UCHAR binding_modes, VOID *stack_ptr, ULONG stack_size, UINT priority)
{
UINT status;
int i;

    /* Initialize the client control block to zero. */
    memset(client_ptr, 0, sizeof(NX_LWM2M_CLIENT));

    /* Create the mutex */
    status = tx_mutex_create(&client_ptr -> nx_lwm2m_client_mutex, "LWM2M Client Mutex", TX_INHERIT);
    if (status != TX_SUCCESS)
    {

        /* Failed to create the mutex, return Failure */
        return(NX_LWM2M_CLIENT_ERROR);
    }

    /* Create the event flags */
    status = tx_event_flags_create(&client_ptr -> nx_lwm2m_client_event_flags, "LWM2M Client Flags");
    if (status != TX_SUCCESS)
    {

        /* Failed to create the event flags, delete all resources */
        tx_mutex_delete(&client_ptr -> nx_lwm2m_client_mutex);

        /* return Failure */
        return(NX_LWM2M_CLIENT_ERROR);
    }

    /* Create the CoAP socket */
    status = nx_udp_socket_create(ip_ptr, (NX_UDP_SOCKET *) &client_ptr -> nx_lwm2m_client_coap_socket, "LWM2M Client CoAP Socket", NX_LWM2M_CLIENT_SOCKET_TOS, NX_DONT_FRAGMENT, NX_LWM2M_CLIENT_SOCKET_TTL, NX_LWM2M_CLIENT_SOCKET_QUEUE_MAX);
    if (status != NX_SUCCESS)
    {

        /* Failed to create the socket, delete all resources */
        tx_event_flags_delete(&client_ptr -> nx_lwm2m_client_event_flags);
        tx_mutex_delete(&client_ptr -> nx_lwm2m_client_mutex);

        /* return Failure */
        return(NX_LWM2M_CLIENT_ERROR);
    }

    /* Bind the CoAP port */
    status = nx_udp_socket_bind((NX_UDP_SOCKET *) &client_ptr -> nx_lwm2m_client_coap_socket, NX_ANY_PORT, NX_NO_WAIT);
    if (status != NX_SUCCESS)
    {

        /* Failed to bind the socket, delete all resources */
        nx_udp_socket_delete((NX_UDP_SOCKET *) &client_ptr -> nx_lwm2m_client_coap_socket);
        tx_event_flags_delete(&client_ptr -> nx_lwm2m_client_event_flags);
        tx_mutex_delete(&client_ptr -> nx_lwm2m_client_mutex);

        /* return port unavailable error */
        return(NX_LWM2M_CLIENT_PORT_UNAVAILABLE);
    }

    /* Set the CoAP receive notify callback */
    client_ptr -> nx_lwm2m_client_coap_socket.client_ptr = client_ptr;
    nx_udp_socket_receive_notify((NX_UDP_SOCKET *) &client_ptr -> nx_lwm2m_client_coap_socket, coap_message_notify);

    /* Store the endpoint name */
    client_ptr -> nx_lwm2m_client_name = name_ptr;
    client_ptr -> nx_lwm2m_client_name_length = name_length;

    /* Store the MSISDN */
    if (msisdn_length)
    {
        client_ptr -> nx_lwm2m_client_msisdn = msisdn_ptr;
        client_ptr -> nx_lwm2m_client_msisdn_length = msisdn_length;
    }

    /* Store the IP instance */
    client_ptr -> nx_lwm2m_client_ip_ptr = ip_ptr;

    /* Store the IP Packet Pool */
    client_ptr -> nx_lwm2m_client_pool_ptr = packet_pool_ptr;

    /* Initialize the Security Object (0) */
#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES
    _nx_lwm2m_client_security_object_create(client_ptr);
#endif

    /* Initialize the Server Object (1) */
    _nx_lwm2m_client_server_object_create(client_ptr);

    /* Initialize the Access Control Object (2) */
#if NX_LWM2M_CLIENT_MAX_ACCESS_CONTROL_INSTANCES
    _nx_lwm2m_client_access_control_object_create(client_ptr);
#endif

    /* Initialize the Device Object (3) */
    _nx_lwm2m_client_device_object_create(client_ptr, binding_modes);

    /* Initialize the linked list of free notifications */
    client_ptr -> nx_lwm2m_client_notifications_free_list = &client_ptr -> nx_lwm2m_client_notifications[0];
    for (i = 0; i < NX_LWM2M_CLIENT_MAX_NOTIFICATIONS-1; i++)
    {
        client_ptr -> nx_lwm2m_client_notifications[i].notify_next = &client_ptr -> nx_lwm2m_client_notifications[i + 1];
    }
    client_ptr -> nx_lwm2m_client_notifications[NX_LWM2M_CLIENT_MAX_NOTIFICATIONS-1].notify_next = NX_NULL;

    /* Initialize empty list of sessions */
    client_ptr -> nx_lwm2m_client_sessions = NX_NULL;

    /* Reset request ID generator */
    /* XXX should be random */
    client_ptr -> nx_lwm2m_client_request_id = 0;

    /* Create the LWM2M Client thread */
    status = tx_thread_create(&client_ptr -> nx_lwm2m_client_thread, "LWM2M Client Thread", _nx_lwm2m_client_thread_entry,
                              (ULONG) client_ptr, stack_ptr, stack_size,
                              priority, priority, TX_NO_TIME_SLICE, TX_AUTO_START);
    if (status != TX_SUCCESS)
    {

        /* Failed to create the thread, delete all resources */
        nx_udp_socket_unbind((NX_UDP_SOCKET *)&client_ptr -> nx_lwm2m_client_coap_socket);
        nx_udp_socket_delete((NX_UDP_SOCKET *)&client_ptr -> nx_lwm2m_client_coap_socket);
        tx_event_flags_delete(&client_ptr -> nx_lwm2m_client_event_flags);
        tx_mutex_delete(&client_ptr -> nx_lwm2m_client_mutex);

        /* Return Failure */
        return(NX_LWM2M_CLIENT_ERROR);
    }

    /* Return NX_SUCCESS */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_delete                             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function deletes a LWM2M Client.                               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_event_flags_set                                                  */
/*    tx_event_flags_get                                                  */
/*    tx_thread_terminate                                                 */
/*    tx_thread_delete                                                    */
/*    nx_udp_socket_unbind                                                */
/*    nx_udp_socket_delete                                                */
/*    tx_event_flags_delete                                               */
/*    tx_mutex_delete                                                     */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_delete(NX_LWM2M_CLIENT *client_ptr)
{
ULONG flags;

    /* Need to terminate the client thread before deleting the LWM2M CLient: */

    /* Set event to tell the client thread to terminate */
    tx_event_flags_set(&client_ptr -> nx_lwm2m_client_event_flags, NX_LWM2M_CLIENT_EVENT_TERM_REQ, TX_OR);

    /* Wait for client thread termination done event */
    tx_event_flags_get(&client_ptr -> nx_lwm2m_client_event_flags, NX_LWM2M_CLIENT_EVENT_TERM_DONE, TX_OR, &flags, TX_WAIT_FOREVER);

    /* Terminate the thread */
    tx_thread_terminate(&client_ptr -> nx_lwm2m_client_thread);

    /* Delete the thread */
    tx_thread_delete(&client_ptr -> nx_lwm2m_client_thread);

    /* Delete all sessions */
    while (client_ptr -> nx_lwm2m_client_sessions != NX_NULL)
    {
        _nx_lwm2m_client_session_delete(client_ptr -> nx_lwm2m_client_sessions);
    }

    /* Unbind and Delete the CoAP socket */
    nx_udp_socket_unbind((NX_UDP_SOCKET *) &client_ptr -> nx_lwm2m_client_coap_socket);
    nx_udp_socket_delete((NX_UDP_SOCKET *) &client_ptr -> nx_lwm2m_client_coap_socket);

    /* Delete the event flags */
    tx_event_flags_delete(&client_ptr -> nx_lwm2m_client_event_flags);

    /* Delete the mutex */
    tx_mutex_delete(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return NX_SUCCESS */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_device_callback_set                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function installs the application callbacks for implementing   */
/*    operations on the LWM2M Device Object resources not handled by the  */
/*    LWM2M Client.                                                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    operation_callback                    The operation callback        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_device_callback_set(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_CLIENT_DEVICE_OPERATION_CALLBACK operation_callback)
{

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Set callbacks pointers */
    client_ptr -> nx_lwm2m_client_device.device_object_user_operation = operation_callback;

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return success */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*   _nx_lwm2m_client_device_error_push                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function adds a new instance to the error code resource of     */
/*    the Device Object.                                                  */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    code                                  The new error code            */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_device_object_error_push                           */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_device_error_push(NX_LWM2M_CLIENT *client_ptr, UCHAR code)
{
UINT status;

    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    status = _nx_lwm2m_client_device_object_error_push(&client_ptr -> nx_lwm2m_client_device, code);

    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_device_error_reset                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function deletes all error code resource instances from the    */
/*    Device Object.                                                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_device_object_error_reset                          */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_device_error_reset(NX_LWM2M_CLIENT *client_ptr)
{

    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    _nx_lwm2m_client_device_object_error_reset(&client_ptr -> nx_lwm2m_client_device);

    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_device_read                        PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the read operation.                         */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    nx_lwm2m_device_object_user_operation                               */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_device_operation                                   */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_device_read(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT *resource_count)
{
NX_LWM2M_CLIENT_DEVICE_OBJECT *device_ptr = (NX_LWM2M_CLIENT_DEVICE_OBJECT *) object_ptr;
NX_LWM2M_CLIENT_RESOURCE *resource_instances = object_ptr -> object_client_ptr -> nx_lwm2m_client_mulitple_temp_resources;
UINT status;
UINT length;
UINT i, j;
UINT num_user_resources;

    NX_PARAMETER_NOT_USED(instance_ptr);

    for (i = 0; i < *resource_count; i++)
    {
        switch (resource[i].resource_id)
        {

        /* Non-readable resources */
        case NX_LWM2M_CLIENT_DEVICE_RESET_ERROR_CODE_ID:

            return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);

        /* Get Error Code */
        case NX_LWM2M_CLIENT_DEVICE_ERROR_CODE_ID:

            for (j = 0; j < device_ptr -> device_object_error_count; j++)
            {

                _nx_lwm2m_client_resource_info_set(&resource_instances[j], (NX_LWM2M_ID)j, NX_NULL);
                _nx_lwm2m_client_resource_integer32_set(&resource_instances[j],  device_ptr -> device_object_error[j]);
            }

            _nx_lwm2m_client_resource_instances_set(&resource[i], resource_instances, device_ptr -> device_object_error_count);
            break;

        /* Get Supported Binding and Modes */
        case NX_LWM2M_CLIENT_DEVICE_BINDING_MODES_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_STRING;
            resource[i].resource_value.resource_bufferdata.resource_buffer_ptr = _nx_lwm2m_client_binding_mode_string_get(device_ptr -> device_object_binding_modes, &length);
            resource[i].resource_value.resource_bufferdata.resource_buffer_length = length;
            break;

        /* Resources implemented by the application */
        default:

            if (device_ptr -> device_object_user_operation == NX_NULL)
            {
                return(NX_LWM2M_CLIENT_NOT_FOUND);
            }
            num_user_resources = 1;
            status = device_ptr -> device_object_user_operation(NX_LWM2M_CLIENT_OBJECT_READ, object_ptr -> object_client_ptr, &resource[i], &num_user_resources, NX_NULL, 0);
            if (status != NX_SUCCESS)
            {
                return(status);
            }
            break;
        }
    }

    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_device_discover                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the discover operation.                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    num_resources_ptr                     Pointer to number of resource */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    nx_lwm2m_device_object_user_operation                               */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_device_operation                                   */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_device_discover(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resources, UINT *resource_count)
{
NX_LWM2M_CLIENT_DEVICE_OBJECT *device_ptr = (NX_LWM2M_CLIENT_DEVICE_OBJECT *) object_ptr;
UINT num_user_resources;
UINT ret;

    NX_PARAMETER_NOT_USED(instance_ptr);

    /* Return the resources implemented by the LWM2M Client */
    if (*resource_count < 3)
    {
        return(NX_LWM2M_CLIENT_BUFFER_TOO_SMALL);
    }

    resources[0].resource_id = NX_LWM2M_CLIENT_DEVICE_ERROR_CODE_ID;
    resources[0].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ;
    resources[0].resource_type = NX_LWM2M_CLIENT_RESOURCE_MULTIPLE;
    resources[0].resource_dim = (UCHAR)device_ptr -> device_object_error_count;
    resources[1].resource_id = NX_LWM2M_CLIENT_DEVICE_RESET_ERROR_CODE_ID;
    resources[1].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_EXECUTABLE;
    resources[2].resource_id = NX_LWM2M_CLIENT_DEVICE_BINDING_MODES_ID;
    resources[2].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ;

    if (device_ptr -> device_object_user_operation == NX_NULL)
    {

        /* No resource implemented by the application */
        num_user_resources = 0;
    }
    else
    {

        /* Add the resources implemented by the application */
        num_user_resources = *resource_count - 3;
        ret = device_ptr -> device_object_user_operation(NX_LWM2M_CLIENT_OBJECT_DISCOVER, object_ptr -> object_client_ptr, &resources[3], &num_user_resources, NX_NULL, 0);
        if (ret != NX_SUCCESS)
        {
            return(ret);
        }
    }

    /* Return total number of resources */
    *resource_count = 3 + num_user_resources;

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_device_write                       PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the write operation.                        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    write_op                              Type of write operation       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    nx_lwm2m_device_object_user_operation                               */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_device_operation                                   */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_device_write(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT *resource_count, VOID *args_ptr, UINT args_length)
{
NX_LWM2M_CLIENT_DEVICE_OBJECT *device_ptr = (NX_LWM2M_CLIENT_DEVICE_OBJECT *) object_ptr;
UINT i;
UINT write_op;

    /* Check the length of argument */
    if (args_length < sizeof(UINT))
        return(NX_LWM2M_CLIENT_BAD_REQUEST);

    /* Get the type of write operation */
    write_op = *(UINT *)args_ptr;

    NX_PARAMETER_NOT_USED(instance_ptr);

    /* XXX replace not supported? */
    if (write_op == NX_LWM2M_CLIENT_OBJECT_WRITE_REPLACE_INSTANCE)
    {
        return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);
    }

    /* Check if resources are valid */
    for (i = 0; i < *resource_count; i++)
    {
        switch (resource[i].resource_id)
        {

        /* Non-writable resources */
        case NX_LWM2M_CLIENT_DEVICE_ERROR_CODE_ID:
        case NX_LWM2M_CLIENT_DEVICE_RESET_ERROR_CODE_ID:
        case NX_LWM2M_CLIENT_DEVICE_BINDING_MODES_ID:

            return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);

        /* Other resources are implemented by the application */
        default:

            break;
        }
    }

    if (write_op == NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP)
    {

        /* Ignore unknown resources during bootstrap */
        return(NX_SUCCESS);
    }

    /* No application resource defined */
    if (device_ptr -> device_object_user_operation == NX_NULL)
    {
        return(NX_LWM2M_CLIENT_NOT_FOUND);
    }

    /* Resources implemented by the application */
    return(device_ptr -> device_object_user_operation(NX_LWM2M_CLIENT_OBJECT_WRITE, object_ptr -> object_client_ptr, resource, resource_count, args_ptr, args_length));
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_device_execute                     PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the execute operation.                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource_id                           The ID of resource            */
/*    args_ptr                              Pointer to arguments          */
/*    args_length                           The length of arguments       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_device_object_error_reset                          */
/*    nx_lwm2m_device_object_user_operation                               */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_device_operation                                   */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_device_execute(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, CHAR *args_ptr, UINT args_length)
{
NX_LWM2M_CLIENT_DEVICE_OBJECT *device_ptr = (NX_LWM2M_CLIENT_DEVICE_OBJECT *) object_ptr;

    NX_PARAMETER_NOT_USED(instance_ptr);

    switch (resource -> resource_id)
    {

    /* Non-executable resources */
    case NX_LWM2M_CLIENT_DEVICE_ERROR_CODE_ID:
    case NX_LWM2M_CLIENT_DEVICE_BINDING_MODES_ID:

        return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);

    /* Reset error code */
    case NX_LWM2M_CLIENT_DEVICE_RESET_ERROR_CODE_ID:

        _nx_lwm2m_client_device_object_error_reset(device_ptr);

        return(NX_SUCCESS);

    /* Resources implemented by the application */
    default:

        if (device_ptr -> device_object_user_operation == NX_NULL)
        {
            return(NX_LWM2M_CLIENT_NOT_FOUND);
        }
        return(device_ptr -> device_object_user_operation(NX_LWM2M_CLIENT_OBJECT_EXECUTE, object_ptr -> object_client_ptr, resource, NX_NULL, args_ptr, args_length));
    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_device_operation                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the object operation.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    operation                             The operation type            */
/*    object_ptr                            Pointer to object             */
/*    object_instance_ptr                   Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    args_ptr                              Pointer to arguments          */
/*    args_length                           The length of arguments       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_device_read                                        */
/*    _nx_lwm2m_client_device_discover                                    */
/*    _nx_lwm2m_client_device_write                                       */
/*    _nx_lwm2m_client_device_execute                                     */
/*    _nx_lwm2m_client_device_write                                       */
/*    _nx_lwm2m_client_device_delete                                      */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_device_operation(UINT operation, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *object_instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT *resource_count, VOID *args_ptr, UINT args_length)
{

    switch (operation)
    {
    case NX_LWM2M_CLIENT_OBJECT_READ:

        /* Call read function */
        return(_nx_lwm2m_client_device_read(object_ptr, object_instance_ptr, resource, resource_count));
    case NX_LWM2M_CLIENT_OBJECT_DISCOVER:

        /* Call discover function */
        return(_nx_lwm2m_client_device_discover(object_ptr, object_instance_ptr, resource, resource_count));
    case NX_LWM2M_CLIENT_OBJECT_WRITE:

        /* Call write function */
        return(_nx_lwm2m_client_device_write(object_ptr, object_instance_ptr, resource, resource_count, args_ptr, args_length));
    case NX_LWM2M_CLIENT_OBJECT_EXECUTE:

        /* Call execute function */
        return(_nx_lwm2m_client_device_execute(object_ptr, object_instance_ptr, resource, args_ptr, args_length));
    default:

        /*Unsupported operation */
        return(NX_LWM2M_CLIENT_NOT_SUPPORTED);

    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_device_object_create               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions initializes the Device Object.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    binding_modes                         The supported binding modes   */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    none                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_add                                         */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_device_object_create(NX_LWM2M_CLIENT *client_ptr, UCHAR binding_modes)
{
NX_LWM2M_CLIENT_DEVICE_OBJECT *device_ptr = &(client_ptr -> nx_lwm2m_client_device);
NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr = &device_ptr -> device_object_instance;

    /* Add Device Object to the LWM2M Client */
    _nx_lwm2m_client_object_add(client_ptr, &(device_ptr -> device_object), NX_LWM2M_CLIENT_DEVICE_OBJECT_ID, _nx_lwm2m_client_device_operation);

    /* Initialize Object data */
    device_ptr -> device_object.object_instances    = instance_ptr;
    device_ptr -> device_object_user_operation               = NX_NULL;
    device_ptr -> device_object_error_count                  = 1;
    device_ptr -> device_object_error[0]                     = 0;
    device_ptr -> device_object_binding_modes                = binding_modes;

    /* Initialize the single Instance */
    instance_ptr -> object_instance_next = NX_NULL;
    instance_ptr -> object_instance_id   = 0;
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_device_object_error_push           PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions pushes a new value to the array of device errors.    */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    device_ptr                            Pointer to device object      */
/*    code                                  The error code                */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_resource_changed                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_device_error_push                                  */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_device_object_error_push(NX_LWM2M_CLIENT_DEVICE_OBJECT *device_ptr, UCHAR code)
{
NX_LWM2M_CLIENT_RESOURCE value;

    if (device_ptr -> device_object_error_count == 1 && device_ptr -> device_object_error[0] == 0)
    {

        /* Replace the "no error" code */
        device_ptr -> device_object_error[0] = code;
    }
    else if (device_ptr -> device_object_error_count < NX_LWM2M_CLIENT_MAX_DEVICE_ERRORS)
    {

        /* Push new error code */
        device_ptr -> device_object_error[device_ptr -> device_object_error_count] = code;
        device_ptr -> device_object_error_count++;
    }
    else
    {
        return(NX_LWM2M_CLIENT_BUFFER_TOO_SMALL);
    }

    /* Handle notifications */
    value.resource_id = NX_LWM2M_CLIENT_DEVICE_ERROR_CODE_ID;
    value.resource_type = NX_LWM2M_CLIENT_RESOURCE_MULTIPLE;
    _nx_lwm2m_client_object_resource_changed((NX_LWM2M_CLIENT_OBJECT *) device_ptr, &device_ptr -> device_object_instance, &value);

    /* Return NX_SUCCESS */
    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_device_object_error_reset          PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function resets the array of device errors.                    */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    device_ptr                            Pointer to device object      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    none                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_resource_changed                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_device_execute                                     */
/*    _nx_lwm2m_client_device_error_reset                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_device_object_error_reset(NX_LWM2M_CLIENT_DEVICE_OBJECT *device_ptr)
{
    if (device_ptr -> device_object_error_count != 1 || device_ptr -> device_object_error[0] != 0)
    {
        NX_LWM2M_CLIENT_RESOURCE value;

        /* Reset to no error code */
        device_ptr -> device_object_error_count  = 1;
        device_ptr -> device_object_error[0]     = 0;

        /* Handle notifications */
        value.resource_id = NX_LWM2M_CLIENT_DEVICE_ERROR_CODE_ID;
        value.resource_type = NX_LWM2M_CLIENT_RESOURCE_MULTIPLE;
        _nx_lwm2m_client_object_resource_changed((NX_LWM2M_CLIENT_OBJECT *) device_ptr, &device_ptr -> device_object_instance, &value);
    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_device_resource_changed            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function signals the LWM2M Client that a resource of the       */
/*    Device Object has changed.                                          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    resource_ptr                          The resource ID and value     */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_resource_changed                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_device_resource_changed(NX_LWM2M_CLIENT *client_ptr, const NX_LWM2M_CLIENT_RESOURCE *resource_ptr)
{
    return(_nx_lwm2m_client_object_resource_changed((NX_LWM2M_CLIENT_OBJECT *) &client_ptr -> nx_lwm2m_client_device, &client_ptr -> nx_lwm2m_client_device.device_object_instance, resource_ptr));
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_read                      PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the read operation.                         */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_operation                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_firmware_read(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT resource_count)
{
NX_LWM2M_CLIENT_FIRMWARE *firmware_ptr = (NX_LWM2M_CLIENT_FIRMWARE *) object_ptr;
NX_LWM2M_CLIENT_RESOURCE *resource_instances = object_ptr -> object_client_ptr -> nx_lwm2m_client_mulitple_temp_resources;
UINT i, j;

    NX_PARAMETER_NOT_USED(instance_ptr);

    for (i = 0; i < resource_count; i++)
    {
        switch (resource[i].resource_id)
        {

        case NX_LWM2M_CLIENT_FIRMWARE_STATE_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = firmware_ptr -> firmware_state;
            break;

        case NX_LWM2M_CLIENT_FIRMWARE_UPDATE_SUPPORTED_OBJECTS_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_BOOLEAN;
            resource[i].resource_value.resource_booleandata = firmware_ptr -> firmware_update_supported_objects;
            break;

        case NX_LWM2M_CLIENT_FIRMWARE_UPDATE_RESULT_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = firmware_ptr -> firmware_result;
            break;

        case NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_NAME_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_STRING;
            resource[i].resource_value.resource_bufferdata.resource_buffer_ptr = firmware_ptr -> firmware_package_name;
            resource[i].resource_value.resource_bufferdata.resource_buffer_length = firmware_ptr -> firmware_package_name_length;
            break;

        case NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_VERSION_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_STRING;
            resource[i].resource_value.resource_bufferdata.resource_buffer_ptr = firmware_ptr -> firmware_package_version;
            resource[i].resource_value.resource_bufferdata.resource_buffer_length = firmware_ptr -> firmware_package_version_length;
            break;

        case NX_LWM2M_CLIENT_FIRMWARE_PROTOCOL_SUPPORT_ID:

            for (j = 0; j < firmware_ptr -> firmware_protocols_count; j++)
            {

                _nx_lwm2m_client_resource_info_set(&resource_instances[j], (NX_LWM2M_ID)j, NX_NULL);
                _nx_lwm2m_client_resource_integer32_set(&resource_instances[j],  firmware_ptr -> firmware_protocols[j]);
            }

            _nx_lwm2m_client_resource_instances_set(&resource[i], resource_instances, firmware_ptr -> firmware_protocols_count);
            break;

        case NX_LWM2M_CLIENT_FIRMWARE_DELIVERY_METHOD_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;

            /* Return which kinds of firmware transfert are supported */
            if ((firmware_ptr -> firmware_package_callback != NX_NULL) &&
                (firmware_ptr -> firmware_package_uri_callback != NX_NULL))
            {

                /* Both are supported */
                resource[i].resource_value.resource_integer32data = 2;
            }
            else if (firmware_ptr -> firmware_package_callback != NX_NULL)
            {

                /* Push only */
                resource[i].resource_value.resource_integer32data = 1;
            }
            else
            {

                /* Pull only */
                resource[i].resource_value.resource_integer32data = 0;
            }
            break;

        /* Non-readable resources */
        case NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_ID:
        case NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_URI_ID:
        case NX_LWM2M_CLIENT_FIRMWARE_UPDATE_ID:

            return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);

        /* Unknown resource */
        default:

            return(NX_LWM2M_CLIENT_NOT_FOUND);
        }
    }

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_discover                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the discover operation.                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    num_resources_ptr                     Pointer to number of resource */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_operation                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_firmware_discover(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resources, UINT *num_resources_ptr)
{
NX_LWM2M_CLIENT_FIRMWARE *firmware_ptr = (NX_LWM2M_CLIENT_FIRMWARE *) object_ptr;
UINT resource_count;

    NX_PARAMETER_NOT_USED(instance_ptr);

    /* Get number of resources */
    if (firmware_ptr -> firmware_protocols_count != 0)
    {
        resource_count = 10;
    }
    else
    {
        resource_count = 9;
    }

    /* Check length of output buffer */
    if (*num_resources_ptr < resource_count)
    {
        return(NX_LWM2M_CLIENT_BUFFER_TOO_SMALL);
    }

    /* Return list of resources */
    resources[0].resource_id        = NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_ID;
    resources[0].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_EXECUTABLE;    /* Write only */
    resources[1].resource_id        = NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_URI_ID;
    resources[1].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_EXECUTABLE;    /* Write only */
    resources[2].resource_id        = NX_LWM2M_CLIENT_FIRMWARE_UPDATE_ID;
    resources[2].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_EXECUTABLE;
    resources[3].resource_id        = NX_LWM2M_CLIENT_FIRMWARE_STATE_ID;
    resources[3].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ;
    resources[4].resource_id        = NX_LWM2M_CLIENT_FIRMWARE_UPDATE_SUPPORTED_OBJECTS_ID;
    resources[4].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ_WRITE;
    resources[5].resource_id        = NX_LWM2M_CLIENT_FIRMWARE_UPDATE_RESULT_ID;
    resources[5].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ;
    resources[6].resource_id        = NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_NAME_ID;
    resources[6].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ;
    resources[7].resource_id        = NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_VERSION_ID;
    resources[7].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ;
    if (firmware_ptr -> firmware_protocols_count != 0)
    {
        resources[8].resource_id        = NX_LWM2M_CLIENT_FIRMWARE_PROTOCOL_SUPPORT_ID;
        resources[8].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ;
        resources[8].resource_type      = NX_LWM2M_CLIENT_RESOURCE_MULTIPLE;
        resources[8].resource_dim       = firmware_ptr -> firmware_protocols_count;
    }
    resources[resource_count-1].resource_id        = NX_LWM2M_CLIENT_FIRMWARE_DELIVERY_METHOD_ID;
    resources[resource_count-1].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ;

    *num_resources_ptr = resource_count;

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_write                     PORTABLE C      */
/*                                                           6.1.10       */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the write operation.                        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    write_op                              Type of write operation       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_string_get                                */
/*    _nx_lwm2m_client_resource_boolean_get                               */
/*    nx_lwm2m_firmware_package_uri_callback                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_operation                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*  01-31-2022     Yuxin Zhou               Modified comment(s), fixed    */
/*                                            compiler warnings,          */
/*                                            resulting in version 6.1.10 */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_firmware_write(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT resource_count, UINT write_op)
{
NX_LWM2M_CLIENT_FIRMWARE *firmware_ptr = (NX_LWM2M_CLIENT_FIRMWARE *) object_ptr;
UINT status;
UINT flags;
const CHAR *uri_ptr = NX_NULL;
UINT uri_length;
NX_LWM2M_BOOL update_supported_objects, old_update_value = 0;
UINT i;

    NX_PARAMETER_NOT_USED(instance_ptr);

    /* Replace or create are not supported */
    if (write_op == NX_LWM2M_CLIENT_OBJECT_WRITE_REPLACE_INSTANCE || write_op == NX_LWM2M_CLIENT_OBJECT_WRITE_CREATE)
    {
        return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);
    }

    /* Check validity of values before updating instance */
    flags = 0;

    for (i = 0 ; i < resource_count; i++)
    {
        switch (resource[i].resource_id)
        {

        case NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_ID:

            if (write_op == NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP)
            {

                /* XXX cannot download package during bootstrap? */
                return(NX_LWM2M_CLIENT_BAD_REQUEST);
            }

            /* XXX TO DO */
            return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);

        case NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_URI_ID:

            if (write_op == NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP)
            {

                /* XXX cannot download package during bootstrap? */
                return(NX_LWM2M_CLIENT_BAD_REQUEST);
            }
            if (firmware_ptr -> firmware_package_uri_callback == NX_NULL)
            {

                /* Not supported */
                return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
            }

            /* Get URI string */
            status =  _nx_lwm2m_client_resource_string_get(&resource[i], &uri_ptr, &uri_length);
            if (status != NX_SUCCESS)
            {
                return(status);
            }
            flags |= 1<<NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_URI_ID;
            break;

        case NX_LWM2M_CLIENT_FIRMWARE_UPDATE_SUPPORTED_OBJECTS_ID:

            status = _nx_lwm2m_client_resource_boolean_get(&resource[i], &update_supported_objects);
            if (status != NX_SUCCESS)
            {
                return(status);
            }
            flags |= 1<<NX_LWM2M_CLIENT_FIRMWARE_UPDATE_SUPPORTED_OBJECTS_ID;
            break;

        /* No-writeable resources */
        case NX_LWM2M_CLIENT_FIRMWARE_UPDATE_ID:
        case NX_LWM2M_CLIENT_FIRMWARE_STATE_ID:
        case NX_LWM2M_CLIENT_FIRMWARE_UPDATE_RESULT_ID:
        case NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_NAME_ID:
        case NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_VERSION_ID:
        case NX_LWM2M_CLIENT_FIRMWARE_PROTOCOL_SUPPORT_ID:
        case NX_LWM2M_CLIENT_FIRMWARE_DELIVERY_METHOD_ID:

            return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);

        /* Unknown resource */
        default:

            if (write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP)
            {
                return(NX_LWM2M_CLIENT_NOT_FOUND);
            }

            /* Ignore unknown resources during bootstrap */
            break;
        }
    }

    /* Update the object */
    if (flags & 1<<NX_LWM2M_CLIENT_FIRMWARE_UPDATE_SUPPORTED_OBJECTS_ID)
    {
        old_update_value = firmware_ptr -> firmware_update_supported_objects;
        firmware_ptr -> firmware_update_supported_objects = update_supported_objects;
    }
    if (flags & 1<<NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_URI_ID)
    {
        status = firmware_ptr -> firmware_package_uri_callback(firmware_ptr, uri_ptr, uri_length);

        if (status != NX_SUCCESS)
        {
            if (flags & 1<<NX_LWM2M_CLIENT_FIRMWARE_UPDATE_SUPPORTED_OBJECTS_ID)
            {

                /* XXX restore previous state as we return an error */
                firmware_ptr -> firmware_update_supported_objects = old_update_value;
            }

            return(status);
        }
    }

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_execute                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the execute operation.                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource_id                           The ID of resource            */
/*    args_ptr                              Pointer to arguments          */
/*    args_length                           The length of arguments       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    nx_lwm2m_firmware_update_callback                                   */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_operation                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_firmware_execute(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_ID resource_id, const CHAR *args_ptr, UINT args_length)
{
NX_LWM2M_CLIENT_FIRMWARE *firmware_ptr = (NX_LWM2M_CLIENT_FIRMWARE *) object_ptr;

    NX_PARAMETER_NOT_USED(instance_ptr);

    switch (resource_id)
    {

    case NX_LWM2M_CLIENT_FIRMWARE_UPDATE_ID:

        return(firmware_ptr -> firmware_update_callback(firmware_ptr, firmware_ptr -> firmware_update_supported_objects, args_ptr, args_length));

    /* Non-executable resources */
    case NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_ID:
    case NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_URI_ID:
    case NX_LWM2M_CLIENT_FIRMWARE_UPDATE_SUPPORTED_OBJECTS_ID:
    case NX_LWM2M_CLIENT_FIRMWARE_STATE_ID:
    case NX_LWM2M_CLIENT_FIRMWARE_UPDATE_RESULT_ID:
    case NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_NAME_ID:
    case NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_VERSION_ID:
    case NX_LWM2M_CLIENT_FIRMWARE_PROTOCOL_SUPPORT_ID:
    case NX_LWM2M_CLIENT_FIRMWARE_DELIVERY_METHOD_ID:

        return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);

    /* Unknown resource */
    default:

        return(NX_LWM2M_CLIENT_NOT_FOUND);
    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_operation                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the object operation.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    operation                             The operation type            */
/*    object_ptr                            Pointer to object             */
/*    object_instance_ptr                   Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    args_ptr                              Pointer to arguments          */
/*    args_length                           The length of arguments       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_read                                      */
/*    _nx_lwm2m_client_firmware_discover                                  */
/*    _nx_lwm2m_client_firmware_write                                     */
/*    _nx_lwm2m_client_firmware_execute                                   */
/*    _nx_lwm2m_client_firmware_write                                     */
/*    _nx_lwm2m_client_firmware_delete                                    */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_operation                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_firmware_operation(UINT operation, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *object_instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT *resource_count, VOID *args_ptr, UINT args_length)
{
UINT *write_op;

    switch (operation)
    {
    case NX_LWM2M_CLIENT_OBJECT_READ:

        /* Call read function */
        return(_nx_lwm2m_client_firmware_read(object_ptr, object_instance_ptr, resource, *resource_count));
    case NX_LWM2M_CLIENT_OBJECT_DISCOVER:

        /* Call discover function */
        return(_nx_lwm2m_client_firmware_discover(object_ptr, object_instance_ptr, resource, resource_count));
    case NX_LWM2M_CLIENT_OBJECT_WRITE:

        /* Check the length of argument */
        if (args_length < sizeof(UINT))
            return(NX_LWM2M_CLIENT_BAD_REQUEST);

        /* Get the type of write operation */
        write_op = (UINT *)args_ptr;

        /* Call write function */
        return(_nx_lwm2m_client_firmware_write(object_ptr, object_instance_ptr, resource, *resource_count, *write_op));
    case NX_LWM2M_CLIENT_OBJECT_EXECUTE:

        /* Call execute function */
        return(_nx_lwm2m_client_firmware_execute(object_ptr, object_instance_ptr, resource -> resource_id, args_ptr, args_length));
    default:

        /*Unsupported operation */
        return(NX_LWM2M_CLIENT_NOT_SUPPORTED);

    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_create                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function initializes the Firmware Update Object.               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    firmware_ptr                          Pointer to object data        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    protocols                             The supported protocols       */
/*    package_callback                      The package callback          */
/*    package_uri_callback                  The package URI callback      */
/*    update_callback                       The update callback           */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_add                                         */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_firmware_create(NX_LWM2M_CLIENT_FIRMWARE *firmware_ptr, NX_LWM2M_CLIENT *client_ptr, UINT protocols, NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_CALLBACK package_callback, NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_URI_CALLBACK package_uri_callback, NX_LWM2M_CLIENT_FIRMWARE_UPDATE_CALLBACK update_callback)
{
int i;
UINT status;

    /* Add Firmware Update Object to the LWM2M Client */
    status = _nx_lwm2m_client_object_add(client_ptr, &(firmware_ptr -> firmware_object), NX_LWM2M_CLIENT_FIRMWARE_OBJECT_ID, _nx_lwm2m_client_firmware_operation);
    if (status)
    {
        return(status);
    }

    /* Initialize Object data */
    firmware_ptr -> firmware_object.object_instances    = &firmware_ptr -> firmware_instance;
    firmware_ptr -> firmware_package_callback                    = package_callback;
    firmware_ptr -> firmware_package_uri_callback                = package_uri_callback;
    firmware_ptr -> firmware_update_callback                     = update_callback;
    firmware_ptr -> firmware_state                               = NX_LWM2M_CLIENT_FIRMWARE_STATE_IDLE;
    firmware_ptr -> firmware_update_supported_objects            = NX_FALSE;
    firmware_ptr -> firmware_result                              = NX_LWM2M_CLIENT_FIRMWARE_RESULT_INIT;
    firmware_ptr -> firmware_protocols_count                     = 0;
    firmware_ptr -> firmware_package_name                        = "";
    firmware_ptr -> firmware_package_name_length                 = 0;
    firmware_ptr -> firmware_package_version                     = "";
    firmware_ptr -> firmware_package_version_length              = 0;

    /* Initialize the single Instance */
    firmware_ptr -> firmware_instance.object_instance_next  = NX_NULL;
    firmware_ptr -> firmware_instance.object_instance_id    = 0;

    /* Set the list of supported protocols */
    protocols &= (1<<NX_LWM2M_CLIENT_FIRMWARE_MAX_PROTOCOLS) - 1;
    i = 0;
    while (protocols != 0)
    {
        if (protocols & 1)
        {

            /* Add protocol */
            firmware_ptr -> firmware_protocols[firmware_ptr -> firmware_protocols_count] = (UCHAR)i;
            firmware_ptr -> firmware_protocols_count++;
        }

        /* Check next bit */
        protocols >>= 1;
        i++;
    }

    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_package_info_set          PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the package information resources of the         */
/*    Firmware Update Object.                                             */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    firmware_ptr                          Pointer to object data        */
/*    name                                  The name of firmware package  */
/*    name_length                           The length of name            */
/*    version                               The version of firmware       */
/*                                          package                       */
/*    version_length                        The length of version         */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_object_resource_changed                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_firmware_package_info_set(NX_LWM2M_CLIENT_FIRMWARE *firmware_ptr, const CHAR *name, UINT name_length, const CHAR *version, UINT version_length)
{
NX_LWM2M_CLIENT_RESOURCE resource;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&((NX_LWM2M_CLIENT_OBJECT *) firmware_ptr) -> object_client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Update the name and version of the firmware package */
    firmware_ptr -> firmware_package_name = name;
    firmware_ptr -> firmware_package_name_length = name_length;
    firmware_ptr -> firmware_package_version = version;
    firmware_ptr -> firmware_package_version_length = version_length;

    /* Notify the server of resources changes */
    resource.resource_id = NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_NAME_ID;
    resource.resource_type = NX_LWM2M_CLIENT_RESOURCE_NONE;
    _nx_lwm2m_client_object_resource_changed((NX_LWM2M_CLIENT_OBJECT *) firmware_ptr, &firmware_ptr -> firmware_instance, &resource);
    resource.resource_id = NX_LWM2M_CLIENT_FIRMWARE_PACKAGE_VERSION_ID;
    _nx_lwm2m_client_object_resource_changed((NX_LWM2M_CLIENT_OBJECT *) firmware_ptr, &firmware_ptr -> firmware_instance, &resource);

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&((NX_LWM2M_CLIENT_OBJECT *) firmware_ptr) -> object_client_ptr -> nx_lwm2m_client_mutex);

    /* Return NX_SUCCESS */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_result_set                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions sets the Update Result resource of the Firmware      */
/*    Update Object.                                                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    firmware_ptr                          Pointer to object data        */
/*    result                                The Update Result             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_object_resource_changed                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_firmware_result_set(NX_LWM2M_CLIENT_FIRMWARE *firmware_ptr, UCHAR result)
{

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&((NX_LWM2M_CLIENT_OBJECT *) firmware_ptr) -> object_client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Set the new value of the update result */
    if (firmware_ptr -> firmware_result != result)
    {
        NX_LWM2M_CLIENT_RESOURCE resource;

        firmware_ptr -> firmware_result = result;

        resource.resource_id = NX_LWM2M_CLIENT_FIRMWARE_UPDATE_RESULT_ID;
        resource.resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
        resource.resource_value.resource_integer32data = result;
        _nx_lwm2m_client_object_resource_changed((NX_LWM2M_CLIENT_OBJECT *) firmware_ptr, &firmware_ptr -> firmware_instance, &resource);
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&((NX_LWM2M_CLIENT_OBJECT *) firmware_ptr) -> object_client_ptr -> nx_lwm2m_client_mutex);

    /* Return NX_SUCCESS */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_firmware_state_set                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions sets the State of the Firmware Update Object.        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    firmware_ptr                          Pointer to object data        */
/*    state                                 The state of the Firmware     */
/*                                          Object                        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_object_resource_changed                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_firmware_state_set(NX_LWM2M_CLIENT_FIRMWARE *firmware_ptr, UCHAR state)
{

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&((NX_LWM2M_CLIENT_OBJECT *) firmware_ptr) -> object_client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Set the new value of the state */
    if (firmware_ptr -> firmware_state != state)
    {
        NX_LWM2M_CLIENT_RESOURCE resource;

        firmware_ptr -> firmware_state = state;

        resource.resource_id = NX_LWM2M_CLIENT_FIRMWARE_STATE_ID;
        resource.resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
        resource.resource_value.resource_integer32data = state;
        _nx_lwm2m_client_object_resource_changed((NX_LWM2M_CLIENT_OBJECT *) firmware_ptr, &firmware_ptr -> firmware_instance, &resource);
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&((NX_LWM2M_CLIENT_OBJECT *) firmware_ptr) -> object_client_ptr -> nx_lwm2m_client_mutex);

    /* Return NX_SUCCESS */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_lock                               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function locks the LWM2M Client to prevent concurent access.   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_lock(NX_LWM2M_CLIENT *client_ptr)
{

    /* Acquire the LWM2M Client mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Always return a success. */
    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_notify_allocate                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function allocates a NX_LWM2M_CLIENT_NOTIFY structure.         */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the allocated structure                                  */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_notify_get                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
NX_LWM2M_CLIENT_NOTIFY * _nx_lwm2m_client_notify_allocate(NX_LWM2M_CLIENT *client_ptr)
{
NX_LWM2M_CLIENT_NOTIFY *notify_ptr;

    /* Dequeue notify structure from free list */
    notify_ptr = client_ptr -> nx_lwm2m_client_notifications_free_list;
    if (notify_ptr != NX_NULL)
    {
        client_ptr -> nx_lwm2m_client_notifications_free_list = notify_ptr -> notify_next;
    }

    return(notify_ptr);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_notify_free                        PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function deallocates a NX_LWM2M_CLIENT_NOTIFY structure.       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    notify_ptr                            Pointer to the structure to   */
/*                                          be deallocated                */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_cleanup                                    */
/*    _nx_lwm2m_client_session_notify_free                                */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_notify_free(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_CLIENT_NOTIFY *notify_ptr)
{

    /* Enqueue the notify structure in free list */
    notify_ptr -> notify_next = client_ptr -> nx_lwm2m_client_notifications_free_list;
    client_ptr -> nx_lwm2m_client_notifications_free_list = notify_ptr;
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*   _nx_lwm2m_client_object_add                          PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function adds a new Object to the LWM2M Client.                */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_ptr                            Pointer to the Object         */
/*    object_id                             The Object ID                 */
/*    object_operation                      Object operation function     */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*    _nx_lwm2m_client_security_object_create                             */
/*    _nx_lwm2m_client_server_object_create                               */
/*    _nx_lwm2m_client_access_control_object_create                       */
/*    _nx_lwm2m_client_device_object_create                               */
/*    _nx_lwm2m_client_firmware_create                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_add(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_ID object_id, 
                                 NX_LWM2M_CLIENT_OBJECT_OPERATION_CALLBACK object_operation)
{
UINT status;
NX_LWM2M_CLIENT_OBJECT *current_ptr;

    /* Reserved value is not allowed */
    if (object_id == NX_LWM2M_CLIENT_RESERVED_ID)
        return(NX_LWM2M_CLIENT_FORBIDDEN);

    /* Set the ID of the new Object */
    object_ptr -> object_id = object_id;

    /* Set operation callback functions */
    object_ptr -> object_operation = object_operation;

    /* Initialize the instances */
    object_ptr -> object_instances = NX_NULL;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Check if this object is the head of object list.  */
    if (client_ptr -> nx_lwm2m_client_object_list_head == NX_NULL)
    {
        object_ptr -> object_client_ptr = client_ptr;
        object_ptr -> object_next = NX_NULL;
        client_ptr -> nx_lwm2m_client_object_list_head = object_ptr;
        status = NX_SUCCESS;
    }
    else
    {

        /* Get pointer to first object */
        current_ptr = client_ptr -> nx_lwm2m_client_object_list_head;

        /* Parse the list of Objects to insert the new object in ID order */
        while (1)
        {
            if ((current_ptr -> object_next == NX_NULL) || 
                (current_ptr -> object_next -> object_id > object_id))
            {

                /* Insert new object in list */
                object_ptr -> object_next = current_ptr -> object_next;
                current_ptr -> object_next = object_ptr;

                /* Set pointer to client */
                object_ptr -> object_client_ptr = client_ptr;

                /* Return Success */
                status = NX_SUCCESS;

                break;
            }
            else if (current_ptr -> object_next -> object_id == object_id)
            {

                /* Duplicated Object ID */
                status = NX_LWM2M_CLIENT_ALREADY_EXIST;

                break;
            }

            /* Get next object */
            current_ptr = current_ptr -> object_next;
        }
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return the status of the operation */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_create                      PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function creates a new Object Instance.                        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id_ptr                       Pointer to the Instance ID    */
/*    num_values                            Number of values to set       */
/*    values                                Pointer to the values         */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_object_ptr_get                                     */
/*    _nx_lwm2m_client_object_create_internal                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_create(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID *instance_id_ptr, UINT num_values, const NX_LWM2M_CLIENT_RESOURCE *values)
{
UINT status;
NX_LWM2M_CLIENT_OBJECT *object_ptr;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Get pointer to the Object implementation */
    object_ptr = _nx_lwm2m_client_object_ptr_get(client_ptr, object_id);
    if (object_ptr == NX_NULL)
    {

        /* Object doesn't exist */
        status = NX_LWM2M_CLIENT_NOT_FOUND;
    }
    else
    {

        /* Create new instance */
        status = _nx_lwm2m_client_object_create_internal(object_ptr, instance_id_ptr, num_values, values, NX_FALSE);
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return the status of the operation */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_create_internal             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function creates a new Object Instance.                        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to the Object         */
/*    instance_id_ptr                       Pointer to the Instance ID    */
/*    num_values                            Number of values to set       */
/*    values                                Pointer to the values         */
/*    bootstrap                             Set to NX_TRUE if called      */
/*                                          during bootstrap              */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_object_create                                      */
/*    _nx_lwm2m_client_session_receive                                    */
/*    _nx_lwm2m_client_session_start                                      */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_create_internal(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_ID *instance_id_ptr, UINT num_values, const NX_LWM2M_CLIENT_RESOURCE *values, NX_LWM2M_BOOL bootstrap)
{
UINT status;
NX_LWM2M_CLIENT_OPERATION_ARGUMENT arguments;
UINT args_len = sizeof(NX_LWM2M_ID);

    if (object_ptr -> object_operation == NX_NULL)
    {

        /* Creation of Instances is not supported */
        return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);
    }

    /* Construct the arguments */
    if (instance_id_ptr == NX_NULL)
    {

        /* Assign an ID if not provided */
        arguments.instance_id = NX_LWM2M_CLIENT_RESERVED_ID;
    }
    else
    {

        /* Get the instance ID to create */
        arguments.instance_id = *instance_id_ptr;
    }

    /* Set bootstrap flag for the internal Objects */
    if (object_ptr -> object_id <= NX_LWM2M_CLIENT_ACCESS_CONTROL_OBJECT_ID)
    {
        arguments.bootstrap = bootstrap;
        args_len = sizeof(NX_LWM2M_CLIENT_OPERATION_ARGUMENT);
    }

    /* Create the new instance */
    status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_CREATE, object_ptr, NX_NULL, (NX_LWM2M_CLIENT_RESOURCE *)values, &num_values, &arguments, args_len);
    if (status != NX_SUCCESS)
    {

        /* Failed to create the instance */
        return(status);
    }

    /* Return assigned ID */
    if (instance_id_ptr != NX_NULL)
    {
        *instance_id_ptr = arguments.instance_id;
    }

    /* Return success */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_delete_all                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function deletes all objects instances of the LWM2M client     */
/*    excepts the given security instance.                                */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    del_object_ptr                        The single object to delete,  */
/*                                          if NULL delete all objects    */
/*    bs_id                                 The ID of the security        */
/*                                          instance to preserve          */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    none                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_receive                                    */
/*    _nx_lwm2m_client_session_start                                      */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_object_delete_all(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_CLIENT_OBJECT *del_object_ptr, NX_LWM2M_ID bs_id)
{
NX_LWM2M_CLIENT_OBJECT *object_ptr;
#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES
NX_LWM2M_CLIENT_OBJECT_INSTANCE *security_ptr;
NX_LWM2M_CLIENT_OBJECT_INSTANCE *prev_ptr;
#else

    NX_PARAMETER_NOT_USED(bs_id);
#endif /* NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES*/


#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES
    if ((bs_id != NX_LWM2M_CLIENT_RESERVED_ID) && ((del_object_ptr == NX_NULL) || (del_object_ptr == (NX_LWM2M_CLIENT_OBJECT *) &client_ptr -> nx_lwm2m_client_security)))
    {

        /* Do not delete the security instance of the bootstrap server */
        prev_ptr = NX_NULL;
        security_ptr = ((NX_LWM2M_CLIENT_OBJECT *) &client_ptr -> nx_lwm2m_client_security) -> object_instances;
        while (security_ptr != NX_NULL)
        {
            if (security_ptr -> object_instance_id == bs_id)
            {

                /* Detach instance from list */
                if (prev_ptr == NX_NULL)
                {
                    ((NX_LWM2M_CLIENT_OBJECT *) &client_ptr -> nx_lwm2m_client_security) -> object_instances = security_ptr -> object_instance_next;
                }
                else
                {
                    prev_ptr -> object_instance_next = security_ptr -> object_instance_next;
                }
                security_ptr -> object_instance_next = NX_NULL;

                break;
            }

            /* Check next instance */
            prev_ptr = security_ptr;
            security_ptr = security_ptr -> object_instance_next;
        }
    }
    else
    {
        security_ptr = NX_NULL;
    }
#endif

    if (del_object_ptr != NX_NULL)
    {

        /* Delete instances of single object */
        object_ptr = del_object_ptr;
    }
    else
    {

        /* Get first object to delete */
        object_ptr = client_ptr -> nx_lwm2m_client_object_list_head;
    }

    while (object_ptr != NX_NULL)
    {

        /* Delete all instances */
        while (object_ptr -> object_instances != NX_NULL)
        {
            if (object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_DELETE, object_ptr, object_ptr -> object_instances, NX_NULL, NX_NULL, NX_NULL, 0) != NX_SUCCESS)
            {
                break;
            }
        }

        if (del_object_ptr != NX_NULL)
        {

            /* Delete a single object */
            break;
        }

        /* Get next object */
        object_ptr = object_ptr -> object_next;
    }

#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES

    /* Restore the security instance of the bootstrap server */
    if (security_ptr != NX_NULL)
    {
        ((NX_LWM2M_CLIENT_OBJECT *)&client_ptr -> nx_lwm2m_client_security) -> object_instances = security_ptr;
    }
#endif
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_delete                      PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function deletes an Object Instance.                           */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id                           The Instance ID               */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_server_session_get                                 */
/*    _nx_lwm2m_client_session_state_update                               */
/*    nx_lwm2m_client_session_state_callback                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_delete(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id)
{
UINT status;
NX_LWM2M_CLIENT_OBJECT *object_ptr;
NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr = NX_NULL;
NX_LWM2M_CLIENT_OBJECT_INSTANCE *server_instance_ptr;
NX_LWM2M_CLIENT_SESSION *session_ptr;


    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Get pointer to the Object implementation */
    object_ptr = _nx_lwm2m_client_object_ptr_get(client_ptr, object_id);
    if (object_ptr != NX_NULL)
    {
        instance_ptr = _nx_lwm2m_client_object_instance_ptr_get(object_ptr, instance_id);
    }
    if ((object_ptr == NX_NULL) || (instance_ptr == NX_NULL))
    {

        /* Object doesn't exist */
        status = NX_LWM2M_CLIENT_NOT_FOUND;
    }
    else
    {

        /* save pointer to instance if it's a server object */
        if (object_ptr -> object_id == NX_LWM2M_CLIENT_SERVER_OBJECT_ID)
        {
            server_instance_ptr = instance_ptr;
        }
        else
        {
            server_instance_ptr = NX_NULL;
        }

        /* delete the instance */
        status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_DELETE, object_ptr, instance_ptr, NX_NULL, NX_NULL, NX_NULL, 0);

        /* Terminate lwm2m session if it's server instance is deleted */
        if ((server_instance_ptr != NX_NULL) && (status == NX_SUCCESS))
        {
            session_ptr = _nx_lwm2m_client_server_session_get(client_ptr, server_instance_ptr);

            if (session_ptr != NX_NULL)
            {

                /* Clear pointer to deleted instance */
                session_ptr -> nx_lwm2m_client_session_server_instance_ptr = NX_NULL;

                if (session_ptr -> nx_lwm2m_client_session_state != NX_LWM2M_CLIENT_SESSION_ERROR)
                {

                    /* Change session state to error */
                    session_ptr -> nx_lwm2m_client_session_error = NX_LWM2M_CLIENT_SERVER_INSTANCE_DELETED;
                    _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_ERROR);

                    /* Report state change to application */
                    session_ptr -> nx_lwm2m_client_session_state_callback(session_ptr, session_ptr -> nx_lwm2m_client_session_state);
                }
            }
        }
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return the status of the operation */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_discover                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the list of resources implemented by an       */
/*    Object.                                                             */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id                           The Instance ID               */
/*    num_resources                         Pointer to the number of      */
/*                                          resources                     */
/*    resources                             Pointer to the resource info  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_object_ptr_get                                     */
/*    _nx_lwm2m_client_object_instance_ptr_get                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_discover(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id, UINT *num_resources, NX_LWM2M_CLIENT_RESOURCE *resources)
{
UINT status;
NX_LWM2M_CLIENT_OBJECT *object_ptr;
NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr = NX_NULL;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Get pointer to the Object Instance */
    object_ptr = _nx_lwm2m_client_object_ptr_get(client_ptr, object_id);
    if (object_ptr != NX_NULL)
    {
        instance_ptr = _nx_lwm2m_client_object_instance_ptr_get(object_ptr, instance_id);
    }
    if ((object_ptr == NX_NULL) || (instance_ptr == NX_NULL))
    {

        /* Object doesn't exist */
        status = NX_LWM2M_CLIENT_NOT_FOUND;
    }
    else
    {

        /* Call the object method */
        status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_DISCOVER, object_ptr, instance_ptr, resources, num_resources, NX_NULL, 0);
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return the status of the operation */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_execute                     PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function performs the execute operation on an Object resource. */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id                           The Instance ID               */
/*    resource_id                           The Resource ID               */
/*    args_ptr                              Pointer to the arguments      */
/*    args_length                           Length of arguments           */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_object_ptr_get                                     */
/*    _nx_lwm2m_client_object_instance_ptr_get                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_execute(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id, NX_LWM2M_ID resource_id, const CHAR *args_ptr, UINT args_length)
{
UINT status;
NX_LWM2M_CLIENT_OBJECT *object_ptr;
NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr = NX_NULL;
NX_LWM2M_CLIENT_RESOURCE resource_ptr;
UINT resource_count;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Get pointer to the Object Instance */
    object_ptr = _nx_lwm2m_client_object_ptr_get(client_ptr, object_id);
    if (object_ptr != NX_NULL)
    {
        instance_ptr = _nx_lwm2m_client_object_instance_ptr_get(object_ptr, instance_id);
    }
    if ((object_ptr == NX_NULL) || (instance_ptr == NX_NULL))
    {

        /* Object doesn't exist */
        status = NX_LWM2M_CLIENT_NOT_FOUND;
    }
    else
    {

        /* Call the object method */
        resource_ptr.resource_id = resource_id;
        resource_count = 1;
        status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_EXECUTE, object_ptr, instance_ptr, &resource_ptr, &resource_count, (VOID *)args_ptr, args_length);
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return the status of the operation */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_instance_add                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function adds a new Object Instance to the Instance list.      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    instance_ptr                          Pointer to the Object Instance*/
/*    instance_id_ptr                       Pointer to the Instance ID    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_object_instance_ptr_get                            */
/*    _nx_lwm2m_client_object_list_changed                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_instance_add(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_ID *instance_id_ptr)
{
NX_LWM2M_CLIENT_OBJECT_INSTANCE *current_ptr;
NX_LWM2M_ID instance_id;
NX_LWM2M_CLIENT *client_ptr = object_ptr -> object_client_ptr;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    if (instance_id_ptr == NX_NULL)
    {

        /* Assign an ID if not provided */
        instance_id = NX_LWM2M_CLIENT_RESERVED_ID;
    }
    else
    {

        /* Get the instance ID to create */
        instance_id = *instance_id_ptr;
    }

    if (instance_id == NX_LWM2M_CLIENT_RESERVED_ID)
    {

        /* Get an unused Instance ID */
        if ((object_ptr -> object_instances == NX_NULL) || 
            (object_ptr -> object_instances -> object_instance_id != 0))
        {
            instance_id = 0;
        }
        else
        {
            current_ptr = object_ptr -> object_instances;
            while (1)
            {
                if ((current_ptr -> object_instance_next == NX_NULL) || 
                    ((current_ptr -> object_instance_next -> object_instance_id - current_ptr -> object_instance_id) > 1))
                {
                    instance_id = (NX_LWM2M_ID)(current_ptr -> object_instance_id + 1);
                    break;
                }

                current_ptr = current_ptr -> object_instance_next;
            }
        }
    }
    else if (_nx_lwm2m_client_object_instance_ptr_get(object_ptr, instance_id) != NX_NULL)
    {

        /* Release the LWM2M Client Mutex */
        tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

        /* The instance ID already exists */
        return(NX_LWM2M_CLIENT_ALREADY_EXIST);
    }

    if (instance_id == NX_LWM2M_CLIENT_RESERVED_ID)
    {

        /* Release the LWM2M Client Mutex */
        tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

        /* All the valid instance ID are used */
        return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
    }

    /* Store the instance ID */
    instance_ptr -> object_instance_id = instance_id;

    /* Insert the instance in the object list */
    if ((object_ptr -> object_instances == NX_NULL) || 
        (object_ptr -> object_instances -> object_instance_id > instance_id))
    {

        /* Set as first instance */
        instance_ptr -> object_instance_next = object_ptr -> object_instances;
        object_ptr -> object_instances = instance_ptr;
    }
    else
    {

        /* Parse list of Object Instances */
        current_ptr = object_ptr -> object_instances;
        while (1)
        {
            if ((current_ptr -> object_instance_next == NX_NULL) || 
                (current_ptr -> object_instance_next -> object_instance_id > instance_id))
            {
                /* Insert new Instance in list */
                instance_ptr -> object_instance_next = current_ptr -> object_instance_next;
                current_ptr -> object_instance_next = instance_ptr;
                break;
            }

            /* Next instance */
            current_ptr = current_ptr -> object_instance_next;
        }
    }

    /* send list of objects in next Update message */
    _nx_lwm2m_client_object_list_changed(client_ptr);

    /* Return assigned ID */
    if (instance_id_ptr != NX_NULL)
    {
        *instance_id_ptr = instance_id;
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_instance_next_get           PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the next Instance of an Object.               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id_ptr                       Pointer to the Instance ID    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_object_ptr_get                                     */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_instance_next_get(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID *instance_id_ptr)
{
UINT status;
NX_LWM2M_CLIENT_OBJECT *object_ptr;
NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr;
NX_LWM2M_ID instance_id;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Get pointer to the Object Instance */
    object_ptr = _nx_lwm2m_client_object_ptr_get(client_ptr, object_id);
    if (object_ptr == NX_NULL)
    {
        status = NX_LWM2M_CLIENT_NOT_FOUND;
    }
    else
    {

        /* Get first instance */
        instance_ptr = object_ptr -> object_instances;

        /* Get requested ID */
        instance_id = *instance_id_ptr;
        if (instance_id != NX_LWM2M_CLIENT_RESERVED_ID)
        {

            /* Find instance just after the requested ID */
            while ((instance_ptr != NX_NULL) && (instance_id >= instance_ptr -> object_instance_id))
            {

                /* Next instance */
                instance_ptr = instance_ptr -> object_instance_next;
            }
        }

        if (instance_ptr == NX_NULL)
        {

            /* Last instance */
            status = NX_LWM2M_CLIENT_NOT_FOUND;
        }
        else 
        {

            /* Return next instance ID */
            *instance_id_ptr = instance_ptr -> object_instance_id;

            status = NX_SUCCESS;
        }
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return the status of the operation */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_instance_ptr_get            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    The function returns the instance of an object with the given ID.   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_id                           The instance ID               */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    instance_ptr                          Pointer to the instance       */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_object_delete                                      */
/*    _nx_lwm2m_client_object_discover                                    */
/*    _nx_lwm2m_client_object_execute                                     */
/*    _nx_lwm2m_client_object_instance_add                                */
/*    _nx_lwm2m_client_object_read                                        */
/*    _nx_lwm2m_client_object_write                                       */
/*    _nx_lwm2m_client_session_register_info_get                          */
/*    _nx_lwm2m_client_session_receive                                    */
/*    _nx_lwm2m_client_session_start                                      */
/*    _nx_lwm2m_client_session_state_update                               */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
NX_LWM2M_CLIENT_OBJECT_INSTANCE * _nx_lwm2m_client_object_instance_ptr_get(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_ID instance_id)
{
NX_LWM2M_CLIENT_OBJECT_INSTANCE *current_ptr;

    /* Find Instance */
    current_ptr = object_ptr -> object_instances;
    while ((current_ptr != NX_NULL) && (instance_id >= current_ptr -> object_instance_id))
    {
        if (instance_id == current_ptr -> object_instance_id)
        {

            /* Return the pointer to the Instance */
            return(current_ptr);
        }

        current_ptr = current_ptr -> object_instance_next;
    }

    /* Instance Not Found */
    return(NX_NULL);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_instance_remove             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function removes an Object Instance from Instance list.        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to the Object         */
/*    instance_ptr                          Pointer to the Instance       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_object_list_changed                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_instance_remove(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr)
{
NX_LWM2M_CLIENT_OBJECT_INSTANCE *prev_ptr = NX_NULL;
NX_LWM2M_CLIENT_OBJECT_INSTANCE *next_ptr;
NX_LWM2M_CLIENT_SESSION *session_ptr;
NX_LWM2M_CLIENT_NOTIFY *notify_ptr;
NX_LWM2M_CLIENT *client_ptr = object_ptr -> object_client_ptr;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Save pointer to next instance */
    next_ptr = instance_ptr -> object_instance_next;

    /* Get preceding instance */
    if (object_ptr -> object_instances != instance_ptr)
    {
        prev_ptr = object_ptr -> object_instances;
        while (prev_ptr != NX_NULL)
        {
            if (prev_ptr -> object_instance_next == instance_ptr)
            {
                break;
            }

            prev_ptr = prev_ptr -> object_instance_next;
        }

        if (prev_ptr == NX_NULL)
        {

            /* Release the LWM2M Client Mutex */
            tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

            /* XXX should not happen */
            return(NX_LWM2M_CLIENT_NOT_FOUND);
        }
    }

    /* Remove deleted instance from linked list */
    if (object_ptr -> object_instances == instance_ptr)
    {
        object_ptr -> object_instances = next_ptr;
    }
    else
    {
        prev_ptr -> object_instance_next = next_ptr;
    }

    /* Delete all notifications for this instance */
    if (instance_ptr != NX_NULL)
    {

        /* Tell all sessions to delete all notifications */
        session_ptr = client_ptr -> nx_lwm2m_client_sessions;
        while (session_ptr != NX_NULL)
        {

            /* Look for existing notification */
            notify_ptr = session_ptr -> nx_lwm2m_client_session_notifications;
            while (notify_ptr != NX_NULL)
            {

                if (notify_ptr -> notify_instance_ptr == instance_ptr)
                {

                    /* Release one notification */
                    _nx_lwm2m_client_session_notify_free(session_ptr, notify_ptr);
                }

                /* Move to next notification */
                notify_ptr = notify_ptr -> notify_next;
            }

            /* Move to next session */
            session_ptr = session_ptr -> nx_lwm2m_client_session_next;
        }
    }

    /* Send list of objects in next Update message */
    _nx_lwm2m_client_object_list_changed(client_ptr);

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_list_changed                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function indicates that the list objects has been updated.     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_update_flags                               */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_object_delete_ptr                                  */
/*    _nx_lwm2m_client_object_instance_add                                */
/*    _nx_lwm2m_client_object_instance_remove                             */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_object_list_changed(NX_LWM2M_CLIENT *client_ptr)
{
NX_LWM2M_CLIENT_SESSION *session_ptr;

    /* Tell all sessions to update the list of objects */
    session_ptr = client_ptr -> nx_lwm2m_client_sessions;
    while (session_ptr != NX_NULL)
    {

        /* Update list of object */
        _nx_lwm2m_client_session_update_flags(session_ptr, NX_LWM2M_CLIENT_SESSION_UPDATE_OBJECTS_LIST);

        /* Get next session */
        session_ptr = session_ptr -> nx_lwm2m_client_session_next;
    }
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_next_get                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the next Object of the LWM2M Client.          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id_ptr                         Pointer to the Object ID      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_next_get(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID *object_id_ptr)
{
UINT status;
NX_LWM2M_CLIENT_OBJECT *object_ptr;
NX_LWM2M_ID object_id;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Get pointer to first object */
    object_ptr = client_ptr -> nx_lwm2m_client_object_list_head;

    /* Get requested ID */
    object_id = *object_id_ptr;
    if (object_id != NX_LWM2M_CLIENT_RESERVED_ID)
    {

        /* Find instance just after the requested ID */
        while ((object_ptr != NX_NULL) && (object_id >= object_ptr -> object_id))
        {

            /* Next object */
            object_ptr = object_ptr -> object_next;
        }
    }

    if (object_ptr == NX_NULL)
    {

        /* Last object */
        status = NX_LWM2M_CLIENT_NOT_FOUND;
    }
    else
    {

        /* Return next object ID */
        *object_id_ptr = object_ptr -> object_id;

        status = NX_SUCCESS;
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return the status of the operation */
    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_ptr_get                     PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the object with the given ID.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The ID of the object          */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    object_ptr                            Pointer to the object         */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_object_create                                      */
/*    _nx_lwm2m_client_object_delete                                      */
/*    _nx_lwm2m_client_object_discover                                    */
/*    _nx_lwm2m_client_object_execute                                     */
/*    _nx_lwm2m_client_object_instance_next_get                           */
/*    _nx_lwm2m_client_object_read                                        */
/*    _nx_lwm2m_client_object_write                                       */
/*    _nx_lwm2m_client_session_register_info_get                          */
/*    _nx_lwm2m_client_session_receive                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
NX_LWM2M_CLIENT_OBJECT * _nx_lwm2m_client_object_ptr_get(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id)
{
NX_LWM2M_CLIENT_OBJECT *current_ptr;

    /* Find Object */
    current_ptr = client_ptr -> nx_lwm2m_client_object_list_head;

    while ((current_ptr != NX_NULL) && (object_id >= current_ptr -> object_id))
    {
        if (object_id == current_ptr -> object_id)
        {

            /* Return the pointer to the Object */
            return(current_ptr);
        }

        current_ptr = current_ptr -> object_next;
    }

    /* Not found */
    return(NX_NULL);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_read                        PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function reads resources from an Object Instance.              */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id                           The Instance ID               */
/*    num_values                            The number of resources       */
/*    values                                Pointer to the resources      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_object_ptr_get                                     */
/*    _nx_lwm2m_client_object_instance_ptr_get                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_read(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id, UINT num_values, NX_LWM2M_CLIENT_RESOURCE *values)
{
UINT status;
NX_LWM2M_CLIENT_OBJECT *object_ptr;
NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr = NX_NULL;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Get pointer to the Object Instance */
    object_ptr = _nx_lwm2m_client_object_ptr_get(client_ptr, object_id);
    if (object_ptr != NX_NULL)
    {
        instance_ptr = _nx_lwm2m_client_object_instance_ptr_get(object_ptr, instance_id);
    }
    if ((object_ptr == NX_NULL) || (instance_ptr == NX_NULL))
    {

        /* Object doesn't exist */
        status = NX_LWM2M_CLIENT_NOT_FOUND;
    }
    else
    {

        /* Call the object method */
        status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_READ, object_ptr, instance_ptr, values, &num_values, NX_NULL, 0);
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return the status of the operation */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*   _nx_lwm2m_client_object_remove                       PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function remove an Object from the LWM2M Client Object list.   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_ptr                            Pointer to Object             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_remove(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_CLIENT_OBJECT *object_ptr)
{
UINT status = NX_LWM2M_CLIENT_NOT_FOUND;
NX_LWM2M_CLIENT_OBJECT *current_ptr;
NX_LWM2M_CLIENT_OBJECT *previous_ptr;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Get pointer to first object */
    current_ptr = client_ptr -> nx_lwm2m_client_object_list_head;
    previous_ptr = current_ptr;

    /* Loop the list of objects to find the corresponding object */
    while (current_ptr)
    {
        if (current_ptr == object_ptr)
        {

            /* Reomve the object from the object list */
            if (current_ptr == client_ptr -> nx_lwm2m_client_object_list_head)
            {
                client_ptr -> nx_lwm2m_client_object_list_head = current_ptr -> object_next;
            }
            else
            {
                previous_ptr -> object_next = current_ptr -> object_next;
            }

            /* Found the object */
            status = NX_SUCCESS;

            break;
        }

        /* Get next object */
        previous_ptr = current_ptr;
        current_ptr = current_ptr -> object_next;
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return the status of the operation */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_number_abs                         PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the absolute value of a notify number.        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    value                                 The value of notify number    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    The absolut value                                                   */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_object_resource_changed                            */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static NX_LWM2M_CLIENT_NOTIFY_NUMBER _nx_lwm2m_client_number_abs(NX_LWM2M_CLIENT_NOTIFY_NUMBER value)
{
    return(value >= 0 ? value : -value);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_resource_changed            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function signals the LWM2M Client that an object resource.     */
/*    value has changed                                                   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to the Object         */
/*    instance_ptr                          Pointer to the Object Instance*/
/*    resource_ptr                          The resource ID and value     */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_session_notify_attributes                          */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*    _nx_lwm2m_client_device_object_error_push                           */
/*    _nx_lwm2m_client_device_object_error_reset                          */
/*    _nx_lwm2m_client_device_resource_changed                            */
/*    _nx_lwm2m_client_firmware_package_info_set                          */
/*    _nx_lwm2m_client_firmware_result_set                                */
/*    _nx_lwm2m_client_firmware_state_set                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_resource_changed(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, const NX_LWM2M_CLIENT_RESOURCE *resource_ptr)
{
NX_LWM2M_CLIENT *client_ptr = object_ptr -> object_client_ptr;
NX_LWM2M_CLIENT_SESSION *session_ptr;
NX_LWM2M_BOOL do_wakeup;

    do_wakeup = NX_FALSE;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Check for notifications on all sessions */
    session_ptr = client_ptr -> nx_lwm2m_client_sessions;
    while (session_ptr != NX_NULL)
    {
        NX_LWM2M_CLIENT_NOTIFY *notify_ptr;

        notify_ptr = session_ptr -> nx_lwm2m_client_session_notifications;
        while (notify_ptr != NX_NULL)
        {
            if (notify_ptr -> notify_object_ptr == object_ptr &&
                (notify_ptr -> notify_instance_ptr == NX_NULL ||
                 (notify_ptr -> notify_instance_ptr == instance_ptr &&
                  (notify_ptr -> notify_resource_id == NX_LWM2M_CLIENT_RESERVED_ID ||
                   notify_ptr -> notify_resource_id == resource_ptr -> resource_id))) &&
                (notify_ptr -> notify_flags & (NX_LWM2M_CLIENT_NOTIFY_OBSERVE|NX_LWM2M_CLIENT_NOTIFY_CHANGED)) == NX_LWM2M_CLIENT_NOTIFY_OBSERVE)
            {
                UINT flags;
                NX_LWM2M_INT32 pmin;
                NX_LWM2M_INT32 pmax;
                NX_LWM2M_CLIENT_NOTIFY_NUMBER gt;
                NX_LWM2M_CLIENT_NOTIFY_NUMBER lt;
                NX_LWM2M_CLIENT_NOTIFY_NUMBER stp;
                NX_LWM2M_CLIENT_NOTIFY_NUMBER value;
                NX_LWM2M_BOOL changed;

                /* Get notify attributes */
                _nx_lwm2m_client_session_notify_attributes(session_ptr, notify_ptr, &flags, &pmin, &pmax, &gt, &lt, &stp);

                /* Get the resource number value if we need to check for conditions */
                if ((flags & (NX_LWM2M_CLIENT_NOTIFY_ATTR_GT|NX_LWM2M_CLIENT_NOTIFY_ATTR_LT|NX_LWM2M_CLIENT_NOTIFY_ATTR_STP)) != 0 && _nx_lwm2m_client_resource_notify_number_get(resource_ptr, &value) == NX_SUCCESS)
                {
                    /* Check for gt/lt/stp conditions */
                    if (((flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_GT) && notify_ptr -> notify_last_value <= gt && value > gt) ||
                        ((flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_LT) && notify_ptr -> notify_last_value >= lt && value < lt) ||
                        ((flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_STP) && _nx_lwm2m_client_number_abs(notify_ptr -> notify_last_value - value) >= stp))
                    {

                        /* Conditions are met */
                        changed = NX_TRUE;
                    }
                    else
                    {

                        /* Conditions are not met */
                        changed = NX_FALSE;
                    }
                }
                else
                {

                    /* No condition or value is not a number, always send notification */
                    changed = NX_TRUE;
                }

                if (changed)
                {

                    /* Resource has changed, arm minimum period timer */
                    if (!(flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_PMIN))
                    {

                        /* Get default minimum period from server object */
                        pmin = session_ptr -> nx_lwm2m_client_session_server_instance_ptr -> server_instance_min_period;
                    }
                    notify_ptr -> notify_flags |= NX_LWM2M_CLIENT_NOTIFY_CHANGED|NX_LWM2M_CLIENT_NOTIFY_TIMER;
                    notify_ptr -> notify_timer = notify_ptr -> notify_last_time + (ULONG)pmin * NX_IP_PERIODIC_RATE;

                    do_wakeup = NX_TRUE;
                }
            }

            notify_ptr = notify_ptr -> notify_next;
        }

        session_ptr = session_ptr -> nx_lwm2m_client_session_next;
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    if (do_wakeup)
    {

        /* Sessions timers have been updated, wakeup client thread */
        tx_event_flags_set(&client_ptr -> nx_lwm2m_client_event_flags, NX_LWM2M_CLIENT_EVENT_WAKEUP, TX_OR);
    }

    /* Return success */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_write                       PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function writes resources to an Object Instance.               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    object_id                             The Object ID                 */
/*    instance_id                           The Instance ID               */
/*    num_values                            The number of resources       */
/*    values                                Pointer to the resources      */
/*    write_op                              The type of write operation   */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_object_ptr_get                                     */
/*    _nx_lwm2m_client_object_instance_ptr_get                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_object_write(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id, UINT num_values, const NX_LWM2M_CLIENT_RESOURCE *values, UINT write_op)
{
UINT status;
NX_LWM2M_CLIENT_OBJECT *object_ptr;
NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr = NX_NULL;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Get pointer to the Object Instance */
    object_ptr = _nx_lwm2m_client_object_ptr_get(client_ptr, object_id);
    if (object_ptr != NX_NULL)
    {
        instance_ptr = _nx_lwm2m_client_object_instance_ptr_get(object_ptr, instance_id);
    }
    if ((object_ptr == NX_NULL) || (instance_ptr == NX_NULL))
    {
        if ((object_ptr != NX_NULL) && (write_op == NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP) && (instance_id != NX_LWM2M_CLIENT_RESERVED_ID))
        {

            /* Create new instance if it's a Bootstrap Write operation */
            status = _nx_lwm2m_client_object_create_internal(object_ptr, &instance_id, num_values, values, NX_TRUE);
        }
        else
        {
            /* Object or Object Instance doesn't exist */
            status = NX_LWM2M_CLIENT_NOT_FOUND;
        }
    }
    else
    {

        /* Call the object method */
        status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_WRITE, object_ptr, instance_ptr, (NX_LWM2M_CLIENT_RESOURCE *)values, &num_values, &write_op, sizeof(UINT));
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return the status of the operation */
    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_boolean_get               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a boolean resource.              */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     bool_ptr                             On return, the boolean value  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_boolean_get(NX_LWM2M_CLIENT_RESOURCE *value, NX_LWM2M_BOOL *bool_ptr)
{
    switch (value -> resource_type)
    {

    case NX_LWM2M_CLIENT_RESOURCE_BOOLEAN:

        /* Direct Value */

        *bool_ptr = value -> resource_value.resource_booleandata;

        return(NX_SUCCESS);

    case NX_LWM2M_CLIENT_RESOURCE_TLV:

        /* TLV Encoding */

        if (value -> resource_value.resource_bufferdata.resource_buffer_length == NX_LWM2M_CLIENT_TLV_LENGTH_BOOLEAN)
        {
            *bool_ptr = *((const UCHAR *) value -> resource_value.resource_bufferdata.resource_buffer_ptr) == 0 ? NX_FALSE : NX_TRUE;

            /* Reset resource to the corresponding type */
            value -> resource_type = NX_LWM2M_CLIENT_RESOURCE_BOOLEAN;
            value -> resource_value.resource_booleandata = *bool_ptr;

            return(NX_SUCCESS);
        }

        /* Invalid TLV length */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);

    default:

        /* Not a boolean value */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);

    }
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_boolean_get               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of a boolean resource.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     bool_data                            The boolean value             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_boolean_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_BOOL bool_data)
{

    /* Set resource type. */
    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_BOOLEAN;

    /* Set resource data. */
    resource_ptr -> resource_value.resource_booleandata = bool_data;

    /* Return */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_dim_get                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function gets the dim of multiple resource.                    */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     dim                                  Dim of the multiple resource  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_dim_get(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, UCHAR *dim)
{

    if (!NX_LWM2M_CLIENT_RESOURCE_IS_MULTIPLE(resource_ptr -> resource_type))
    {

        /* Not a multiple resource */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);
    }

    /* Get dim of multiple resource. */
    *dim = resource_ptr -> resource_dim;

    /* Return */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_dim_set                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the dim of multiple resource.                    */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     dim                                  Dim of the multiple resource  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_dim_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, UCHAR dim)
{

    /* Set resource type. */
    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_MULTIPLE;

    /* Set dim of multiple resource. */
    resource_ptr -> resource_dim = dim;

    /* Return */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_float32_get               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a 32-bit floating point resource.*/
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     float32_ptr                          On return, the floating point */
/*                                          value                         */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer32_get                             */
/*    memcpy                                                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_float32_get(NX_LWM2M_CLIENT_RESOURCE *value, NX_LWM2M_FLOAT32 *float32_ptr)
{
    switch (value -> resource_type)
    {

    case NX_LWM2M_CLIENT_RESOURCE_FLOAT32:

        *float32_ptr = value -> resource_value.resource_float32data;

        return(NX_SUCCESS);

    case NX_LWM2M_CLIENT_RESOURCE_TLV:

        /* TLV Encoding, check value length */
        switch (value -> resource_value.resource_bufferdata.resource_buffer_length)
        {

        case NX_LWM2M_CLIENT_TLV_LENGTH_FLOAT32:
            {

                /* Read 32-bit value */
                NX_LWM2M_INT32 as_integer32;

                _nx_lwm2m_client_resource_integer32_get(value, &as_integer32);

                memcpy(float32_ptr, &as_integer32, sizeof(NX_LWM2M_FLOAT32)); /* Use case of memcpy is verified. */

                /* Reset resource to the corresponding type */
                value -> resource_type = NX_LWM2M_CLIENT_RESOURCE_FLOAT32;
                value -> resource_value.resource_float32data = *float32_ptr;

                return(NX_SUCCESS);
            }

        default:

            /* Invalid TLV length */
            return(NX_LWM2M_CLIENT_BAD_ENCODING);

        }

    default:

        /* Not a floating point value */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);

    }
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_float32_set               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of a 32-bit floating point resource.   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     float32_data                         The floating data             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_float32_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_FLOAT32 float32_data)
{

    /* Set resource type. */
    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_FLOAT32;

    /* Set resource data. */
    resource_ptr -> resource_value.resource_float32data = float32_data;

    /* Return */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_float64_get               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a 64-bit floating point resource.*/
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     float64_ptr                          On return, the floating point */
/*                                          value                         */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer32_get                             */
/*    _nx_lwm2m_client_resource_integer64_get                             */
/*    memcpy                                                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_float64_get(NX_LWM2M_CLIENT_RESOURCE *value, NX_LWM2M_FLOAT64 *float64_ptr)
{
    switch (value -> resource_type)
    {

    case NX_LWM2M_CLIENT_RESOURCE_FLOAT32:

        *float64_ptr = (NX_LWM2M_FLOAT64) value -> resource_value.resource_float32data;

        return(NX_SUCCESS);

    case NX_LWM2M_CLIENT_RESOURCE_FLOAT64:

        *float64_ptr = value -> resource_value.resource_float64data;

        return(NX_SUCCESS);

    case NX_LWM2M_CLIENT_RESOURCE_TLV:

        /* TLV Encoding, check value length */
        switch (value -> resource_value.resource_bufferdata.resource_buffer_length)
        {

        case NX_LWM2M_CLIENT_TLV_LENGTH_FLOAT32:
            {

                /* Read 32-bit value, convert to 64-bit floating point */
                NX_LWM2M_INT32 as_integer32;
                NX_LWM2M_FLOAT32 as_float32;

                _nx_lwm2m_client_resource_integer32_get(value, &as_integer32);

                memcpy(&as_float32, &as_integer32, sizeof(NX_LWM2M_FLOAT32)); /* Use case of memcpy is verified. */

                *float64_ptr = (NX_LWM2M_FLOAT64) as_float32;

                break;
            }

        case NX_LWM2M_CLIENT_TLV_LENGTH_FLOAT64:
            {

                /* Read 64-bit value, convert to 64-bit floating point */
                NX_LWM2M_INT64 as_integer64;

                _nx_lwm2m_client_resource_integer64_get(value, &as_integer64);

                memcpy(float64_ptr, &as_integer64, sizeof(NX_LWM2M_FLOAT64));  /* Use case of memcpy is verified. */

                break;
            }

        default:

            /* Invalid TLV length */
            return(NX_LWM2M_CLIENT_BAD_ENCODING);

        }

        /* Reset resource to the corresponding type */
        value -> resource_type = NX_LWM2M_CLIENT_RESOURCE_FLOAT64;
        value -> resource_value.resource_float64data = *float64_ptr;

        return(NX_SUCCESS);

    default:

        /* Not a floating point value */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);

    }
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_float64_set               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of a 64-bit floating point resource.   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     float64_data                         The floating point value      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_float64_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_FLOAT64 float64_data)
{

    /* Set resource type. */
    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_FLOAT64;

    /* Set resource data. */
    resource_ptr -> resource_value.resource_float64data = float64_data;

    /* Return */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_info_get                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function gets the information of resource.                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     resource_id                          ID of the resource            */
/*     operation                            Orperation of the resource    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_info_get(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_ID *resource_id, ULONG *operation)
{

    /* Set resource id. */
    *resource_id = resource_ptr -> resource_id;

    /* Set resource operation. */
    if (operation)
    {
        *operation = resource_ptr -> resource_operation;
    }

    /* Return */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_info_set                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the information of resource.                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     resource_id                          ID of the resource            */
/*     operation                            Orperation of the resource    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_info_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_ID resource_id, ULONG operation)
{

    /* Set resource type. */
    resource_ptr -> resource_id = resource_id;

    /* Set resource operation. */
    resource_ptr -> resource_operation = operation;

    /* Reset resource type. */
    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_NONE;

    /* Return */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_instances_get             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function gets the multiple resource instances.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     resource_instances                   Pointer to the instances      */
/*     count                                Count of the instances        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_tlv_header_parse                                   */
/*    memcpy                                                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_instances_get(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_CLIENT_RESOURCE *resource_instances, UINT *count)
{
const UCHAR *ptr;
UCHAR resource_type = 0;
UINT i;
UINT buffer_length;

    /* Check type of resource */
    if (!NX_LWM2M_CLIENT_RESOURCE_IS_MULTIPLE(resource_ptr -> resource_type) || resource_ptr -> resource_dim == 0)
    {

        /* Not a multiple resource */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);
    }

    /* Check the size of input buffer */
    if (*count < resource_ptr -> resource_dim)
    {
        return(NX_LWM2M_CLIENT_NO_MEMORY);
    }

    /* Set count for multiple resource */
    *count = resource_ptr -> resource_dim;

    /* Return the instance value */
    switch (resource_ptr -> resource_type)
    {

    case NX_LWM2M_CLIENT_RESOURCE_MULTIPLE:

        /* Array of NX_LWM2M_CLIENT_RESOURCE structures */
        memcpy(resource_instances, /* Use case of memcpy is verified. */
               resource_ptr -> resource_value.resource_bufferdata.resource_buffer_ptr,
               resource_ptr -> resource_value.resource_bufferdata.resource_buffer_length);

        break;

    case NX_LWM2M_CLIENT_RESOURCE_MULTIPLE_TLV:

        /* TLV resource array */

        /* Begin of TLV buffer */
        ptr =  resource_ptr -> resource_value.resource_bufferdata.resource_buffer_ptr;

        /* Walk through the instances until the index is reached */
        for (i = 0 ; i < *count; i++)
        {

            /* Parse TLV encoded instance */
            ptr = _nx_lwm2m_client_tlv_header_parse(ptr, NX_NULL, &resource_type, &(resource_instances[i].resource_id), &buffer_length);

            /* The TLV data has already been validated by _nx_lwm2m_tlv_resource_parse */
            if (ptr == NX_NULL || resource_type != NX_LWM2M_CLIENT_TLV_RESOURCE_INSTANCE)
            {
                return(NX_LWM2M_CLIENT_BAD_ENCODING);
            }

            /* Set the type and value of resource instance */
            resource_instances[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_TLV;
            resource_instances[i].resource_value.resource_bufferdata.resource_buffer_ptr = ptr;
            resource_instances[i].resource_value.resource_bufferdata.resource_buffer_length = buffer_length;

            /* Adjust pointer to next instance */
            ptr += buffer_length;
        }

        break;

    default:

        /* Invalid type */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);
    }

    /* Return success */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_instances_set             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the multiple resource instances.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     resource_instances                   Pointer to the instances      */
/*     count                                Count of the instances        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_instances_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_CLIENT_RESOURCE *resource_instances, UINT count)
{

    /* Set resource type */
    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_MULTIPLE;

    /* Set dim for multiple resource */
    resource_ptr -> resource_dim = (UCHAR)count;

    /* Set value fot multiple resource */
    resource_ptr -> resource_value.resource_bufferdata.resource_buffer_ptr = resource_instances;
    resource_ptr -> resource_value.resource_bufferdata.resource_buffer_length = count * sizeof(NX_LWM2M_CLIENT_RESOURCE);

    /* Return success */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer32_get             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a 32-bit integer resource.       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     int32_ptr                            On return, the integer value  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_integer32_get(NX_LWM2M_CLIENT_RESOURCE *value, NX_LWM2M_INT32 *int32_ptr)
{
    switch (value -> resource_type)
    {

    case NX_LWM2M_CLIENT_RESOURCE_INTEGER32:

        /* 32-bit integer value */
        *int32_ptr = value -> resource_value.resource_integer32data;

        return(NX_SUCCESS);

    case NX_LWM2M_CLIENT_RESOURCE_TLV:

        /* TLV Encoding */

        switch (value -> resource_value.resource_bufferdata.resource_buffer_length)
        {

        case NX_LWM2M_CLIENT_TLV_LENGTH_INTEGER8:

            /* Read 8-bit integer value */
            *int32_ptr = *((signed char *) value -> resource_value.resource_bufferdata.resource_buffer_ptr);

            break;

        case NX_LWM2M_CLIENT_TLV_LENGTH_INTEGER16:
            {

                /* Read 16-bit integer value */
                const UCHAR *byte_ptr = (const UCHAR *) value -> resource_value.resource_bufferdata.resource_buffer_ptr;

                *int32_ptr = (SHORT) ((((USHORT) byte_ptr[0]) << 8) |
                                       ((USHORT) byte_ptr[1]));

               break;
            }

        case NX_LWM2M_CLIENT_TLV_LENGTH_INTEGER32:
            {

                /* Read 32-bit integer value */
                const UCHAR *byte_ptr = (const UCHAR *) value -> resource_value.resource_bufferdata.resource_buffer_ptr;

                *int32_ptr = (LONG) ((((ULONG) byte_ptr[0]) << 24) |
                                     (((ULONG) byte_ptr[1]) << 16) |
                                     (((ULONG) byte_ptr[2]) << 8)  |
                                      ((ULONG) byte_ptr[3]));

                break;
            }

        default:

            /* Invalid TLV length */
            return(NX_LWM2M_CLIENT_BAD_ENCODING);

        }

        /* Reset resource to the corresponding type */
        value -> resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
        value -> resource_value.resource_integer32data = *int32_ptr;
        return(NX_SUCCESS);

    default:

        /* Not an Integer value */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);

    }
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer32_set             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of a 32-bit integer resource.          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     int32_data                           The integer value             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_integer32_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_INT32 int32_data)
{

    /* Set resource type. */
    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;

    /* Set resource data. */
    resource_ptr -> resource_value.resource_integer32data = int32_data;

    /* Return */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer64_get             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a 64-bit integer resource.       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     int64_ptr                            On return, the integer value  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer32_get                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_integer64_get(NX_LWM2M_CLIENT_RESOURCE *value, NX_LWM2M_INT64 *int64_ptr)
{
    switch (value -> resource_type)
    {

    case NX_LWM2M_CLIENT_RESOURCE_INTEGER32:

        *int64_ptr = value -> resource_value.resource_integer32data;

        return(NX_SUCCESS);

    case NX_LWM2M_CLIENT_RESOURCE_INTEGER64:

        *int64_ptr = value -> resource_value.resource_integer64data;

        return(NX_SUCCESS);

    case NX_LWM2M_CLIENT_RESOURCE_TLV:

        /* TLV Encoding */

        switch (value -> resource_value.resource_bufferdata.resource_buffer_length)
        {

        case NX_LWM2M_CLIENT_TLV_LENGTH_INTEGER8:
        case NX_LWM2M_CLIENT_TLV_LENGTH_INTEGER16:
        case NX_LWM2M_CLIENT_TLV_LENGTH_INTEGER32:
            {

                /* Read 32-bit integer value, convert to 64-bit */
                NX_LWM2M_INT32 as_integer32;

                _nx_lwm2m_client_resource_integer32_get(value, &as_integer32);

                *int64_ptr = as_integer32;

                break;
            }

        case NX_LWM2M_CLIENT_TLV_LENGTH_INTEGER64:
            {

                /* Read 64-bit integer value */
                const UCHAR *byte_ptr = (const UCHAR *) value -> resource_value.resource_bufferdata.resource_buffer_ptr;

                *int64_ptr = (NX_LWM2M_INT64) ((((NX_LWM2M_UINT64) byte_ptr[0]) << 56) |
                                               (((NX_LWM2M_UINT64) byte_ptr[1]) << 48) |
                                               (((NX_LWM2M_UINT64) byte_ptr[2]) << 40) |
                                               (((NX_LWM2M_UINT64) byte_ptr[3]) << 32) |
                                               (((NX_LWM2M_UINT64) byte_ptr[4]) << 24) |
                                               (((NX_LWM2M_UINT64) byte_ptr[5]) << 16) |
                                               (((NX_LWM2M_UINT64) byte_ptr[6]) << 8)  |
                                                ((NX_LWM2M_UINT64) byte_ptr[7]));

                break;
            }

        default:

            /* Invalid TLV length */
            return(NX_LWM2M_CLIENT_BAD_ENCODING);

        }

        /* Reset resource to the corresponding type */
        value -> resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER64;
        value -> resource_value.resource_integer64data = *int64_ptr;
        return(NX_SUCCESS);

    default:

        /* Not an Integer value */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);

    }
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer64_set             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of a 64-bit integer resource.          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     int64_data                           The integer value             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_integer64_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_INT64 int64_data)
{

    /* Set resource type. */
    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER64;

    /* Set resource data. */
    resource_ptr -> resource_value.resource_integer64data = int64_data;

    /* Return */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_notify_number_get         PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a resource as a notify number.   */
/*    The resource value must be an integer or a floating point.          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     number_ptr                           On return, the number value   */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_object_resource_changed                            */
/*    _nx_lwm2m_client_object_observe                                     */
/*    _nx_lwm2m_client_session_step                                       */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_notify_number_get(const NX_LWM2M_CLIENT_RESOURCE *resource_ptr, NX_LWM2M_CLIENT_NOTIFY_NUMBER *number_ptr)
{

    switch (resource_ptr -> resource_type)
    {

    case NX_LWM2M_CLIENT_RESOURCE_INTEGER32:

        *number_ptr = (NX_LWM2M_CLIENT_NOTIFY_NUMBER) resource_ptr -> resource_value.resource_integer32data;
        break;

    case NX_LWM2M_CLIENT_RESOURCE_INTEGER64:

        *number_ptr = (NX_LWM2M_CLIENT_NOTIFY_NUMBER) resource_ptr -> resource_value.resource_integer64data;
        break;

    case NX_LWM2M_CLIENT_RESOURCE_FLOAT32:

        *number_ptr = (NX_LWM2M_CLIENT_NOTIFY_NUMBER) resource_ptr -> resource_value.resource_float32data;
        break;

    case NX_LWM2M_CLIENT_RESOURCE_FLOAT64:

        *number_ptr = (NX_LWM2M_CLIENT_NOTIFY_NUMBER) resource_ptr -> resource_value.resource_float64data;
        break;

    default:

        /* Not a number */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);

    }

    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_objlnk_get                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of an Object Link resource.         */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     object_id                            On return, the object ID      */
/*     instance_id                          On return, the instance ID    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_objlnk_get(NX_LWM2M_CLIENT_RESOURCE *value, NX_LWM2M_ID *object_id, NX_LWM2M_ID *instance_id)
{
    switch (value -> resource_type)
    {

    case NX_LWM2M_CLIENT_RESOURCE_OBJLNK:

        *object_id = value -> resource_value.resource_objlnkdata.objlnk_object_id;
        *instance_id = value -> resource_value.resource_objlnkdata.objlnk_instance_id;

        return(NX_SUCCESS);

    case NX_LWM2M_CLIENT_RESOURCE_TLV:

        /* TLV Encoding */

        if (value -> resource_value.resource_bufferdata.resource_buffer_length == NX_LWM2M_CLIENT_TLV_LENGTH_OBJLNK)
        {
            const UCHAR *byte_ptr = (const UCHAR *) value -> resource_value.resource_bufferdata.resource_buffer_ptr;

            /* read 2 16-bit integers */
            *object_id   = (NX_LWM2M_ID) ((byte_ptr[0] << 8) | byte_ptr[1]);
            *instance_id = (NX_LWM2M_ID) ((byte_ptr[2] << 8) | byte_ptr[3]);

            /* Reset resource to the corresponding type */
            value -> resource_type = NX_LWM2M_CLIENT_RESOURCE_OBJLNK;
            value -> resource_value.resource_objlnkdata.objlnk_object_id = *object_id;
            value -> resource_value.resource_objlnkdata.objlnk_instance_id = *instance_id;
            return(NX_SUCCESS);
        }

        /* Invalid TLV length */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);

    default:

        /* Not an Objlnk value */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);

    }
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_objlnk_set                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of an Object Link resource.            */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     object_id                            The object ID                 */
/*     instance_id                          The instance ID               */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_objlnk_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr,  NX_LWM2M_ID object_id, NX_LWM2M_ID instance_id)
{

    /* Set resource type. */
    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_OBJLNK;

    /* Set resource data. */
    resource_ptr -> resource_value.resource_objlnkdata.objlnk_object_id= object_id;
    resource_ptr -> resource_value.resource_objlnkdata.objlnk_instance_id = instance_id;

    /* Return */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_opaque_get                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of an opaque resource.              */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     opaque_ptr                           On return, a pointer to the   */
/*                                          opaque data                   */
/*     opaque_length                        On return, the length of the  */
/*                                          opaque data                   */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_opaque_get(NX_LWM2M_CLIENT_RESOURCE *value, const VOID **opaque_ptr, UINT *opaque_length)
{
    switch (value -> resource_type)
    {

    case NX_LWM2M_CLIENT_RESOURCE_OPAQUE:
    case NX_LWM2M_CLIENT_RESOURCE_TLV:

        /* Opaque data stored in resource buffer */
        *opaque_ptr     = value -> resource_value.resource_bufferdata.resource_buffer_ptr;
        *opaque_length  = value -> resource_value.resource_bufferdata.resource_buffer_length;

        /* Reset resource to the corresponding type */
        value -> resource_type = NX_LWM2M_CLIENT_RESOURCE_OPAQUE;

        return(NX_SUCCESS);
    
    default:

        /* Not an opaque value */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);

    }
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_opaque_set                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of an opaque resource.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     resource_ptr                         Pointer to the resource       */
/*     opaque_ptr                           Apointer to the opaque data   */
/*     opaque_length                        The length of the opaque data */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_opaque_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, VOID *opaque_ptr, UINT opaque_length)
{

    /* Set resource type. */
    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_OPAQUE;

    /* Set resource data. */
    resource_ptr -> resource_value.resource_bufferdata.resource_buffer_ptr = opaque_ptr;
    resource_ptr -> resource_value.resource_bufferdata.resource_buffer_length = opaque_length;

    /* Return */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_string_get                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the value of a string resource.               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     value                                Pointer to the resource       */
/*     string_ptr                           On return, a pointer to the   */
/*                                          string                        */
/*     string_length                        On return, the length of the  */
/*                                          string                        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_string_get(NX_LWM2M_CLIENT_RESOURCE *value, const CHAR **string_ptr, UINT *string_length)
{
    switch (value -> resource_type)
    {

    case NX_LWM2M_CLIENT_RESOURCE_STRING:
    case NX_LWM2M_CLIENT_RESOURCE_TLV:
    case NX_LWM2M_CLIENT_RESOURCE_TEXT:

        /* String stored in resource buffer */
        *string_ptr     = value -> resource_value.resource_bufferdata.resource_buffer_ptr;
        *string_length  = value -> resource_value.resource_bufferdata.resource_buffer_length;

        /* Reset resource to the corresponding type */
        value -> resource_type = NX_LWM2M_CLIENT_RESOURCE_STRING;

        return(NX_SUCCESS);

    default:

        /* Not a string value */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);

    }
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_resource_string_set                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sets the value of a string resource.                  */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*     Resource_ptr                         Pointer to the resource       */
/*     string_ptr                           A pointer to the string       */
/*     string_length                        Length of the string          */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_resource_string_set(NX_LWM2M_CLIENT_RESOURCE *resource_ptr, CHAR *string_ptr, UINT string_length)
{

    /* Set resource type. */
    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_STRING;

    /* Set resource data. */
    resource_ptr -> resource_value.resource_bufferdata.resource_buffer_ptr = string_ptr;
    resource_ptr -> resource_value.resource_bufferdata.resource_buffer_length = string_length;

    return(NX_SUCCESS);
}



#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_security_read                      PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the read operation.                         */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    object_instance_ptr                   Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_security_operation                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_security_read(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *object_instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT resource_count)
{
NX_LWM2M_CLIENT_SECURITY_INSTANCE *instance_ptr = (NX_LWM2M_CLIENT_SECURITY_INSTANCE *) object_instance_ptr;
UINT i;

    NX_PARAMETER_NOT_USED(object_ptr);

    for (i = 0; i < resource_count; i++)
    {
        switch (resource[i].resource_id)
        {

        case NX_LWM2M_CLIENT_SECURITY_URI_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_STRING;
            resource[i].resource_value.resource_bufferdata.resource_buffer_ptr = instance_ptr -> security_instance_server_uri;
            resource[i].resource_value.resource_bufferdata.resource_buffer_length = instance_ptr -> security_instance_server_uri_len;
            break;

        case NX_LWM2M_CLIENT_SECURITY_BOOTSTRAP_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_BOOLEAN;
            resource[i].resource_value.resource_booleandata = instance_ptr -> security_instance_bootstrap;
            break;

        case NX_LWM2M_CLIENT_SECURITY_MODE_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = instance_ptr -> security_instance_mode;
            break;

        case NX_LWM2M_CLIENT_SECURITY_SHORT_SERVER_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = instance_ptr -> security_instance_server_id;
            break;

        case NX_LWM2M_CLIENT_SECURITY_HOLD_OFF_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = instance_ptr -> security_instance_hold_off;
            break;

        case NX_LWM2M_CLIENT_SECURITY_PUBLIC_KEY_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_OPAQUE;
            resource[i].resource_value.resource_bufferdata.resource_buffer_ptr = instance_ptr -> security_instance_pub_key_or_id;
            resource[i].resource_value.resource_bufferdata.resource_buffer_length = instance_ptr -> security_instance_pub_key_or_id_len;
            break;

        case NX_LWM2M_CLIENT_SECURITY_SERVER_PUBLIC_KEY_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_OPAQUE;
            resource[i].resource_value.resource_bufferdata.resource_buffer_ptr = instance_ptr -> security_instance_server_pub_key;
            resource[i].resource_value.resource_bufferdata.resource_buffer_length = instance_ptr -> security_instance_server_pub_key_len;
            break;

        case NX_LWM2M_CLIENT_SECURITY_SECRET_KEY_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_OPAQUE;
            resource[i].resource_value.resource_bufferdata.resource_buffer_ptr = instance_ptr -> security_instance_secret_key;
            resource[i].resource_value.resource_bufferdata.resource_buffer_length = instance_ptr -> security_instance_secret_key_len;
            break;

        /* Unknown resource */
        default:

            return(NX_LWM2M_CLIENT_NOT_FOUND);
        }
    }
    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_security_discover                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the discover operation.                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    object_instance_ptr                   Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    num_resources_ptr                     Pointer to number of resource */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_security_operation                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_security_discover(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *object_instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resources, UINT *resource_count)
{

    NX_PARAMETER_NOT_USED(object_ptr);
    NX_PARAMETER_NOT_USED(object_instance_ptr);
    NX_PARAMETER_NOT_USED(resource_count);
    NX_PARAMETER_NOT_USED(resources);

    /* Dont need to implement discover method for security object */
    return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_security_write                     PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the write operation.                        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    object_instance_ptr                   Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    write_op                              Type of write operation       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer32_get                             */
/*    _nx_lwm2m_client_resource_opaque_get                                */
/*    _nx_lwm2m_client_resource_string_get                                */
/*    _nx_lwm2m_client_resource_boolean_get                               */
/*    memcpy                                                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_security_operation                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_security_write(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *object_instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT resource_count, UINT write_op)
{
NX_LWM2M_CLIENT_SECURITY_INSTANCE *instance_ptr = (NX_LWM2M_CLIENT_SECURITY_INSTANCE *) object_instance_ptr;
UINT status;
UINT i;
#ifdef NX_SECURE_ENABLE_DTLS
UCHAR *key_ptr;
UINT key_len;
#endif /* NX_SECURE_ENABLE_DTLS */

    NX_PARAMETER_NOT_USED(object_ptr);

    /* Writing to security object is only allowed during bootstrap */
    if (write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP)
    {
        return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
    }

    for (i = 0; i < resource_count; i++)
    {
        switch (resource[i].resource_id)
        {

        case NX_LWM2M_CLIENT_SECURITY_URI_ID:
            {
                const CHAR *uri_ptr;
                UINT uri_length;

                status = _nx_lwm2m_client_resource_string_get(&resource[i], &uri_ptr, &uri_length);
                if (status != NX_SUCCESS)
                {
                    return(status);
                }

                /* Check for maximum URI length */
                if (uri_length >= NX_LWM2M_CLIENT_SECURITY_MAX_SERVER_URI)
                {
                    return(NX_LWM2M_CLIENT_BUFFER_TOO_SMALL);
                }

                /* Copy URI into instance */
                if (uri_length > 0)
                {
                    memcpy(instance_ptr -> security_instance_server_uri, uri_ptr, uri_length); /* Use case of memcpy is verified. */
                }

                /* Write null at end of string */
                instance_ptr -> security_instance_server_uri[uri_length] = 0;
                instance_ptr -> security_instance_server_uri_len = uri_length;

                break;
            }

        case NX_LWM2M_CLIENT_SECURITY_BOOTSTRAP_ID:

            status = _nx_lwm2m_client_resource_boolean_get(&resource[i], &instance_ptr -> security_instance_bootstrap);
            if (status != NX_SUCCESS)
            {
                return(status);
            }
            break;

        case NX_LWM2M_CLIENT_SECURITY_MODE_ID:
            {
                NX_LWM2M_INT32 security_mode;

                status = _nx_lwm2m_client_resource_integer32_get(&resource[i], &security_mode);
                if (status != NX_SUCCESS)
                {
                    return(status);
                }

                instance_ptr -> security_instance_mode = (UCHAR) security_mode;

                break;
            }

#ifdef NX_SECURE_ENABLE_DTLS
        case NX_LWM2M_CLIENT_SECURITY_PUBLIC_KEY_ID:

            /* Get key opaque data */
            status = _nx_lwm2m_client_resource_opaque_get(&resource[i], (const VOID **)(&key_ptr), &key_len);
            if (status != NX_SUCCESS)
            {
                return(status);
            }

            /* Check key length */
            if (key_len > NX_LWM2M_CLIENT_SECURITY_MAX_PUBLIC_KEY_OR_IDENTITY)
            {
                return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
            }

            /* Store key data */
            memcpy(instance_ptr -> security_instance_pub_key_or_id, key_ptr, key_len); /* Use case of memcpy is verified. */
            instance_ptr -> security_instance_pub_key_or_id_len = key_len;

            break;

        case NX_LWM2M_CLIENT_SECURITY_SERVER_PUBLIC_KEY_ID:

            /* Get key opaque data */
            status = _nx_lwm2m_client_resource_opaque_get(&resource[i], (const VOID **)(&key_ptr), &key_len);
            if (status != NX_SUCCESS)
            {
                return(status);
            }

            /* Check key length */
            if (key_len > NX_LWM2M_CLIENT_SECURITY_MAX_SERVER_PUBLIC_KEY)
            {
                return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
            }

            /* Store key data */
            memcpy(instance_ptr -> security_instance_server_pub_key, key_ptr, key_len); /* Use case of memcpy is verified. */
            instance_ptr -> security_instance_server_pub_key_len = key_len;

            break;

        case NX_LWM2M_CLIENT_SECURITY_SECRET_KEY_ID:

            /* Get key opaque data */
            status = _nx_lwm2m_client_resource_opaque_get(&resource[i], (const VOID **)(&key_ptr), &key_len);
            if (status != NX_SUCCESS)
            {
                return(status);
            }

            /* Check key length */
            if (key_len > NX_LWM2M_CLIENT_SECURITY_MAX_SECRET_KEY)
            {
                return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
            }

            /* Store key data */
            memcpy(instance_ptr -> security_instance_secret_key, key_ptr, key_len); /* Use case of memcpy is verified. */
            instance_ptr -> security_instance_secret_key_len = key_len;

            break;

#endif /* NX_SECURE_ENABLE_DTLS */

        case NX_LWM2M_CLIENT_SECURITY_SHORT_SERVER_ID:
            {
                NX_LWM2M_INT32 short_server_id;

                status = _nx_lwm2m_client_resource_integer32_get(&resource[i], &short_server_id);
                if (status != NX_SUCCESS)
                {
                    return(status);
                }

                /* Short server ID must be in range 0-65535 */
                if (short_server_id < 0 || short_server_id > 65535)
                {
                    return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
                }

                instance_ptr -> security_instance_server_id = (NX_LWM2M_ID) short_server_id;

                break;
            }

        case NX_LWM2M_CLIENT_SECURITY_HOLD_OFF_ID:

            status = _nx_lwm2m_client_resource_integer32_get(&resource[i], &instance_ptr -> security_instance_hold_off);
            if (status != NX_SUCCESS)
            {
                return(status);
            }
            break;

        default:

            /* Ignore unknown resources (bootstrap write) */
            break;
        }
    }

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_security_execute                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the execute operation.                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource_id                           The ID of resource            */
/*    args_ptr                              Pointer to arguments          */
/*    args_length                           The length of arguments       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_security_operation                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_security_execute(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_ID resource_id, const CHAR *args_ptr, UINT args_length)
{

    NX_PARAMETER_NOT_USED(object_ptr);
    NX_PARAMETER_NOT_USED(instance_ptr);
    NX_PARAMETER_NOT_USED(resource_id);
    NX_PARAMETER_NOT_USED(args_ptr);
    NX_PARAMETER_NOT_USED(args_length);

    /* cannot execute any resource */
    return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_security_create                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the create operation.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_id                           Pointer to object instance ID */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    bootstrap                             Flag of bootstrap step        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_security_write                                     */
/*    _nx_lwm2m_client_object_instance_add                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_security_operation                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_security_create(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_ID *instance_id, NX_LWM2M_CLIENT_RESOURCE *values_ptr, UINT resource_count, NX_LWM2M_BOOL bootstrap)
{
NX_LWM2M_CLIENT_SECURITY_OBJECT *security_ptr = (NX_LWM2M_CLIENT_SECURITY_OBJECT *) object_ptr;
NX_LWM2M_CLIENT_SECURITY_INSTANCE *instance_ptr;
UINT status;

    /* Only allow security instance creation during boostrap */
    if (!bootstrap)
    {
        return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
    }

    /* Allocate a new instance */
    instance_ptr = (NX_LWM2M_CLIENT_SECURITY_INSTANCE *) security_ptr -> security_object_instances_free_list;
    if (instance_ptr == NX_NULL)
    {
        return(NX_LWM2M_CLIENT_NO_MEMORY);
    }

    /* Initialize resources with default values */
    instance_ptr -> security_instance_server_id      = 0;
    instance_ptr -> security_instance_bootstrap      = NX_FALSE;
    instance_ptr -> security_instance_mode           = NX_LWM2M_CLIENT_SECURITY_MODE_UNDEFINED;
    instance_ptr -> security_instance_server_uri[0]  = 0;
    instance_ptr -> security_instance_hold_off       = 0;

    /* Write the provided resources */
    status = _nx_lwm2m_client_security_write((NX_LWM2M_CLIENT_OBJECT *) security_ptr, (NX_LWM2M_CLIENT_OBJECT_INSTANCE *) instance_ptr, values_ptr, resource_count, NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP);
    if (status != NX_SUCCESS)
    {
        return(status);
    }

    /* Update free instance list */
    security_ptr -> security_object_instances_free_list = instance_ptr -> security_instance.object_instance_next;

    /* Add new instance to the instance list */
    status = _nx_lwm2m_client_object_instance_add(object_ptr, &(instance_ptr -> security_instance), instance_id);
    if (status != NX_SUCCESS)
    {

        /* Revert the free instance list */
        security_ptr -> security_object_instances_free_list = &(instance_ptr -> security_instance);
        return(status);
    }

    /* Return success */
    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_security_delete                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the delete operation.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_instance_remove                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_security_operation                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_security_delete(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr)
{
NX_LWM2M_CLIENT_SECURITY_OBJECT *security_ptr = (NX_LWM2M_CLIENT_SECURITY_OBJECT *) object_ptr;
UINT status;

    /* Remove the instance from instance list */
    status = _nx_lwm2m_client_object_instance_remove(object_ptr, instance_ptr);
    if (status)
    {
        return(status);
    }

    /* Release the instance */
    instance_ptr -> object_instance_next = security_ptr -> security_object_instances_free_list;
    security_ptr -> security_object_instances_free_list = instance_ptr;

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_security_operation                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the object operation.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    operation                             The operation type            */
/*    object_ptr                            Pointer to object             */
/*    object_instance_ptr                   Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    args_ptr                              Pointer to arguments          */
/*    args_length                           The length of arguments       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_security_read                                      */
/*    _nx_lwm2m_client_security_discover                                  */
/*    _nx_lwm2m_client_security_write                                     */
/*    _nx_lwm2m_client_security_execute                                   */
/*    _nx_lwm2m_client_security_write                                     */
/*    _nx_lwm2m_client_security_delete                                    */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_security_operation(UINT operation, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *object_instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT *resource_count, VOID *args_ptr, UINT args_length)
{
UINT *write_op;
NX_LWM2M_CLIENT_OPERATION_ARGUMENT *arguments;

    switch (operation)
    {
    case NX_LWM2M_CLIENT_OBJECT_READ:

        /* Call read function */
        return(_nx_lwm2m_client_security_read(object_ptr, object_instance_ptr, resource, *resource_count));
    case NX_LWM2M_CLIENT_OBJECT_DISCOVER:

        /* Call discover function */
        return(_nx_lwm2m_client_security_discover(object_ptr, object_instance_ptr, resource, resource_count));
    case NX_LWM2M_CLIENT_OBJECT_WRITE:

        /* Get the type of write operation */
        write_op = (UINT *)args_ptr;

        /* Call write function */
        return(_nx_lwm2m_client_security_write(object_ptr, object_instance_ptr, resource, *resource_count, *write_op));
    case NX_LWM2M_CLIENT_OBJECT_EXECUTE:

        /* Call execute function */
        return(_nx_lwm2m_client_security_execute(object_ptr, object_instance_ptr, resource -> resource_id, args_ptr, args_length));
    case NX_LWM2M_CLIENT_OBJECT_CREATE:

        /*  Get arguments */
        arguments = (NX_LWM2M_CLIENT_OPERATION_ARGUMENT *)args_ptr;

        /* Call create function */
        return(_nx_lwm2m_client_security_create(object_ptr, &(arguments -> instance_id), resource, *resource_count, arguments -> bootstrap));
    case NX_LWM2M_CLIENT_OBJECT_DELETE:

        /* Call delete function */
        return(_nx_lwm2m_client_security_delete(object_ptr, object_instance_ptr));
    default:

        /*Unsupported operation */
        return(NX_LWM2M_CLIENT_NOT_SUPPORTED);

    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_security_object_create             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions initializes the Security Object.                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    none                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_add                                         */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_create                                             */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_security_object_create(NX_LWM2M_CLIENT *client_ptr)
{
#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES > 1
int i;
#endif
NX_LWM2M_CLIENT_SECURITY_OBJECT *security_ptr = &(client_ptr -> nx_lwm2m_client_security);

    /* Add Security Object to the LWM2M Client */
    _nx_lwm2m_client_object_add(client_ptr, &(security_ptr -> security_object), NX_LWM2M_CLIENT_SECURITY_OBJECT_ID, _nx_lwm2m_client_security_operation);

    /* Initialize the list of free instances */
    security_ptr -> security_object_instances_free_list = (NX_LWM2M_CLIENT_OBJECT_INSTANCE *) &security_ptr -> security_object_instances[0];
#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES > 1
    for (i = 0; i < NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES-1; i++)
    {
        security_ptr -> security_object_instances[i].security_instance.object_instance_next = (NX_LWM2M_CLIENT_OBJECT_INSTANCE *) &security_ptr -> security_object_instances[i+1];
    }
#endif
    security_ptr -> security_object_instances[NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES-1].security_instance.object_instance_next = NX_NULL;
}

#endif  /* NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES */



/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_server_read                        PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the read operation.                         */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_server_operation                                   */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_server_read(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT resource_count)
{
NX_LWM2M_CLIENT_SERVER_INSTANCE *server_ptr = (NX_LWM2M_CLIENT_SERVER_INSTANCE *) instance_ptr;
UINT length;
UINT i;

    NX_PARAMETER_NOT_USED(object_ptr);

    for (i = 0; i < resource_count; i++)
    {
        switch (resource[i].resource_id)
        {

        case NX_LWM2M_CLIENT_SERVER_SHORT_SERVER_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = server_ptr -> server_instance_short_id;
            break;

        case NX_LWM2M_CLIENT_SERVER_LIFETIME_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = server_ptr -> server_instance_lifetime;
            break;

        case NX_LWM2M_CLIENT_SERVER_MIN_PERIOD_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = server_ptr -> server_instance_min_period;
            break;

        case NX_LWM2M_CLIENT_SERVER_MAX_PERIOD_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = server_ptr -> server_instance_max_period;
            break;

        case NX_LWM2M_CLIENT_SERVER_DISABLE_TIMEOUT_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_INTEGER32;
            resource[i].resource_value.resource_integer32data = server_ptr -> server_instance_disable_timeout;
            break;

        case NX_LWM2M_CLIENT_SERVER_STORING_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_BOOLEAN;
            resource[i].resource_value.resource_booleandata = server_ptr -> server_instance_storing;
            break;

        case NX_LWM2M_CLIENT_SERVER_BINDING_ID:

            resource[i].resource_type = NX_LWM2M_CLIENT_RESOURCE_STRING;
            resource[i].resource_value.resource_bufferdata.resource_buffer_ptr = _nx_lwm2m_client_binding_mode_string_get(server_ptr -> server_instance_binding, &length);
            resource[i].resource_value.resource_bufferdata.resource_buffer_length = length;
            break;

        /* Non-readable resources */
        case NX_LWM2M_CLIENT_SERVER_DISABLE_ID:
        case NX_LWM2M_CLIENT_SERVER_UPDATE_ID:

            return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);

        /* Unknown resource */
        default:

            return(NX_LWM2M_CLIENT_NOT_FOUND);
        }
    }

    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_server_discover                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the discover operation.                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    num_resources_ptr                     Pointer to number of resource */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_server_operation                                   */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_server_discover(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resources, UINT *resource_count)
{

    NX_PARAMETER_NOT_USED(object_ptr);
    NX_PARAMETER_NOT_USED(instance_ptr);

    /* Check length of output buffer */
    if (*resource_count < 9)
    {
        return(NX_LWM2M_CLIENT_BUFFER_TOO_SMALL);
    }

    /* Return list of resources */
    resources[0].resource_id        = NX_LWM2M_CLIENT_SERVER_SHORT_SERVER_ID;
    resources[0].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ_WRITE;
    resources[1].resource_id        = NX_LWM2M_CLIENT_SERVER_LIFETIME_ID;
    resources[1].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ_WRITE;
    resources[2].resource_id        = NX_LWM2M_CLIENT_SERVER_MIN_PERIOD_ID;
    resources[2].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ_WRITE;
    resources[3].resource_id        = NX_LWM2M_CLIENT_SERVER_MAX_PERIOD_ID;
    resources[3].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ_WRITE;
    resources[4].resource_id        = NX_LWM2M_CLIENT_SERVER_DISABLE_ID;
    resources[4].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_EXECUTABLE;
    resources[5].resource_id        = NX_LWM2M_CLIENT_SERVER_DISABLE_TIMEOUT_ID;
    resources[5].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ_WRITE;
    resources[6].resource_id        = NX_LWM2M_CLIENT_SERVER_STORING_ID;
    resources[6].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ_WRITE;
    resources[7].resource_id        = NX_LWM2M_CLIENT_SERVER_BINDING_ID;
    resources[7].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ_WRITE;
    resources[8].resource_id        = NX_LWM2M_CLIENT_SERVER_UPDATE_ID;
    resources[8].resource_operation = NX_LWM2M_CLIENT_RESOURCE_OPERATION_EXECUTABLE;

    *resource_count = 9;

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_server_write                       PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the write operation.                        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    object_instance_ptr                   Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    write_op                              Type of write operation       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_resource_integer32_get                             */
/*    _nx_lwm2m_client_resource_boolean_get                               */
/*    _nx_lwm2m_client_resource_string_get                                */
/*    _nx_lwm2m_client_binding_mode_string_parse                          */
/*    _nx_lwm2m_client_server_session_get                                 */
/*    _nx_lwm2m_client_session_update_flags                               */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_server_operation                                   */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_server_write(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *object_instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT resource_count, UINT write_op)
{
NX_LWM2M_CLIENT_SERVER_INSTANCE *instance_ptr = (NX_LWM2M_CLIENT_SERVER_INSTANCE *) object_instance_ptr;
UINT status;
ULONG flags;
NX_LWM2M_INT32 short_server_id;
NX_LWM2M_INT32 lifetime;
NX_LWM2M_INT32 min_period;
NX_LWM2M_INT32 max_period;
NX_LWM2M_INT32 disable_timeout;
NX_LWM2M_BOOL storing;
UCHAR binding = 0;
UINT i;

    /* XXX replace not supported? */
    if (write_op == NX_LWM2M_CLIENT_OBJECT_WRITE_REPLACE_INSTANCE)
    {
        return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);
    }

    /* Check validity of values before updating instance */
    flags = 0;

    for (i = 0 ; i < resource_count; i++)
    {
        switch (resource[i].resource_id)
        {

        case NX_LWM2M_CLIENT_SERVER_SHORT_SERVER_ID:

            /* Resource is read-only, can only be set during boostrap or instance creation */
            if (write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP && write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_CREATE)
            {
                return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);
            }
            status = _nx_lwm2m_client_resource_integer32_get(&resource[i], &short_server_id);
            if (status != NX_SUCCESS)
            {
                return(status);
            }

            /* Short server ID must be in range 1-65535 */
            if (short_server_id < 1 || short_server_id > 65535)
            {
                return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
            }
            flags |= 1<<NX_LWM2M_CLIENT_SERVER_SHORT_SERVER_ID;
            break;

        case NX_LWM2M_CLIENT_SERVER_LIFETIME_ID:

            status = _nx_lwm2m_client_resource_integer32_get(&resource[i], &lifetime);
            if (status != NX_SUCCESS)
            {
                return(status);
            }
            flags |= 1<<NX_LWM2M_CLIENT_SERVER_LIFETIME_ID;
            break;

        case NX_LWM2M_CLIENT_SERVER_MIN_PERIOD_ID:

            status = _nx_lwm2m_client_resource_integer32_get(&resource[i], &min_period);
            if (status != NX_SUCCESS)
            {
                return(status);
            }
            flags |= 1<<NX_LWM2M_CLIENT_SERVER_MIN_PERIOD_ID;
            break;

        case NX_LWM2M_CLIENT_SERVER_MAX_PERIOD_ID:

            status = _nx_lwm2m_client_resource_integer32_get(&resource[i], &max_period);
            if (status != NX_SUCCESS)
            {
                return(status);
            }
            flags |= 1<<NX_LWM2M_CLIENT_SERVER_MAX_PERIOD_ID;
            break;

        case NX_LWM2M_CLIENT_SERVER_DISABLE_TIMEOUT_ID:

            status = _nx_lwm2m_client_resource_integer32_get(&resource[i], &disable_timeout);
            if (status != NX_SUCCESS)
            {
                return(status);
            }
            flags |= 1<<NX_LWM2M_CLIENT_SERVER_DISABLE_TIMEOUT_ID;
            break;

        case NX_LWM2M_CLIENT_SERVER_STORING_ID:

            status = _nx_lwm2m_client_resource_boolean_get(&resource[i], &storing);
            if (status != NX_SUCCESS)
            {
                return(status);
            }
            flags |= 1<<NX_LWM2M_CLIENT_SERVER_STORING_ID;
            break;

        case NX_LWM2M_CLIENT_SERVER_BINDING_ID:
            {
                const CHAR *binding_str;
                UINT binding_strlen;
                status = _nx_lwm2m_client_resource_string_get(&resource[i], &binding_str, &binding_strlen);
                if (status != NX_SUCCESS)
                {
                    return(status);
                }

                /* Parse and check binding string */
                binding = _nx_lwm2m_client_binding_mode_string_parse(binding_str, binding_strlen);
                if (binding != NX_LWM2M_CLIENT_BINDING_U &&
                    binding != NX_LWM2M_CLIENT_BINDING_UQ &&
                    binding != NX_LWM2M_CLIENT_BINDING_S &&
                    binding != NX_LWM2M_CLIENT_BINDING_SQ &&
                    binding != NX_LWM2M_CLIENT_BINDING_US &&
                    binding != NX_LWM2M_CLIENT_BINDING_UQS)
                {

                    /* Not a valid binding mode */
                    return(NX_LWM2M_CLIENT_NOT_ACCEPTABLE);
                }
                flags |= 1<<NX_LWM2M_CLIENT_SERVER_BINDING_ID;
                break;
            }


        /* Cannot write to executable resources */
        case NX_LWM2M_CLIENT_SERVER_DISABLE_ID:
        case NX_LWM2M_CLIENT_SERVER_UPDATE_ID:

            return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);

        /* Unknown resource */
        default:

            if (write_op != NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP)
            {
                return(NX_LWM2M_CLIENT_NOT_FOUND);
            }

            /* Ignore unknown resources during bootstrap */
            break;
        }
    }

    /* Update the instance */

    if (flags & 1<<NX_LWM2M_CLIENT_SERVER_SHORT_SERVER_ID)
    {
        instance_ptr -> server_instance_short_id         = (NX_LWM2M_ID) short_server_id;
    }
    if (flags & 1<<NX_LWM2M_CLIENT_SERVER_STORING_ID)
    {
        instance_ptr -> server_instance_storing          = storing;
    }
    if ((flags & 1<<NX_LWM2M_CLIENT_SERVER_BINDING_ID) && instance_ptr -> server_instance_binding != binding)
    {
        NX_LWM2M_CLIENT_SESSION *session_ptr;

        instance_ptr -> server_instance_binding          = binding;

        /* Update session binding if we are currently registered to the server */
        session_ptr = _nx_lwm2m_client_server_session_get(object_ptr -> object_client_ptr, object_instance_ptr);
        if (session_ptr != NX_NULL)
        {
            _nx_lwm2m_client_session_update_flags(session_ptr, NX_LWM2M_CLIENT_SESSION_UPDATE_BINDING_MODE);
        }
    }
    if ((flags & 1<<NX_LWM2M_CLIENT_SERVER_LIFETIME_ID) && instance_ptr -> server_instance_lifetime != lifetime)
    {
        NX_LWM2M_CLIENT_SESSION *session_ptr;

        instance_ptr -> server_instance_lifetime         = lifetime;

        /* Update session lifetime if we are currently registered to the server */
        session_ptr = _nx_lwm2m_client_server_session_get(object_ptr -> object_client_ptr, object_instance_ptr);
        if (session_ptr != NX_NULL)
        {
            _nx_lwm2m_client_session_update_flags(session_ptr, NX_LWM2M_CLIENT_SESSION_UPDATE_LIFETIME);
        }
    }
    if (flags & 1<<NX_LWM2M_CLIENT_SERVER_MIN_PERIOD_ID)
    {
        instance_ptr -> server_instance_min_period       = min_period;
    }
    if (flags & 1<<NX_LWM2M_CLIENT_SERVER_MAX_PERIOD_ID)
    {
        instance_ptr -> server_instance_max_period       = max_period;
    }
    if (flags & 1<<NX_LWM2M_CLIENT_SERVER_DISABLE_TIMEOUT_ID)
    {
        instance_ptr -> server_instance_disable_timeout  = disable_timeout;
    }

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_server_execute                     PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the execute operation.                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource_id                           The ID of resource            */
/*    args_ptr                              Pointer to arguments          */
/*    args_length                           The length of arguments       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_server_session_get                                 */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_server_operation                                   */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_server_execute(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_ID resource_id, const CHAR *args_ptr, UINT args_length)
{
NX_LWM2M_CLIENT_SESSION *session_ptr;

    NX_PARAMETER_NOT_USED(args_ptr);
    NX_PARAMETER_NOT_USED(args_length);

    switch (resource_id)
    {

    /* Non-executable resources */
    case NX_LWM2M_CLIENT_SERVER_SHORT_SERVER_ID:
    case NX_LWM2M_CLIENT_SERVER_LIFETIME_ID:
    case NX_LWM2M_CLIENT_SERVER_MIN_PERIOD_ID:
    case NX_LWM2M_CLIENT_SERVER_MAX_PERIOD_ID:
    case NX_LWM2M_CLIENT_SERVER_DISABLE_TIMEOUT_ID:
    case NX_LWM2M_CLIENT_SERVER_STORING_ID:
    case NX_LWM2M_CLIENT_SERVER_BINDING_ID:

        return(NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED);

    /* Executable resources */
    case NX_LWM2M_CLIENT_SERVER_DISABLE_ID:
    case NX_LWM2M_CLIENT_SERVER_UPDATE_ID:
        {

            /* Disable/update the session if it is actually registered */
            session_ptr = _nx_lwm2m_client_server_session_get(object_ptr -> object_client_ptr, instance_ptr);
            if ((session_ptr != NX_NULL) &&
                ((session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_REGISTERED) ||
                 (session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_UPDATING)))
            {
                if (resource_id == NX_LWM2M_CLIENT_SERVER_DISABLE_ID)
                {
                    session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_DISABLE;
                }
                else
                {
                    session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_UPDATE;
                }

                /* Wakeup client thread */
                tx_event_flags_set(&object_ptr -> object_client_ptr -> nx_lwm2m_client_event_flags, NX_LWM2M_CLIENT_EVENT_WAKEUP, TX_OR);
            }

            /* Return success */
            /* XXX should we return an error if the client is not registered? */
            return(NX_SUCCESS);
        }

    /* unknown resource */
    default:

        return(NX_LWM2M_CLIENT_NOT_FOUND);
    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_server_create                      PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the create operation.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_id                           Pointer to object instance ID */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    bootstrap                             Flag of bootstrap step        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_server_write                                       */
/*    _nx_lwm2m_client_object_instance_add                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_server_operation                                   */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_server_create(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_ID *instance_id, NX_LWM2M_CLIENT_RESOURCE *values_ptr, UINT resource_count, NX_LWM2M_BOOL bootstrap)
{
NX_LWM2M_CLIENT_SERVER_OBJECT *server_ptr = (NX_LWM2M_CLIENT_SERVER_OBJECT *) object_ptr;
NX_LWM2M_CLIENT_SERVER_INSTANCE *instance_ptr;
UINT status;

    /* Allocate a new instance */
    instance_ptr = (NX_LWM2M_CLIENT_SERVER_INSTANCE *) server_ptr -> server_object_instances_free_list;
    if (instance_ptr == NX_NULL)
    {
        return(NX_LWM2M_CLIENT_NO_MEMORY);
    }

    /* Initialize resources with default values */
    instance_ptr -> server_instance_short_id         = 0;
    instance_ptr -> server_instance_storing          = NX_TRUE;
    instance_ptr -> server_instance_binding          = NX_LWM2M_CLIENT_BINDING_U;
    instance_ptr -> server_instance_lifetime         = 86400;
    instance_ptr -> server_instance_min_period       = 0;
    instance_ptr -> server_instance_max_period       = 0;
    instance_ptr -> server_instance_disable_timeout  = 86400;

    /* Write the provided resources */
    status = _nx_lwm2m_client_server_write((NX_LWM2M_CLIENT_OBJECT *) server_ptr, (NX_LWM2M_CLIENT_OBJECT_INSTANCE *) instance_ptr, values_ptr, resource_count, bootstrap ? NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP : NX_LWM2M_CLIENT_OBJECT_WRITE_CREATE);
    if (status != NX_SUCCESS)
    {
        return(status);
    }

    /* Update free instance list */
    server_ptr -> server_object_instances_free_list = instance_ptr -> server_instance.object_instance_next;

    /* Add new instance to the instance list */
    status = _nx_lwm2m_client_object_instance_add(object_ptr, &(instance_ptr -> server_instance), instance_id);
    if (status != NX_SUCCESS)
    {

        /* Revert the free instance list */
        server_ptr -> server_object_instances_free_list = &(instance_ptr -> server_instance);
        return(status);
    }

    /* Return success */
    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_server_delete                      PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the delete operation.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_instance_remove                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_server_operation                                   */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_server_delete(NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr)
{
NX_LWM2M_CLIENT_SERVER_OBJECT *server_ptr = (NX_LWM2M_CLIENT_SERVER_OBJECT *) object_ptr;
UINT status;

    /* Remove the instance from instance list */
    status = _nx_lwm2m_client_object_instance_remove(object_ptr, instance_ptr);
    if (status)
    {
        return(status);
    }

    /* Release the instance */
    instance_ptr -> object_instance_next = server_ptr -> server_object_instances_free_list;
    server_ptr -> server_object_instances_free_list = instance_ptr;

    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_server_operation                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes the object operation.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    operation                             The operation type            */
/*    object_ptr                            Pointer to object             */
/*    object_instance_ptr                   Pointer to object instance    */
/*    resource                              Pointer to resource           */
/*    resource_count                        The number of resource        */
/*    args_ptr                              Pointer to arguments          */
/*    args_length                           The length of arguments       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_server_read                                        */
/*    _nx_lwm2m_client_server_discover                                    */
/*    _nx_lwm2m_client_server_write                                       */
/*    _nx_lwm2m_client_server_execute                                     */
/*    _nx_lwm2m_client_server_write                                       */
/*    _nx_lwm2m_client_server_delete                                      */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_server_operation(UINT operation, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *object_instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource, UINT *resource_count, VOID *args_ptr, UINT args_length)
{
UINT *write_op;
NX_LWM2M_CLIENT_OPERATION_ARGUMENT *arguments;

    switch (operation)
    {
    case NX_LWM2M_CLIENT_OBJECT_READ:

        /* Call read function */
        return(_nx_lwm2m_client_server_read(object_ptr, object_instance_ptr, resource, *resource_count));
    case NX_LWM2M_CLIENT_OBJECT_DISCOVER:

        /* Call discover function */
        return(_nx_lwm2m_client_server_discover(object_ptr, object_instance_ptr, resource, resource_count));
    case NX_LWM2M_CLIENT_OBJECT_WRITE:

        /* Get the type of write operation */
        write_op = (UINT *)args_ptr;

        /* Call write function */
        return(_nx_lwm2m_client_server_write(object_ptr, object_instance_ptr, resource, *resource_count, *write_op));
    case NX_LWM2M_CLIENT_OBJECT_EXECUTE:

        /* Call execute function */
        return(_nx_lwm2m_client_server_execute(object_ptr, object_instance_ptr, resource -> resource_id, args_ptr, args_length));
    case NX_LWM2M_CLIENT_OBJECT_CREATE:

        /*  Get arguments */
        arguments = (NX_LWM2M_CLIENT_OPERATION_ARGUMENT *)args_ptr;

        /* Call create function */
        return(_nx_lwm2m_client_server_create(object_ptr, &(arguments -> instance_id), resource, *resource_count, arguments -> bootstrap));
    case NX_LWM2M_CLIENT_OBJECT_DELETE:

        /* Call delete function */
        return(_nx_lwm2m_client_server_delete(object_ptr, object_instance_ptr));
    default:

        /*Unsupported operation */
        return(NX_LWM2M_CLIENT_NOT_SUPPORTED);

    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_server_object_create               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions initializes the Server Object.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    none                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_add                                         */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_create                                             */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_server_object_create(NX_LWM2M_CLIENT *client_ptr)
{
#if NX_LWM2M_CLIENT_MAX_SERVER_INSTANCES > 1
int i;
#endif
NX_LWM2M_CLIENT_SERVER_OBJECT *server_ptr = &(client_ptr -> nx_lwm2m_client_server);

    /* Add Server Object to the LWM2M Client */
    _nx_lwm2m_client_object_add(client_ptr, &(server_ptr -> server_object), NX_LWM2M_CLIENT_SERVER_OBJECT_ID, _nx_lwm2m_client_server_operation);

    /* Initialize the list of free instances */
    server_ptr -> server_object_instances_free_list = (NX_LWM2M_CLIENT_OBJECT_INSTANCE *) &server_ptr -> server_object_instances[0];
#if NX_LWM2M_CLIENT_MAX_SERVER_INSTANCES > 1
    for (i = 0; i < NX_LWM2M_CLIENT_MAX_SERVER_INSTANCES-1; i++)
    {
        server_ptr -> server_object_instances[i].server_instance.object_instance_next = (NX_LWM2M_CLIENT_OBJECT_INSTANCE *) &server_ptr -> server_object_instances[i+1];
    }
#endif
    server_ptr -> server_object_instances[NX_LWM2M_CLIENT_MAX_SERVER_INSTANCES-1].server_instance.object_instance_next = NX_NULL;
}



/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_server_session_get                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    Return the session associated with a server instance.               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LWM2M client       */
/*    server_instance_ptr                   Pointer to the server         */
/*                                          instance                      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    session_ptr                           The pointer to the session    */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_object_delete                                      */
/*    _nx_lwm2m_client_server_write                                       */
/*    _nx_lwm2m_client_server_execute                                     */
/*    _nx_lwm2m_client_session_receive                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
NX_LWM2M_CLIENT_SESSION *_nx_lwm2m_client_server_session_get(NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *server_instance_ptr)
{
NX_LWM2M_CLIENT_SESSION *session_ptr;

    /* Look if the instance is attached to a session */
    session_ptr = client_ptr -> nx_lwm2m_client_sessions;
    while (session_ptr != NX_NULL)
    {
        if (session_ptr -> nx_lwm2m_client_session_server_instance_ptr == (NX_LWM2M_CLIENT_SERVER_INSTANCE *) server_instance_ptr)
        {
            return(session_ptr);
        }

        session_ptr = session_ptr -> nx_lwm2m_client_session_next;
    }

    return(NX_NULL);
}

#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_bootstrap                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function starts a session with a Bootstrap server.             */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    security_id                           The Security Instance ID of   */
/*                                          the bootstrap server          */
/*    ip_address                            The IP address of the server  */
/*    port                                  The UDP port of the server    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_start                                      */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_bootstrap(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_ID security_id, const NXD_ADDRESS *ip_address, UINT port)
{

    /* Start bootstrap. */
#ifdef NX_SECURE_ENABLE_DTLS
    return(_nx_lwm2m_client_session_start(session_ptr, NX_TRUE, security_id, ip_address, port, NX_NULL));
#else
    return(_nx_lwm2m_client_session_start(session_ptr, NX_TRUE, security_id, ip_address, port));
#endif /* NX_SECURE_ENABLE_DTLS */
}

#ifdef NX_SECURE_ENABLE_DTLS

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_bootstrap_dtls             PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function starts a session with a Bootstrap server using a      */
/*    secure channel.                                                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    security_id                           The Security Instance ID of   */
/*                                          the bootstrap server          */
/*    ip_address                            The IP address of the server  */
/*    port                                  The UDP port of the server    */
/*    dtls_session_ptr                      The pointer to the DTLS       */
/*                                          session                       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_start                                      */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_bootstrap_dtls(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_ID security_id, const NXD_ADDRESS *ip_address, UINT port, NX_SECURE_DTLS_SESSION *dtls_session_ptr)
{

    /* Start bootstrap. */
    return(_nx_lwm2m_client_session_start(session_ptr, NX_TRUE, security_id, ip_address, port, dtls_session_ptr));
}

#endif /* NX_SECURE_ENABLE_DTLS */
#endif /* NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES */


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_cleanup                    PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function cleans up the LWM2M session.                          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    none                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    nx_secure_dtls_session_end                                          */
/*    nx_secure_dtls_session_reset                                        */
/*    nx_udp_socket_unbind                                                */
/*    nx_udp_socket_delete                                                */
/*    _nx_lwm2m_client_notify_free                                        */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_state_update                               */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_session_cleanup(NX_LWM2M_CLIENT_SESSION *session_ptr)
{
NX_LWM2M_CLIENT_NOTIFY * notify_ptr;
UINT state = session_ptr -> nx_lwm2m_client_session_state;

    if ((state == NX_LWM2M_CLIENT_SESSION_DISABLED) && 
        (session_ptr -> nx_lwm2m_client_session_server_instance_ptr -> server_instance_disable_timeout > 0))
    {

        /* Arm disable timeout */
        session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_WAIT;

        session_ptr -> nx_lwm2m_client_session_timer = tx_time_get() + (ULONG)(session_ptr -> nx_lwm2m_client_session_server_instance_ptr -> server_instance_disable_timeout) * NX_IP_PERIODIC_RATE;
    }
    else
    {

        /* Set session to idle state */
        session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_IDLE;
    }

#ifdef NX_SECURE_ENABLE_DTLS

    /* Close DTLS session */
    if (session_ptr -> nx_lwm2m_client_session_dtls_session != NX_NULL)
    {
        if (state == NX_LWM2M_CLIENT_SESSION_DISABLED ||
            state == NX_LWM2M_CLIENT_SESSION_DEREGISTERED ||
            state == NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_FINISHED)
        {

            /* We have deregistered from the server,
                properly shutdown the DTLS session */
            nx_secure_dtls_session_end(session_ptr -> nx_lwm2m_client_session_dtls_session, NX_LWM2M_CLIENT_DTLS_END_TIMEOUT);
        }
        else
        {

            /* lwm2m session was aborted, just reset the DTLS session */
            nx_secure_dtls_session_reset(session_ptr -> nx_lwm2m_client_session_dtls_session);
        }

        /*
            * keep the DTLS session and socket if we are disabled so we are
            * able to reconnect again to the server when disable timer expires,
            * otherwise release all DTLS resources
            */
        if ((state != NX_LWM2M_CLIENT_SESSION_DISABLED) ||
            (session_ptr -> nx_lwm2m_client_session_server_instance_ptr -> server_instance_disable_timeout == 0))
        {

            /* Delete reference to DTLS session */
            session_ptr -> nx_lwm2m_client_session_dtls_session = NX_NULL;

            /* Unbind the DTLS session socket */
            nx_udp_socket_unbind((NX_UDP_SOCKET *)&session_ptr -> nx_lwm2m_client_session_dtls_socket);

            /* Delete the DTLS session socket */
            nx_udp_socket_delete((NX_UDP_SOCKET *)&session_ptr -> nx_lwm2m_client_session_dtls_socket);
        }
    }
#endif

    /* Free notifications */
    while (1)
    {

        /* Get next notification */
        notify_ptr = session_ptr -> nx_lwm2m_client_session_notifications;

        if (notify_ptr == NX_NULL)
        {

            /* All notifications have been deleted */
            break;
        }

        /* Detach notification */
        session_ptr -> nx_lwm2m_client_session_notifications = notify_ptr -> notify_next;

        /* Free the structure */
        _nx_lwm2m_client_notify_free(session_ptr -> nx_lwm2m_client_session_client_ptr, notify_ptr);

    }
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_create                     PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions creates a LWM2M Client Session.                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    client_ptr                            Pointer to LWM2M Client       */
/*    state_callback                        The session state callback    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_create(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_CLIENT *client_ptr, NX_LWM2M_CLIENT_SESSION_STATE_CALLBACK state_callback)
{

    /* Initialize the session control block to zero. */
    memset(session_ptr, 0, sizeof(NX_LWM2M_CLIENT_SESSION));

    /* Set the pointer to the LWM2M Client */
    session_ptr -> nx_lwm2m_client_session_client_ptr = client_ptr;

    /* Reset the pointer to the server object instance */
    session_ptr -> nx_lwm2m_client_session_server_instance_ptr = NX_NULL;

    /* Set the state callback */
    session_ptr -> nx_lwm2m_client_session_state_callback = state_callback;

    /* Set the initial state */
    session_ptr -> nx_lwm2m_client_session_state = NX_LWM2M_CLIENT_SESSION_INIT;

    /* Set the initial sub-state */
    session_ptr -> nx_lwm2m_client_session_state = NX_LWM2M_CLIENT_SESSION_SUBSTATE_IDLE;

    /* Reset Uri-Path of server */
    session_ptr -> nx_lwm2m_client_session_uri_path_length = 0;

    /* Reset flags */
    session_ptr -> nx_lwm2m_client_session_flags = 0;

    /* Reset error code */
    session_ptr -> nx_lwm2m_client_session_error = NX_SUCCESS;

    /* Clear list of notifications */
    session_ptr -> nx_lwm2m_client_session_notifications = NX_NULL;

#ifdef NX_SECURE_ENABLE_DTLS
    /* Clear pointer to DTLS session */
    session_ptr -> nx_lwm2m_client_session_dtls_session = NX_NULL;
#endif /* NX_SECURE_ENABLE_DTLS */

    /* Acquire the client mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Attach the new session to the client */
    session_ptr -> nx_lwm2m_client_session_next = client_ptr -> nx_lwm2m_client_sessions;
    client_ptr -> nx_lwm2m_client_sessions = session_ptr;

    /* Release the client mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Retun success */
    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_delete                     PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function deletes a LWM2M Client Session.                       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_session_state_update                               */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_delete(NX_LWM2M_CLIENT_SESSION *session_ptr)
{
NX_LWM2M_CLIENT *client_ptr = session_ptr -> nx_lwm2m_client_session_client_ptr;

    /* Acquire the client mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Clear all resources attached to this session */
    if (session_ptr -> nx_lwm2m_client_session_state != NX_LWM2M_CLIENT_SESSION_INIT)
    {
        _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_INIT);
    }

    /* Detach the session from the client */
    if (client_ptr -> nx_lwm2m_client_sessions == session_ptr)
    {

        /* First session of the list */
        client_ptr -> nx_lwm2m_client_sessions = session_ptr -> nx_lwm2m_client_session_next;
    }
    else
    {
        NX_LWM2M_CLIENT_SESSION *current_ptr = client_ptr -> nx_lwm2m_client_sessions;
        while (current_ptr != NX_NULL)
        {

            /* Find the preceding session */
            if (current_ptr -> nx_lwm2m_client_session_next == session_ptr)
            {
                current_ptr -> nx_lwm2m_client_session_next = session_ptr -> nx_lwm2m_client_session_next;
                break;
            }

            current_ptr = current_ptr -> nx_lwm2m_client_session_next;
        }
    }

    /* Release mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    return(NX_SUCCESS);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_deregister                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function deregisters the Client from a LWM2M Server.           */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_deregister(NX_LWM2M_CLIENT_SESSION *session_ptr)
{
UINT status = NX_SUCCESS;
NX_LWM2M_CLIENT *client_ptr = session_ptr -> nx_lwm2m_client_session_client_ptr;


    /* Acquire the client mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Check session state */
    if ((session_ptr -> nx_lwm2m_client_session_state != NX_LWM2M_CLIENT_SESSION_REGISTERED &&
         session_ptr -> nx_lwm2m_client_session_state != NX_LWM2M_CLIENT_SESSION_UPDATING) ||
        session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_DEREGISTER)
    {
        status = NX_LWM2M_CLIENT_NOT_REGISTERED;
    }
    else
    {

        /* Tell session to start deregistering */
        session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_DEREGISTER;
    }

    /* Release the client mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    if (status == NX_SUCCESS)
    {

        /* Wakeup client thread */
        tx_event_flags_set(&client_ptr -> nx_lwm2m_client_event_flags, NX_LWM2M_CLIENT_EVENT_WAKEUP, TX_OR);
    }

    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_error_get                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the error code of a LWM2M Session.            */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                The error code                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_error_get(NX_LWM2M_CLIENT_SESSION *session_ptr)
{
NX_LWM2M_CLIENT *client_ptr = session_ptr -> nx_lwm2m_client_session_client_ptr;
UINT error;

    /* Acquire the client mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Get the error code */
    error = session_ptr -> nx_lwm2m_client_session_error;

    /* Release the client mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return the error code */
    return(error);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_notify_attributes          PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function returns the attributes of a notification, including   */
/*    the inherited attributes.                                           */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    notify_ptr                            Pointer to the notification   */
/*    flags_ptr                             On return, the attributes set */
/*    pmin_ptr                              On return, the pmin attribute */
/*    pmax_ptr                              On return, the pmax attribute */
/*    gt_ptr                                On return, the gt attribute   */
/*    lt_ptr                                On return, the lt attribute   */
/*    stp_ptr                               On return, the stp attribute  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    none                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_object_resource_changed                            */
/*    _nx_lwm2m_client_session_receive                                    */
/*    _nx_lwm2m_client_session_step                                       */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_session_notify_attributes(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_CLIENT_NOTIFY *notify_ptr, UINT *flags_ptr, NX_LWM2M_INT32 *pmin_ptr, NX_LWM2M_INT32 *pmax_ptr, NX_LWM2M_CLIENT_NOTIFY_NUMBER *gt_ptr, NX_LWM2M_CLIENT_NOTIFY_NUMBER *lt_ptr, NX_LWM2M_CLIENT_NOTIFY_NUMBER *stp_ptr)
{
NX_LWM2M_CLIENT_NOTIFY *current_ptr;
UINT object_flags;

    /* Initialize attributes */
    *flags_ptr  = notify_ptr -> notify_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_FLAGS;
    *pmin_ptr   = notify_ptr -> notify_attr_pmin;
    *pmax_ptr   = notify_ptr -> notify_attr_pmax;
    *gt_ptr     = notify_ptr -> notify_attr_gt;
    *lt_ptr     = notify_ptr -> notify_attr_lt;
    *stp_ptr    = notify_ptr -> notify_attr_stp;

    if (notify_ptr -> notify_instance_ptr == NX_NULL)
    {

        /* Object notification, no inherited attribute */
        return;
    }

    /* Look for inherited attributes from object and object instance */
    object_flags = 0;
    current_ptr = session_ptr -> nx_lwm2m_client_session_notifications;
    while (current_ptr != NX_NULL)
    {
        if ((current_ptr != notify_ptr) && (current_ptr -> notify_object_ptr == notify_ptr -> notify_object_ptr) && 
            (current_ptr -> notify_resource_id == NX_LWM2M_CLIENT_RESERVED_ID) && 
            ((current_ptr -> notify_instance_ptr == NX_NULL) || (current_ptr -> notify_instance_ptr== notify_ptr -> notify_instance_ptr)))
        {
            UINT flags;
            if (current_ptr -> notify_instance_ptr == NX_NULL)
            {

                /* Object attributes, do not overwrite attributes already set */
                flags = current_ptr -> notify_flags & ~(*flags_ptr) & NX_LWM2M_CLIENT_NOTIFY_ATTR_FLAGS;
                object_flags = flags;
            }
            else
            {

                /* Instance attributes, can only overwrite object attributes */
                flags = current_ptr -> notify_flags & (~(*flags_ptr) | object_flags) & NX_LWM2M_CLIENT_NOTIFY_ATTR_FLAGS;
            }
            if (flags != 0)
            {
                *flags_ptr |= flags;
                if (flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_PMIN)
                {
                    *pmin_ptr   = current_ptr -> notify_attr_pmin;
                }
                if (flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_PMAX)
                {
                    *pmax_ptr   = current_ptr -> notify_attr_pmax;
                }
                if (flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_GT)
                {
                    *gt_ptr     = current_ptr -> notify_attr_gt;
                }
                if (flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_LT)
                {
                    *lt_ptr     = current_ptr -> notify_attr_lt;
                }
                if (flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_STP)
                {
                    *stp_ptr    = current_ptr -> notify_attr_stp;
                }
            }
        }

        /* Next notification */
        current_ptr = current_ptr -> notify_next;
    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_notify_free                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function releases a notification from a LWM2M session.         */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    notify_ptr                            Pointer to the notification   */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    none                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_notify_free                                        */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_receive                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_session_notify_free(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_CLIENT_NOTIFY *notify_ptr)
{
NX_LWM2M_CLIENT_NOTIFY *current_ptr;

    /* Detach the structure from the list */
    current_ptr = session_ptr -> nx_lwm2m_client_session_notifications;
    if (current_ptr == notify_ptr)
    {
        session_ptr -> nx_lwm2m_client_session_notifications = notify_ptr -> notify_next;
    }
    else
    {
        while (current_ptr != NX_NULL)
        {
            if (current_ptr -> notify_next == notify_ptr)
            {
                current_ptr -> notify_next = notify_ptr -> notify_next;

                break;
            }

            current_ptr = current_ptr -> notify_next;
        }
    }

    /* Free structure */
    _nx_lwm2m_client_notify_free(session_ptr -> nx_lwm2m_client_session_client_ptr, notify_ptr);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_notify_get                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function gets an existing notification structure or allocates  */
/*    a new one if requested.                                             */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    object_ptr                            Pointer to the Object         */
/*    instance_ptr                          Pointer to the Instance, can  */
/*                                          be NULL                       */
/*    resource_id                           The resource ID               */
/*    create                                If True, creates the          */
/*                                          notification if it doesnt     */
/*                                          exists                        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    notify_ptr                            Pointer to the notification   */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_notify_allocate                                    */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_write_attributes                                   */
/*    _nx_lwm2m_client_session_send_response                              */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
NX_LWM2M_CLIENT_NOTIFY * _nx_lwm2m_client_session_notify_get(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_ID resource_id, NX_LWM2M_BOOL create)
{
NX_LWM2M_CLIENT_NOTIFY *notify_ptr;

    /* Look for existing notification */
    notify_ptr = session_ptr -> nx_lwm2m_client_session_notifications;
    while (notify_ptr != NX_NULL)
    {
        if (notify_ptr -> notify_object_ptr == object_ptr && notify_ptr -> notify_instance_ptr == instance_ptr && notify_ptr -> notify_resource_id == resource_id)
        {
            break;
        }

        notify_ptr = notify_ptr -> notify_next;
    }

    if ((notify_ptr != NX_NULL) || (!create))
    {

        /* Notification found or creation not asked */
        return(notify_ptr);
    }

    /* Create a new notification */
    notify_ptr = _nx_lwm2m_client_notify_allocate(session_ptr -> nx_lwm2m_client_session_client_ptr);

    if (notify_ptr == NX_NULL)
    {

        /* Failed to allocate notification */
        return(NX_NULL);
    }

    /* Initialize structure */
    notify_ptr -> notify_object_ptr      = object_ptr;
    notify_ptr -> notify_instance_ptr    = instance_ptr;
    notify_ptr -> notify_resource_id     = resource_id;
    notify_ptr -> notify_flags           = 0;

    /* Insert structure in the list */
    notify_ptr -> notify_next = session_ptr -> nx_lwm2m_client_session_notifications;
    session_ptr -> nx_lwm2m_client_session_notifications = notify_ptr;

    return(notify_ptr);
}

/* size of Physical+IP+UDP headers */
#ifdef FEATURE_NX_IPV6
#define LWM2M_PACKET(session_ptr)   (session_ptr -> nx_lwm2m_client_session_server_address.nxd_ip_version == NX_IP_VERSION_V4 ? NX_IPv4_UDP_PACKET : NX_IPv6_UDP_PACKET)
#else
#define LWM2M_PACKET(session_ptr)   NX_UDP_PACKET
#endif


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_packet_allocate            PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions allocates and initializes a packet from the IP pool. */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    packet_ptr                            The allocated packet          */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
NX_PACKET *_nx_lwm2m_client_session_packet_allocate(NX_LWM2M_CLIENT_SESSION *session_ptr)
{
NX_PACKET *packet_ptr;
UINT status;

    /* Allocate a packet from the pool */
#ifdef NX_SECURE_ENABLE_DTLS
    if (session_ptr -> nx_lwm2m_client_session_dtls_session != NX_NULL)
    {
        status = nx_secure_dtls_packet_allocate(session_ptr -> nx_lwm2m_client_session_dtls_session, session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_pool_ptr, &packet_ptr, NX_NO_WAIT);
    }
    else
#endif /* NX_SECURE_ENABLE_DTLS */
    {
        status = nx_packet_allocate(session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_pool_ptr, &packet_ptr, LWM2M_PACKET(session_ptr), NX_NO_WAIT);
    }
    if (status != NX_SUCCESS)
    {

        /* Failed to allocate the packet */
        return(NX_NULL);
    }

    /* Set nx_packet_append_ptr to the max message length */
    packet_ptr -> nx_packet_append_ptr = packet_ptr -> nx_packet_data_end - NX_PHYSICAL_TRAILER;

    /* Check for max LWM2M MTU */
    if ((packet_ptr -> nx_packet_append_ptr - packet_ptr -> nx_packet_data_start) > 
        (NX_PHYSICAL_HEADER + NX_LWM2M_CLIENT_MTU))
    {
        packet_ptr -> nx_packet_append_ptr = packet_ptr -> nx_packet_data_start + NX_PHYSICAL_HEADER + NX_LWM2M_CLIENT_MTU;
    }

    /* Return pointer to the allocated packet */
    return(packet_ptr);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_get_register_options               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function parses parses the registering response options.       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LwM2M session      */
/*    ptr                                   Pointer to the start of buffer*/
/*    ptr_max                               Pointer to the end of buffer  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the end of registering response option                   */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_coap_option_parse                                  */
/*    memcpy                                                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_receive                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static const VOID *_nx_lwm2m_client_get_register_options(NX_LWM2M_CLIENT_SESSION *session_ptr, const VOID *ptr, const VOID *ptr_max)
{
UINT option;

    option = 0;
    while (1)
    {
        const VOID *value_ptr;
        UINT value_length;

        ptr = _nx_lwm2m_client_coap_option_parse(ptr, ptr_max, &option, &value_ptr, &value_length);
        if ((ptr == NX_NULL) || (option == NX_LWM2M_CLIENT_COAP_OPTION_NONE))
        {
            break;
        }

        if (option == NX_LWM2M_CLIENT_COAP_OPTION_LOCATION_PATH)
        {

            /* Get uri-path of server */
            const CHAR *uri_start;
            const CHAR *uri_end;
            UINT uri_length;

            /* Path must begin with '/rd' */
            if (value_length != 2 || ((const CHAR *) value_ptr)[0] != 'r' || ((const CHAR *) value_ptr)[1] != 'd')
            {

                /* Invalid path */
                return(NX_NULL);
            }

            /* Get next part of the path */
            uri_start = ptr;
            while (1)
            {
                ptr = _nx_lwm2m_client_coap_option_parse(ptr, ptr_max, &option, &value_ptr, &value_length);
                if (ptr == NX_NULL)
                {

                    /* Invalid option */
                    return(NX_NULL);
                }

                if (option != NX_LWM2M_CLIENT_COAP_OPTION_LOCATION_PATH)
                {
                    break;
                }
            }

            /* End of uri-path */
            uri_end = ptr;

            /* Get uri-path length */
            uri_length = (UINT)(uri_end - uri_start);
            if (uri_length > NX_LWM2M_CLIENT_MAX_COAP_URI_PATH)
            {

                /* XXX path too long */
                return(NX_NULL);
            }

            /* Copy uri-path into session structure */
            session_ptr -> nx_lwm2m_client_session_uri_path_length = uri_length;
            if (uri_length != 0)
            {
                memcpy(session_ptr -> nx_lwm2m_client_session_uri_path, uri_start, uri_length); /* Use case of memcpy is verified. */
            }
        }

        /* Last option */
        if (option == NX_LWM2M_CLIENT_COAP_OPTION_NONE)
        {
            break;
        }

        /* Ignore other options... */
    }

    return(ptr);
}


/* options in request packet */
#define REQUEST_OPTIONS_INSTANCE_ID         (1<<0)
#define REQUEST_OPTIONS_RESOURCE_ID         (1<<1)
#define REQUEST_OPTIONS_ACCEPT              (1<<2)
#define REQUEST_OPTIONS_CONTENT_FORMAT      (1<<3)
#define REQUEST_OPTIONS_ROOT_PATH           (1<<4)
#define REQUEST_OPTIONS_BS_PATH             (1<<5)
#define REQUEST_OPTIONS_OBSERVE             (1<<6)

/* attributes */
#define ATTR_FLAG_PMIN                      (1<<0)
#define ATTR_FLAG_PMAX                      (1<<1)
#define ATTR_FLAG_GT                        (1<<2)
#define ATTR_FLAG_LT                        (1<<3)
#define ATTR_FLAG_STP                       (1<<4)

/* list of supported attributes */
static const struct {
    const CHAR *    name;
    CHAR            name_len;
    CHAR            flag;
} attr_list[] = {
    { "pmin", 4, ATTR_FLAG_PMIN },
    { "pmax", 4, ATTR_FLAG_PMAX },
    { "gt",   2, ATTR_FLAG_GT   },
    { "lt",   2, ATTR_FLAG_LT   },
    { "st",   2, ATTR_FLAG_STP  }
};


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_get_request_options                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function parses request options.                               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LwM2M session      */
/*    ptr                                   Pointer to the start of buffer*/
/*    ptr_max                               Pointer to the end of buffer  */
/*    options_ptr                           Pointer to the request option */
/*    status_ptr                            Returned status               */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the end of request option                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_coap_option_parse                                  */
/*    _nx_lwm2m_client_strconv_parse_id                                   */
/*    _nx_lwm2m_client_strconv_parse_int32                                */
/*    _nx_lwm2m_client_strconv_parse_notify_number                        */
/*    memcmp                                                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_receive                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static const VOID *_nx_lwm2m_client_get_request_options(NX_LWM2M_CLIENT_SESSION *session_ptr, const VOID *ptr, const VOID *ptr_max, NX_LWM2M_CLIENT_REQUEST_OPTIONS *options_ptr, UINT *status_ptr)
{
UINT option;
NX_LWM2M_BOOL has_path;
const VOID *value_ptr;
UINT value_length;

    options_ptr -> flags = 0;
    options_ptr -> attr_set = 0;
    options_ptr -> attr_unset = 0;
    has_path = NX_FALSE;

    option = 0;
    while (1)
    {

        /* Get next option */
        ptr = _nx_lwm2m_client_coap_option_parse(ptr, ptr_max, &option, &value_ptr, &value_length);
        if (ptr == NX_NULL)
        {
            break;
        }

        /* Uri-path option, Bootstrap session */
        if ((option == NX_LWM2M_CLIENT_COAP_OPTION_URI_PATH) && (session_ptr -> nx_lwm2m_client_session_server_instance_ptr == NX_NULL))
        {

            /* Look for '/bs' path */
            if (value_length == 2 && ((const CHAR *) value_ptr)[0] == 'b' && ((const CHAR *) value_ptr)[1] == 's')
            {

                /* Get next option */
                ptr = _nx_lwm2m_client_coap_option_parse(ptr, ptr_max, &option, &value_ptr, &value_length);
                if (ptr == NX_NULL)
                {
                    break;
                }

                if (option == NX_LWM2M_CLIENT_COAP_OPTION_URI_PATH)
                {

                    /* Invalid path */
                    *status_ptr = NX_LWM2M_CLIENT_NOT_FOUND;
                    return(NX_NULL);
                }

                options_ptr -> flags |= REQUEST_OPTIONS_BS_PATH;

                has_path = NX_TRUE;
            }
        }

        /* Uri-Path option, Object location path */
        if (option == NX_LWM2M_CLIENT_COAP_OPTION_URI_PATH)
        {

            /* Path has the form: /{Object ID}/{Instance ID}/{Resource ID} */

            /* Get {Object ID} */
            if (_nx_lwm2m_client_strconv_parse_id(value_ptr, value_length, &options_ptr -> object_id) != NX_SUCCESS)
            {

                /* Invalid ID */
                *status_ptr = NX_LWM2M_CLIENT_NOT_FOUND;
                return(NX_NULL);
            }
            has_path = NX_TRUE;

            /* Get next option */
            ptr = _nx_lwm2m_client_coap_option_parse(ptr, ptr_max, &option, &value_ptr, &value_length);
            if (ptr == NX_NULL)
            {
                break;
            }
            if (option == NX_LWM2M_CLIENT_COAP_OPTION_URI_PATH)
            {

                /* Get {Instance ID} */
                if (_nx_lwm2m_client_strconv_parse_id(value_ptr, value_length, &options_ptr -> instance_id) != NX_SUCCESS)
                {

                    /* Invalid ID */
                    *status_ptr = NX_LWM2M_CLIENT_NOT_FOUND;
                    return(NX_NULL);
                }
                options_ptr -> flags |= REQUEST_OPTIONS_INSTANCE_ID;

                /* Get next option */
                ptr = _nx_lwm2m_client_coap_option_parse(ptr, ptr_max, &option, &value_ptr, &value_length);
                if (ptr == NX_NULL)
                {
                    break;
                }
                if (option == NX_LWM2M_CLIENT_COAP_OPTION_URI_PATH)
                {

                    /* Get {Resource ID} */
                    if (_nx_lwm2m_client_strconv_parse_id(value_ptr, value_length, &options_ptr -> resource_id) != NX_SUCCESS)
                    {

                        /* Invalid ID */
                        *status_ptr = NX_LWM2M_CLIENT_NOT_FOUND;
                        return(NX_NULL);
                    }
                    options_ptr -> flags |= REQUEST_OPTIONS_RESOURCE_ID;

                    /* Get next option */
                    ptr = _nx_lwm2m_client_coap_option_parse(ptr, ptr_max, &option, &value_ptr, &value_length);
                    if (ptr == NX_NULL)
                    {
                        break;
                    }
                    if (option == NX_LWM2M_CLIENT_COAP_OPTION_URI_PATH)
                    {

                        /* Invalid path */
                        *status_ptr = NX_LWM2M_CLIENT_NOT_FOUND;
                        return(NX_NULL);
                    }
                }
            }
        } /* NX_LWM2M_CLIENT_COAP_OPTION_URI_PATH */

        /* Accept / Content-Format / Observe */
        if (option == NX_LWM2M_CLIENT_COAP_OPTION_CONTENT_FORMAT ||
            option == NX_LWM2M_CLIENT_COAP_OPTION_ACCEPT ||
            option == NX_LWM2M_CLIENT_COAP_OPTION_OBSERVE)
        {
            UINT format;
            UINT i;
            const UCHAR *tmp_ptr;

            /* Read uint option value */
            format = 0;
            tmp_ptr = value_ptr;
            for (i = 0; i < value_length; i++)
            {
                format <<= 8;
                format |= *tmp_ptr++;
            }

            if (option == NX_LWM2M_CLIENT_COAP_OPTION_CONTENT_FORMAT)
            {
                if (options_ptr -> flags & REQUEST_OPTIONS_CONTENT_FORMAT)
                {

                    /* Invalid repeated option */
                    ptr = NX_NULL;
                    break;
                }

                options_ptr -> flags |= REQUEST_OPTIONS_CONTENT_FORMAT;
                options_ptr -> content_format = format;
            }
            else if (option == NX_LWM2M_CLIENT_COAP_OPTION_ACCEPT)
            {
                if (options_ptr -> flags & REQUEST_OPTIONS_ACCEPT)
                {

                    /* Invalid repeated option */
                    ptr = NX_NULL;
                    break;
                }

                options_ptr -> flags |= REQUEST_OPTIONS_ACCEPT;
                options_ptr -> accept = format;
            }
            else
            {
                if (options_ptr -> flags & REQUEST_OPTIONS_OBSERVE)
                {

                    /* Invalid repeated option */
                    ptr = NX_NULL;
                    break;
                }

                options_ptr -> flags |= REQUEST_OPTIONS_OBSERVE;
                options_ptr -> observe = format;
            }
        }

        /* URI Query */
        if (option == NX_LWM2M_CLIENT_COAP_OPTION_URI_QUERY)
        {
            UINT i;

            /* check for known attribute */
            for (i = 0; i < sizeof(attr_list)/sizeof(attr_list[0]); i++)
            {
                if (value_length >= (UINT) attr_list[i].name_len && memcmp(value_ptr, attr_list[i].name, (UINT)(attr_list[i].name_len)) == 0)
                {

                    /* Must be of the form 'attr=value' or 'attr' */
                    if (value_length > (UINT) attr_list[i].name_len && *((const CHAR *) value_ptr + attr_list[i].name_len) != '=')
                    {

                        /* Ignore */
                        continue;
                    }

                    if (options_ptr -> attr_set & attr_list[i].flag || options_ptr -> attr_unset & attr_list[i].flag)
                    {

                        /* Attribute already set */
                        ptr = NX_NULL;
                    }
                    else
                    {
                        if (value_length == (UINT)(attr_list[i].name_len))
                        {

                            /* Unset attribute */
                            options_ptr -> attr_unset |= attr_list[i].flag;
                        }
                        else
                        {
                            UINT status;

                            /* Set attribute */
                            options_ptr -> attr_set |= attr_list[i].flag;

                            /* Get value */
                            value_ptr = (const CHAR *) value_ptr + attr_list[i].name_len + 1;
                            value_length -= (UINT)(attr_list[i].name_len) + 1;
                            switch (attr_list[i].flag)
                            {

                            case ATTR_FLAG_PMIN:

                                status = _nx_lwm2m_client_strconv_parse_int32(value_ptr, value_length, &options_ptr -> pmin);
                                break;

                            case ATTR_FLAG_PMAX:

                                status = _nx_lwm2m_client_strconv_parse_int32(value_ptr, value_length, &options_ptr -> pmax);
                                break;

                            case ATTR_FLAG_GT:

                                status = _nx_lwm2m_client_strconv_parse_notify_number(value_ptr, value_length, &options_ptr -> gt);
                                break;

                            case ATTR_FLAG_LT:

                                status = _nx_lwm2m_client_strconv_parse_notify_number(value_ptr, value_length, &options_ptr -> lt);
                                break;

                            case ATTR_FLAG_STP:

                                status = _nx_lwm2m_client_strconv_parse_notify_number(value_ptr, value_length, &options_ptr -> stp);
                                break;

                            default:

                                /* XXX should not happen */
                                status = NX_LWM2M_CLIENT_ERROR;
                                break;
                            }

                            if (status != NX_SUCCESS)
                            {

                                /* Invalid value */
                                ptr = NX_NULL;
                            }
                        }
                    }

                    break;
                }
            }

            if (ptr == NX_NULL)
            {
                break;
            }
        }

        /* Last option? */
        if (option == NX_LWM2M_CLIENT_COAP_OPTION_NONE)
        {
            break;
        }

        /* Ignore other options... */
    }

    /* Bad request (XXX should we return bad option?) */
    if ((ptr == NX_NULL) || (!has_path && session_ptr -> nx_lwm2m_client_session_server_instance_ptr != NX_NULL))
    {
        *status_ptr = NX_LWM2M_CLIENT_BAD_REQUEST;
        return(NX_NULL);
    }

    /* Root path? */
    if (!has_path)
    {
        options_ptr -> flags |= REQUEST_OPTIONS_ROOT_PATH;
    }

    /* No instance ID ? */
    if (!(options_ptr -> flags & REQUEST_OPTIONS_INSTANCE_ID))
    {
        options_ptr -> instance_id = NX_LWM2M_CLIENT_RESERVED_ID;
    }

    /* No resource ID ? */
    if (!(options_ptr -> flags & REQUEST_OPTIONS_RESOURCE_ID))
    {
        options_ptr -> resource_id = NX_LWM2M_CLIENT_RESERVED_ID;
    }

    /* Success, return pointer to payload */
    *status_ptr = NX_SUCCESS;
    return(ptr);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_parse_tlv_instance                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function parses values of a TLV instance.                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LwM2M session      */
/*    ptr                                   Pointer to the start of buffer*/
/*    ptr_max                               Pointer to the end of buffer  */
/*    num_values_ptr                        The returned number           */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_tlv_resource_parse                                 */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_receive                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_parse_tlv_instance(NX_LWM2M_CLIENT_SESSION *session_ptr, const UCHAR *ptr, const UCHAR *ptr_max, UINT *num_values_ptr)
{
NX_LWM2M_CLIENT_RESOURCE *values;
UINT num_values;

    values = session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_temp_resources;
    num_values = 0;
    while (ptr < ptr_max)
    {
        if (num_values >= NX_LWM2M_CLIENT_MAX_RESOURCES)
        {
            /* XXX too many values */
            return(NX_LWM2M_CLIENT_BUFFER_TOO_SMALL);
        }

        ptr = _nx_lwm2m_client_tlv_resource_parse(ptr, ptr_max, values++);
        if (ptr == NX_NULL)
        {
            /* invalid TLV encoding */
            return(NX_LWM2M_CLIENT_BAD_REQUEST);
        }

        num_values++;
    }

    /* return number of values */
    *num_values_ptr = num_values;

    /* return success */
    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_object_observe                     PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function registers/deregisters observation.                    */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LwM2M session      */
/*    options_ptr                           Pointer to the request option */
/*    token_ptr                             Pointer to token              */
/*    token_length                          The length of token           */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource_ptr                          Pointer to resource           */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_notify_get                                 */
/*    _nx_lwm2m_client_session_notify_free                                */
/*    _nx_lwm2m_client_resource_notify_number_get                         */
/*    _nx_lwm2m_client_session_notify_attributes                          */
/*    memcpy                                                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_receive                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static NX_LWM2M_BOOL _nx_lwm2m_client_object_observe(NX_LWM2M_CLIENT_SESSION *session_ptr, const NX_LWM2M_CLIENT_REQUEST_OPTIONS *options_ptr, const VOID *token_ptr, UINT token_length, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource_ptr)
{
NX_LWM2M_CLIENT_NOTIFY *notify_ptr;
UINT attr_flags;
NX_LWM2M_INT32 attr_pmin;
NX_LWM2M_INT32 attr_pmax;
NX_LWM2M_CLIENT_NOTIFY_NUMBER attr_gt;
NX_LWM2M_CLIENT_NOTIFY_NUMBER attr_lt;
NX_LWM2M_CLIENT_NOTIFY_NUMBER attr_stp;

    if (!(options_ptr -> flags & REQUEST_OPTIONS_OBSERVE) || (options_ptr -> observe != 0 && options_ptr -> observe != 1))
    {
        /* unknown operation */
        return(NX_FALSE);
    }

    /* get the notify structure, allocate new one in case of registration */
    notify_ptr = _nx_lwm2m_client_session_notify_get(session_ptr, object_ptr, instance_ptr, resource_ptr == NX_NULL ? NX_LWM2M_CLIENT_RESERVED_ID : resource_ptr -> resource_id, options_ptr -> observe == 0);

    if (notify_ptr == NX_NULL)
    {
        /* notification doesn't exist, or allocation failure */
        return(NX_FALSE);
    }

    /* clear current observer state, keep only attributes */
    notify_ptr -> notify_flags &= NX_LWM2M_CLIENT_NOTIFY_ATTR_FLAGS;

    if (options_ptr -> observe != 0)
    {
        /* deregister, free notification if it doesn't store attributes */
        if (notify_ptr -> notify_flags == 0)
        {
            _nx_lwm2m_client_session_notify_free(session_ptr, notify_ptr);
        }

        return(NX_FALSE);
    }

    /* store observer information */
    notify_ptr -> notify_flags |= NX_LWM2M_CLIENT_NOTIFY_OBSERVE | token_length;
    if (token_length != 0)
    {
        /* save token */
        memcpy(notify_ptr -> notify_token, token_ptr, token_length); /* Use case of memcpy is verified. */
    }

    /* set format of response */
    if ((options_ptr -> flags & REQUEST_OPTIONS_ACCEPT) && options_ptr -> accept == NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV)
    {
        notify_ptr -> notify_flags |= NX_LWM2M_CLIENT_NOTIFY_ACCEPT_TLV;
    }

    /* initialize last message ID with ID of last request (ensure it dosen't correspond to another notify message) */
    notify_ptr -> notify_last_id = session_ptr -> nx_lwm2m_client_session_request_id;

    /* set last response time */
    notify_ptr -> notify_last_time = tx_time_get();

    /* set last number value */
    if (resource_ptr != NX_NULL)
    {
        _nx_lwm2m_client_resource_notify_number_get(resource_ptr, &notify_ptr -> notify_last_value);
    }

    /* enable timer if pmax is defined */
    _nx_lwm2m_client_session_notify_attributes(session_ptr, notify_ptr, &attr_flags, &attr_pmin, &attr_pmax, &attr_gt, &attr_lt, &attr_stp);
    if (!(attr_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_PMAX))
    {
        /* get default maximum period from server object */
        attr_pmax = session_ptr -> nx_lwm2m_client_session_server_instance_ptr -> server_instance_max_period;
    }

    if (attr_pmax > 0)
    {
        /* arm timer */
        notify_ptr -> notify_flags |= NX_LWM2M_CLIENT_NOTIFY_TIMER;
        notify_ptr -> notify_timer = notify_ptr -> notify_last_time + ((ULONG)attr_pmax * NX_IP_PERIODIC_RATE);
    }

    return(NX_TRUE);
}


/* XXX */
#define ACK_TIMEOUT     (10 * NX_IP_PERIODIC_RATE)

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_observe_reset                      PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function resets the observation.                               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LwM2M session      */
/*    msg_id                                The message ID                */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_notify_free                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_receive                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static VOID _nx_lwm2m_client_observe_reset(NX_LWM2M_CLIENT_SESSION *session_ptr, USHORT msg_id)
{
NX_LWM2M_CLIENT_NOTIFY *notify_ptr;
ULONG now;

    /* get current time */
    now = tx_time_get();

    /* check if the ID corresponds to a previous notify message */
    notify_ptr = session_ptr -> nx_lwm2m_client_session_notifications;
    while (notify_ptr != NX_NULL)
    {
        if ((notify_ptr -> notify_flags & NX_LWM2M_CLIENT_NOTIFY_OBSERVE) && notify_ptr -> notify_last_id == msg_id && (LONG) (now - notify_ptr -> notify_last_time) < (LONG)ACK_TIMEOUT)
        {
            /* deregister the observer, clear current observer state, keep only attributes */
            notify_ptr -> notify_flags &= NX_LWM2M_CLIENT_NOTIFY_ATTR_FLAGS;

            /* free notification if it doesn't store attributes */
            if (notify_ptr -> notify_flags == 0)
            {
                _nx_lwm2m_client_session_notify_free(session_ptr, notify_ptr);
            }

            return;
        }

        notify_ptr = notify_ptr -> notify_next;
    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_write_attributes                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function writes the object attributes.                         */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LwM2M session      */
/*    options_ptr                           Pointer to the request option */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource_ptr                          Pointer to resource           */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Status                                                              */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_notify_free                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_receive                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_write_attributes(NX_LWM2M_CLIENT_SESSION *session_ptr, const NX_LWM2M_CLIENT_REQUEST_OPTIONS *options_ptr, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_CLIENT_RESOURCE *resource_ptr)
{
NX_LWM2M_CLIENT_NOTIFY *notify_ptr;

    /* get the notify structure, allocate new one if some attributes are to be set */
    notify_ptr = _nx_lwm2m_client_session_notify_get(session_ptr, object_ptr, instance_ptr, resource_ptr == NX_NULL ? NX_LWM2M_CLIENT_RESERVED_ID : resource_ptr -> resource_id, options_ptr -> attr_set != 0);

    if (notify_ptr == NX_NULL)
    {
        /* notification doesn't exist, or allocation failure */
        return(options_ptr -> attr_set != 0 ? NX_LWM2M_CLIENT_NO_MEMORY : NX_SUCCESS);
    }

    /* set attributes */
    if (options_ptr -> attr_set & ATTR_FLAG_PMIN)
    {
        notify_ptr -> notify_flags |= NX_LWM2M_CLIENT_NOTIFY_ATTR_PMIN;
        notify_ptr -> notify_attr_pmin = options_ptr -> pmin;
    }
    if (options_ptr -> attr_set & ATTR_FLAG_PMAX)
    {
        notify_ptr -> notify_flags |= NX_LWM2M_CLIENT_NOTIFY_ATTR_PMAX;
        notify_ptr -> notify_attr_pmax = options_ptr -> pmax;
    }
    if (options_ptr -> attr_set & ATTR_FLAG_GT)
    {
        notify_ptr -> notify_flags |= NX_LWM2M_CLIENT_NOTIFY_ATTR_GT;
        notify_ptr -> notify_attr_gt = options_ptr -> gt;
    }
    if (options_ptr -> attr_set & ATTR_FLAG_LT)
    {
        notify_ptr -> notify_flags |= NX_LWM2M_CLIENT_NOTIFY_ATTR_LT;
        notify_ptr -> notify_attr_lt = options_ptr -> lt;
    }
    if (options_ptr -> attr_set & ATTR_FLAG_STP)
    {
        notify_ptr -> notify_flags |= NX_LWM2M_CLIENT_NOTIFY_ATTR_STP;
        notify_ptr -> notify_attr_stp = options_ptr -> stp;
    }

    /* clear attributes */
    if (options_ptr -> attr_unset & ATTR_FLAG_PMIN)
    {
        notify_ptr -> notify_flags &= (UINT)(~NX_LWM2M_CLIENT_NOTIFY_ATTR_PMIN);
    }
    if (options_ptr -> attr_unset & ATTR_FLAG_PMAX)
    {
        notify_ptr -> notify_flags &= (UINT)(~NX_LWM2M_CLIENT_NOTIFY_ATTR_PMAX);
    }
    if (options_ptr -> attr_unset & ATTR_FLAG_GT)
    {
        notify_ptr -> notify_flags &= (UINT)(~NX_LWM2M_CLIENT_NOTIFY_ATTR_GT);
    }
    if (options_ptr -> attr_unset & ATTR_FLAG_LT)
    {
        notify_ptr -> notify_flags &= (UINT)(~NX_LWM2M_CLIENT_NOTIFY_ATTR_LT);
    }
    if (options_ptr -> attr_unset & ATTR_FLAG_STP)
    {
        notify_ptr -> notify_flags &= (UINT)(~NX_LWM2M_CLIENT_NOTIFY_ATTR_STP);
    }

    /* free structure if all attributes and observe flag are cleared */
    if ((notify_ptr -> notify_flags & (NX_LWM2M_CLIENT_NOTIFY_OBSERVE|NX_LWM2M_CLIENT_NOTIFY_ATTR_FLAGS)) == 0)
    {
        _nx_lwm2m_client_session_notify_free(session_ptr, notify_ptr);
    }

    return(NX_SUCCESS);
}


/* Define the LWM2M operations */
#define LWM2M_OPERATION_BS_WRITE        0
#define LWM2M_OPERATION_BS_DELETE       1
#define LWM2M_OPERATION_BS_DELETE_ALL   2
#define LWM2M_OPERATION_BS_DISCOVER     3
#define LWM2M_OPERATION_BS_FINISH       4
#define LWM2M_OPERATION_READ            5
#define LWM2M_OPERATION_DISCOVER        6
#define LWM2M_OPERATION_WRITE_UPDATE    7
#define LWM2M_OPERATION_WRITE_REPLACE   8
#define LWM2M_OPERATION_WRITE_ATTRS     9
#define LWM2M_OPERATION_EXECUTE         10
#define LWM2M_OPERATION_CREATE          11
#define LWM2M_OPERATION_DELETE          12


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_receive                    PORTABLE C      */
/*                                                           6.1.10       */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes a packet received by a LWM2M session.       */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    packet_ptr                            Pointer to the received packet*/
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    none                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_coap_header_parse                                  */
/*    _nx_lwm2m_client_session_state_update                               */
/*    _nx_lwm2m_client_get_register_options                               */
/*    _nx_lwm2m_client_observe_reset                                      */
/*    _nx_lwm2m_client_object_ptr_get                                     */
/*    _nx_lwm2m_client_object_instance_ptr_get                            */
/*    _nx_lwm2m_client_tlv_resource_parse                                 */
/*    _nx_lwm2m_client_object_observe                                     */
/*    _nx_lwm2m_client_parse_tlv_instance                                 */
/*    _nx_lwm2m_client_write_attributes                                   */
/*    _nx_lwm2m_client_object_create_internal                             */
/*    _nx_lwm2m_client_object_delete_all                                  */
/*    _nx_lwm2m_client_server_session_get                                 */
/*    _nx_lwm2m_client_session_send_response                              */
/*    nx_packet_release                                                   */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_thread_entry                                       */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*  01-31-2022     Yuxin Zhou               Modified comment(s), supported*/
/*                                            token and processing        */
/*                                            confirmable response,       */
/*                                            resulting in version 6.1.10 */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_session_receive(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_PACKET *packet_ptr)
{
UCHAR msg_type;
UCHAR msg_code;
USHORT msg_id;
const VOID *token_ptr;
UINT token_length;
const UCHAR *ptr;
const UCHAR *ptr_max;
UINT i;

    /* check if we need to parse the packet */
    if (session_ptr -> nx_lwm2m_client_session_substate <= NX_LWM2M_CLIENT_SESSION_SUBSTATE_REGISTER || session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_DISABLED)
    {
        /* Discard packet */
        nx_packet_release(packet_ptr);

        /* End of processing */
        return;
    }

#ifndef NX_DISABLE_PACKET_CHAIN
    /* Don't support packet chain. */
    if (packet_ptr -> nx_packet_next != NX_NULL)
    {

        /* Discard packet */
        nx_packet_release(packet_ptr);

        /* End of processing */
        return;
    }
#endif

    /* start of message */
    ptr = packet_ptr -> nx_packet_prepend_ptr;

    /* end of message */
    ptr_max = packet_ptr -> nx_packet_append_ptr;

    /* Decode the CoAP header */
    ptr = _nx_lwm2m_client_coap_header_parse(ptr, ptr_max, &msg_type, &msg_code, &msg_id, &token_ptr, &token_length);
    if (ptr != NX_NULL)
    {
        /* check type of message: */
        if ((msg_type == NX_LWM2M_CLIENT_COAP_TYPE_ACK) || (msg_type == NX_LWM2M_CLIENT_COAP_TYPE_RST) ||
            (msg_type == NX_LWM2M_CLIENT_COAP_TYPE_CON && (msg_code & NX_LWM2M_CLIENT_COAP_CLASS_MASK) != NX_LWM2M_CLIENT_COAP_CLASS_REQUEST))
        {
            /* Response to our last request? */
            if (session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_REQUEST_SENT && 
                (session_ptr -> nx_lwm2m_client_session_request_id == msg_id || msg_type == NX_LWM2M_CLIENT_COAP_TYPE_CON))
            {
                if (msg_type == NX_LWM2M_CLIENT_COAP_TYPE_RST || (msg_code & NX_LWM2M_CLIENT_COAP_CLASS_MASK) != NX_LWM2M_CLIENT_COAP_CLASS_SUCCESS)
                {
                    /* Request failed, set error state */
                    /* XXX get coap error code */
                    session_ptr -> nx_lwm2m_client_session_error = NX_LWM2M_CLIENT_ERROR;
                    _nx_lwm2m_client_session_state_update(session_ptr, session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_REQUESTING ? NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_ERROR : NX_LWM2M_CLIENT_SESSION_ERROR);
                }
                else if ((token_length == sizeof(session_ptr -> nx_lwm2m_client_session_token)) && 
                         (memcmp(token_ptr, session_ptr -> nx_lwm2m_client_session_token, token_length) == 0)) /* Use case of memcmp is verified. */
                {
                    /* Success, update client state */
                    switch (session_ptr -> nx_lwm2m_client_session_state)
                    {

                    case NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_REQUESTING:

                        /* set state to bootstrap initiated */
                        _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_INITIATED);

                        break;

                    case NX_LWM2M_CLIENT_SESSION_REGISTERING:

                        /* get server location */
                        ptr = _nx_lwm2m_client_get_register_options(session_ptr, ptr, ptr_max);
                        if (ptr == NX_NULL)
                        {
                            /* Invalid response */
                            /* XXX define error code */
                            session_ptr -> nx_lwm2m_client_session_error = NX_LWM2M_CLIENT_ERROR;
                            _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_ERROR);

                            break;
                        }

                        /* fall through */

                    case NX_LWM2M_CLIENT_SESSION_UPDATING:

                        /* set state to registered */
                        _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_REGISTERED);

                        break;

                    case NX_LWM2M_CLIENT_SESSION_DISABLING:

                        /* set state to disabled */
                        _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_DISABLED);

                        break;

                    case NX_LWM2M_CLIENT_SESSION_DEREGISTERING:

                        /* set state to de-registered */
                        _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_DEREGISTERED);

                        break;

                    default:

                        /* XXX SHOULD NOY GO HERE */
                        NX_ASSERT(0);

                    }

                    /* Send an empty ACK to the response in confirmable type.  */
                    if (msg_type == NX_LWM2M_CLIENT_COAP_TYPE_CON)
                    {
                        _nx_lwm2m_client_session_send_response(session_ptr, 0, msg_id, NX_TRUE, NX_NULL, 0, NX_NULL, NX_NULL, NX_NULL, NX_NULL, NX_NULL);
                    }
                }

                /* report new state to state callback */
                session_ptr -> nx_lwm2m_client_session_state_callback(session_ptr, session_ptr -> nx_lwm2m_client_session_state);
            }
            else if (msg_type == NX_LWM2M_CLIENT_COAP_TYPE_RST)
            {
                /* Reset notification? */
                _nx_lwm2m_client_observe_reset(session_ptr, msg_id);
            }
        }
        else if ((msg_code >= NX_LWM2M_CLIENT_COAP_REQUEST_GET && msg_code <= NX_LWM2M_CLIENT_COAP_REQUEST_DELETE) &&
                 (session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_WAITING || session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_REQUESTING || session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_INITIATED || session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_REGISTERED || session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_UPDATING))
        {

            /* Handle request */
            NX_LWM2M_CLIENT_REQUEST_OPTIONS *options = &(session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_request_options);
            UINT status;
            UCHAR response_code;
            UINT response_format = 0;
            NX_LWM2M_BOOL observe = NX_FALSE;
            NX_LWM2M_CLIENT_OBJECT *object_ptr;
            NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr;
            NX_LWM2M_CLIENT_RESOURCE resource;
            NX_LWM2M_CLIENT_RESOURCE *resource_ptr;
            UINT lwm2m_operation = 0;
            NX_LWM2M_CLIENT_SESSION *state_changed_session_ptr;
            UINT write_type;
            UINT num_values;

            /* set pointer to session that has changed state */
            state_changed_session_ptr = NX_NULL;

            /* get request options */
            ptr = _nx_lwm2m_client_get_request_options(session_ptr, ptr, ptr_max, options, &status);

            /* Check Object ID/Instance ID/Resource ID */
            if (status == NX_SUCCESS)
            {
                /* get pointer to object */
                if (options->flags & (REQUEST_OPTIONS_BS_PATH|REQUEST_OPTIONS_ROOT_PATH))
                {
                    /* bootstrap specific paths, no target object */
                    object_ptr = NX_NULL;
                    instance_ptr = NX_NULL;
                    resource_ptr = NX_NULL;
                }
                else
                {
                    object_ptr = _nx_lwm2m_client_object_ptr_get(session_ptr -> nx_lwm2m_client_session_client_ptr, options->object_id);
                    if (object_ptr == NX_NULL)
                    {
                        /* object doesn't exist */
                        status = NX_LWM2M_CLIENT_NOT_FOUND;

                        /* also clear instance and resource pointers */
                        instance_ptr = NX_NULL;
                        resource_ptr = NX_NULL;
                    }
                    else
                    {
                        if (options->flags & REQUEST_OPTIONS_INSTANCE_ID)
                        {
                            /* get pointer to instance if it exists */
                            instance_ptr = _nx_lwm2m_client_object_instance_ptr_get(object_ptr, options->instance_id);

                            if (options->flags & REQUEST_OPTIONS_RESOURCE_ID)
                            {
                                /* get resource */
                                resource_ptr = &resource;
                                if (options->flags & REQUEST_OPTIONS_CONTENT_FORMAT)
                                {
                                    /* get resource value */
                                    if (options->content_format == NX_LWM2M_CLIENT_COAP_CONTENT_TEXT_PLAIN || options->content_format == NX_LWM2M_CLIENT_COAP_CONTENT_OCTET_STREAM)
                                    {
                                        /* text/plain or octet-stream (opaque) */
                                        resource_ptr -> resource_id = options->resource_id;
                                        resource_ptr -> resource_type = options->content_format == NX_LWM2M_CLIENT_COAP_CONTENT_TEXT_PLAIN ? NX_LWM2M_CLIENT_RESOURCE_TEXT : NX_LWM2M_CLIENT_RESOURCE_OPAQUE;
                                        resource_ptr -> resource_value.resource_bufferdata.resource_buffer_ptr = ptr;
                                        resource_ptr -> resource_value.resource_bufferdata.resource_buffer_length = (UINT)(ptr_max - ptr);
                                    }
                                    else if (options->content_format == NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV)
                                    {

                                        /* TLV resource value */
                                         if ((_nx_lwm2m_client_tlv_resource_parse(ptr, ptr_max, resource_ptr) == NX_NULL) ||
                                             (resource_ptr -> resource_id != options -> resource_id) ||
                                             ((resource_ptr -> resource_type != NX_LWM2M_CLIENT_RESOURCE_TLV) && 
                                              (resource_ptr -> resource_type != NX_LWM2M_CLIENT_RESOURCE_MULTIPLE_TLV)))
                                         {

                                            /* unsupported content format */
                                            status = NX_LWM2M_CLIENT_BAD_REQUEST;
                                         }
                                    }
                                    else
                                    {
                                        /* unsupported content format */
                                        status = NX_LWM2M_CLIENT_BAD_REQUEST;
                                    }
                                }
                                else
                                {
                                    /* no content */
                                    resource_ptr -> resource_id = options->resource_id;
                                    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_NONE;
                                    resource_ptr -> resource_value.resource_bufferdata.resource_buffer_ptr = NX_NULL;
                                    resource_ptr -> resource_value.resource_bufferdata.resource_buffer_length = 0;
                                }
                            }
                            else
                            {
                                resource_ptr = NX_NULL;
                            }
                        }
                        else
                        {
                            instance_ptr = NX_NULL;
                            resource_ptr = NX_NULL;
                        }
                    }
                }
            }
            else
            {
                object_ptr = NX_NULL;
                instance_ptr = NX_NULL;
                resource_ptr = NX_NULL;
            }

            if (status == NX_SUCCESS)
            {
                /* select the type of LWM2M operation: */

                if (session_ptr -> nx_lwm2m_client_session_server_instance_ptr == NX_NULL)
                {
                    /* Bootstrap session */

                    if (options->flags & REQUEST_OPTIONS_BS_PATH)
                    {
                        if (msg_code == NX_LWM2M_CLIENT_COAP_REQUEST_POST)
                        {
                            /* POST /bs -> Bootstrap Finish */
                            lwm2m_operation = LWM2M_OPERATION_BS_FINISH;
                        }
                        else
                        {
                            /* '/bs' path only allowed with POST method */
                            status = NX_LWM2M_CLIENT_BAD_REQUEST;
                        }
                    }
                    else if (options->flags & REQUEST_OPTIONS_ROOT_PATH)
                    {
                        if (msg_code == NX_LWM2M_CLIENT_COAP_REQUEST_DELETE)
                        {
                            /* XXX delete / seems to be removed from the final LWM2M spec */
                            /* keep it for now in case all servers are not updated */

                            /* DELETE / -> Bootstrap Delete All */
                            lwm2m_operation = LWM2M_OPERATION_BS_DELETE_ALL;
                        }
                        else
                        if ((msg_code == NX_LWM2M_CLIENT_COAP_REQUEST_GET) &&
                            (options->flags & REQUEST_OPTIONS_ACCEPT) &&
                            (options->accept == NX_LWM2M_CLIENT_COAP_CONTENT_LINK_FORMAT))
                        {
                            /* DISCOVER / */
                            lwm2m_operation = LWM2M_OPERATION_BS_DISCOVER;
                        }
                        else
                        {
                            /* '/' path only allowed with DELETE and DISCOVER method */
                            status = NX_LWM2M_CLIENT_BAD_REQUEST;
                        }
                    }
                    else if (msg_code == NX_LWM2M_CLIENT_COAP_REQUEST_GET)
                    {
                        if ((options->flags & REQUEST_OPTIONS_ACCEPT) &&
                            (options->accept == NX_LWM2M_CLIENT_COAP_CONTENT_LINK_FORMAT) &&
                            !(options->flags & REQUEST_OPTIONS_INSTANCE_ID))
                        {
                            /* Bootstrap Discover */
                            lwm2m_operation = LWM2M_OPERATION_BS_DISCOVER;
                        }
                        else
                        {
                            status = NX_LWM2M_CLIENT_BAD_REQUEST;
                        }
                    }
                    else if (msg_code == NX_LWM2M_CLIENT_COAP_REQUEST_PUT)
                    {
                        if (resource_ptr == NX_NULL &&
                            (!(options->flags & REQUEST_OPTIONS_CONTENT_FORMAT) ||
                             options->content_format != NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV))
                        {
                            /* Content format must be TLV */
                            status = NX_LWM2M_CLIENT_NOT_ACCEPTABLE;
                        }
                        else
                        {
                            /* Bootstrap Write */
                            lwm2m_operation = LWM2M_OPERATION_BS_WRITE;
                        }
                    }
                    else if (msg_code == NX_LWM2M_CLIENT_COAP_REQUEST_DELETE &&
                             resource_ptr == NX_NULL)
                    {
                        /* Bootstrap Delete */
                        lwm2m_operation = LWM2M_OPERATION_BS_DELETE;
                    }
                    else
                    {
                        status = NX_LWM2M_CLIENT_BAD_REQUEST;
                    }
                }
                else if (msg_code == NX_LWM2M_CLIENT_COAP_REQUEST_GET)
                {
                    if ((options->flags & REQUEST_OPTIONS_ACCEPT) &&
                        (options->accept == NX_LWM2M_CLIENT_COAP_CONTENT_LINK_FORMAT))
                    {
                        /* Discover */
                        lwm2m_operation = LWM2M_OPERATION_DISCOVER;
                    }
                    else
                    {
                        /* Read */
                        lwm2m_operation = LWM2M_OPERATION_READ;
                    }
                }
                else if (msg_code == NX_LWM2M_CLIENT_COAP_REQUEST_PUT)
                {
                    if (options->flags & REQUEST_OPTIONS_CONTENT_FORMAT)
                    {
                        if (!(options->flags & REQUEST_OPTIONS_INSTANCE_ID))
                        {
                            /* missing Instance ID */
                            status = NX_LWM2M_CLIENT_BAD_REQUEST;
                        }
                        else if (resource_ptr == NX_NULL && options->content_format != NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV)
                        {
                            /* Invalid content format */
                            status = NX_LWM2M_CLIENT_NOT_ACCEPTABLE;
                        }
                        else
                        {
                            /* Write Replace */
                            lwm2m_operation = LWM2M_OPERATION_WRITE_REPLACE;
                        }
                    }
                    else
                    {
                        /* Write Attributes */
                        lwm2m_operation = LWM2M_OPERATION_WRITE_ATTRS;
                    }
                }
                else if (msg_code == NX_LWM2M_CLIENT_COAP_REQUEST_POST)
                {
                    if (resource_ptr != NX_NULL)
                    {
                        if (resource_ptr -> resource_type != NX_LWM2M_CLIENT_RESOURCE_NONE &&
                            resource_ptr -> resource_type != NX_LWM2M_CLIENT_RESOURCE_TEXT)
                        {
                            /* execute arguments must be in text/plain format */
                            status = NX_LWM2M_CLIENT_NOT_ACCEPTABLE;
                        }
                        else
                        {
                            /* Execute */
                            lwm2m_operation = LWM2M_OPERATION_EXECUTE;
                        }
                    }
                    else if (!(options->flags & REQUEST_OPTIONS_CONTENT_FORMAT) ||
                             options->content_format != NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV)
                    {
                        /* Invalid content format (must be TLV) */
                        status = NX_LWM2M_CLIENT_NOT_ACCEPTABLE;
                    }
                    else if (options->flags & REQUEST_OPTIONS_INSTANCE_ID)
                    {
                        /* Write Update */
                        lwm2m_operation = LWM2M_OPERATION_WRITE_UPDATE;
                    }
                    else
                    {
                        /* Create */
                        lwm2m_operation = LWM2M_OPERATION_CREATE;
                    }
                }
                else /* NX_LWM2M_CLIENT_COAP_REQUEST_DELETE */
                {
                    if ((options->flags & REQUEST_OPTIONS_INSTANCE_ID) &&
                        resource_ptr == NX_NULL)
                    {
                        /* Delete */
                        lwm2m_operation = LWM2M_OPERATION_DELETE;
                    }
                    else
                    {
                        status = NX_LWM2M_CLIENT_BAD_REQUEST;
                    }
                }
            }

            /* Access to Security Object is forbidden except in bootstrap mode */
            if (status == NX_SUCCESS &&
                session_ptr -> nx_lwm2m_client_session_server_instance_ptr != NX_NULL &&
                object_ptr -> object_id == NX_LWM2M_CLIENT_SECURITY_OBJECT_ID)
            {
                status = NX_LWM2M_CLIENT_UNAUTHORIZED;
            }

            /* check if the provided instance is valid */
            if (status == NX_SUCCESS &&
                (options->flags & REQUEST_OPTIONS_INSTANCE_ID) &&
                instance_ptr == NX_NULL &&
                (lwm2m_operation == LWM2M_OPERATION_READ ||
                 lwm2m_operation == LWM2M_OPERATION_DISCOVER ||
                 lwm2m_operation == LWM2M_OPERATION_WRITE_UPDATE ||
                 lwm2m_operation == LWM2M_OPERATION_WRITE_REPLACE ||
                 lwm2m_operation == LWM2M_OPERATION_WRITE_ATTRS ||
                 lwm2m_operation == LWM2M_OPERATION_EXECUTE ||
                 lwm2m_operation == LWM2M_OPERATION_DELETE))
            {
                /* instance doesn't exist */
                status = NX_LWM2M_CLIENT_NOT_FOUND;
            }

            if (status == NX_SUCCESS)
            {

                /* check for server initiated bootstrap */
                if ((lwm2m_operation == LWM2M_OPERATION_BS_WRITE ||
                     lwm2m_operation == LWM2M_OPERATION_BS_DELETE ||
                     lwm2m_operation == LWM2M_OPERATION_BS_DELETE_ALL ||
                     lwm2m_operation == LWM2M_OPERATION_BS_DISCOVER) &&
                    session_ptr -> nx_lwm2m_client_session_state < NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_INITIATED)
                {
                    /* set state to bootstrap initiated */
                    _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_INITIATED);

                    /* call state change callback */
                    state_changed_session_ptr = session_ptr;
                }

                /* do the LWM2M operation: */
                switch (lwm2m_operation)
                {

                /* Read */
                case LWM2M_OPERATION_READ:

                    if (resource_ptr != NX_NULL)
                    {
                        /* read single value */
                        num_values = 1;
                        status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_READ, object_ptr, instance_ptr, resource_ptr, &num_values, NX_NULL, 0);
                        if (status == NX_SUCCESS)
                        {
                            /* select the content-format of the response: */
                            if (options->flags & REQUEST_OPTIONS_ACCEPT)
                            {
                                /* check if format is acceptable: */
                                if ((options->accept == NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV) ||
                                    (options->accept == NX_LWM2M_CLIENT_COAP_CONTENT_TEXT_PLAIN && resource_ptr -> resource_type == NX_LWM2M_CLIENT_RESOURCE_STRING) ||
                                    (options->accept == NX_LWM2M_CLIENT_COAP_CONTENT_OCTET_STREAM && resource_ptr -> resource_type == NX_LWM2M_CLIENT_RESOURCE_OPAQUE))
                                {
                                    /* use requested format */
                                    response_format = options->accept;
                                }
                                else
                                {
                                    /* invalid format */
                                    status = NX_LWM2M_CLIENT_NOT_ACCEPTABLE;
                                }
                            }
                            else if (resource_ptr -> resource_type == NX_LWM2M_CLIENT_RESOURCE_STRING)
                            {
                                /* text value */
                                response_format = NX_LWM2M_CLIENT_COAP_CONTENT_TEXT_PLAIN;
                            }
                            else if (resource_ptr -> resource_type == NX_LWM2M_CLIENT_RESOURCE_OPAQUE)
                            {
                                /* opaque value */
                                response_format = NX_LWM2M_CLIENT_COAP_CONTENT_OCTET_STREAM;
                            }
                            else
                            {
                                /* use tlv for other values */
                                response_format = NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV;
                            }
                        }
                    }
                    else if ((options->flags & REQUEST_OPTIONS_ACCEPT) && options->accept != NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV)
                    {
                        /* invalid format */
                        status = NX_LWM2M_CLIENT_NOT_ACCEPTABLE;
                    }
                    else
                    {
                        /* return instance(s) in tlv format */
                        response_format = NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV;
                    }

                    /* Observe option */
                    if (status == NX_SUCCESS && (options->flags & REQUEST_OPTIONS_OBSERVE))
                    {
                        /* Register */
                        observe = _nx_lwm2m_client_object_observe(session_ptr, options, token_ptr, token_length, object_ptr, instance_ptr, resource_ptr);
                    }

                    break;

                /* Discover */
                case LWM2M_OPERATION_BS_DISCOVER:
                case LWM2M_OPERATION_DISCOVER:

                    if (resource_ptr != NX_NULL)
                    {
                        /* return single resource, check if it's exists */
                        NX_LWM2M_CLIENT_RESOURCE *resources = session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_temp_resources;
                        UINT num_infos;

                        /* get list of supported resources */
                        num_infos = NX_LWM2M_CLIENT_MAX_RESOURCES;
                        status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_DISCOVER, object_ptr, instance_ptr, resources, &num_infos, NX_NULL, 0);
                        if (status == NX_SUCCESS)
                        {

                            /* find resource ID */
                            for (i = 0; i < num_infos; i++)
                            {
                                if (resources[i].resource_id == resource_ptr -> resource_id)
                                {
                                    break;
                                }
                            }
                            if (i < num_infos)
                            {
                                if (resources[i].resource_type == NX_LWM2M_CLIENT_RESOURCE_MULTIPLE)
                                {
                                    /* set dimension of multiple resource */
                                    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_MULTIPLE;
                                    resource_ptr -> resource_dim = resources[i].resource_dim;
                                }
                                else
                                {
                                    /* not a multiple resource */
                                    resource_ptr -> resource_type = NX_LWM2M_CLIENT_RESOURCE_NONE;
                                }
                            }
                            else
                            {
                                /* resource ID not found */
                                status = NX_LWM2M_CLIENT_NOT_FOUND;
                            }
                        }
                    }

                    if (status == NX_SUCCESS)
                    {
                        /* return list of IDs and attributes in link format */
                        response_format = NX_LWM2M_CLIENT_COAP_CONTENT_LINK_FORMAT;
                    }

                    break;

                /* Write */
                case LWM2M_OPERATION_WRITE_UPDATE:
                case LWM2M_OPERATION_WRITE_REPLACE:

                    if (resource_ptr != NX_NULL)
                    {

                        /* Set write type */
                        if (lwm2m_operation == LWM2M_OPERATION_WRITE_REPLACE)
                            write_type = NX_LWM2M_CLIENT_OBJECT_WRITE_REPLACE_RESOURCE;
                        else
                            write_type = NX_LWM2M_CLIENT_OBJECT_WRITE_UPDATE;

                        /* write a single value */
                        num_values = 1;
                        status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_WRITE, object_ptr, instance_ptr, resource_ptr, &num_values, &write_type, sizeof(UINT));
                    }
                    else
                    {
                        /* the TLV payload must contains an object instance */

                        /* parse list of TLV values */
                        status = _nx_lwm2m_client_parse_tlv_instance(session_ptr, ptr, ptr_max, &num_values);
                        if (status == NX_SUCCESS)
                        {

                            /* Set write type */
                            if (lwm2m_operation == LWM2M_OPERATION_WRITE_REPLACE)
                                write_type = NX_LWM2M_CLIENT_OBJECT_WRITE_REPLACE_INSTANCE;
                            else
                                write_type = NX_LWM2M_CLIENT_OBJECT_WRITE_UPDATE;

                            /* write resources values */
                            status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_WRITE, object_ptr, instance_ptr, 
                                                                             session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_temp_resources, 
                                                                             &num_values, &write_type, sizeof(UINT));
                        }
                    }

                    break;

                /* Write Attributes */
                case LWM2M_OPERATION_WRITE_ATTRS:

                    if (resource_ptr != NX_NULL)
                    {
                        /* check that resource exists and is readable */
                        num_values = 1;
                        status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_READ, object_ptr, instance_ptr, resource_ptr, &num_values, NX_NULL, 0);
                        if (status != NX_SUCCESS)
                        {
                            break;
                        }
                    }

                    /* set attributes */
                    status = _nx_lwm2m_client_write_attributes(session_ptr, options, object_ptr, instance_ptr, resource_ptr);
                    break;

                /* Execute */
                case LWM2M_OPERATION_EXECUTE:

                    /* call object method */
                    num_values = 1;
                    if (resource_ptr -> resource_type == NX_LWM2M_CLIENT_RESOURCE_TEXT)
                    {
                        /* pass the text arguments to the method */
                        status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_EXECUTE, object_ptr, instance_ptr, resource_ptr, &num_values,
                                                                         (VOID *)resource_ptr -> resource_value.resource_bufferdata.resource_buffer_ptr,
                                                                         resource_ptr -> resource_value.resource_bufferdata.resource_buffer_length);
                    }
                    else
                    {
                        /* no arguments */
                        status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_EXECUTE, object_ptr, instance_ptr, resource_ptr, &num_values, NX_NULL, 0);
                    }

                    break;

                /* Bootstrap Write */
                case LWM2M_OPERATION_BS_WRITE:

                    if (resource_ptr != NX_NULL)
                    {
                        /* Write a single value */
                        if (instance_ptr == NX_NULL)
                        {
                            /* Create the instance */
                            status = _nx_lwm2m_client_object_create_internal(object_ptr, &(options->instance_id), 1, resource_ptr, NX_TRUE);
                        }
                        else
                        {
                            /* Replace resource of an existing instance */
                            write_type = NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP;
                            num_values = 1;
                            status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_WRITE, object_ptr, instance_ptr, resource_ptr, &num_values, &write_type, sizeof(UINT));
                        }

                        break;
                    }
                    else if (options->flags & REQUEST_OPTIONS_INSTANCE_ID)
                    {
                        /* Write a list of values to an instance */

                        /* parse list of TLV values */
                        status = _nx_lwm2m_client_parse_tlv_instance(session_ptr, ptr, ptr_max, &num_values);
                        if (status == NX_SUCCESS)
                        {
                            if (instance_ptr == NX_NULL)
                            {
                                /* Create the instance */
                                status = _nx_lwm2m_client_object_create_internal(object_ptr, &(options->instance_id), num_values, session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_temp_resources, NX_TRUE);
                            }
                            else
                            {
                                /* Write to an existing instance */
                                write_type = NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP;
                                status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_WRITE, object_ptr, instance_ptr, 
                                                                                 session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_temp_resources, 
                                                                                 &num_values, &write_type, sizeof(UINT));
                            }
                        }
                        break;
                    }

                    /* fall through */

                /* Create */
                case LWM2M_OPERATION_CREATE:
                    /* check if the value contains an instance ID */
                    if (ptr == ptr_max)
                    {
                        /* empty TLV payload */
                        status = NX_LWM2M_CLIENT_BAD_REQUEST;

                        break;
                    }

                    if ((ptr[0] & NX_LWM2M_CLIENT_TLV_TYPE_ID_TYPE_MASK) != NX_LWM2M_CLIENT_TLV_OBJECT_INSTANCE)
                    {
                        /* Create a new instance with an assigned ID */

                        /* get list of values */
                        status = _nx_lwm2m_client_parse_tlv_instance(session_ptr, ptr, ptr_max, &num_values);
                        if (status == NX_SUCCESS)
                        {
                            /* create new instance */
                            status = _nx_lwm2m_client_object_create_internal(object_ptr, NX_NULL, num_values, session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_temp_resources, NX_FALSE);
                        }
                    }
                    else
                    {
                        /* create instance(s) */
                        while (ptr < ptr_max)
                        {
                            NX_LWM2M_CLIENT_RESOURCE instance;

                            /* get object instance */
                            ptr = _nx_lwm2m_client_tlv_resource_parse(ptr, ptr_max, &instance);
                            if ((ptr == NX_NULL) || (instance.resource_type != NX_LWM2M_CLIENT_RESOURCE_OBJECT_INSTANCE_TLV))
                            {
                                /* Invalid TLV value */
                                status = NX_LWM2M_CLIENT_BAD_REQUEST;

                                break;
                            }

                            /* get list of values */
                            status = _nx_lwm2m_client_parse_tlv_instance(session_ptr, instance.resource_value.resource_bufferdata.resource_buffer_ptr, (const UCHAR *) instance.resource_value.resource_bufferdata.resource_buffer_ptr + instance.resource_value.resource_bufferdata.resource_buffer_length, &num_values);
                            if (status != NX_SUCCESS)
                            {
                                break;
                            }

                            /* check if instance already exists */
                            instance_ptr = _nx_lwm2m_client_object_instance_ptr_get(object_ptr, instance.resource_id);
                            if (instance_ptr != NX_NULL)
                            {
                                if (lwm2m_operation == LWM2M_OPERATION_BS_WRITE)
                                {
                                    /* write to existing instance (Bootstrap Write) */
                                    write_type = NX_LWM2M_CLIENT_OBJECT_WRITE_BOOTSTRAP;
                                    status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_WRITE, object_ptr, instance_ptr, 
                                                                                     session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_temp_resources, 
                                                                                     &num_values, &write_type, sizeof(UINT));
                                }
                                else
                                {
                                    /* instance already exist */
                                    status = NX_LWM2M_CLIENT_ALREADY_EXIST;
                                }
                            }
                            else
                            {
                                /* create new instance */
                                status = _nx_lwm2m_client_object_create_internal(object_ptr, &instance.resource_id, num_values, session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_temp_resources, NX_FALSE);
                            }
                            if (status != NX_SUCCESS)
                            {
                                break;
                            }
                        }
                    }

                    break;

                /* Delete */
                case LWM2M_OPERATION_BS_DELETE:
                    if (!(options->flags & REQUEST_OPTIONS_INSTANCE_ID)) {
                        /* delete all instances of the object, except security instance of bootstrap server */
                        _nx_lwm2m_client_object_delete_all(session_ptr -> nx_lwm2m_client_session_client_ptr, object_ptr, session_ptr -> nx_lwm2m_client_session_server_id);

                        break;
                    }

                    /* fall through */

                case LWM2M_OPERATION_DELETE:
                    {
                        NX_LWM2M_CLIENT_OBJECT_INSTANCE *server_instance_ptr;

                        /* save pointer to instance if it's a server object */
                        if (object_ptr -> object_id == NX_LWM2M_CLIENT_SERVER_OBJECT_ID)
                        {
                            server_instance_ptr = instance_ptr;
                        }
                        else
                        {
                            server_instance_ptr = NX_NULL;
                        }

                        /* Delete the instance */
                        if (instance_ptr != NX_NULL)
                        {
                            status = object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_DELETE, object_ptr, instance_ptr, NX_NULL, NX_NULL, NX_NULL, 0);
                        }

                        /* Terminate lwm2m session if it's server instance is deleted */
                        if (server_instance_ptr != NX_NULL && status == NX_SUCCESS)
                        {
                            NX_LWM2M_CLIENT_SESSION *deleted_session_ptr;

                            deleted_session_ptr = _nx_lwm2m_client_server_session_get(session_ptr -> nx_lwm2m_client_session_client_ptr, server_instance_ptr);

                            if (deleted_session_ptr != NX_NULL)
                            {
                                /* clear pointer to deleted instance */
                                deleted_session_ptr -> nx_lwm2m_client_session_server_instance_ptr = NX_NULL;

                                if (deleted_session_ptr -> nx_lwm2m_client_session_state != NX_LWM2M_CLIENT_SESSION_ERROR)
                                {
                                    /* change session state to error */
                                    deleted_session_ptr -> nx_lwm2m_client_session_error = NX_LWM2M_CLIENT_SERVER_INSTANCE_DELETED;
                                    _nx_lwm2m_client_session_state_update(deleted_session_ptr, NX_LWM2M_CLIENT_SESSION_ERROR);

                                    /* report state change to application */
                                    state_changed_session_ptr = deleted_session_ptr;
                                }
                            }
                        }

                        break;
                    }

                /* Delete All */
                case LWM2M_OPERATION_BS_DELETE_ALL:

                    /* Delete all instances, except security instance of bootstrap server */
                    _nx_lwm2m_client_object_delete_all(session_ptr -> nx_lwm2m_client_session_client_ptr, NX_NULL, session_ptr -> nx_lwm2m_client_session_server_id);

                    break;

                /* Bootstrap Finish */
                case LWM2M_OPERATION_BS_FINISH:

                    break;

                default:

                    status = NX_LWM2M_CLIENT_BAD_REQUEST;

                    break;

                }

            }

            /* set response code */
            switch (status)
            {

            /* Successful operation */
            case NX_SUCCESS:

                /* Return status code corresponding to the operation: */
                switch (lwm2m_operation)
                {

                /* Read, Discover -> 2.05 Content */
                case LWM2M_OPERATION_READ:
                case LWM2M_OPERATION_DISCOVER:
                case LWM2M_OPERATION_BS_DISCOVER:

                    response_code = NX_LWM2M_CLIENT_COAP_STATUS_CONTENT;
                    break;

                /* Create -> 2.01 Created */
                case LWM2M_OPERATION_CREATE:

                    response_code = NX_LWM2M_CLIENT_COAP_STATUS_CREATED;
                    break;

                /* Delete -> 2.02 Deleted */
                case LWM2M_OPERATION_DELETE:
                case LWM2M_OPERATION_BS_DELETE:
                case LWM2M_OPERATION_BS_DELETE_ALL:

                    response_code = NX_LWM2M_CLIENT_COAP_STATUS_DELETED;
                    break;

                /* Other operations: 2.04 Changed */
                default:

                    response_code = NX_LWM2M_CLIENT_COAP_STATUS_CHANGED;
                    break;

                }

                break;

            /* 4.01 Unauthorized */
            case NX_LWM2M_CLIENT_UNAUTHORIZED:

                response_code = NX_LWM2M_CLIENT_COAP_STATUS_UNAUTHORIZED;
                break;

            /* 4.04 Not Found */
            case NX_LWM2M_CLIENT_NOT_FOUND:

                response_code = NX_LWM2M_CLIENT_COAP_STATUS_NOT_FOUND;
                break;

            /* 4.05 Method Not Allowed */
            case NX_LWM2M_CLIENT_METHOD_NOT_ALLOWED:

                response_code = NX_LWM2M_CLIENT_COAP_STATUS_METHOD_NOT_ALLOWED;
                break;

            /* 4.06 Not Acceptable */
            case NX_LWM2M_CLIENT_NOT_ACCEPTABLE:

                response_code = NX_LWM2M_CLIENT_COAP_STATUS_NOT_ACCEPTABLE;
                break;

            /* 5.00 Internal server error */
            case NX_LWM2M_CLIENT_BUFFER_TOO_SMALL:
            case NX_LWM2M_CLIENT_NO_MEMORY:

                response_code = NX_LWM2M_CLIENT_COAP_STATUS_INTERNAL_SERVER_ERROR;
                break;

            /* Other errors: 4.00 Bad Request */
            case NX_LWM2M_CLIENT_BAD_REQUEST:
            default:

                response_code = NX_LWM2M_CLIENT_COAP_STATUS_BAD_REQUEST;
                break;

            }

            /* Send response */
            _nx_lwm2m_client_session_send_response(session_ptr, response_code, msg_id, NX_TRUE, token_ptr, token_length, observe, response_format, object_ptr, instance_ptr, resource_ptr);

            if (status == NX_SUCCESS && lwm2m_operation == LWM2M_OPERATION_BS_FINISH)
            {

                /* update session state */
                if (session_ptr -> nx_lwm2m_client_session_state < NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_FINISHED)
                {

                    _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_FINISHED);

                    state_changed_session_ptr = session_ptr;
                }
            }

            /* Rearm bootstrap idle timer if bootstrap is initiated */
            if (session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_INITIATED)
            {
                session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_WAIT;
                session_ptr -> nx_lwm2m_client_session_timer = tx_time_get() + NX_LWM2M_CLIENT_BOOTSTRAP_IDLE_TIMER;
            }

            /* Report session state changes to application */
            if (state_changed_session_ptr != NX_NULL)
            {
                state_changed_session_ptr -> nx_lwm2m_client_session_state_callback(state_changed_session_ptr, session_ptr -> nx_lwm2m_client_session_state);
            }
        }
    }

    /* End of message processing, release packet */
    nx_packet_release(packet_ptr);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_register                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function registers the Client to a LWM2M server.               */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    server_id                             The Short Server ID of the    */
/*                                          LWM2M server                  */
/*    ip_address                            The IP address of the server  */
/*    port                                  The UDP port of the server    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_start                                      */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_register(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_ID server_id, const NXD_ADDRESS *ip_address, UINT port)
{
#ifdef NX_SECURE_ENABLE_DTLS
    return(_nx_lwm2m_client_session_start(session_ptr, NX_FALSE, server_id, ip_address, port, NX_NULL));
#else
    return(_nx_lwm2m_client_session_start(session_ptr, NX_FALSE, server_id, ip_address, port));
#endif /* NX_SECURE_ENABLE_DTLS */
}

#ifdef NX_SECURE_ENABLE_DTLS

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_register_dtls              PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function registers the Client to a LWM2M server using a        */
/*    secure channel.                                                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    server_id                             The Short Server ID of the    */
/*                                          LWM2M server                  */
/*    ip_address                            The IP address of the server  */
/*    port                                  The UDP port of the server    */
/*    dtls_session_ptr                      The pointer to the DTLS       */
/*                                          session                       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_start                                      */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_register_dtls(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_ID server_id, const NXD_ADDRESS *ip_address, UINT port, NX_SECURE_DTLS_SESSION *dtls_session_ptr)
{
    return(_nx_lwm2m_client_session_start(session_ptr, NX_FALSE, server_id, ip_address, port, dtls_session_ptr));
}

#endif /* NX_SECURE_ENABLE_DTLS */

#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_register_info_get          PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function gets the server and security information for the next */
/*    registration step.                                                  */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    security_instance_id                  Security instance ID          */
/*    server_id                             Server ID                     */
/*    server_uri                            Server URI                    */
/*    server_uri_len                        Length of server URI          */
/*    security_mode                         Security Mode                 */
/*    pub_key_or_id                         Public key or ID              */
/*    pub_key_or_id_len                     Length of public key or ID    */
/*    server_pub_key                        Server public key             */
/*    server_pub_key_len                    Length of server public key   */
/*    secret_key                            Secret key                    */
/*    secret_key_len                        Length of secret key          */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_object_ptr_get                                     */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_register_info_get(NX_LWM2M_CLIENT_SESSION *session_ptr, UINT security_instance_id, NX_LWM2M_ID *server_id, CHAR **server_uri, UINT *server_uri_len, UCHAR *security_mode,
                                                UCHAR **pub_key_or_id, UINT *pub_key_or_id_len, UCHAR **server_pub_key, UINT *server_pub_key_len, UCHAR **secret_key, UINT *secret_key_len)
{
UINT status;
NX_LWM2M_CLIENT *client_ptr = session_ptr -> nx_lwm2m_client_session_client_ptr;
NX_LWM2M_CLIENT_OBJECT *object_ptr;
NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr;
NX_LWM2M_CLIENT_SECURITY_INSTANCE *security_instance_ptr;

    /* Acquire the LWM2M Client Mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Get pointer to the Object Instance */
    object_ptr = _nx_lwm2m_client_object_ptr_get(client_ptr, NX_LWM2M_CLIENT_SECURITY_OBJECT_ID);
    if (object_ptr == NX_NULL)
    {
        status = NX_LWM2M_CLIENT_NOT_FOUND;
    }
    else
    {

        /* Get first instance */
        instance_ptr = object_ptr -> object_instances;

        while (instance_ptr != NX_NULL)
        {

            /* If instance ID is NX_LWM2M_CLIENT_RESERVED_ID, find the first non-bootstrap server;
               If not, find the non-bootstrap server with the corresponded instance ID */
            if (!(((NX_LWM2M_CLIENT_SECURITY_INSTANCE *)instance_ptr) -> security_instance_bootstrap) &&
                ((security_instance_id == NX_LWM2M_CLIENT_RESERVED_ID) ||
                 (security_instance_id == instance_ptr -> object_instance_id)))
            {
                break;
            }

            /* Next instance */
            instance_ptr = instance_ptr -> object_instance_next;
        }

        if (instance_ptr == NX_NULL)
        {

            /* Last instance */
            status = NX_LWM2M_CLIENT_NOT_FOUND;
        }
        else
        {

            /* Found the instance of requested ID */
            security_instance_ptr = (NX_LWM2M_CLIENT_SECURITY_INSTANCE *)instance_ptr;

            /* Set server ID */
            *server_id = security_instance_ptr -> security_instance_server_id;

            /* Set server URI */
            *server_uri = security_instance_ptr -> security_instance_server_uri;
            *server_uri_len = security_instance_ptr -> security_instance_server_uri_len;

            /* Set security mode */
            *security_mode = security_instance_ptr -> security_instance_mode;

            /* Set security keys */
            if (pub_key_or_id && pub_key_or_id_len)
            {
                *pub_key_or_id = security_instance_ptr -> security_instance_pub_key_or_id;
                *pub_key_or_id_len = security_instance_ptr -> security_instance_pub_key_or_id_len;
            }

            if (server_pub_key && server_pub_key_len)
            {
                *server_pub_key = security_instance_ptr -> security_instance_server_pub_key;
                *server_pub_key_len = security_instance_ptr -> security_instance_server_pub_key_len;
            }

            if (secret_key && secret_key_len)
            {
                *secret_key = security_instance_ptr -> security_instance_secret_key;
                *secret_key_len = security_instance_ptr -> security_instance_secret_key_len;
            }

            status = NX_SUCCESS;
        }
    }

    /* Release the LWM2M Client Mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    /* Return the status of the operation */
    return(status);
}
#endif /* NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES */

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_send                       PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sends a packets from a LWM2M session.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    packet_ptr                            Pointer to the packet to send */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    nx_secure_dtls_session_send                                         */
/*    nxd_udp_socket_send                                                 */
/*    nx_packet_release                                                   */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_send_request                               */
/*    _nx_lwm2m_client_session_send_response                              */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_send(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_PACKET *packet_ptr)
{
UINT status;

#ifdef NX_SECURE_ENABLE_DTLS
    if (session_ptr -> nx_lwm2m_client_session_dtls_session != NX_NULL)
    {

        /* Send the packet through the DTLS session */
        status = nx_secure_dtls_session_send(session_ptr -> nx_lwm2m_client_session_dtls_session, packet_ptr, &session_ptr -> nx_lwm2m_client_session_server_address, session_ptr -> nx_lwm2m_client_session_server_port);
    }
    else
#endif /* NX_SECURE_ENABLE_DTLS */
    {

        /* Send the packet through the CoAP socket */
        status = nxd_udp_socket_send((NX_UDP_SOCKET *) &session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_coap_socket, packet_ptr, &session_ptr -> nx_lwm2m_client_session_server_address, session_ptr -> nx_lwm2m_client_session_server_port);
    }

    if (status != NX_SUCCESS)
    {

        /* Cannot send packet, discard */
        nx_packet_release(packet_ptr);

        /* Return invalid server address error */
        return(NX_LWM2M_CLIENT_ADDRESS_ERROR);
    }

    /* Return success */
    return(NX_SUCCESS);
}

/* Initial retransmit timer */
#define RETRANSMIT_TIMER        2*NX_IP_PERIODIC_RATE

/* Max retransmit timer */
#define RETRANSMIT_TIMER_MAX    10*NX_IP_PERIODIC_RATE

/* Max retransmit count */
#define RETRANSMIT_COUNT        5


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_send_request               PORTABLE C      */
/*                                                           6.1.10       */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sends a request packet from a LWM2M session.          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    first                                 Set to True if it's not a     */
/*                                          retransmission                */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_packet_allocate                            */
/*    _nx_lwm2m_client_coap_option_header_add                             */
/*    _nx_lwm2m_client_strconv_format_int32                               */
/*    _nx_lwm2m_client_binding_mode_string_get                            */
/*    _nx_lwm2m_client_corelink_path_add                                  */
/*    _nx_lwm2m_client_session_send                                       */
/*    tx_time_get                                                         */
/*    memcpy                                                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_state_update                               */
/*    _nx_lwm2m_client_session_step                                       */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*  01-31-2022     Yuxin Zhou               Modified comment(s), supported*/
/*                                            token and processing        */
/*                                            confirmable response,       */
/*                                            resulting in version 6.1.10 */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_send_request(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_BOOL first)
{
NX_PACKET *packet_ptr;
UINT status;
UINT i;

    /* Check that current state is valid for sending a request */
    NX_ASSERT(session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_REQUESTING || session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_REGISTERING || session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_UPDATING || session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_DISABLING || session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_DEREGISTERING);

    status = NX_SUCCESS;

    if (first)
    {

        /* New request */
        session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_REQUEST_SENT;

        /* Initialize retransmission paramters */
        session_ptr -> nx_lwm2m_client_session_retransmit_count = RETRANSMIT_COUNT;
        session_ptr -> nx_lwm2m_client_session_retransmit_timer = RETRANSMIT_TIMER;

        /* Get new request ID */
        session_ptr -> nx_lwm2m_client_session_request_id = ++session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_request_id;
    }
    else
    {

        /* Retransmit request */

        /* Update retransmission counter */
        session_ptr -> nx_lwm2m_client_session_retransmit_count--;
        if (session_ptr -> nx_lwm2m_client_session_retransmit_count == 0)
        {

            /* Max number of retransmissions reached, */
            /* Report error to application */
            status = NX_LWM2M_CLIENT_TIMED_OUT;
        }
        else
        {

            /* Increase retransmission timer */
            session_ptr -> nx_lwm2m_client_session_retransmit_timer *= 2;
            if (session_ptr -> nx_lwm2m_client_session_retransmit_timer > RETRANSMIT_TIMER_MAX)
            {
                session_ptr -> nx_lwm2m_client_session_retransmit_timer = RETRANSMIT_TIMER_MAX;
            }
        }
    }

    if (status == NX_SUCCESS)
    {
        UCHAR *ptr;
        UINT last_option;
        UINT update_flags;

        /* Allocate a pack for the CoAP request message */
        packet_ptr = _nx_lwm2m_client_session_packet_allocate(session_ptr);
        if (packet_ptr == NX_NULL)
        {

            /* Failed to allocate the packet, will try to retransmited later */
            return(NX_SUCCESS);
        }

        /* Start of message */
        ptr = packet_ptr -> nx_packet_prepend_ptr;

        /* Check that the packet is large enought for the CoAP header */
        if (ptr + 4 > packet_ptr -> nx_packet_append_ptr)
        {
            nx_packet_release(packet_ptr);
            status = NX_LWM2M_CLIENT_BUFFER_TOO_SMALL;
        }
        else
        {

            /* Set CoAP header and type of request */
            *ptr++ = NX_LWM2M_CLIENT_COAP_VERSION_1 | NX_LWM2M_CLIENT_COAP_TYPE_CON | NX_LWM2M_CLIENT_COAP_TOKEN_LEN;
            if (session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_DISABLING || session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_DEREGISTERING)
            {

                /* De-register */
                *ptr++ = NX_LWM2M_CLIENT_COAP_REQUEST_DELETE;
            }
            else
            {

                /* Bootstrap request, Register, Update */
                *ptr++ = NX_LWM2M_CLIENT_COAP_REQUEST_POST;
            }
            *ptr++ = (UCHAR) (session_ptr -> nx_lwm2m_client_session_request_id >> 8);
            *ptr++ = (UCHAR)  session_ptr -> nx_lwm2m_client_session_request_id;

            /* Set token */
            for (i = 0; i < sizeof(session_ptr -> nx_lwm2m_client_session_token); i++)
            {
                session_ptr -> nx_lwm2m_client_session_token[i] = (UCHAR) NX_RAND();
                *ptr++ = session_ptr -> nx_lwm2m_client_session_token[i];
            }

            /* Add options */
            last_option = 0;

            /* Uri-Path: always of the form /bs or /rd [/...] */
            ptr = _nx_lwm2m_client_coap_option_header_add(ptr, packet_ptr -> nx_packet_append_ptr, &last_option, NX_LWM2M_CLIENT_COAP_OPTION_URI_PATH, 2);
            if ((ptr == NX_NULL) || ((ptr + 2) > packet_ptr -> nx_packet_append_ptr))
            {
                nx_packet_release(packet_ptr);
                status = NX_LWM2M_CLIENT_BUFFER_TOO_SMALL;
            }
            else
            {
                if (session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_REQUESTING)
                {
                    *ptr++ = 'b';
                    *ptr++ = 's';
                }
                else
                {
                    *ptr++ = 'r';
                    *ptr++ = 'd';
                }
            }
        }

        if ((status == NX_SUCCESS) &&
            ((session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_UPDATING) || 
             (session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_DISABLING) || 
             (session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_DEREGISTERING)))
        {

            /* Check packet payload */
            if ((ptr + session_ptr -> nx_lwm2m_client_session_uri_path_length) > packet_ptr -> nx_packet_append_ptr)
            {
                nx_packet_release(packet_ptr);
                status = NX_LWM2M_CLIENT_BUFFER_TOO_SMALL;
            }
            else
            {

                /* Add server Uri-Path option(s) if we are registered */
                memcpy(ptr, session_ptr -> nx_lwm2m_client_session_uri_path, session_ptr -> nx_lwm2m_client_session_uri_path_length); /* Use case of memcpy is verified. */
                ptr += session_ptr -> nx_lwm2m_client_session_uri_path_length;
            }
        }

        /* Get Register/Update optional parameters and payload */
        if (session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_REGISTERING || session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_UPDATING)
        {
            update_flags = session_ptr -> nx_lwm2m_client_session_flags;
        }
        else
        {
            update_flags = 0;
        }

        /* Add Content-Format option if we include a payload */
        if ((status == NX_SUCCESS) && (update_flags & NX_LWM2M_CLIENT_SESSION_UPDATE_OBJECTS_LIST))
        {

            /* Check packet payload */
            if ((ptr + NX_LWM2M_CLIENT_COAP_CONTENT_LINK_FORMAT) > packet_ptr -> nx_packet_append_ptr)
            {
                nx_packet_release(packet_ptr);
                status = NX_LWM2M_CLIENT_BUFFER_TOO_SMALL;
            }
            else
            {
                ptr = _nx_lwm2m_client_coap_option_header_add(ptr, packet_ptr -> nx_packet_append_ptr, &last_option, NX_LWM2M_CLIENT_COAP_OPTION_CONTENT_FORMAT, 1);
                *ptr++ = NX_LWM2M_CLIENT_COAP_CONTENT_LINK_FORMAT;
            }
        }

        /* Uri-Query options: */

        /* ep={Endpoint Client Name} only in Bootstrap request and Register */
        if ((status == NX_SUCCESS) &&
            ((session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_REQUESTING) || 
             (session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_REGISTERING)))
        {
            UINT client_name_length;

            client_name_length = session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_name_length;
            ptr = _nx_lwm2m_client_coap_option_header_add(ptr, packet_ptr -> nx_packet_append_ptr, &last_option, NX_LWM2M_CLIENT_COAP_OPTION_URI_QUERY, 3 + client_name_length);
            if ((ptr == NX_NULL) || ((ptr + 3 + client_name_length) > packet_ptr -> nx_packet_append_ptr))
            {
                nx_packet_release(packet_ptr);
                status = NX_LWM2M_CLIENT_BUFFER_TOO_SMALL;
            }
            else
            {
                *ptr++ = 'e';
                *ptr++ = 'p';
                *ptr++ = '=';
                memcpy(ptr, session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_name, client_name_length); /* Use case of memcpy is verified. */
                ptr += client_name_length;
            }
        }

        /* lt={Lifetime} */
        if ((status == NX_SUCCESS) && (update_flags & NX_LWM2M_CLIENT_SESSION_UPDATE_LIFETIME))
        {
            CHAR lifetime[NX_LWM2M_CLIENT_STRCONV_INT32_MAXLEN];
            UINT lifetime_length;

            NX_ASSERT(session_ptr -> nx_lwm2m_client_session_server_instance_ptr != NX_NULL);

            /* Format lifetime value */
            lifetime_length = _nx_lwm2m_client_strconv_format_int32(lifetime, sizeof(lifetime), session_ptr -> nx_lwm2m_client_session_server_instance_ptr -> server_instance_lifetime);

            /* Write option */
            ptr = _nx_lwm2m_client_coap_option_header_add(ptr, packet_ptr -> nx_packet_append_ptr, &last_option, NX_LWM2M_CLIENT_COAP_OPTION_URI_QUERY, 3 + lifetime_length);
            if ((ptr == NX_NULL) || ((ptr + 3 + lifetime_length) > packet_ptr -> nx_packet_append_ptr))
            {
                nx_packet_release(packet_ptr);
                status = NX_LWM2M_CLIENT_BUFFER_TOO_SMALL;
            }
            else
            {
                *ptr++ = 'l';
                *ptr++ = 't';
                *ptr++ = '=';
                memcpy(ptr, lifetime, lifetime_length); /* Use case of memcpy is verified. */
                ptr += lifetime_length;
            }
        }

        /* sms={MSISDN} */
        if ((status == NX_SUCCESS) && (update_flags & NX_LWM2M_CLIENT_SESSION_UPDATE_MSISDN))
        {
            UINT msisdn_length;

            /* MSISDN is allowed to be NULL if device has no SMS number */
            msisdn_length = session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_msisdn_length;

            /* Write option */
            ptr = _nx_lwm2m_client_coap_option_header_add(ptr, packet_ptr -> nx_packet_append_ptr, &last_option, NX_LWM2M_CLIENT_COAP_OPTION_URI_QUERY, 4 + msisdn_length);
            if ((ptr == NX_NULL) || ((ptr + 4 + msisdn_length) > packet_ptr -> nx_packet_append_ptr))
            {
                nx_packet_release(packet_ptr);
                status = NX_LWM2M_CLIENT_BUFFER_TOO_SMALL;
            }
            else
            {
                *ptr++ = 's';
                *ptr++ = 'm';
                *ptr++ = 's';
                *ptr++ = '=';
                if (msisdn_length != 0)
                {
                    memcpy(ptr, session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_msisdn, msisdn_length); /* Use case of memcpy is verified. */
                    ptr += msisdn_length;
                }
            }
        }

        /* lwm2m={version} only in Register */
        if ((status == NX_SUCCESS) && (session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_REGISTERING))
        {
            static const CHAR version[] = "lwm2m=1.0";
            UINT version_length = sizeof(version)-1;

            ptr = _nx_lwm2m_client_coap_option_header_add(ptr, packet_ptr -> nx_packet_append_ptr, &last_option, NX_LWM2M_CLIENT_COAP_OPTION_URI_QUERY, version_length);
            if ((ptr == NX_NULL) || ((ptr + 4 + version_length) > packet_ptr -> nx_packet_append_ptr))
            {
                nx_packet_release(packet_ptr);
                status = NX_LWM2M_CLIENT_BUFFER_TOO_SMALL;
            }
            else
            {
                memcpy(ptr, version, version_length); /* Use case of memcpy is verified. */
                ptr += version_length;
            }
        }

        /* b={binding} */
        if ((status == NX_SUCCESS) && (update_flags & NX_LWM2M_CLIENT_SESSION_UPDATE_BINDING_MODE))
        {
            const CHAR *binding;
            UINT binding_length;

            NX_ASSERT(session_ptr -> nx_lwm2m_client_session_server_instance_ptr != NX_NULL);

            /* Get binding mode from server object instance */
            binding = _nx_lwm2m_client_binding_mode_string_get(session_ptr -> nx_lwm2m_client_session_server_instance_ptr -> server_instance_binding, &binding_length);

            /* Write option */
            ptr = _nx_lwm2m_client_coap_option_header_add(ptr, packet_ptr -> nx_packet_append_ptr, &last_option, NX_LWM2M_CLIENT_COAP_OPTION_URI_QUERY, 2 + binding_length);
            if ((ptr == NX_NULL) || ((ptr + 2 + binding_length) > packet_ptr -> nx_packet_append_ptr))
            {
                nx_packet_release(packet_ptr);
                status = NX_LWM2M_CLIENT_BUFFER_TOO_SMALL;
            }
            else
            {
                *ptr++ = 'b';
                *ptr++ = '=';
                memcpy(ptr, binding, binding_length); /* Use case of memcpy is verified. */
                ptr += binding_length;
            }
        }

        /* Add payload (list of objects) */
        if ((status == NX_SUCCESS) && (update_flags & NX_LWM2M_CLIENT_SESSION_UPDATE_OBJECTS_LIST))
        {
            NX_LWM2M_CLIENT_OBJECT *object_ptr;
            UINT object_first;

            /* Check packet payload */
            if (ptr >= packet_ptr -> nx_packet_append_ptr)
            {
                status = NX_LWM2M_CLIENT_BUFFER_TOO_SMALL;
            }
            else
            {

                /* Mark the begining of the payload */
                *ptr++ = NX_LWM2M_CLIENT_COAP_PAYLOAD_MARKER;

                /* Write the list of objects (do not include security object 0) */
                object_first = NX_TRUE;
                object_ptr = (NX_LWM2M_CLIENT_OBJECT *) &session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_server;
                while (object_ptr != NX_NULL)
                {
                    if (object_ptr -> object_instances == NX_NULL)
                    {

                        /* Object with no instance */

                        /* Write separator */
                        if (!object_first)
                        {
                            if (ptr >= packet_ptr -> nx_packet_append_ptr)
                            {

                                /* Not enought space */
                                ptr = NX_NULL;
                                break;
                            }
                            *ptr++ = ',';
                        }

                        /* write object path */
                        ptr = (UCHAR *) _nx_lwm2m_client_corelink_path_add((CHAR *) ptr, (CHAR *) packet_ptr -> nx_packet_append_ptr, object_ptr -> object_id, NX_LWM2M_CLIENT_RESERVED_ID, NX_LWM2M_CLIENT_RESERVED_ID);

                    }
                    else
                    {
                        NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr;

                        /* List object instances */
                        instance_ptr = object_ptr -> object_instances;
                        while (instance_ptr != NX_NULL)
                        {

                            /* Write separator */
                            if (!object_first)
                            {
                                if (ptr >= packet_ptr -> nx_packet_append_ptr)
                                {

                                    /* Not enought space */
                                    ptr = NX_NULL;
                                    break;
                                }
                                *ptr++ = ',';
                            }

                            /* Write object instance path */
                            ptr = (UCHAR *) _nx_lwm2m_client_corelink_path_add((CHAR *) ptr, (CHAR *) packet_ptr -> nx_packet_append_ptr, object_ptr -> object_id, instance_ptr -> object_instance_id, NX_LWM2M_CLIENT_RESERVED_ID);
                            if (ptr == NX_NULL)
                            {
                                break;
                            }

                            /* Reset the flag because an instance was added.  */
                            object_first = NX_FALSE;

                            /* Get next instance */
                            instance_ptr = instance_ptr -> object_instance_next;
                        }
                    }

                    if (ptr == NX_NULL)
                    {
                        break;
                    }

                    object_first = NX_FALSE;
                    object_ptr = object_ptr -> object_next;
                }

                if (ptr == NX_NULL)
                {

                    /* Cannot write payload */
                    status = NX_LWM2M_CLIENT_BUFFER_TOO_SMALL;
                }
            }
        }

        if (status == NX_SUCCESS)
        {

            /* Set final length of message */
            packet_ptr -> nx_packet_length = (ULONG)(ptr - packet_ptr -> nx_packet_prepend_ptr);
            packet_ptr -> nx_packet_append_ptr = ptr;

            /* Transmit packet to UDP/DTLS */
            status = _nx_lwm2m_client_session_send(session_ptr, packet_ptr);
        }
        else
        {

            /* Failed to make request, release packet */
            nx_packet_release(packet_ptr);
        }
    }

    if (status != NX_SUCCESS)
    {

        /* Cannot send the packet to the server due to a network error, abort session */
        if (session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_REQUESTING)
        {
            session_ptr -> nx_lwm2m_client_session_state = NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_ERROR;
        }
        else
        {
            session_ptr -> nx_lwm2m_client_session_state = NX_LWM2M_CLIENT_SESSION_ERROR;
        }
        session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_IDLE;
        session_ptr -> nx_lwm2m_client_session_error = status;
    }
    else
    {

        /* Rearm retransmission timer */
        session_ptr -> nx_lwm2m_client_session_timer = tx_time_get() + session_ptr -> nx_lwm2m_client_session_retransmit_timer;
    }

    return(status);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_get_attributes                     PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function gets attributes.                                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LwM2M session      */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource_id                           Resource ID                   */
/*    flags_ptr                             Returned flags                */
/*    pmin_ptr                              Returned pmin                 */
/*    pmax_ptr                              Returned pmax                 */
/*    gt_ptr                                Returned gt                   */
/*    lt_ptr                                Returned lt                   */
/*    stp_ptr                               Returned stp                  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_notify_get                                 */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_discover_resource                                  */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static VOID _nx_lwm2m_client_get_attributes(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_ID resource_id, UINT *flags_ptr, NX_LWM2M_INT32 *pmin_ptr, NX_LWM2M_INT32 *pmax_ptr, NX_LWM2M_CLIENT_NOTIFY_NUMBER *gt_ptr, NX_LWM2M_CLIENT_NOTIFY_NUMBER *lt_ptr, NX_LWM2M_CLIENT_NOTIFY_NUMBER *stp_ptr)
{
NX_LWM2M_CLIENT_NOTIFY *notify_ptr;

    /* Get notifications structure for the object/instance/resource */
    notify_ptr = _nx_lwm2m_client_session_notify_get(session_ptr, object_ptr, instance_ptr, resource_id, NX_FALSE);

    if (notify_ptr == NX_NULL)
    {

        /* No attributes */
        return;
    }

    /* Update flags */
    *flags_ptr |= notify_ptr -> notify_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_FLAGS;

    /* Update attributes */
    if (notify_ptr -> notify_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_PMIN)
    {
        *pmin_ptr = notify_ptr -> notify_attr_pmin;
    }
    if (notify_ptr -> notify_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_PMAX)
    {
        *pmax_ptr = notify_ptr -> notify_attr_pmax;
    }
    if (notify_ptr -> notify_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_GT)
    {
        *gt_ptr = notify_ptr -> notify_attr_gt;
    }
    if (notify_ptr -> notify_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_LT)
    {
        *lt_ptr = notify_ptr -> notify_attr_lt;
    }
    if (notify_ptr -> notify_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_STP)
    {
        *stp_ptr = notify_ptr -> notify_attr_stp;
    }
}



/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_discover_resource                  PORTABLE C      */
/*                                                           6.1.10       */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes discover resource.                          */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LwM2M session      */
/*    ptr                                   Pointer to the start of buffer*/
/*    ptr_max                               Pointer to the end of buffer  */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    resource_ptr                          Pointer to resource           */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the end of payload                                       */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_corelink_path_add                                  */
/*    _nx_lwm2m_client_get_attributes                                     */
/*    _nx_lwm2m_client_corelink_attributes_add                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_send_response                              */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*  01-31-2022     Yuxin Zhou               Modified comment(s), fixed    */
/*                                            compiler warnings,          */
/*                                            resulting in version 6.1.10 */
/*                                                                        */
/**************************************************************************/
static UCHAR *_nx_lwm2m_client_discover_resource(NX_LWM2M_CLIENT_SESSION *session_ptr, UCHAR *ptr, UCHAR *ptr_max, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, const NX_LWM2M_CLIENT_RESOURCE *resource_ptr)
{
UINT flags;
NX_LWM2M_INT32 pmin = 0;
NX_LWM2M_INT32 pmax = 0;
NX_LWM2M_CLIENT_NOTIFY_NUMBER gt = 0;
NX_LWM2M_CLIENT_NOTIFY_NUMBER lt = 0;
NX_LWM2M_CLIENT_NOTIFY_NUMBER stp = 0;

    /* Write resource path */
    ptr = (UCHAR *) _nx_lwm2m_client_corelink_path_add((CHAR *) ptr, (CHAR *) ptr_max, object_ptr -> object_id, instance_ptr -> object_instance_id, resource_ptr -> resource_id);
    if (ptr == NX_NULL)
    {

        /* Not enougth space in buffer */
        return(NX_NULL);
    }

    /* Reset attributes flags */
    flags = 0;

    /* Get object attributes */
    _nx_lwm2m_client_get_attributes(session_ptr, object_ptr, NX_NULL, NX_LWM2M_CLIENT_RESERVED_ID, &flags, &pmin, &pmax, &gt, &lt, &stp);

    /* Get instance attributes */
    _nx_lwm2m_client_get_attributes(session_ptr, object_ptr, instance_ptr, NX_LWM2M_CLIENT_RESERVED_ID, &flags, &pmin, &pmax, &gt, &lt, &stp);

    /* Get resource attributes */
    _nx_lwm2m_client_get_attributes(session_ptr, object_ptr, instance_ptr, resource_ptr -> resource_id, &flags, &pmin, &pmax, &gt, &lt, &stp);

    /* Write resource attributes */
    return((UCHAR *)_nx_lwm2m_client_corelink_attributes_add((CHAR *) ptr, (CHAR *) ptr_max, 0, resource_ptr, flags, pmin, pmax, gt, lt, stp));
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_discover_instance                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes discover single instance.                   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LwM2M session      */
/*    ptr                                   Pointer to the start of buffer*/
/*    ptr_max                               Pointer to the end of buffer  */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    show_attributes                       Flag of show attributes       */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the end of payload                                       */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_notify_get                                 */
/*    _nx_lwm2m_client_corelink_path_add                                  */
/*    _nx_lwm2m_client_corelink_attributes_add                            */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_discover_instances                                 */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UCHAR *_nx_lwm2m_client_discover_instance(NX_LWM2M_CLIENT_SESSION *session_ptr, UCHAR *ptr, UCHAR *ptr_max, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, NX_LWM2M_BOOL show_attributes)
{
NX_LWM2M_CLIENT_RESOURCE infos[NX_LWM2M_CLIENT_MAX_RESOURCES];
UINT num_infos;
UINT i;

    /* get list of resources for this instance */
    num_infos = NX_LWM2M_CLIENT_MAX_RESOURCES;
    if (object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_DISCOVER, object_ptr, instance_ptr, infos, &num_infos, NX_NULL, 0) != NX_SUCCESS)
    {
        /* cannot get list of resources */
        return(NX_NULL);
    }

    for (i = 0; i < num_infos; i++)
    {
        if (i != 0)
        {
            /* check buffer space for separator */
            if (ptr >= ptr_max)
            {
                ptr = NX_NULL;

                break;
            }

            /* write separator */
            *ptr++ = ',';
        }

        /* write resource path */
        ptr = (UCHAR *) _nx_lwm2m_client_corelink_path_add((CHAR *) ptr, (CHAR *) ptr_max, object_ptr -> object_id, instance_ptr -> object_instance_id, infos[i].resource_id);
        if (ptr == NX_NULL)
        {
            /* not enougth space in buffer */
            return(NX_NULL);
        }

        /* write resource attributes */
        if (show_attributes)
        {
            NX_LWM2M_CLIENT_NOTIFY *notify_ptr;

            /* get resources attributes */
            notify_ptr = _nx_lwm2m_client_session_notify_get(session_ptr, object_ptr, instance_ptr, infos[i].resource_id, NX_FALSE);

            /* write resource attributes */
            if (notify_ptr != NX_NULL)
            {
                /* write notification attributes */
                ptr = (UCHAR *) _nx_lwm2m_client_corelink_attributes_add((CHAR *) ptr, (CHAR *) ptr_max, 0, &infos[i], notify_ptr -> notify_flags, notify_ptr -> notify_attr_pmin, notify_ptr -> notify_attr_pmax, notify_ptr -> notify_attr_gt, notify_ptr -> notify_attr_lt, notify_ptr -> notify_attr_stp);
            }
            else
            {
                /* write only dimension attribute if the resource is multiple */
                ptr = (UCHAR *) _nx_lwm2m_client_corelink_attributes_add((CHAR *) ptr, (CHAR *) ptr_max, 0, &infos[i], 0, 0, 0, 0, 0, 0);
            }
            if (ptr == NX_NULL)
            {
                /* not enougth space in buffer */
                break;
            }
        }
    }

    return(ptr);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_discover_instances                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes discover object instances.                  */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LwM2M session      */
/*    ptr                                   Pointer to the start of buffer*/
/*    ptr_max                               Pointer to the end of buffer  */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the end of payload                                       */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_notify_get                                 */
/*    _nx_lwm2m_client_corelink_path_add                                  */
/*    _nx_lwm2m_client_corelink_attributes_add                            */
/*    _nx_lwm2m_client_discover_instance                                  */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_send_response                              */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UCHAR *_nx_lwm2m_client_discover_instances(NX_LWM2M_CLIENT_SESSION *session_ptr, UCHAR *ptr, UCHAR *ptr_max, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr)
{
    NX_LWM2M_CLIENT_NOTIFY *notify_ptr;

    /* get object/instance attributes */
    notify_ptr = _nx_lwm2m_client_session_notify_get(session_ptr, object_ptr, instance_ptr, NX_LWM2M_CLIENT_RESERVED_ID, NX_FALSE);

    if (notify_ptr != NX_NULL && (notify_ptr -> notify_flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_FLAGS) != 0)
    {
        /* write object/instance path */
        ptr = (UCHAR *) _nx_lwm2m_client_corelink_path_add((CHAR *) ptr, (CHAR *) ptr_max, object_ptr -> object_id, instance_ptr == NX_NULL ? NX_LWM2M_CLIENT_RESERVED_ID : instance_ptr -> object_instance_id, NX_LWM2M_CLIENT_RESERVED_ID);

        /* write object/instance attributes */
        ptr = (UCHAR *) _nx_lwm2m_client_corelink_attributes_add((CHAR *) ptr, (CHAR *) ptr_max, 0, NX_NULL, notify_ptr -> notify_flags, notify_ptr -> notify_attr_pmin, notify_ptr -> notify_attr_pmax, notify_ptr -> notify_attr_gt, notify_ptr -> notify_attr_lt, notify_ptr -> notify_attr_stp);
        if ((ptr == NX_NULL) || (ptr >= ptr_max))
        {
            /* not enougth space in buffer */
            return(NX_NULL);
        }

        /* write separator */
        *ptr++ = ',';
    }

    if (instance_ptr != NX_NULL)
    {
        /* return a single instance */
        return _nx_lwm2m_client_discover_instance(session_ptr, ptr, ptr_max, object_ptr, instance_ptr, NX_TRUE);
    }

    /* return all instances */
    instance_ptr = object_ptr -> object_instances;
    while (instance_ptr != NX_NULL)
    {
        /* write instance information */
        ptr = _nx_lwm2m_client_discover_instance(session_ptr, ptr, ptr_max, object_ptr, instance_ptr, NX_FALSE);

        if (ptr == NX_NULL)
        {
            break;
        }

        /* get next instance */
        instance_ptr = instance_ptr -> object_instance_next;

        if (instance_ptr != NX_NULL)
        {
            /* check buffer space for separator */
            if (ptr >= ptr_max)
            {
                ptr = NX_NULL;

                break;
            }

            /* write separator */
            *ptr++ = ',';
        }
    }

    return(ptr);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_bootstrap_discover                 PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function writes bootstrap discover response.                   */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LwM2M session      */
/*    ptr                                   Pointer to the start of buffer*/
/*    ptr_max                               Pointer to the end of buffer  */
/*    object_ptr                            Pointer to object             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the end of payload                                       */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_corelink_attributes_add                            */
/*    _nx_lwm2m_client_corelink_path_add                                  */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_send_response                              */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UCHAR *_nx_lwm2m_client_bootstrap_discover(NX_LWM2M_CLIENT_SESSION *session_ptr, UCHAR *ptr, UCHAR *ptr_max, NX_LWM2M_CLIENT_OBJECT *object_ptr)
{
NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr;

    /* get first instance */
    instance_ptr = object_ptr -> object_instances;

    if (instance_ptr == NX_NULL)
    {
        /* object with no instance, just report object path */

        /* write separator */
        if (ptr >= ptr_max)
        {
            return(NX_NULL);
        }
        *ptr++ = ',';

        /* write object path */
        ptr = (UCHAR *) _nx_lwm2m_client_corelink_path_add((CHAR *) ptr, (CHAR *) ptr_max, object_ptr -> object_id, NX_LWM2M_CLIENT_RESERVED_ID, NX_LWM2M_CLIENT_RESERVED_ID);

        return(ptr);
    }

    /* write list of instances */
    while (instance_ptr != NX_NULL)
    {
        NX_LWM2M_ID ssid;

        /* write separator */
        if (ptr >= ptr_max)
        {
            return(NX_NULL);
        }
        *ptr++ = ',';

        /* write object/instance path */
        ptr = (UCHAR *) _nx_lwm2m_client_corelink_path_add((CHAR *) ptr, (CHAR *) ptr_max, object_ptr -> object_id, instance_ptr -> object_instance_id, NX_LWM2M_CLIENT_RESERVED_ID);
        if (ptr == NX_NULL)
        {
            return(NX_NULL);
        }

        /* check for short server id attribute */
#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES
        if (object_ptr == (NX_LWM2M_CLIENT_OBJECT *) &session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_security)
        {
            if (((NX_LWM2M_CLIENT_SECURITY_INSTANCE *) instance_ptr) -> security_instance_bootstrap)
            {
                ssid = 0;   /* bootstrap server */
            }
            else
            {
                ssid = ((NX_LWM2M_CLIENT_SECURITY_INSTANCE *) instance_ptr) -> security_instance_server_id;
            }
        }
        else
#endif
        if (object_ptr == (NX_LWM2M_CLIENT_OBJECT *) &session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_server)
        {
            ssid = ((NX_LWM2M_CLIENT_SERVER_INSTANCE *) instance_ptr) -> server_instance_short_id;
        }
        else
        {
            ssid = 0;
        }
        if (ssid != 0)
        {
            /* write attribute */
            ptr = (UCHAR *) _nx_lwm2m_client_corelink_attributes_add((CHAR *) ptr, (CHAR *) ptr_max, ssid, NX_NULL, 0, 0, 0, 0, 0, 0);
            if (ptr == NX_NULL)
            {
                return(NX_NULL);
            }
        }

        /* get next instance */
        instance_ptr = instance_ptr -> object_instance_next;
    }

    return(ptr);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_read_instance                      PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes reading object instance.                    */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    client_ptr                            Pointer to LwM2M client       */
/*    ptr                                   Pointer to the start of buffer*/
/*    ptr_max                               Pointer to the end of buffer  */
/*    object_ptr                            Pointer to object             */
/*    instance_ptr                          Pointer to object instance    */
/*    is_object                             Flag of reading object        */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    Pointer to the end of payload                                       */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    memcpy                                                              */
/*    _nx_lwm2m_client_tlv_resource_add                                   */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_send_response                              */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UCHAR *_nx_lwm2m_client_read_instance(NX_LWM2M_CLIENT *client_ptr, UCHAR *ptr, UCHAR *ptr_max, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, UINT is_object)
{
NX_LWM2M_CLIENT_RESOURCE *resources = client_ptr -> nx_lwm2m_client_temp_resources;
UINT num_resources;
UINT i;
NX_LWM2M_CLIENT_RESOURCE object_resource;

    /* Discover to get info of all the resources */
    num_resources = NX_LWM2M_CLIENT_MAX_RESOURCES;
    if (object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_DISCOVER, object_ptr, instance_ptr, resources, &num_resources, NX_NULL, 0) != NX_SUCCESS)
    {
        return(NX_NULL);
    }

    /* Remove unreadble resources */
    for (i = 0; i < num_resources;)
    {
        if (!(resources[i].resource_operation & NX_LWM2M_CLIENT_RESOURCE_OPERATION_READ))
        {
            if (i == (num_resources - 1))
            {
                num_resources--;
                break;
            }

            memcpy(&resources[i], &resources[num_resources - 1], sizeof(NX_LWM2M_CLIENT_RESOURCE)); /* Use case of memcpy is verified. */
            num_resources--;
        }
        else
        {
            i++;
        }
    }

    /* Read resources */
    if ((num_resources == 0) ||
        (object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_READ, object_ptr, instance_ptr, resources, &num_resources, NX_NULL, 0) != NX_SUCCESS))
    {
        return(NX_NULL);
    }

    /* Add resources to payload */
    if (is_object)
    {
        object_resource.resource_id = instance_ptr -> object_instance_id;
        object_resource.resource_type = NX_LWM2M_CLIENT_RESOURCE_OBJECT_INSTANCE;
        object_resource.resource_dim = (UCHAR)num_resources;
        object_resource.resource_value.resource_bufferdata.resource_buffer_ptr = resources;
        object_resource.resource_value.resource_bufferdata.resource_buffer_length = num_resources * sizeof(NX_LWM2M_CLIENT_RESOURCE);

        ptr = _nx_lwm2m_client_tlv_resource_add(ptr, ptr_max, &object_resource);
    }
    else
    {
        for (i = 0; i < num_resources; i++)
        {
            ptr = _nx_lwm2m_client_tlv_resource_add(ptr, ptr_max, &resources[i]);
        }
    }

    return(ptr);

}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_send_response              PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function sends a response packet from a LWM2M session.         */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    code                                  The response code             */
/*    id                                    The message id                */
/*    ack                                   Set to True if it's an ACK,   */
/*                                          send a NON otherwise          */
/*    token_ptr                             Pointer to the token          */
/*    token_length                          Length of token               */
/*    observe                               Set to true to include an     */
/*                                          Observe option                */
/*    format                                Format of payload:            */
/*    object_ptr                            Pointer to Object to include  */
/*                                          as payload                    */
/*    instance_ptr                          Pointer to Object Instance    */
/*    resource_ptr                          Pointer to Object Resource    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_packet_allocate                            */
/*    _nx_lwm2m_client_coap_option_uint_add                               */
/*    _nx_lwm2m_client_bootstrap_discover                                 */
/*    _nx_lwm2m_client_discover_resource                                  */
/*    _nx_lwm2m_client_discover_instances                                 */
/*    _nx_lwm2m_client_tlv_resource_add                                   */
/*    _nx_lwm2m_client_read_instance                                      */
/*    _nx_lwm2m_client_session_send                                       */
/*    memcpy                                                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_session_receive                                    */
/*    _nx_lwm2m_client_session_step                                       */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_send_response(NX_LWM2M_CLIENT_SESSION *session_ptr, UCHAR code, USHORT id, NX_LWM2M_BOOL ack, const VOID *token_ptr, UINT token_length, NX_LWM2M_BOOL observe, UINT format, NX_LWM2M_CLIENT_OBJECT *object_ptr, NX_LWM2M_CLIENT_OBJECT_INSTANCE *instance_ptr, const NX_LWM2M_CLIENT_RESOURCE *resource_ptr)
{
    NX_PACKET *packet_ptr;
    UCHAR *ptr;

    /* allocate a pack for the CoAP request message */
    packet_ptr = _nx_lwm2m_client_session_packet_allocate(session_ptr);
    if (packet_ptr == NX_NULL)
    {
        /* Failed to allocate the packet */
        return(NX_LWM2M_CLIENT_NO_MEMORY);
    }

    /* start of message */
    ptr = packet_ptr -> nx_packet_prepend_ptr;

    /* Check that the packet is large enought for the CoAP header */
    if (ptr + 4 + token_length > packet_ptr -> nx_packet_append_ptr)
    {
        nx_packet_release(packet_ptr);
        return(NX_LWM2M_CLIENT_BUFFER_TOO_SMALL);
    }

    /* set CoAP header */
    *ptr++ = (UCHAR)((UINT)NX_LWM2M_CLIENT_COAP_VERSION_1 | (UINT)(ack ? NX_LWM2M_CLIENT_COAP_TYPE_ACK : NX_LWM2M_CLIENT_COAP_TYPE_NON) | token_length);
    *ptr++ = code;
    *ptr++ = (UCHAR) (id >> 8);
    *ptr++ = (UCHAR)  id;
    if (token_length > 0)
    {
        memcpy(ptr, token_ptr, token_length); /* Use case of memcpy is verified. */
        ptr += token_length;
    }

    /* add content */
    if (code == NX_LWM2M_CLIENT_COAP_STATUS_CONTENT)
    {
        UINT last_option;
        UCHAR *coap_header_end_ptr;

        /* save pointer to end of coap header */
        coap_header_end_ptr = ptr;

        /* initialize last option number */
        last_option = 0;

        /* Include Observe option */
        if (observe)
        {
            ULONG sequence;

            /* generate 24-bit sequence number from system clock */
            sequence = tx_time_get() & 0xffffff;

            /* add option */
            ptr = _nx_lwm2m_client_coap_option_uint_add(ptr, packet_ptr -> nx_packet_append_ptr, &last_option, NX_LWM2M_CLIENT_COAP_OPTION_OBSERVE, sequence);
            if (ptr == NX_NULL)
            {
                nx_packet_release(packet_ptr);
                return(NX_LWM2M_CLIENT_NO_MEMORY);
            }
        }

        /* Include Content-Format option */
        ptr = _nx_lwm2m_client_coap_option_uint_add(ptr, packet_ptr -> nx_packet_append_ptr, &last_option, NX_LWM2M_CLIENT_COAP_OPTION_CONTENT_FORMAT, format);
        if ((ptr == NX_NULL) || (ptr >= packet_ptr -> nx_packet_append_ptr))
        {
            nx_packet_release(packet_ptr);
            return(NX_LWM2M_CLIENT_NO_MEMORY);
        }

        /* end of options */
        *ptr++ = NX_LWM2M_CLIENT_COAP_PAYLOAD_MARKER;

        /* write payload */
        switch (format)
        {

        case NX_LWM2M_CLIENT_COAP_CONTENT_LINK_FORMAT:

            /* Return list of IDs and attributes */
            if (session_ptr -> nx_lwm2m_client_session_server_instance_ptr == NX_NULL)
            {
                /* Bootstrap Discover */

                /* Check packet payload */
                if ((ptr + 11) > packet_ptr -> nx_packet_append_ptr)
                {
                    ptr = NX_NULL;
                    break;
                }

                /* write enabler version */
                memcpy(ptr, "lwm2m=\"1.0\"", 11); /* Use case of memcpy is verified. */
                ptr += 11;

                /* write list of object instances */
                if (object_ptr == NX_NULL)
                {
                    /* discover all objects */
                    NX_LWM2M_CLIENT_OBJECT *current_ptr;
#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES
                    current_ptr = (NX_LWM2M_CLIENT_OBJECT *) &session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_security;
#else
                    current_ptr = (NX_LWM2M_CLIENT_OBJECT *) &session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_server;
#endif
                    while (current_ptr != NX_NULL)
                    {
                        ptr = _nx_lwm2m_client_bootstrap_discover(session_ptr, ptr, packet_ptr -> nx_packet_append_ptr, current_ptr);
                        if (ptr == NX_NULL)
                        {
                            break;
                        }

                        /* get next object */
                        current_ptr = current_ptr -> object_next;
                    }
                }
                else
                {
                    /* discover single object */
                    ptr = _nx_lwm2m_client_bootstrap_discover(session_ptr, ptr, packet_ptr -> nx_packet_append_ptr, object_ptr);
                }
            }
            else if (resource_ptr != NX_NULL)
            {
                /* return attributes of a single resource */
                ptr = _nx_lwm2m_client_discover_resource(session_ptr, ptr, packet_ptr -> nx_packet_append_ptr, object_ptr, instance_ptr, resource_ptr);
            }
            else
            {
                /* return attributes of object instance(s) */
                ptr = _nx_lwm2m_client_discover_instances(session_ptr, ptr, packet_ptr -> nx_packet_append_ptr, object_ptr, instance_ptr);
            }
            break;

        case NX_LWM2M_CLIENT_COAP_CONTENT_TEXT_PLAIN:

            /* Return single text string value */
            if (resource_ptr != NX_NULL &&
                resource_ptr -> resource_type == NX_LWM2M_CLIENT_RESOURCE_STRING)
            {
                UINT length;

                length = resource_ptr -> resource_value.resource_bufferdata.resource_buffer_length;
                if (ptr + length > packet_ptr -> nx_packet_append_ptr)
                {
                    /* payload too big */
                    ptr = NX_NULL;
                }
                else if (length != 0)
                {
                    memcpy(ptr, resource_ptr -> resource_value.resource_bufferdata.resource_buffer_ptr, length); /* Use case of memcpy is verified. */
                    ptr += length;
                }
            }
            break;

        case NX_LWM2M_CLIENT_COAP_CONTENT_OCTET_STREAM:

            /* Return single opaque value */
            if (resource_ptr != NX_NULL &&
                resource_ptr -> resource_type == NX_LWM2M_CLIENT_RESOURCE_OPAQUE)
            {
                if (ptr + resource_ptr -> resource_value.resource_bufferdata.resource_buffer_length > packet_ptr -> nx_packet_append_ptr)
                {
                    /* payload too big */
                    ptr = NX_NULL;
                }
                else if (resource_ptr -> resource_value.resource_bufferdata.resource_buffer_length != 0)
                {
                    memcpy(ptr, resource_ptr -> resource_value.resource_bufferdata.resource_buffer_ptr, resource_ptr -> resource_value.resource_bufferdata.resource_buffer_length); /* Use case of memcpy is verified. */
                    ptr += resource_ptr -> resource_value.resource_bufferdata.resource_buffer_length;
                }
            }
            break;

        case NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV:

            if (resource_ptr != NX_NULL)
            {
                /* Return single value */
                ptr = _nx_lwm2m_client_tlv_resource_add(ptr, packet_ptr -> nx_packet_append_ptr, resource_ptr);
            }
            else if (instance_ptr != NX_NULL)
            {
                /* Return full instance */
                ptr = _nx_lwm2m_client_read_instance(session_ptr -> nx_lwm2m_client_session_client_ptr, ptr, packet_ptr -> nx_packet_append_ptr, object_ptr, instance_ptr, NX_FALSE);
            }
            else
            {
                /* Return all instances */
                instance_ptr = object_ptr -> object_instances;
                while (instance_ptr != NX_NULL)
                {
                    ptr = _nx_lwm2m_client_read_instance(session_ptr -> nx_lwm2m_client_session_client_ptr, ptr, packet_ptr -> nx_packet_append_ptr, object_ptr, instance_ptr, NX_TRUE);

                    if (ptr == NX_NULL)
                    {
                        break;
                    }

                    instance_ptr = instance_ptr -> object_instance_next;
                }
            }
            break;

        default:

            /* Unknown format */
            break;
        }

        if (ptr == NX_NULL)
        {
            /* failed to write payload, change response code */
            /* XXX is there a better error code? */
            packet_ptr -> nx_packet_prepend_ptr[1] = NX_LWM2M_CLIENT_COAP_STATUS_INTERNAL_SERVER_ERROR;

            /* send only coap header */
            ptr = coap_header_end_ptr;
        }
    }

    /* set final length of message */
    packet_ptr -> nx_packet_length = (ULONG)(ptr - packet_ptr -> nx_packet_prepend_ptr);
    packet_ptr -> nx_packet_append_ptr = ptr;

    /* Transmit packet to UDP/DTLS */
    return(_nx_lwm2m_client_session_send(session_ptr, packet_ptr));
}

#ifdef NX_SECURE_ENABLE_DTLS

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    coaps_message_notify                                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function processes secure CoAP socket receive notification.    */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    socket_ptr                            Pointer to the CoAP socket    */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_event_flags_set                                                  */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    UDP stack                                                           */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static VOID coaps_message_notify(NX_UDP_SOCKET *socket_ptr)
{

    /* Get pointer to LWM2M Client */
    NX_LWM2M_CLIENT *client_ptr = ((NX_LWM2M_CLIENT_SOCKET *) socket_ptr) -> client_ptr;

    /* Set Secure CoAP Message event flag */
    tx_event_flags_set(&client_ptr -> nx_lwm2m_client_event_flags, NX_LWM2M_CLIENT_EVENT_COAPS_MESSAGE, TX_OR);
}
#endif /* NX_SECURE_ENABLE_DTLS */


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_start                      PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function starts a LWM2M session with a Bootstrap server or a   */
/*    LWM2M device management server.                                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    bootstrap                             Set to True if it's a         */
/*                                          bootstrap session             */
/*    server_id                             The Bootstrap Security ID or  */
/*                                          the Short Server ID           */
/*    ip_address                            The IP address of the server  */
/*    port                                  The UDP port of the server    */
/*    dtls_session_ptr                      The pointer to the DTLS       */
/*                                          session if COAPS is used      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    tx_event_flags_set                                                  */
/*    _nx_lwm2m_client_session_state_update                               */
/*    _nx_lwm2m_client_resource_info_set                                  */
/*    _nx_lwm2m_client_resource_integer32_set                             */
/*    _nx_lwm2m_client_object_create_internal                             */
/*    _nx_lwm2m_client_object_instance_ptr_get                            */
/*    _nx_lwm2m_client_object_delete_all                                  */
/*    _nx_lwm2m_client_resource_boolean_set                               */
/*    nx_udp_socket_create                                                */
/*    nx_udp_socket_bind                                                  */
/*    nx_udp_socket_delete                                                */
/*    nx_udp_socket_receive_notify                                        */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
#ifdef NX_SECURE_ENABLE_DTLS
UINT _nx_lwm2m_client_session_start(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_BOOL bootstrap, NX_LWM2M_ID server_id, const NXD_ADDRESS *ip_address, UINT port, NX_SECURE_DTLS_SESSION *dtls_session_ptr)
#else
UINT _nx_lwm2m_client_session_start(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_BOOL bootstrap, NX_LWM2M_ID server_id, const NXD_ADDRESS *ip_address, UINT port)
#endif /* NX_SECURE_ENABLE_DTLS */
{
NX_LWM2M_CLIENT *client_ptr = session_ptr -> nx_lwm2m_client_session_client_ptr;
UINT status;
NX_LWM2M_CLIENT_SERVER_INSTANCE *server_instance_ptr;
NX_LWM2M_ID instance_id;
NX_LWM2M_CLIENT_RESOURCE *resources = client_ptr -> nx_lwm2m_client_temp_resources;

    status = NX_SUCCESS;

    /* Acquire the client mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Clear previous state resources */
    if (session_ptr -> nx_lwm2m_client_session_state != NX_LWM2M_CLIENT_SESSION_INIT)
    {
        _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_INIT);
    }

    if (!bootstrap)
    {

        /* find LWM2M server instance corresponding to the short server ID */
        server_instance_ptr = (NX_LWM2M_CLIENT_SERVER_INSTANCE *) client_ptr -> nx_lwm2m_client_server.server_object.object_instances;
        while (server_instance_ptr != NX_NULL)
        {
            if (server_instance_ptr -> server_instance_short_id == server_id)
            {
                break;
            }

            server_instance_ptr = (NX_LWM2M_CLIENT_SERVER_INSTANCE *) server_instance_ptr -> server_instance.object_instance_next;
        }

        if (server_instance_ptr == NX_NULL)
        {

            /* Create a server instance */
            /* Setup the info for register server */
            _nx_lwm2m_client_resource_info_set(&resources[0], NX_LWM2M_CLIENT_SERVER_SHORT_SERVER_ID, NX_NULL);
            _nx_lwm2m_client_resource_integer32_set(&resources[0], server_id);
            _nx_lwm2m_client_resource_info_set(&resources[1], NX_LWM2M_CLIENT_SERVER_LIFETIME_ID, NX_NULL);
            _nx_lwm2m_client_resource_integer32_set(&resources[1], NX_LWM2M_CLIENT_LIFE_TIME);

            /* Create a server instance for register server */
            instance_id = NX_LWM2M_CLIENT_RESERVED_ID;
            status = _nx_lwm2m_client_object_create_internal(&(client_ptr -> nx_lwm2m_client_server.server_object), 
                                                             &instance_id, 2, resources, NX_TRUE);
            if (status)
            {
                return(status);
            }

            /* Get the pointer of created server instace */
            server_instance_ptr = (NX_LWM2M_CLIENT_SERVER_INSTANCE *)_nx_lwm2m_client_object_instance_ptr_get(&(client_ptr -> nx_lwm2m_client_server.server_object), instance_id);
        }
    }
    else
    {

#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES
        /* Clear all the security instances */
        if (client_ptr -> nx_lwm2m_client_security.security_object.object_instances != NX_NULL)
        {
            _nx_lwm2m_client_object_delete_all(client_ptr, &(client_ptr -> nx_lwm2m_client_security.security_object), NX_LWM2M_CLIENT_RESERVED_ID);
        }

        /* Setup the info for bootstrap server */
        _nx_lwm2m_client_resource_info_set(&resources[0], NX_LWM2M_CLIENT_SECURITY_BOOTSTRAP_ID, NX_NULL);
        _nx_lwm2m_client_resource_boolean_set(&resources[0], NX_TRUE);
        _nx_lwm2m_client_resource_info_set(&resources[1], NX_LWM2M_CLIENT_SECURITY_HOLD_OFF_ID, NX_NULL);
        _nx_lwm2m_client_resource_integer32_set(&resources[1], NX_LWM2M_CLIENT_HOLD_OFF);

        /* Create a security instance for bootstrap server */
        instance_id = server_id;
        status = _nx_lwm2m_client_object_create_internal(&(client_ptr -> nx_lwm2m_client_security.security_object), 
                                                         &instance_id, 2, resources, NX_TRUE);
        if (status)
        {
            return(status);
        }
#endif

        /* Bootstrap server, no related instance */
        server_instance_ptr = NX_NULL;
    }

    if (status == NX_SUCCESS)
    {

        /* Set pointer to Server Object Instance */
        session_ptr -> nx_lwm2m_client_session_server_instance_ptr = server_instance_ptr;

        /* Set address of server */
        session_ptr -> nx_lwm2m_client_session_server_address = *ip_address;

        /* Set port number */
        session_ptr -> nx_lwm2m_client_session_server_port = port;

        /* Set security/short server ID */
        session_ptr -> nx_lwm2m_client_session_server_id = server_id;

        /* Set substate to 'bootstrap' or 'register' */
        session_ptr -> nx_lwm2m_client_session_substate = bootstrap ? NX_LWM2M_CLIENT_SESSION_SUBSTATE_BOOTSTRAP : NX_LWM2M_CLIENT_SESSION_SUBSTATE_REGISTER;

        /* Reset error code */
        session_ptr -> nx_lwm2m_client_session_error = NX_SUCCESS;

#ifdef NX_SECURE_ENABLE_DTLS
        if (dtls_session_ptr != NX_NULL)
        {

            /* Create UDP socket for the DTLS session */
            status = nx_udp_socket_create(client_ptr -> nx_lwm2m_client_ip_ptr, (NX_UDP_SOCKET *) &session_ptr -> nx_lwm2m_client_session_dtls_socket, "LWM2M Client Secure CoAP Socket", NX_LWM2M_CLIENT_SOCKET_TOS, NX_DONT_FRAGMENT, NX_LWM2M_CLIENT_SOCKET_TTL, NX_LWM2M_CLIENT_SOCKET_QUEUE_MAX);
            if (status != NX_SUCCESS)
            {
                status = NX_LWM2M_CLIENT_ERROR;
            }
            else
            {

                /* Set the local port of the UDP socket*/
                status = nx_udp_socket_bind((NX_UDP_SOCKET *) &session_ptr -> nx_lwm2m_client_session_dtls_socket, NX_ANY_PORT, NX_NO_WAIT);
                if (status != NX_SUCCESS)
                {
                    status = NX_LWM2M_CLIENT_PORT_UNAVAILABLE;

                    nx_udp_socket_delete((NX_UDP_SOCKET *) &session_ptr -> nx_lwm2m_client_session_dtls_socket);
                }
                else
                {

                    /* Set the COAPS receive notify callback */
                    session_ptr -> nx_lwm2m_client_session_dtls_socket.client_ptr = client_ptr;
                    nx_udp_socket_receive_notify((NX_UDP_SOCKET *) &session_ptr -> nx_lwm2m_client_session_dtls_socket, coaps_message_notify);

                    /* Set the DTLS session pointer */
                    session_ptr -> nx_lwm2m_client_session_dtls_session = dtls_session_ptr;
                }
            }
        }
#endif /* NX_SECURE_ENABLE_DTLS */
    }

    /* Release the client mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    if (status == NX_SUCCESS)
    {

        /* Wakeup client thread */
        tx_event_flags_set(&client_ptr -> nx_lwm2m_client_event_flags, NX_LWM2M_CLIENT_EVENT_WAKEUP, TX_OR);
    }

    return(status);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_state_update               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function updates the state of a LWM2M session.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    state                                 The new state of the session  */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    none                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    nx_secure_dtls_client_session_start                                 */
/*    _nx_lwm2m_client_session_cleanup                                    */
/*    _nx_lwm2m_client_session_send_request                               */
/*    _nx_lwm2m_client_object_instance_ptr_get                            */
/*    tx_time_get                                                         */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_object_delete                                      */
/*    _nx_lwm2m_client_session_receive                                    */
/*    _nx_lwm2m_client_session_delete                                     */
/*    _nx_lwm2m_client_session_start                                      */
/*    _nx_lwm2m_client_session_step                                       */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_session_state_update(NX_LWM2M_CLIENT_SESSION *session_ptr, UINT state)
{
#ifdef NX_SECURE_ENABLE_DTLS
UINT status;
#endif
#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES
NX_LWM2M_INT32 hold_off = 0;
#endif

    /* Update state and sub-state of session */
    session_ptr -> nx_lwm2m_client_session_state = state;

    switch (state)
    {

    case NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_REQUESTING:

#ifdef NX_SECURE_ENABLE_DTLS
        /* DTLS session */
        if (session_ptr -> nx_lwm2m_client_session_dtls_session != NX_NULL)
        {

            /* Establish the DTLS session */
            status = nx_secure_dtls_client_session_start(session_ptr -> nx_lwm2m_client_session_dtls_session, (NX_UDP_SOCKET *) &session_ptr -> nx_lwm2m_client_session_dtls_socket, 
                                                         &(session_ptr -> nx_lwm2m_client_session_server_address), session_ptr -> nx_lwm2m_client_session_server_port, NX_LWM2M_CLIENT_DTLS_START_TIMEOUT);
            if (status != NX_SUCCESS)
            {

                /* Failed to connect to DTLS server */
                session_ptr -> nx_lwm2m_client_session_error = NX_LWM2M_CLIENT_DTLS_ERROR;
                session_ptr -> nx_lwm2m_client_session_state = NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_ERROR;
                _nx_lwm2m_client_session_cleanup(session_ptr);
                return;
            }
        }
#endif

        /* Send the first request */
        _nx_lwm2m_client_session_send_request(session_ptr, NX_TRUE);

        return;

    case NX_LWM2M_CLIENT_SESSION_REGISTERING:

#ifdef NX_SECURE_ENABLE_DTLS
        /* DTLS session */
        if (session_ptr -> nx_lwm2m_client_session_dtls_session != NX_NULL)
        {

            /* Establish the DTLS session */
            status = nx_secure_dtls_client_session_start(session_ptr -> nx_lwm2m_client_session_dtls_session, (NX_UDP_SOCKET *) &session_ptr -> nx_lwm2m_client_session_dtls_socket, 
                                                         &(session_ptr -> nx_lwm2m_client_session_server_address), session_ptr -> nx_lwm2m_client_session_server_port, NX_LWM2M_CLIENT_DTLS_START_TIMEOUT);
            if (status != NX_SUCCESS)
            {

                /* Failed to connect to DTLS server */
                session_ptr -> nx_lwm2m_client_session_error = NX_LWM2M_CLIENT_DTLS_ERROR;
                session_ptr -> nx_lwm2m_client_session_state = NX_LWM2M_CLIENT_SESSION_ERROR;
                _nx_lwm2m_client_session_cleanup(session_ptr);
                return;
            }
        }
#endif

        /* Send lifetime, binding mode and list of objects in request */
        session_ptr -> nx_lwm2m_client_session_flags |= NX_LWM2M_CLIENT_SESSION_UPDATE_LIFETIME|NX_LWM2M_CLIENT_SESSION_UPDATE_BINDING_MODE|NX_LWM2M_CLIENT_SESSION_UPDATE_OBJECTS_LIST;

        /* Send MSISDN parameter only if defined */
        if (session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_msisdn != NX_NULL)
        {
            session_ptr -> nx_lwm2m_client_session_flags |= NX_LWM2M_CLIENT_SESSION_UPDATE_MSISDN;
        }
        else
        {
            session_ptr -> nx_lwm2m_client_session_flags &= (UINT)(~NX_LWM2M_CLIENT_SESSION_UPDATE_MSISDN);
        }

        /* Reset Uri-Path of server */
        session_ptr -> nx_lwm2m_client_session_uri_path_length = 0;

        /* Send the first request */
        _nx_lwm2m_client_session_send_request(session_ptr, NX_TRUE);

        return;

    case NX_LWM2M_CLIENT_SESSION_UPDATING:
    case NX_LWM2M_CLIENT_SESSION_DISABLING:
    case NX_LWM2M_CLIENT_SESSION_DEREGISTERING:

        /* Send the first request */
        _nx_lwm2m_client_session_send_request(session_ptr, NX_TRUE);

        return;

    case NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_WAITING:
        {

            /* Set substate for bootstrap */
            session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_WAIT;

#if NX_LWM2M_CLIENT_MAX_SECURITY_INSTANCES

            /* Get hold-off timer from security object if it exists */
            if (session_ptr -> nx_lwm2m_client_session_server_id != NX_LWM2M_CLIENT_RESERVED_ID)
            {
                NX_LWM2M_CLIENT_SECURITY_INSTANCE *security_ptr;

                security_ptr = (NX_LWM2M_CLIENT_SECURITY_INSTANCE *) _nx_lwm2m_client_object_instance_ptr_get((NX_LWM2M_CLIENT_OBJECT *) &session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_security, session_ptr -> nx_lwm2m_client_session_server_id);
                if (security_ptr != NX_NULL)
                {
                    hold_off = security_ptr -> security_instance_hold_off;
                }
            }

            if (hold_off)
            {

                /* Start bootstrap after hold-off time */
                session_ptr -> nx_lwm2m_client_session_timer = tx_time_get() + (ULONG)(hold_off) * NX_IP_PERIODIC_RATE;

                return;
            }
#endif

            /* Hold-off timer disabled, start bootstrap after one tick */
            session_ptr -> nx_lwm2m_client_session_timer = tx_time_get() + 1;

            return;
        }

    case NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_INITIATED:

        /* Arm bootstrap completion timer */
        session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_WAIT;

        session_ptr -> nx_lwm2m_client_session_timer = tx_time_get() + NX_LWM2M_CLIENT_BOOTSTRAP_IDLE_TIMER;

        return;

    case NX_LWM2M_CLIENT_SESSION_REGISTERED:

        /* Arm update timer at half of the session lifetime */
        session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_WAIT;

        session_ptr -> nx_lwm2m_client_session_timer = tx_time_get() + (ULONG)(session_ptr -> nx_lwm2m_client_session_server_instance_ptr -> server_instance_lifetime)/2 * NX_IP_PERIODIC_RATE;

        /* Clear list of parameters to report to server */
        session_ptr -> nx_lwm2m_client_session_flags &= (UINT)(~NX_LWM2M_CLIENT_SESSION_UPDATE_ALL);

        return;

    /* NX_LWM2M_CLIENT_SESSION_INIT */
    /* NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_FINISHED */
    /* NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_ERROR */
    /* NX_LWM2M_CLIENT_SESSION_DEREGISTERED */
    /* NX_LWM2M_CLIENT_SESSION_DISABLED */
    /* NX_LWM2M_CLIENT_SESSION_ERROR */
    default:

        /* Reset the session */
        _nx_lwm2m_client_session_cleanup(session_ptr);

        return;

    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_step                       PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This functions checks if the session state needs to be updated      */
/*    due to application request or timer expiration.                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    has_timer_ptr                         On return, set to NX_TRUE if  */
/*                                          the session has active timers */
/*    timer_expire_ptr                      On return, the expiration     */
/*                                          time of the next active timer */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    NX_TRUE                               The session callback has been */
/*                                          called,                       */
/*    NX_FALSE                              The session callback has not  */
/*                                          been called                   */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_session_state_update                               */
/*    _nx_lwm2m_client_session_send_request                               */
/*    _nx_lwm2m_client_resource_notify_number_get                         */
/*    _nx_lwm2m_client_session_notify_attributes                          */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_thread_entry                                       */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
NX_LWM2M_BOOL _nx_lwm2m_client_session_step(NX_LWM2M_CLIENT_SESSION *session_ptr, NX_LWM2M_BOOL *has_timer_ptr, ULONG *timer_expire_ptr)
{
ULONG now;
ULONG next_timer;

    if (session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_IDLE)
    {

        /* Idle session, got nothing to do. */
        return(NX_FALSE);
    }

    /* Check for state transition request from application */
    if (session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_BOOTSTRAP ||
        session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_REGISTER ||
        session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_UPDATE ||
        session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_DISABLE ||
        session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_DEREGISTER)
    {

        /* Update current state */
        switch (session_ptr -> nx_lwm2m_client_session_substate)
        {

        case NX_LWM2M_CLIENT_SESSION_SUBSTATE_BOOTSTRAP:

            _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_WAITING);
            break;

        case NX_LWM2M_CLIENT_SESSION_SUBSTATE_REGISTER:

            _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_REGISTERING);
            break;

        case NX_LWM2M_CLIENT_SESSION_SUBSTATE_UPDATE:

            _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_UPDATING);
            break;

        case NX_LWM2M_CLIENT_SESSION_SUBSTATE_DISABLE:

            _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_DISABLING);
            break;

        /* NX_LWM2M_CLIENT_SESSION_SUBSTATE_DEREGISTER */
        default:

            _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_DEREGISTERING);
            break;

        }

        /* Report new state to state callback */
        session_ptr -> nx_lwm2m_client_session_state_callback(session_ptr, session_ptr -> nx_lwm2m_client_session_state);

        return(NX_TRUE);
    }

    /* Check for session timer expiration */
    now = tx_time_get();

    if ((LONG) (session_ptr -> nx_lwm2m_client_session_timer - now) <= 0)
    {
        if (session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_REQUEST_SENT)
        {

            /* Request retransmission timer expired */

            /* Re-send the request */
            if (_nx_lwm2m_client_session_send_request(session_ptr, NX_FALSE) != NX_SUCCESS)
            {

                /* Report the error to the state callback */
                session_ptr -> nx_lwm2m_client_session_state_callback(session_ptr, session_ptr -> nx_lwm2m_client_session_state);

                return(NX_TRUE);
            }
        }
        else
        {

            /* substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_WAIT */
            /* Session state timer expired */

            /* Get type of timer event */
            switch (session_ptr -> nx_lwm2m_client_session_state)
            {

            case NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_WAITING:

                /* Hold off timer expired, start client initiated bootstrap */
                _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_REQUESTING);
                break;

            case NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_INITIATED:

                /* No request from bootstrap server for too long, report error */
                session_ptr -> nx_lwm2m_client_session_error = NX_LWM2M_CLIENT_TIMED_OUT;

                _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_BOOTSTRAP_ERROR);
                break;

            case NX_LWM2M_CLIENT_SESSION_REGISTERED:

                /* Registration timer expired, send update */
                _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_UPDATING);
                break;

            case NX_LWM2M_CLIENT_SESSION_DISABLED:

                /* Disabled timer expired, register to server again */
                _nx_lwm2m_client_session_state_update(session_ptr, NX_LWM2M_CLIENT_SESSION_REGISTERING);
                break;

            default:

                /* XXX SHOULD NOT GO HERE */
                NX_ASSERT(0);
            }

            /* Report new state to state callback */
            session_ptr -> nx_lwm2m_client_session_state_callback(session_ptr, session_ptr -> nx_lwm2m_client_session_state);

            return(NX_TRUE);
        }
    }

    /* Update the next timer expiration */
    next_timer = *timer_expire_ptr;
    if (!(*has_timer_ptr) || (LONG) (session_ptr -> nx_lwm2m_client_session_timer - next_timer) < 0)
    {
        next_timer = session_ptr -> nx_lwm2m_client_session_timer;
    }

    /* Process Notifications */
    if (session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_REGISTERED || session_ptr -> nx_lwm2m_client_session_state == NX_LWM2M_CLIENT_SESSION_UPDATING)
    {
        NX_LWM2M_CLIENT_NOTIFY *notify_ptr;

        notify_ptr = session_ptr -> nx_lwm2m_client_session_notifications;
        while (notify_ptr != NX_NULL)
        {
            if (notify_ptr -> notify_flags & NX_LWM2M_CLIENT_NOTIFY_TIMER)
            {
                /* notify timer is active */

                if ((LONG) (notify_ptr -> notify_timer - now) <= 0)
                {
                    NX_LWM2M_CLIENT_RESOURCE resource;
                    UINT resource_count;
                    UINT response_format;
                    UINT flags;
                    NX_LWM2M_INT32 pmin;
                    NX_LWM2M_INT32 pmax;
                    NX_LWM2M_CLIENT_NOTIFY_NUMBER gt;
                    NX_LWM2M_CLIENT_NOTIFY_NUMBER lt;
                    NX_LWM2M_CLIENT_NOTIFY_NUMBER stp;

                    /* Timer has expired, send notify message */

                    if (notify_ptr -> notify_resource_id != NX_LWM2M_CLIENT_RESERVED_ID)
                    {

                        /* Send a single value */
                        resource.resource_id = notify_ptr -> notify_resource_id;
                        resource_count = 1;
                        if (notify_ptr -> notify_object_ptr -> object_operation(NX_LWM2M_CLIENT_OBJECT_READ, notify_ptr -> notify_object_ptr, notify_ptr -> notify_instance_ptr, 
                                                                                                  &resource, &resource_count, NX_NULL, 0) == NX_SUCCESS)
                        {

                            /* Save value if it's a number */
                            _nx_lwm2m_client_resource_notify_number_get(&resource, &notify_ptr -> notify_last_value);
                        }
                        else
                        {

                            /* XXX should not happen */
                            resource.resource_type = NX_LWM2M_CLIENT_RESOURCE_NONE;
                        }
                        /* select response format */
                        if (notify_ptr -> notify_flags & NX_LWM2M_CLIENT_NOTIFY_ACCEPT_TLV)
                        {

                            /* Server wanted TLV */
                            response_format = NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV;
                        }
                        else if (resource.resource_type == NX_LWM2M_CLIENT_RESOURCE_STRING)
                        {

                            /* Text value */
                            response_format = NX_LWM2M_CLIENT_COAP_CONTENT_TEXT_PLAIN;
                        }
                        else if (resource.resource_type == NX_LWM2M_CLIENT_RESOURCE_OPAQUE)
                        {

                            /* Opaque value */
                            response_format = NX_LWM2M_CLIENT_COAP_CONTENT_OCTET_STREAM;
                        }
                        else
                        {

                            /* Use TLV for all other types */
                            response_format = NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV;
                        }
                    }
                    else
                    {

                        /* Return instance(s) in tlv format */
                        response_format = NX_LWM2M_CLIENT_COAP_CONTENT_VND_OMA_LWM2M_TLV;
                    }

                    /* Get new message id */
                    notify_ptr -> notify_last_id = ++session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_request_id;

                    /* Save notification time */
                    notify_ptr -> notify_last_time = tx_time_get();

                    /* Send notify message */
                    _nx_lwm2m_client_session_send_response(session_ptr, NX_LWM2M_CLIENT_COAP_STATUS_CONTENT, notify_ptr -> notify_last_id,
                                                           NX_FALSE, notify_ptr -> notify_token, (notify_ptr -> notify_flags & NX_LWM2M_CLIENT_NOTIFY_TOKEN_LEN),
                                                           NX_TRUE, response_format, notify_ptr -> notify_object_ptr, notify_ptr -> notify_instance_ptr,
                                                           (notify_ptr -> notify_resource_id == NX_LWM2M_CLIENT_RESERVED_ID ? NX_NULL : &resource));

                    /* Clear CHANGED flag */
                    notify_ptr -> notify_flags &= (UINT)(~NX_LWM2M_CLIENT_NOTIFY_CHANGED);

                    /* We need to rearm timer if pmax is defined */
                    _nx_lwm2m_client_session_notify_attributes(session_ptr, notify_ptr, &flags, &pmin, &pmax, &gt, &lt, &stp);
                    if (!(flags & NX_LWM2M_CLIENT_NOTIFY_ATTR_PMAX))
                    {

                        /* Get default maximum period from server object */
                        pmax = session_ptr -> nx_lwm2m_client_session_server_instance_ptr -> server_instance_max_period;
                    }

                    if (pmax > 0)
                    {

                        /* Rearm timer */
                        notify_ptr -> notify_timer = notify_ptr -> notify_last_time + (ULONG)(pmax) * NX_IP_PERIODIC_RATE;
                        if ((LONG) (notify_ptr -> notify_timer - next_timer) < 0)
                        {

                            /* Update next timer value if current notify timer expires before */
                            next_timer = notify_ptr -> notify_timer;
                        }
                    }
                    else
                    {

                        /* Clear timer flag */
                        notify_ptr -> notify_flags &= (UINT)(~NX_LWM2M_CLIENT_NOTIFY_TIMER);
                    }
                }
                else if ((LONG) (notify_ptr -> notify_timer - next_timer) < 0)
                {

                    /* Update next timer value if current notify timer expires before */
                    next_timer = notify_ptr -> notify_timer;
                }
            }

            notify_ptr = notify_ptr -> notify_next;
        }
    }

    /* Return next timer expiration */
    *has_timer_ptr = NX_TRUE;
    *timer_expire_ptr = next_timer;

    return(NX_FALSE);
}

/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_update                     PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function updates the Client to a LWM2M Server.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    tx_event_flags_set                                                  */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    Application Code                                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_session_update(NX_LWM2M_CLIENT_SESSION *session_ptr)
{
NX_LWM2M_CLIENT *client_ptr = session_ptr -> nx_lwm2m_client_session_client_ptr;
UINT status;
NX_LWM2M_BOOL do_wakeup;

    status = NX_SUCCESS;
    do_wakeup = NX_FALSE;

    /* Acquire the client mutex */
    tx_mutex_get(&client_ptr -> nx_lwm2m_client_mutex, TX_WAIT_FOREVER);

    /* Check session state */
    if ((session_ptr -> nx_lwm2m_client_session_state != NX_LWM2M_CLIENT_SESSION_REGISTERED &&
         session_ptr -> nx_lwm2m_client_session_state != NX_LWM2M_CLIENT_SESSION_UPDATING) ||
        session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_DISABLE ||
        session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_DEREGISTER)
    {
        status = NX_LWM2M_CLIENT_NOT_REGISTERED;
    }
    else if (session_ptr -> nx_lwm2m_client_session_state != NX_LWM2M_CLIENT_SESSION_UPDATING &&
             session_ptr -> nx_lwm2m_client_session_substate != NX_LWM2M_CLIENT_SESSION_SUBSTATE_UPDATE)
    {

        /* Tell session to start updating */
        session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_UPDATE;
        do_wakeup = NX_TRUE;
    }

    /* Release the client mutex */
    tx_mutex_put(&client_ptr -> nx_lwm2m_client_mutex);

    if (do_wakeup)
    {

        /* Wakeup client thread */
        tx_event_flags_set(&client_ptr -> nx_lwm2m_client_event_flags, NX_LWM2M_CLIENT_EVENT_WAKEUP, TX_OR);
    }

    return(status);
}

/* Define the maximum delay before sending an update message */
#define MAX_UPDATE_DELAY     NX_IP_PERIODIC_RATE


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_session_update_flags               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function updates the information that needs to be sent to the  */
/*    the next Client Update message.                                     */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    session_ptr                           Pointer to LWM2M session      */
/*    flags                                 The information that has      */
/*                                          changed since last update     */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    none                                                                */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_time_get                                                         */
/*    tx_event_flags_set                                                  */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_object_list_changed                                */
/*    _nx_lwm2m_client_server_write                                       */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
VOID _nx_lwm2m_client_session_update_flags(NX_LWM2M_CLIENT_SESSION *session_ptr, UINT flags)
{
ULONG now;

    /* Only update registered sessions */
    if ((session_ptr -> nx_lwm2m_client_session_state != NX_LWM2M_CLIENT_SESSION_REGISTERED &&
         session_ptr -> nx_lwm2m_client_session_state != NX_LWM2M_CLIENT_SESSION_UPDATING) ||
        session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_DISABLE ||
        session_ptr -> nx_lwm2m_client_session_substate == NX_LWM2M_CLIENT_SESSION_SUBSTATE_DEREGISTER)
    {

        /* Not registered or currently de-registring/disabling */
        return;
    }

    /* Set update flags */
    session_ptr -> nx_lwm2m_client_session_flags |= flags;

    /* Schedule the sending of the update message */
    now = tx_time_get();
    if (session_ptr -> nx_lwm2m_client_session_substate != NX_LWM2M_CLIENT_SESSION_SUBSTATE_WAIT || ((LONG) (session_ptr -> nx_lwm2m_client_session_timer - now)) > (LONG)MAX_UPDATE_DELAY)
    {

        /* Arm update timer */
        session_ptr -> nx_lwm2m_client_session_state = NX_LWM2M_CLIENT_SESSION_REGISTERED;
        session_ptr -> nx_lwm2m_client_session_substate = NX_LWM2M_CLIENT_SESSION_SUBSTATE_WAIT;
        session_ptr -> nx_lwm2m_client_session_timer = now + MAX_UPDATE_DELAY;

        /* Wakeup client thread */
        tx_event_flags_set(&session_ptr -> nx_lwm2m_client_session_client_ptr -> nx_lwm2m_client_event_flags, NX_LWM2M_CLIENT_EVENT_WAKEUP, TX_OR);
    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_strconv_format_id                  PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function converts an object ID to a string.                    */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    string_ptr                            The destination buffer        */
/*    string_length                         The length of the buffer      */
/*    id                                    The object ID                 */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    The length of the returned string                                   */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    _nx_lwm2m_client_strconv_format_int32                               */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_corelink_path_add                                  */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_strconv_format_id(CHAR *string_ptr, UINT string_length, NX_LWM2M_ID id)
{

    /* Same format as integer */
    return(_nx_lwm2m_client_strconv_format_int32(string_ptr, string_length, id));
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_format_uint32                      PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function formats unsigned 32-bit integer.                      */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    string_ptr                            The destination buffer        */
/*    string_length                         The length of the buffer      */
/*    value                                 The integer value             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    The length of the returned string                                   */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_strconv_format_int32                               */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
static UINT _nx_lwm2m_client_format_uint32(CHAR *string_ptr, UINT string_length, ULONG value)
{
UINT length;
CHAR *ptr;

    /* Fast path for values 0-9 */
    if (value < 10)
    {
        if (string_length == 0)
        {
            return(0);
        }
        *string_ptr = (CHAR)('0' + value);
        return(1);
    }

    /* Compute length of string representation */
    if (value >= 1000000000UL)
    {
        length = 10;
    }
    else if (value >= 100000000UL)
    {
        length = 9;
    }
    else if (value >= 10000000UL)
    {
        length = 8;
    }
    else if (value >= 1000000UL)
    {
        length = 7;
    }
    else if (value >= 100000UL)
    {
        length = 6;
    }
    else if (value >= 10000UL)
    {
        length = 5;
    }
    else if (value >= 1000UL)
    {
        length = 4;
    }
    else if (value >= 100UL)
    {
        length = 3;
    }
    else
    {
        length = 2;
    }

    if (length > string_length)
    {

        /* Buffer too small */
        return(0);
    }

    /* Write digits in reverse order */
    ptr = string_ptr + length - 1;
    while (1)
    {
        *ptr-- = (CHAR)('0' + (value % 10));

        if (ptr < string_ptr)
        {
            return(length);
        }

        value = value / 10;
    }
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_strconv_format_int32               PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function converts a 32-bit integer to a string                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    string_ptr                            The destination buffer        */
/*    string_length                         The length of the buffer      */
/*    value                                 The integer value             */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    The length of the returned string                                   */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    tx_mutex_get                                                        */
/*    tx_mutex_put                                                        */
/*    _nx_lwm2m_client_access_control_delete                              */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_write_attribute                                    */
/*    _nx_lwm2m_client_session_send_request                               */
/*    _nx_lwm2m_client_strconv_format_notify_number                       */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_strconv_format_int32(CHAR *string_ptr, UINT string_length, NX_LWM2M_INT32 value)
{
UINT ret;

    if (value < 0)
    {

        if (string_length < 2)
        {

            /* Need at least 2 characters */
            return(0);
        }

        /* Write minus sign */
        *string_ptr++ = '-';

        /* Write absolute value */
        ret = _nx_lwm2m_client_format_uint32(string_ptr, string_length-1, (ULONG) (-value));

        /* Return total length or error */
        return(ret != 0 ? ret + 1 : 0);
    }

    /* Write positive value */
    return(_nx_lwm2m_client_format_uint32(string_ptr, string_length, (ULONG) value));
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_strconv_format_notify_number       PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function converts a notify number to a string.                 */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    string_ptr                            The destination buffer        */
/*    string_length                         The length of the buffer      */
/*    value                                 The number value              */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    The length of the returned string                                   */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    memcpy                                                              */
/*    sprintf                                                             */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_write_attribute                                    */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_strconv_format_notify_number(CHAR *string_ptr, UINT string_length, NX_LWM2M_CLIENT_NOTIFY_NUMBER value)
{
UINT    digit_num = 1;
INT     integer;
INT     decimal[NX_LWM2M_CLIENT_DOUBLE_DECIMAL_PLACE_DIGITS] = {0};
INT     index = 0;
INT     precision = NX_LWM2M_CLIENT_DOUBLE_DECIMAL_PLACE_DIGITS;
double  temp;
UINT    length = 0;
INT     exponent = 0;
INT     exp = 0;

    /* Check if it's positive or negative */
    if (value < 0)
    {
        if (string_length < 2)
        {
            return(0);
        }
        string_ptr[length++] = '-';
        value *= -1;
    }
    else if (string_length == 0)
    {
        return(0);
    }

    /* Check if using exponent */
    if (value >= 10000000000000000.)
    {
        exp = 1;
    }
    else if (value < 0.001 && value > 0)
    {
        exp = 2;
    }

    if (!exp)
    {

        /* Get integer number */
        temp = value;
        while (temp >= 10)
        {
            temp /= 10;
            digit_num++;
        }

        /* Check string length */
        if (string_length < (digit_num + length))
        {
            return(0);
        }

        /* Save the integer part to output buffer */
        if (digit_num > 1)
        {
            for (temp = 1, index = 1; index < (INT)digit_num; index++)
            {
                temp = temp * 10;
            }
            for (index = 0; index < (INT)digit_num; index++)
            {
                integer = (INT)(value / temp);
                string_ptr[length++] = (CHAR)(integer + '0');
                value -= integer * temp;
                temp /= 10;
            }
        }
        else
        {
            integer = (INT)value;
            string_ptr[length++] = (CHAR)(integer + '0');
            value -= integer;
        }
    }
    else
    {

        if (exp == 1)
        {
            while (value >= 10)
            {
                value /= 10;
                exponent++;
            }
        }
        else
        {
            while (value < 1)
            {
                value = value * 10;
                exponent++;
            }
        }

        string_ptr[length++] = (CHAR)((INT)value + '0');
        value -= (INT)value;
    }

    /* Save the decimal to 'decimal' array */
    for (index = 0; index < precision; index++)
    {
        decimal[index] = (INT)(value * 10);
        value = value * 10 - decimal[index];
    }

    for (index = precision - 1; index >= 0; index--)
    {
        if (decimal[index] == 0)
        {
            precision--;
        }
        else
        {
            break;
        }
    }

    if (precision)
    {

        /* Check string length */
        if (string_length < (length + (UINT)precision + 1))
        {
            return(0);
        }

        string_ptr[length++] = '.';
    }

    /* Save the decimal to output buffer */
    for (index = 0; index < precision; index++)
    {
        string_ptr[length++] = (CHAR)(decimal[index] + '0');
    }

    /* Save the exponent part */
    if (exp)
    {
        if (string_length < (length + 2))
        {
            return(0);
        }
        string_ptr[length++] = 'E';

        if (exp == 1)
        {
            string_ptr[length++] = '+';
        }
        else
        {
            string_ptr[length++] = '-';
        }

        digit_num = _nx_lwm2m_client_strconv_format_int32(string_ptr + length, string_length - length, exponent);
        if (digit_num == 0)
        {
            return(0);
        }
        length += digit_num;
    }

    return(length);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_strconv_parse_id                   PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function parses a string representing an object ID.            */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    string_ptr                            Pointer to the string         */
/*    string_length                         The length of the string      */
/*    id_ptr                                On return, the parsed value   */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_get_request_options                                */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_strconv_parse_id(const CHAR *string_ptr, UINT string_length, NX_LWM2M_ID *id_ptr)
{
const CHAR *string_ptr_max;
CHAR c;
ULONG id;

    /* At least one character */
    if (string_length == 0)
    {
        return(NX_LWM2M_CLIENT_BAD_ENCODING);
    }

    /* Get pointer to end of string */
    string_ptr_max = string_ptr + string_length;

    /* Get first character */
    c = *string_ptr++;

    /* Convert first character to digit */
    if (c < '0' || c > '9')
    {

        /* Invalid character */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);
    }
    id = (ULONG) (c - '0');

    /* Check for invalid leading zero */
    if (id == 0 && string_ptr != string_ptr_max)
    {
        return(NX_LWM2M_CLIENT_BAD_ENCODING);
    }

    /* Parse the rest of the string */
    while (string_ptr < string_ptr_max)
    {
        if (id > 6553)
        {

            /* Max range reached */
            return(NX_LWM2M_CLIENT_BAD_ENCODING);
        }

        /* Get next character */
        c = *string_ptr++;

        /* Convert to digit */
        if (c < '0' || c > '9')
        {

            /* Invalid character */
            return(NX_LWM2M_CLIENT_BAD_ENCODING);
        }
        id = id*10 + (ULONG) (c - '0');
    }

    /* ID must be in range 0-65534 (65535 is reserved) */
    if (id > 65534)
    {
        return(NX_LWM2M_CLIENT_BAD_ENCODING);
    }

    /* Return ID */
    *id_ptr = (NX_LWM2M_ID) id;

    /* Return Success */
    return(NX_SUCCESS);
}


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_strconv_parse_int32                PORTABLE C      */
/*                                                           6.1.3        */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Yuxin Zhou, Microsoft Corporation                                   */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */
/*    This function parses a string representing a 32-bit integer.        */
/*                                                                        */
/*  INPUT                                                                 */
/*                                                                        */
/*    string_ptr                            Pointer to the string         */
/*    string_length                         The length of the string      */
/*    value_ptr                             On return, the parsed value   */
/*                                                                        */
/*  OUTPUT                                                                */
/*                                                                        */
/*    status                                Completion status             */
/*                                                                        */
/*  CALLS                                                                 */
/*                                                                        */
/*    None                                                                */
/*                                                                        */
/*  CALLED BY                                                             */
/*                                                                        */
/*    _nx_lwm2m_client_get_request_options                                */
/*    _nx_lwm2m_client_strconv_parse_notify_number                        */
/*                                                                        */
/*  RELEASE HISTORY                                                       */
/*                                                                        */
/*    DATE              NAME                      DESCRIPTION             */
/*                                                                        */
/*  12-31-2020     Yuxin Zhou               Initial Version 6.1.3         */
/*                                                                        */
/**************************************************************************/
UINT _nx_lwm2m_client_strconv_parse_int32(const CHAR *string_ptr, UINT string_length, NX_LWM2M_INT32 *value_ptr)
{
const CHAR *string_ptr_max;
NX_LWM2M_BOOL negative;
CHAR c;
ULONG value;

    /* At least one character */
    if (string_length == 0)
    {
        return(NX_LWM2M_CLIENT_BAD_ENCODING);
    }

    /* Get pointer to end of string */
    string_ptr_max = string_ptr + string_length;

    /* Get first character */
    c = *string_ptr++;
    if (c == '-')
    {

        /* Minus sign character, must be followed by digit */
        if (string_ptr >= string_ptr_max)
        {
            return(NX_LWM2M_CLIENT_BAD_ENCODING);
        }

        negative = NX_TRUE;

        value = 0;
    }
    else if (c >= '0' && c <= '9')
    {
        negative = NX_FALSE;

        value = (ULONG)(c - '0');
    }
    else
    {

        /* Invalid character */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);
    }

    while (string_ptr < string_ptr_max)
    {

        /* Get next character */
        c = *string_ptr++;

        if (c < '0' || c > '9' || value > 214748364UL)
        {

            /* Invalid character or number too big */
            return(NX_LWM2M_CLIENT_BAD_ENCODING);
        }

        value *= 10;
        value += (ULONG)(c - '0');
    }

    /* Convert to 32-bit signed integer */
    if (negative)
    {
        if (value > 2147483648UL)
        {

            /* Absolute value too big */
            return(NX_LWM2M_CLIENT_BAD_ENCODING);
        }
        *value_ptr = (NX_LWM2M_INT32) - (LONG) value;

        return(NX_SUCCESS);
    }
    if (value > 2147483647UL)
    {

        /* Number too big */
        return(NX_LWM2M_CLIENT_BAD_ENCODING);
    }
    *value_ptr = (NX_LWM2M_INT32) value;

    return(NX_SUCCESS);
}


#define EXP_MAX             308


/**************************************************************************/
/*                                                                        */
/*  FUNCTION                                               RELEASE        */
/*                                                                        */
/*    _nx_lwm2m_client_strconv_parse_notify_number        PORTABLE C      */
/*                                                          