/**************************************************************************/
/*                                                                        */
/*       Copyright (c) Microsoft Corporation. All rights reserved.        */
/*                                                                        */
/*       This software is licensed under the Microsoft Software License   */
/*       Terms for Microsoft Azure RTOS. Full text of the license can be  */
/*       found in the LICENSE file at https://aka.ms/AzureRTOS_EULA       */
/*       and in the root directory of this software.                      */
/*                                                                        */
/**************************************************************************/


/**************************************************************************/
/**************************************************************************/
/**                                                                       */ 
/** USBX Component                                                        */ 
/**                                                                       */
/**   Utility                                                             */
/**                                                                       */
/**************************************************************************/
/**************************************************************************/


/* Include necessary system files.  */

#define UX_SOURCE_CODE

#include "ux_api.h"


/**************************************************************************/ 
/*                                                                        */ 
/*  FUNCTION                                               RELEASE        */ 
/*                                                                        */ 
/*    _ux_utility_thread_schedule_other                   PORTABLE C      */ 
/*                                                           6.1          */
/*  AUTHOR                                                                */
/*                                                                        */
/*    Chaoqiong Xiao, Microsoft Corporation                               */
/*                                                                        */
/*  DESCRIPTION                                                           */
/*                                                                        */ 
/*    This function force the scheduling of all other threads.            */ 
/*                                                                        */ 
/*  INPUT                                                                 */ 
/*                                                                        */ 
/*    caller_priority                        Priority to restore.          */ 
/*                                                                        */ 
/*  OUTPUT                                                                */ 
/*                                                                        */ 
/*    Completion Status                                                   */ 
/*                                                                        */ 
/*  CALLS                                                                 */ 
/*                                                                        */ 
/*    tx_thread_identify                    ThreadX identify              */ 
/*    tx_thread_priority_change             ThreadX priority change       */ 
/*    tx_thread_relinquish                  ThreadX relinquish            */ 
/*                                                                        */ 
/*  CALLED BY                                                             */ 
/*                                                                        */ 
/*    USBX Components                                                     */ 
/*                                                                        */ 
/*  RELEASE HISTORY                                                       */ 
/*                                                                        */ 
/*    DATE              NAME                      DESCRIPTION             */ 
/*                                                                        */ 
/*  05-19-2020     Chaoqiong Xiao           Initial Version 6.0           */
/*  09-30-2020     Chaoqiong Xiao           Modified comment(s),          */
/*                                            used UX prefix to refer to  */
/*                                            TX symbols instead of using */
/*                                            them directly,              */
/*                                            resulting in version 6.1    */
/*                                                                        */
/**************************************************************************/
UINT  _ux_utility_thread_schedule_other(UINT caller_priority)
{

UINT        status;
UINT        old_priority;
UX_THREAD   *my_thread;

    UX_PARAMETER_NOT_USED(caller_priority);

    /* Call TX to know my own tread.  */
    my_thread = tx_thread_identify();

    /* Call ThreadX to change thread priority .  */
    status =  tx_thread_priority_change(my_thread, _ux_system -> ux_system_thread_lowest_priority, &old_priority);
    
    /* Check for error.  */
    if (status == TX_SUCCESS)
    {
    
        /* Wait until all other threads passed into the scheduler. */
        _ux_utility_thread_relinquish();
    
        /* And now return the priority of the thread to normal.  */
        status =  tx_thread_priority_change(my_thread, old_priority, &old_priority);
        
    }

    /* Return completion status.  */
    return(status);
}

