/*
 * Copyright 2021 Embedded Artists AB
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */


#include "fsl_common.h"
#include "fsl_gpio.h"
#include "fsl_iomuxc.h"
#include "pin_mux.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

// 0xf8b0 = IOMUX_PULL_UP22K | IOMUX_ODE | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW
// 0x10b0 = IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW
// 0xB0A9 = IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(5) | IOMUX_SLEW_FAST
// 0xB0B9 = IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(7) | IOMUX_SLEW_FAST
// 0xB0E9 = IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST
// 0xB829 = IOMUX_PULL_UP100K | IOMUX_ODE | IOMUX_SPD_LOW | IOMUX_DSE(5) | IOMUX_SLEW_FAST
// 0x31   = IOMUX_PULL_NONE | IOMUX_SPD_LOW | IOMUX_DSE(6) | IOMUX_SLEW_FAST
// 0x17089= IOMUX_HYS | IOMUX_PULL_UP47K | IOMUX_SPD_FAST | IOMUX_DSE(1) | IOMUX_SLEW_FAST
// 0x1B0B0= IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW

/* GPIO Pad Ctrl except for SNVS pins */
#define IOMUX_HYS         (1<<16)        /* hysteresis enabled */
#define IOMUX_PULL_UP100K (0xb<<12)      /* 100K pull up + enable pull */
#define IOMUX_PULL_UP47K  (0x7<<12)      /* 47K pull up + enable pull */
#define IOMUX_PULL_UP22K  (0xf<<12)      /* 22K pull up + enable pull */
#define IOMUX_PULL_DOWN   (0x3<<12)      /* 100K pull down + enable pull */
#define IOMUX_KEEPER_DOWN (0x1<<12)      /* 100K pull down + Keeper enabled */
#define IOMUX_KEEPER_UP   (0x9<<12)      /* 100K pull up + Keeper enabled */
#define IOMUX_PULL_NONE   (0x0<<12)      /* pull disabled */
#define IOMUX_ODE         (1<<11)        /* open drain enabled */
#define IOMUX_SPD_LOW     (0<<6)         /* speed LOW (50MHz) */
#define IOMUX_SPD_MED     (1<<6)         /* speed MEDIUM (100MHz) */
#define IOMUX_SPD_FAST    (2<<6)         /* speed FAST (100MHz) */
#define IOMUX_SPD_MAX     (3<<6)         /* speed MAX (200MHz) */
#define IOMUX_DSE(__x)    (((__x)&7)<<3) /* drive strength */
#define IOMUX_SLEW_FAST   (1<<0)         /* fast slew rate */
#define IOMUX_SLEW_SLOW   (0<<0)         /* slow slew rate */

/* GPIO_SNVS */
#define SNVS_HYS         (1<<16)        /* hysteresis enabled */
#define SNVS_PULL_UP100K (0xb<<12)      /* 100K pull up + enable pull */
#define SNVS_PULL_UP47K  (0x7<<12)      /* 47K pull up + enable pull */
#define SNVS_PULL_UP22K  (0xf<<12)      /* 22K pull up + enable pull */
#define SNVS_PULL_DOWN   (0x3<<12)        /* 100K pull down + enable pull */
#define SNVS_KEEPER      (0x1<<12)        /* Keeper enabled */
#define SNVS_PULL_NONE   (0x0<<12)        /* pull disabled */
#define SNVS_ODE         (1<<11)        /* open drain enabled */
#define SNVS_SPD_MED     (2<<6)         /* speed MEDIUM (100MHz) */
#define SNVS_DSE(__x)    (((__x)&7)<<3) /* drive strength */
#define SNVS_SLEW_FAST   (1<<0)         /* fast slew rate */
#define SNVS_SLEW_SLOW   (0<<0)         /* slow slew rate */

#define MUST_FIX_THIS  (-1)

typedef enum {
    input, out_high, out_low,
} gpio_type_t;

/*******************************************************************************
 * Variables
 ******************************************************************************/

/* Guard to prevent multiple initialization of pins in cases where main()
 * calls multiple BOARD_Init* functions, like for example both
 * BOARD_InitPins() and BOARD_InitLpuartPins().
 */
static bool first_time_calling = true;

/*******************************************************************************
 * Code
 ******************************************************************************/

/* Convenience function so that both calls to SetPinMux and SetPinConfig can
 * be done on one line without repetition
 */
static void SETUP(uint32_t muxRegister,
                  uint32_t muxMode,
                  uint32_t inputRegister,
                  uint32_t inputDaisy,
                  uint32_t configRegister,
                  uint32_t inputOnfield,
                  uint32_t configValue) {

    *((volatile uint32_t*) muxRegister) =
            IOMUXC_SW_MUX_CTL_PAD_MUX_MODE(muxMode) | IOMUXC_SW_MUX_CTL_PAD_SION(inputOnfield);

    if (inputRegister != 0UL) {
        *((volatile uint32_t*) inputRegister) = inputDaisy;
    }
    if (configRegister != 0UL) {
        *((volatile uint32_t*) configRegister) = configValue;
    }
}

/* Same as SETUP() but also configures GPIO for pin
 */
static void SETUPG(uint32_t muxRegister,
                   uint32_t muxMode,
                   uint32_t inputRegister,
                   uint32_t inputDaisy,
                   uint32_t configRegister,
                   uint32_t inputOnfield,
                   uint32_t configValue,
                   GPIO_Type *port,
                   uint32_t pin,
                   gpio_type_t type) {

    *((volatile uint32_t*) muxRegister) =
            IOMUXC_SW_MUX_CTL_PAD_MUX_MODE(muxMode) | IOMUXC_SW_MUX_CTL_PAD_SION(inputOnfield);

    if (inputRegister != 0UL) {
        *((volatile uint32_t*) inputRegister) = inputDaisy;
    }
    if (configRegister != 0UL) {
        *((volatile uint32_t*) configRegister) = configValue;
    }
    gpio_pin_config_t cfg = {
        .direction     = (type == input) ? kGPIO_DigitalInput : kGPIO_DigitalOutput,
        .interruptMode = kGPIO_NoIntmode,
        .outputLogic   = (type == out_high) ? 1 : 0,
    };
    GPIO_PinInit(port, pin, &cfg);
}


/* FUNCTION ************************************************************************************************************
 *
 * Function Name : BOARD_InitPins_uCOM_Board
 * Description   : Initializes pins available on the iMXRT1176 uCOM Board
 *
 * END ****************************************************************************************************************/
static void BOARD_InitPins_uCOM_Board(void) {
    CLOCK_EnableClock(kCLOCK_Iomuxc);

    /* UART-A, Main console UART */
    SETUP(IOMUXC_GPIO_AD_B0_12_LPUART1_TX, 0, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
    SETUP(IOMUXC_GPIO_AD_B0_13_LPUART1_RX, 1, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);

    /* UART-B */
    SETUP(IOMUXC_GPIO_AD_B1_02_LPUART2_TX, 0, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
    SETUP(IOMUXC_GPIO_AD_B1_03_LPUART2_RX, 1, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);

    /* UART-C, Used by Microbus and M.2 Bluetooth */
    SETUP(IOMUXC_GPIO_AD_B1_06_LPUART3_TX,    0, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
    SETUP(IOMUXC_GPIO_AD_B1_07_LPUART3_RX,    1, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
    SETUP(IOMUXC_GPIO_AD_B1_04_LPUART3_CTS_B, 0, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
    SETUP(IOMUXC_GPIO_AD_B1_05_LPUART3_RTS_B, 1, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);

    /* WDOG */
    SETUPG(IOMUXC_GPIO_B1_13_GPIO2_IO29, 0, IOMUX_PULL_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW, GPIO2, 29, out_low);

    /* BOOT */
    SETUPG(IOMUXC_GPIO_AD_B0_04_GPIO1_IO04, 0, IOMUX_PULL_NONE | IOMUX_DSE(3) | IOMUX_SLEW_SLOW, GPIO1, 4, input);
    SETUPG(IOMUXC_GPIO_AD_B0_05_GPIO1_IO05, 0, IOMUX_PULL_NONE | IOMUX_DSE(3) | IOMUX_SLEW_SLOW, GPIO1, 5, input);

    /* I2C-A */
    SETUP(IOMUXC_GPIO_AD_B1_01_LPI2C1_SDA, 1, IOMUX_PULL_UP22K | IOMUX_ODE | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
    SETUP(IOMUXC_GPIO_AD_B1_00_LPI2C1_SCL, 1, IOMUX_PULL_UP22K | IOMUX_ODE | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);

    /* SDRAM */
    // Skip as it is done either by debug script or DCD. See BOARD_InitSemcPins() below.

    /* ENET (Onboard PHY)
     *
     * ENET_INT: GPIO1.0
     * ENET_RST: optionally available on expansion connector JB pin 37
     */
    SETUPG(IOMUXC_GPIO_AD_B0_00_GPIO1_IO00, 0, IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(5) | IOMUX_SLEW_FAST, GPIO1, 0, input);
    SETUP(IOMUXC_GPIO_B1_04_ENET_RX_DATA00, 0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B1_05_ENET_RX_DATA01, 0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B1_06_ENET_RX_EN,     0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B1_07_ENET_TX_DATA00, 0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B1_08_ENET_TX_DATA01, 0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B1_09_ENET_TX_EN,     0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B1_10_ENET_REF_CLK,   1, IOMUX_PULL_NONE | IOMUX_SPD_LOW | IOMUX_DSE(6) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B1_11_ENET_RX_ER,     0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_40_ENET_MDC,      0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_41_ENET_MDIO,     0, IOMUX_PULL_UP100K | IOMUX_ODE | IOMUX_SPD_LOW | IOMUX_DSE(5) | IOMUX_SLEW_FAST);

    /* SPI-A, Available on Microbus connector and expansion connector "A" */
    SETUP(IOMUXC_GPIO_SD_B1_06_LPSPI2_PCS0, 0U, IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(6) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_SD_B1_07_LPSPI2_SCK,  0U, IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(6) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_SD_B1_08_LPSPI2_SD0,  0U, IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(6) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_SD_B1_09_LPSPI2_SDI,  0U, IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(6) | IOMUX_SLEW_FAST);

    /* From Hello World */
    SETUP(IOMUXC_GPIO_AD_B0_10_ARM_TRACE_SWO, 0, IOMUX_KEEPER_UP | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_FAST);

    /* Setup muxing to allow use of GPIO1 and GPIO2 instead of GPIO6 and GPIO7 */
    IOMUXC_GPR->GPR26 = ((IOMUXC_GPR->GPR26
            & (~(IOMUXC_GPR_GPR26_GPIO_MUX1_GPIO_SEL_MASK))) /* Mask bits to zero which are setting */
            | IOMUXC_GPR_GPR26_GPIO_MUX1_GPIO_SEL(0x00U) /* GPIO1 and GPIO6 share same IO MUX function, GPIO_MUX1 selects one GPIO function: 0x00U */
    );
    IOMUXC_GPR->GPR27 = ((IOMUXC_GPR->GPR27
            & (~(IOMUXC_GPR_GPR27_GPIO_MUX2_GPIO_SEL_MASK))) /* Mask bits to zero which are setting */
            | IOMUXC_GPR_GPR27_GPIO_MUX2_GPIO_SEL(0x00U) /* GPIO2 and GPIO7 share same IO MUX function, GPIO_MUX2 selects one GPIO function: 0x00U */
    );
}

/* FUNCTION ************************************************************************************************************
 *
 * Function Name : BOARD_InitPins_uCOM_Carrier_Board
 * Description   : Initializes pins available on the uCOM Carrier Board
 *
 * END ****************************************************************************************************************/
static void BOARD_InitPins_uCOM_Carrier_Board(void) {
    if (first_time_calling) {
        first_time_calling = false;
        BOARD_InitPins_uCOM_Board();

        /* SW5/WAKEUP button */
        SETUPG(IOMUXC_SNVS_WAKEUP_GPIO5_IO00, 1, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(4) | IOMUX_SLEW_SLOW, GPIO5, 0, input);

        /* USER LED (turned off) */
        SETUPG(IOMUXC_GPIO_AD_B1_08_GPIO1_IO24, 0, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW, GPIO1, 24, out_low);

        /* I2C-B */
        SETUP(IOMUXC_GPIO_SD_B1_10_LPI2C2_SDA, 1, IOMUX_PULL_UP22K | IOMUX_ODE | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_SD_B1_11_LPI2C2_SCL, 1, IOMUX_PULL_UP22K | IOMUX_ODE | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);

        /* uSD Card
         * SD_CD:  GPIO2.28
         * SD_RST: GPIO5.2
         */
        SETUPG(IOMUXC_SNVS_PMIC_STBY_REQ_GPIO5_IO02, 0, SNVS_SPD_MED | SNVS_DSE(6) | SNVS_SLEW_SLOW, GPIO5, 2, out_high);
        SETUPG(IOMUXC_GPIO_B1_12_GPIO2_IO28,         0, IOMUX_HYS | IOMUX_PULL_UP47K | IOMUX_SPD_FAST | IOMUX_DSE(1) | IOMUX_SLEW_FAST, GPIO2, 28, input);
        SETUP(IOMUXC_GPIO_B1_14_USDHC1_VSELECT,      0, IOMUX_HYS | IOMUX_PULL_UP47K | IOMUX_SPD_FAST | IOMUX_DSE(4) | IOMUX_SLEW_FAST);
        SETUP(IOMUXC_GPIO_SD_B0_00_USDHC1_CMD,       0, IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
        SETUP(IOMUXC_GPIO_SD_B0_01_USDHC1_CLK,       0, IOMUX_PULL_NONE | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_FAST);
        SETUP(IOMUXC_GPIO_SD_B0_02_USDHC1_DATA0,     0, IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
        SETUP(IOMUXC_GPIO_SD_B0_03_USDHC1_DATA1,     0, IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
        SETUP(IOMUXC_GPIO_SD_B0_04_USDHC1_DATA2,     0, IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
        SETUP(IOMUXC_GPIO_SD_B0_05_USDHC1_DATA3,     0, IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(7) | IOMUX_SLEW_FAST);

        /* QSPI on uCOM Carrier Board */
        SETUP(IOMUXC_GPIO_AD_B1_10_FLEXSPIA_DATA03, 0, IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(6) | IOMUX_SLEW_FAST);
        SETUP(IOMUXC_GPIO_AD_B1_11_FLEXSPIA_DATA02, 0, IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(6) | IOMUX_SLEW_FAST);
        SETUP(IOMUXC_GPIO_AD_B1_12_FLEXSPIA_DATA01, 0, IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(6) | IOMUX_SLEW_FAST);
        SETUP(IOMUXC_GPIO_AD_B1_13_FLEXSPIA_DATA00, 0, IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(6) | IOMUX_SLEW_FAST);
        SETUP(IOMUXC_GPIO_AD_B1_14_FLEXSPIA_SCLK,   0, IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(6) | IOMUX_SLEW_FAST);
        SETUP(IOMUXC_GPIO_AD_B1_15_FLEXSPIA_SS0_B,  0, IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(6) | IOMUX_SLEW_FAST);

        /* USB1 */
        SETUP(IOMUXC_GPIO_AD_B0_01_USB_OTG1_ID,  1, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_AD_B0_02_USB_OTG1_PWR, 1, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_AD_B0_03_USB_OTG1_OC,  1, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);

        /* USB2 is connected to a USB hub on the uCOM Carrier board hence OTG2_OC and OTG2_PWR are not used. */

        /* FLEXCAN2 - No transceiver but the pins are available on expansion connector J15-37 and J15-38 */
        SETUP(IOMUXC_GPIO_AD_B0_14_FLEXCAN2_TX, 1, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_AD_B0_15_FLEXCAN3_RX, 1, IOMUX_KEEPER_DOWN | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);

        /*
         * The parallel RGB interface pins overlap with the interface to the
         * 100/10Mbit Ethernet-PHY Adapter. The RGB and the adapter both use
         * the "D" connector on the uCOM Carrier Board so only one can be
         * used at a time.
         *
         * CTP_IRQ: GPIO3.0
         * CTP_RST: GPIO3.1
         */
        SETUPG(IOMUXC_GPIO_SD_B1_00_GPIO3_IO00, 0, IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(5) | IOMUX_SLEW_FAST, GPIO3, 0, input);
        SETUPG(IOMUXC_GPIO_SD_B1_01_GPIO3_IO01, 0, IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(5) | IOMUX_SLEW_FAST, GPIO3, 1, out_high);
        SETUP(IOMUXC_GPIO_B0_00_LCD_CLK,    0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_01_LCD_ENABLE, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_02_LCD_HSYNC,  0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_03_LCD_VSYNC,  0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_04_LCD_DATA00, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_05_LCD_DATA01, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_06_LCD_DATA02, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_07_LCD_DATA03, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_08_LCD_DATA04, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_09_LCD_DATA05, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_10_LCD_DATA06, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_11_LCD_DATA07, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_12_LCD_DATA08, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_13_LCD_DATA09, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_14_LCD_DATA10, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B0_15_LCD_DATA11, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B1_00_LCD_DATA12, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B1_01_LCD_DATA13, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B1_02_LCD_DATA14, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);
        SETUP(IOMUXC_GPIO_B1_03_LCD_DATA15, 0U, IOMUX_HYS | IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_SLOW);

        /* Setup muxing to allow use of GPIO3 instead of GPIO8 */
        IOMUXC_GPR->GPR28 = ((IOMUXC_GPR->GPR28
                & (~(IOMUXC_GPR_GPR28_GPIO_MUX3_GPIO_SEL_MASK))) /* Mask bits to zero which are setting */
                | IOMUXC_GPR_GPR28_GPIO_MUX3_GPIO_SEL(0x00U) /* GPIO3 and GPIO8 share same IO MUX function, GPIO_MUX3 selects one GPIO function: 0x00U */
        );
    }
}

void BOARD_InitPins(void) {
    BOARD_InitPins_uCOM_Carrier_Board();
}

void BOARD_InitBootPins(void) {
    BOARD_InitPins_uCOM_Carrier_Board();
}

void BOARD_InitEnetPins(void) {
    BOARD_InitPins_uCOM_Carrier_Board();
}

void BOARD_InitEnet2Pins(void) {
    BOARD_InitPins_uCOM_Carrier_Board();

    /* ENET 2 (100/10Mbit Ethernet-PHY Adapter)
     *
     * The parallel RGB interface pins overlap with the interface to the
     * 100/10Mbit Ethernet-PHY Adapter. The RGB and the adapter both use
     * the "D" connector on the uCOM Carrier Board so only one can be
     * used at a time.
     *
     * ENET_INT: GPIO3.0   (initialized above when intializing the Parallel RGB interface)
     * ENET_RST: GPIO3.1   (initialized above when intializing the Parallel RGB interface)
     */
    //SETUPG(IOMUXC_GPIO_SD_B1_00_GPIO3_IO00, 0, IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(5) | IOMUX_SLEW_FAST, GPIO3, 0, input);
    //SETUPG(IOMUXC_GPIO_SD_B1_01_GPIO3_IO01, 0, IOMUX_PULL_UP100K | IOMUX_SPD_FAST | IOMUX_DSE(5) | IOMUX_SLEW_FAST, GPIO3, 1, out_high);
    SETUP(IOMUXC_GPIO_B1_01_ENET2_RDATA00,  0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B1_02_ENET2_RDATA01,  0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B1_03_ENET2_RX_EN,    0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B0_12_ENET2_TDATA00,  0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B0_13_ENET2_TDATA01,  0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B0_14_ENET2_TX_EN,    0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B0_15_ENET2_REF_CLK2, 1, IOMUX_PULL_NONE | IOMUX_SPD_LOW | IOMUX_DSE(6) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B1_00_ENET2_RX_ER,    0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B0_00_ENET2_MDC,      0, IOMUX_PULL_UP100K | IOMUX_SPD_MAX | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_B0_01_ENET2_MDIO,     0, IOMUX_PULL_UP100K | IOMUX_ODE | IOMUX_SPD_LOW | IOMUX_DSE(5) | IOMUX_SLEW_FAST);
}

void BOARD_InitCSIPins(void) {
    BOARD_InitPins_uCOM_Carrier_Board();
}

void BOARD_InitI2C1Pins(void) {
    BOARD_InitPins_uCOM_Carrier_Board();
}

void BOARD_I2C_ConfigurePins(void) {
    BOARD_InitPins_uCOM_Carrier_Board();
}

void BOARD_InitSemcPins(void) {
    BOARD_InitPins_uCOM_Carrier_Board();

    /*
     * The SEMC pins are normally setup by either the DCD or by the debugger's
     * initialization scripts. This function MUST NEVER be called from code
     * running in SDRAM as it will corrupt the data.
     */
    SETUP(IOMUXC_GPIO_EMC_00_SEMC_DATA00, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_01_SEMC_DATA01, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_02_SEMC_DATA02, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_03_SEMC_DATA03, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_04_SEMC_DATA04, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_05_SEMC_DATA05, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_06_SEMC_DATA06, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_07_SEMC_DATA07, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_08_SEMC_DM00,   0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_09_SEMC_ADDR00, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_10_SEMC_ADDR01, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_11_SEMC_ADDR02, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_12_SEMC_ADDR03, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_13_SEMC_ADDR04, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_14_SEMC_ADDR05, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_15_SEMC_ADDR06, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_16_SEMC_ADDR07, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_17_SEMC_ADDR08, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_18_SEMC_ADDR09, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_19_SEMC_ADDR11, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_20_SEMC_ADDR12, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_21_SEMC_BA0,    0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_22_SEMC_BA1,    0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_23_SEMC_ADDR10, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_24_SEMC_CAS,    0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_25_SEMC_RAS,    0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_26_SEMC_CLK,    0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_27_SEMC_CKE,    0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_28_SEMC_WE,     0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_29_SEMC_CS0,    0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_30_SEMC_DATA08, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_31_SEMC_DATA09, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_32_SEMC_DATA10, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_33_SEMC_DATA11, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_34_SEMC_DATA12, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_35_SEMC_DATA13, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_36_SEMC_DATA14, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_37_SEMC_DATA15, 0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_38_SEMC_DM01,   0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    SETUP(IOMUXC_GPIO_EMC_39_SEMC_DQS,    1U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST);
    //SETUP(IOMUXC_GPIO_EMC_40_SEMC_RDY,    0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST); // Used as MDC for ENET
    //SETUP(IOMUXC_GPIO_EMC_41_SEMC_CSX00,  0U, IOMUX_HYS | IOMUX_KEEPER_DOWN | IOMUX_SPD_MAX | IOMUX_DSE(7) | IOMUX_SLEW_FAST); // Used as MDIO for ENET
}

void BOARD_InitParallelRGBPins(void) {
    BOARD_InitPins_uCOM_Carrier_Board();
}

void BOARD_InitWiFiPins(void) {
    BOARD_InitPins_uCOM_Carrier_Board();

    /* Wi-Fi using a uSD-M2 Adapter (LBEE0ZZ1WE-TEMP) from Murata
     * inserted in uSD connector J34. The M.2 socket (J33) on the
     * uCOM Carrier Board is not used.
     *
     * USDHC1_VSELECT: GPIO2.30
     * WL_REG_ON:      GPIO3.2  (see readme file in project for how to connect this pin)
     * All other pins already configured as part of setting up the SDCard interface above
     *
     * USDHC1_VSELECT: NVCC_SD fixed at 1.8V (by setting this signal high)
     * This pin is used as an GPIO for Wi-Fi examples but must be switched to USDHC1_VSELECT
     * when using SDCARD examples, see BOARD_InitPins_uCOM_Carrier_Board() above.
     */
    SETUPG(IOMUXC_GPIO_B1_14_GPIO2_IO30, 0, IOMUX_PULL_NONE | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_FAST, GPIO2, 30, out_high);

    SETUPG(IOMUXC_GPIO_SD_B1_02_GPIO3_IO02, 0, IOMUX_PULL_NONE | IOMUX_SPD_FAST | IOMUX_DSE(6) | IOMUX_SLEW_FAST, GPIO3, 2, out_low);
}

/***********************************************************************************************************************
 * EOF
 **********************************************************************************************************************/
