/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2020 NXP
 * All rights reserved.
 *
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*******************************************************************************
 * Includes
 ******************************************************************************/
#include "lwip/opt.h"

#if LWIP_IPV4 && LWIP_RAW && LWIP_SOCKET

#include "ping.h"
#include "lwip/netifapi.h"
#include "lwip/tcpip.h"
#include "netif/ethernet.h"
#include "enet_ethernetif.h"

#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "mac_addr.h"
#include "fsl_phy.h"

#include "fsl_phyksz8081.h"
#include "fsl_enet_mdio.h"
#include "fsl_gpio.h"
#include "fsl_iomuxc.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
/* IP address configuration. */
#define configIP_ADDR0 192
#define configIP_ADDR1 168
#define configIP_ADDR2 0
#define configIP_ADDR3 102

/* Netmask configuration. */
#define configNET_MASK0 255
#define configNET_MASK1 255
#define configNET_MASK2 255
#define configNET_MASK3 0

/* Gateway address configuration. */
#define configGW_ADDR0 192
#define configGW_ADDR1 168
#define configGW_ADDR2 0
#define configGW_ADDR3 100

/* MAC address configuration. */
#define configMAC_ADDR                     \
    {                                      \
        0x02, 0x12, 0x13, 0x10, 0x15, 0x11 \
    }

//#define EXAMPLE_USE_100M_ENET_ADAPTER
#ifdef EXAMPLE_USE_100M_ENET_ADAPTER
    /* Address of PHY interface. */
    #define EXAMPLE_PHY_ADDRESS BOARD_ENET2_PHY_ADDRESS
    /*! @brief Network interface initialization function. */
    #define EXAMPLE_NETIF_INIT_FN ethernetif1_init
#else
    /* Address of PHY interface. */
    #define EXAMPLE_PHY_ADDRESS BOARD_ENET0_PHY_ADDRESS
    /*! @brief Network interface initialization function. */
    #define EXAMPLE_NETIF_INIT_FN ethernetif0_init
#endif

/* MDIO operations. */
#define EXAMPLE_MDIO_OPS enet_ops

/* PHY operations. */
#define EXAMPLE_PHY_OPS phyksz8081_ops

/* ENET clock frequency. */
#define EXAMPLE_CLOCK_FREQ CLOCK_GetFreq(kCLOCK_IpgClk)


/*! @brief Stack size of the temporary lwIP initialization thread. */
#define INIT_THREAD_STACKSIZE 1024

/*! @brief Priority of the temporary lwIP initialization thread. */
#define INIT_THREAD_PRIO DEFAULT_THREAD_PRIO

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

static mdio_handle_t mdioHandle = {.ops = &EXAMPLE_MDIO_OPS};
static phy_handle_t phyHandle   = {.phyAddr = EXAMPLE_PHY_ADDRESS, .mdioHandle = &mdioHandle, .ops = &EXAMPLE_PHY_OPS};

/*******************************************************************************
 * Code
 ******************************************************************************/
void BOARD_InitModuleClock(void)
{
    const clock_enet_pll_config_t config = {
#ifdef EXAMPLE_USE_100M_ENET_ADAPTER
        .enableClkOutput    = false,
        .enableClkOutput25M = true,
        .loopDivider1       = 1,
#else
        .enableClkOutput    = true,
        .enableClkOutput25M = false,
        .loopDivider        = 1,
#endif
    };
    CLOCK_InitEnetPll(&config);
}

void delay(void)
{
    volatile uint32_t i = 0;
    for (i = 0; i < 1000000; ++i)
    {
        __asm("NOP"); /* delay */
    }
}



/*!
 * @brief Initializes lwIP stack.
 */
static void stack_init(void *arg)
{
    static struct netif netif;
    ip4_addr_t netif_ipaddr, netif_netmask, netif_gw;
    ethernetif_config_t enet_config = {
        .phyHandle  = &phyHandle,
        .macAddress = configMAC_ADDR,
    };

    LWIP_UNUSED_ARG(arg);
    mdioHandle.resource.csrClock_Hz = EXAMPLE_CLOCK_FREQ;

    IP4_ADDR(&netif_ipaddr, configIP_ADDR0, configIP_ADDR1, configIP_ADDR2, configIP_ADDR3);
    IP4_ADDR(&netif_netmask, configNET_MASK0, configNET_MASK1, configNET_MASK2, configNET_MASK3);
    IP4_ADDR(&netif_gw, configGW_ADDR0, configGW_ADDR1, configGW_ADDR2, configGW_ADDR3);

    tcpip_init(NULL, NULL);

#ifdef EXAMPLE_USE_100M_ENET_ADAPTER
    MAC_Read(MAC_E2PROM_I2C_7BIT_ADDR_ADAPTER, enet_config.macAddress);
#else
    MAC_Read(MAC_E2PROM_I2C_7BIT_ADDR_ONBOARD, enet_config.macAddress);
#endif
    netifapi_netif_add(&netif, &netif_ipaddr, &netif_netmask, &netif_gw, &enet_config, EXAMPLE_NETIF_INIT_FN,
                       tcpip_input);
    netifapi_netif_set_default(&netif);
    netifapi_netif_set_up(&netif);

    PRINTF("\r\n************************************************\r\n");
    PRINTF(" PING example\r\n");
    PRINTF("************************************************\r\n");
    PRINTF(" IPv4 Address     : %u.%u.%u.%u\r\n", ((u8_t *)&netif_ipaddr)[0], ((u8_t *)&netif_ipaddr)[1],
           ((u8_t *)&netif_ipaddr)[2], ((u8_t *)&netif_ipaddr)[3]);
    PRINTF(" IPv4 Subnet mask : %u.%u.%u.%u\r\n", ((u8_t *)&netif_netmask)[0], ((u8_t *)&netif_netmask)[1],
           ((u8_t *)&netif_netmask)[2], ((u8_t *)&netif_netmask)[3]);
    PRINTF(" IPv4 Gateway     : %u.%u.%u.%u\r\n", ((u8_t *)&netif_gw)[0], ((u8_t *)&netif_gw)[1],
           ((u8_t *)&netif_gw)[2], ((u8_t *)&netif_gw)[3]);
    PRINTF("************************************************\r\n");

    ping_init(&netif_gw);

    vTaskDelete(NULL);
}

/*!
 * @brief Main function
 */
int main(void)
{
    gpio_pin_config_t gpio_config = {kGPIO_DigitalOutput, 0, kGPIO_NoIntmode};

    BOARD_ConfigMPU();
    BOARD_InitBootPins();
    BOARD_InitBootClocks();
    BOARD_InitDebugConsole();
    BOARD_InitModuleClock();

#ifdef EXAMPLE_USE_100M_ENET_ADAPTER
    BOARD_InitEnet2Pins();
    PRINTF("Using the 100/10Mbps Ethernet-PHY Adapter\r\n");
    IOMUXC_EnableMode(IOMUXC_GPR, kIOMUXC_GPR_ENET2TxClkOutputDir, true);

    GPIO_PinInit(GPIO3, 1, &gpio_config);
    GPIO_PinInit(GPIO3, 0, &gpio_config);
    /* pull up the ENET_INT before RESET. */
    GPIO_WritePinOutput(GPIO3, 0, 1);
    GPIO_WritePinOutput(GPIO3, 1, 0);
    delay();
    GPIO_WritePinOutput(GPIO3, 1, 1);
#else
    BOARD_InitEnetPins();
    PRINTF("Using the onboard 100/10Mbps Ethernet-PHY\r\n");
    IOMUXC_EnableMode(IOMUXC_GPR, kIOMUXC_GPR_ENET1TxClkOutputDir, true);

    /* The Ethernet-PHY is resetted at power on (POR) but there is no direct GPIO
       on the iMXRT1064 uCOM board directly connected to the Ethernet-PHY so it
       cannot be resetted here.
       If you want to control the reset sequence then:
       1) pick an unused GPIO that is also available on the uCOM Carrier Board
          e.g. IOMUXC_GPIO_AD_B1_08_GPIO1_IO24, which can be found on connector
          "A" J15 pin 40
       2) Use a jumper cable to connect J15 pin 40 (your GPIO pin) with J15 pin 66
          (the reset pin for the Ethernet-PHY)
       3) declare the pin in pin_mux.c
       4) uncomment and modify the code below
    */
    //GPIO_PinInit(GPIO1, 24, &gpio_config);
    //GPIO_WritePinOutput(GPIO1, 24, 0);
    //delay();
    //GPIO_WritePinOutput(GPIO1, 24, 1);
#endif

    mdioHandle.resource.csrClock_Hz = EXAMPLE_CLOCK_FREQ;

    /* Initialize lwIP from thread */
    if (sys_thread_new("main", stack_init, NULL, INIT_THREAD_STACKSIZE, INIT_THREAD_PRIO) == NULL)
        LWIP_ASSERT("main(): Task creation failed.", 0);

    vTaskStartScheduler();

    /* Will not get here unless a task calls vTaskEndScheduler ()*/
    return 0;
}
#endif
