"""
   Copyright 2017 The TensorFlow Authors. All Rights Reserved.
   Copyright 2019-2020 NXP
   
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================
"""

import pandas as pd
import numpy as np

import argparse

from sklearn.model_selection import train_test_split

from tensorflow import keras
from tensorflow.keras import layers
import tensorflow.lite as tflite

FLAGS = None

def preprocess_data(data):
    #data = (FLAGS.scale_range/3*(data - data.mean())/data.std()).clip(-1,1)
    data = data.diff()[5:]
    data[['ax','ay','az']] /= 10
    return data


def main(FLAGS):
    chanels = ['wx', 'wy', 'wz', 'ax', 'ay', 'az']
    patch_size = 5
   
    # load input data
    data = pd.read_csv(FLAGS.input_data,
                       delimiter = ',', dtype = np.int, header=0)
    data = preprocess_data(data)
    
    # prepare patches   
    samples_nb = int(np.floor(data.shape[0] / float(patch_size))) * patch_size
    patch = np.reshape(data[chanels][:samples_nb].values,
                       (-1,len(chanels)*patch_size))
                   
    # split into train and test set
    train_set, test_set = train_test_split(patch, test_size=0.1,
                                           random_state=42)

    if FLAGS.train_keras_model:
        # create autoencoder model
        input_dim = test_set.shape[1]   
        encoding_dim = 16 
    
        autoencoder = keras.Sequential(
                [
                    keras.Input(shape=(input_dim, )),
                    layers.Dense(encoding_dim, activation='tanh', 
                            activity_regularizer=keras.regularizers.l1(10e-7)),
                    layers.Dense(int(encoding_dim / 2), activation='tanh'),                
                    layers.Dense(int(encoding_dim / 2), activation='tanh'),
                    layers.Dense(input_dim, activation='tanh')
                ]
        )
        
        autoencoder.compile(optimizer='adamax',
                            loss='mean_squared_error', 
                            metrics=['categorical_accuracy'])
        
        # set training parameters
        nb_epoch = 300
        batch_size = 10
    
        # train model
        history = autoencoder.fit(train_set,train_set,
                                  epochs=nb_epoch,
                                  batch_size=batch_size,
                                  shuffle=True,
                                  validation_data=(test_set,test_set),
                                  verbose=1)
        threshold = 3 * history.history['val_loss'][-1]    
        
        # save keras model
        autoencoder.save(FLAGS.keras_model)    
    else:
      # Recreate the exact same model, including weights and optimizer.
      autoencoder = keras.models.load_model(FLAGS.keras_model)
      
      predictions = autoencoder.predict(test_set)
      error = np.mean(np.square(predictions - test_set), 1)
      threshold = 3 * np.mean(error)
      
      
    # convert keras model into tf lite model and store it
    converter = tflite.TFLiteConverter.from_keras_model(autoencoder)
    tflite_model = converter.convert()
    open(FLAGS.tflite_model, "wb").write(tflite_model)
    
    # convert to C source code                  
    with open(FLAGS.tflite_model, 'rb') as f_in:
        x = [hex(b) for b in f_in.read()]
        with open(FLAGS.c_model, 'w') as f_out:
            f_out.write(("// This file is generated by adt_train.py\n\n" +
                         "const char adt_model[] = {{{}}};\n" +
                         "unsigned int adt_model_len = {};")
                        .format(", ".join(x), len(x)))
            
    with open('parameters.h', 'w') as f:
            f.write('// This file is generated by adt_train.py\n\n')
            f.write('const int PATCH_SIZE = %d;\n'%(patch_size))
            f.write('const int  NUM_CHANELS = %d;\n'%(len(chanels)))
            f.write('const float THRESHOLD = %f;\n'%(threshold))
            
    if len(FLAGS.validation_data) > 0:
        data = pd.read_csv(FLAGS.validation_data,
                           delimiter = ',', dtype = np.int, header=0)
        
        if len(FLAGS.validation_data_c) > 0:
            c_source_contain = ("// This file is generated by adt_train.py\n\n" +
                                '#ifndef COMMANDS_H_\n' +            
                                '#define COMMANDS_H_\n\n'+
                                'const int16_t in_data[] = {{\n {}}};\n\n' +
                                '#endif /* COMMANDS_H_ */')
                                 
            c_content = np.array2string(data[chanels].values.flatten(),
                                        threshold = data[chanels].size,
                                        separator=',')[1:-1]
            with open(FLAGS.validation_data_c, 'w') as f:
                f.write(c_source_contain.format(c_content))
                                
        
        # normalize data
        data = preprocess_data(data)
        
        # prepare patches
        samples_nb = int(np.floor(data.shape[0] / float(patch_size))) * patch_size
        patch = np.reshape(data[chanels][:samples_nb].values,
                           (-1,len(chanels)*patch_size))
        
        # evaluate
        predictions = autoencoder.predict(patch)
        error = np.mean(np.square(predictions - patch), 1)
        
        for i, err in enumerate(error):
            print(i, err)            
        
            
        if (FLAGS.vizualize):
            import matplotlib.pyplot as plt
            
            x = np.arange(predictions.shape[0])
            x_dense = np.arange(0,predictions.shape[0],0.02)
            y = np.interp(x_dense, x, error)
            y_a = np.ma.masked_less(y, threshold)
            
            fig, ax = plt.subplots()
            ax.plot(x_dense, y, '-k')
            ax.plot(x_dense, y_a, '-r')
            ax.set_xticks(np.arange(0,201,10))
            ax.grid(which='major', axis='x', linestyle='--')
            
            

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument(
      '--input_data',
      type=str,
      default='input_data.txt',
      help='Location of input training data.')
    
    parser.add_argument(
      '--keras_model',
      type=str,
      default='keras_model.h5',
      help='Location where output keras model will be stored.')    
    
    parser.add_argument(
      '--train_keras_model',
      type=lambda x: True if x in ('1', 'yes', 'y') else False,
      default=True,
      help='If True than train a new model else use an existing model.')
    
    parser.add_argument(
      '--tflite_model',
      type=str,
      default='adt_model.tflite',
      help='Location where output tflite model will be stored.')
        
    parser.add_argument(
      '--c_model',
      type=str,
      default='adt_model.h',
      help='Location where model as c source code will be stored.')
      
    parser.add_argument(
      '--validation_data',
      type=str,
      default='',
      help='If specified, evaluate validation data.')
    
    parser.add_argument(
      '--validation_data_c',
      type=str,
      default='commands.h',
      help='If specified, store validation data as c variable.')
    
    parser.add_argument(
      '--vizualize',
      type=lambda x: True if x in ('1', 'yes', 'y') else False,
      default=False,
      help='Show evaluation data.')
      
    FLAGS, unparsed = parser.parse_known_args()
    main(FLAGS)
