/*
 * Copyright 2022 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_common.h"
#include "fsl_mu.h"
#include "tee_fault_common.h"
#include "pin_mux.h"
#include "board.h"
#include "fsl_debug_console.h"

#include "resource_config.h"
#include "fsl_soc_src.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define DEMO_MU                MUB
#define DEMO_INVALID_DATA_ADDR 0x20361000

/*******************************************************************************
 * Variables
 ******************************************************************************/
volatile uint8_t userOption = 0U;

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
void DEMO_SwitchToUntrustedDomain(void);

/*******************************************************************************
 * Code
 ******************************************************************************/
/*
 * For the dual core project, generally primary core starts first, initializes
 * the system, then starts the secondary core to run.
 * In the case that debugging dual-core at the same time (for example, using IAR+DAPLink),
 * the secondary core is started by debugger. Then the secondary core might
 * run when the primary core initialization not finished. The SRC->GPR is used
 * here to indicate whether secondary core could go. When started, the secondary core
 * should check and wait the flag in SRC->GPR, the primary core sets the
 * flag in SRC->GPR when its initialization work finished.
 */
#define DEMO_SECONDARY_CORE_GO_FLAG 0xa5a5a5a5u
#define DEMO_SECONDARY_CORE_SRC_GPR kSRC_GeneralPurposeRegister20

void BOARD_WaitAndClearSecondaryCoreGoFlag(void)
{
    while (DEMO_SECONDARY_CORE_GO_FLAG != SRC_GetGeneralPurposeRegister(SRC, DEMO_SECONDARY_CORE_SRC_GPR))
    {
    }

    SRC_SetGeneralPurposeRegister(SRC, DEMO_SECONDARY_CORE_SRC_GPR, 0x0);
}

/*
 * XRDC2 domain is decided by the MATCH and MASK. The MATCH for each master is:
 *
 * xrdc2_mdac_axi_m7  For READ access
 *                    M7MCM.pid[7:0] (MCM register)
 *                    apc_axi_m7.r_xid[5] (AXI bus ARPROT[2]: 1-instruction, 0-data)
 *                    apc_axi_m7.r_xid[4] (reserved)
 *                    apc_axi_m7.r_xid[3] (APC region hit: 1-hit, 0-no hit)
 *                    apc_axi_m7.r_xid[2:0] (AXI bus ARID[3:0])
 *
 *                    For WRITE access
 *                    M7MCM.pid[7:0] (MCM register)
 *                    apc_axi_m7.w_xid[5] (AXI bus AWPROT[2]: 1-instruction, 0-data)
 *                    apc_axi_m7.w_xid[4] (reserved)
 *                    apc_axi_m7.w_xid[3] (APC region hit: 1-hit, 0-no hit)
 *                    apc_axi_m7.w_xid[2] (reserved)
 *                    apc_axi_m7.w_xid[1:0] (AXI bus AWID[2:0])
 *
 * xrdc2_mdac_ahb_m7  Both READ and WRITE access
 *                    apc_ahb_m7.xid[5] (AHBP bus ~HPROT[0]: 1-instruction, 0-data)
 *                    apc_ahb_m7.xid[4] (reserved)
 *                    apc_ahb_m7.xid[3] (APC region hit: 1-hit, 0-no hit)
 *                    apc_ahb_m7.xid[2:0] (APC region hit number: 000-region 0 hit, 001-region 1 hit,111-region 7 hit)
 *
 * xrdc2_mdac_edma_m7 Both READ and WRITE access
 *                    edma.dma_channel_id[4:0]
 *                    edma.dma_hmaster[3:0] (depends on DCHMID enable bit in eDMA)
 *                    apc_edma_m7.xid[5] (AHB bus ~HPROT[0]: 1-instruction, 0-data)
 *                    apc_edma_m7.xid[4] (reserved)
 *                    apc_edma_m7.xid[3] (APC region hit: 1-hit, 0-no hit)
 *                    apc_edma_m7.xid[2:0] (APC region hit number: 000-region 0 hit, 001-region 1 hit,111-region 7 hit)
 *
 * xrdc2_mdac_caam   For READ access
 *                   caam.caam_mst_aricid[3:0] (generated by CAAM)
 *                   apc_caam.r_xid[5] (AXI bus ARPROT[2]: 1-instruction, 0-data)
 *                   apc_caam.r_xid[4] (reserved)
 *                   apc_caam.r_xid[3] (APC region hit: 1-hit, 0-no hit)
 *                   apc_caam.r_xid[2:0] (APC region hit number: 000-region 0 hit, 001-region 1 hit,111-region 7 hit)
 *
 *                   For WRITE access
 *                   caam.caam_mst_awicid[3:0] (generated by CAAM)
 *                   apc_caam.w_xid[5] (AXI bus AWPROT[2]: 1-instruction, 0-data)
 *                   apc_caam.w_xid[4] (reserved)
 *                   apc_caam.w_xid[3] (APC region hit: 1-hit, 0-no hit)
 *                   apc_caam.w_xid[2:0] (APC region hit number: 000-region 0 hit, 001-region 1 hit,111-region 7 hit)
 *
 * xrdc2_mdac_enet_1g_tx    apc_enet_1g_tx.r_xid[5:0]    apc_enet_1g_tx.w_xid[5:0]
 * xrdc2_mdac_enet_1g_rx    apc_enet_1g_rx.r_xid[5:0]    apc_enet_1g_rx.w_xid[5:0]
 * xrdc2_mdac_enet          apc_enet.xid[5:0]
 * xrdc2_mdac_enet_qos      apc_enet_qos.r_xid[5:0]      apc_enet_qos.w_xid[5:0]
 * xrdc2_mdac_usdhc1        apc_usdhc1.r_xid[5:0]        apc_usdhc1.w_xid[5:0]
 * xrdc2_mdac_usdhc2        apc_usdhc2.r_xid[5:0]        apc_usdhc2.w_xid[5:0]
 * xrdc2_mdac_usb           apc_usb.r_xid[5:0]           apc_usb.w_xid[5:0]
 * xrdc2_mdac_gc355         apc_gc355.r_xid[5:0]         apc_gc355.w_xid[5:0]
 * xrdc2_mdac_pxp           apc_pxp.r_xid[5:0]           apc_pxp.w_xid[5:0]
 * xrdc2_mdac_lcdif1        apc_lcdif1.r_xid[5:0]        apc_lcdif1.w_xid[5:0]
 * xrdc2_mdac_lcdif2        apc_lcdif2.r_xid[5:0]        apc_lcdif2.w_xid[5:0]
 * xrdc2_mdac_csi           apc_csi.r_xid[5:0]           apc_csi.w_xid[5:0]
 *
 * xrdc2_mdac_ahbc_m4 Both READ and WRITE access
 *                    cm4_imxrt.mcm_pid[7:0] (M4 MCM register)
 *                    apc_ahbc_m4.xid[5] (AHBC bus ~HPROT[0]: 1-instruction, 0-data)
 *                    apc_ahbc_m4.xid[4] (reserved)
 *                    apc_ahbc_m4.xid[3] (APC region hit: 1-hit, 0-no hit)
 *                    apc_ahbc_m4.xid[2:0] (APC region hit number: 000-region 0 hit, 001-region 1 hit,111-region 7 hit)
 *
 * xrdc2_mdac_ahbs_m4 Both READ and WRITE access
 *                    cm4_imxrt.mcm_pid[7:0] (M4 MCM register)
 *                    apc_ahbs_m4.xid[5] (AHBS bus ~HPROT[0]: 1-instruction, 0-data)
 *                    apc_ahbs_m4.xid[4] (reserved)
 *                    apc_ahbs_m4.xid[3] (APC region hit: 1-hit, 0-no hit)
 *                    apc_ahbs_m4.xid[2:0] (APC region hit number: 000-region 0 hit, 001-region 1 hit,111-region 7 hit)
 *
 * xrdc2_mdac_edma_m4 edma_lpsr.dma_channel_id[4:0]
 *                    edma_lpsr.dma_hmaster[3:0]
 *                    apc_edma_m4.xid[5:0]
 *
 * ============================================================================
 *
 * AXI bug ARID and AWID used for xrdc2_mdac_axi_m7:
 *
 * ARID[2:0]
 * 0b000         Normal Non-cacheable, Device and Strong-ordered reads.
 * 0b011, 0b010  Data cache line fill
 * 0b100         Instruction fetch
 *
 * AWID[1:0]
 * 0b00          Normal Non-cacheable writes, all shared exclusive writes,
 *               for example from STREX.
 * 0b01          Normal cacheable writes to Write-Through and Write-Back,
 *               Non Write-Allocate memory.
 * 0b10          Device and Strong-ordered writes.
 * 0b11          Evictions to Normal cacheable Write-Back memory.
 *
 * ============================================================================
 *
 * For CM4, in XRDC2 configuration, the M4MCM.pid determines the domain ID.
 * M4MCM.pid = 0 is Domain 0 (trusted domain).
 * M4MCM.pid = 1 is Domain 1 (untrusted domain).
 *
 * M4MCM.pid address is 0xE0080030.
 */
void DEMO_SwitchToUntrustedDomain(void)
{
    *(volatile uint32_t *)0xe0080030 = 1;
}

static void Fault_Handler(void)
{
    switch (userOption)
    {
        case DEMO_MU_MSG_INVALID_DATA_ACCESS:
            MU_SendMsg(DEMO_MU, DEMO_MU_CH, DEMO_MU_MSG_INVALID_DATA_ACCESS_DONE);
            break;

        case DEMO_MU_MSG_INVALID_PARAM:
            MU_SendMsg(DEMO_MU, DEMO_MU_CH, DEMO_MU_MSG_INVALID_PARAM_DONE);
            break;

        default:
            break;
    }

    while (1)
        ;
}

void HardFault_Handler(void)
{
    Fault_Handler();
}

void BusFault_Handler(void)
{
    Fault_Handler();
}

static void DEMO_InvalidDataAccess(void)
{
    /*
     * The DEMO_INVALID_DATA_ADDR is inaccessible for untrusted domain,
     * so the access results to hardfault.
     */
    (*(volatile uint32_t *)DEMO_INVALID_DATA_ADDR)++;

    /* Should never get here */
    while (1)
        ;
}

static void DEMO_InvalidParameters(void)
{
    /*
     * The DEMO_INVALID_DATA_ADDR is inaccessible for untrusted domain,
     * so the access results to hardfault.
     */
    memcpy((char *)DEMO_INVALID_DATA_ADDR, (char *)(DEMO_INVALID_DATA_ADDR + 4), 4);

    /* Should never get here */
    while (1)
        ;
}

int main(void)
{
    BOARD_WaitAndClearSecondaryCoreGoFlag();
    BOARD_InitBootTEE();
    BOARD_InitBootPins();

    MU_Init(DEMO_MU);

    /* Notify core 0 that ready for the demo. */
    MU_SendMsg(DEMO_MU, DEMO_MU_CH, DEMO_MU_MSG_CORE1_READY);

    /* Handle the command from core 0. */
    while (1)
    {
        userOption = MU_ReceiveMsg(DEMO_MU, DEMO_MU_CH);

        switch (userOption)
        {
            case DEMO_MU_MSG_INVALID_DATA_ACCESS:
                DEMO_SwitchToUntrustedDomain();
                DEMO_InvalidDataAccess();
                break;

            case DEMO_MU_MSG_INVALID_PARAM:
                DEMO_SwitchToUntrustedDomain();
                DEMO_InvalidParameters();
                break;

            default:
                break;
        }
    }
}
