/*
 * Copyright 2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*  Standard C Included Files */
#include <stdio.h>
#include <string.h>
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "fsl_debug_console.h"
#include "fsl_lpi2c.h"
#include "edid.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/* Get frequency of lpi2c clock */
#define LPI2C_CLOCK_FREQUENCY (CLOCK_GetFreq(kCLOCK_OscRc48MDiv2))

#define I2C_BAUDRATE   100000U
#define EDID_I2C       LPI2C1
#define MAIN_ADDR      0x3d
#define EDID_ADDR      0x3f

#define NUM_EDID_BYTES_TO_READ  256

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

static uint8_t edid_data[NUM_EDID_BYTES_TO_READ] = {0};
static struct edid decoded = {0};

/*******************************************************************************
 * Code
 ******************************************************************************/

static status_t read_regs(uint8_t addr_7bit, uint8_t reg, uint8_t* data, size_t len)
{
    lpi2c_master_transfer_t xfer = {0};
    status_t res;
    assert(data);

    xfer.slaveAddress = addr_7bit;
    xfer.direction = kLPI2C_Read;
    xfer.subaddress = reg;
    xfer.subaddressSize = 1;
    xfer.data = data;
    xfer.dataSize = len;
    xfer.flags = kLPI2C_TransferDefaultFlag;

    res = LPI2C_MasterTransferBlocking(EDID_I2C, &xfer);
    if (res != kStatus_Success) {
        PRINTF("Failed to read %u bytes from reg %u of %u\r\n", len, reg, addr_7bit);
    }
    return res;
}

static status_t read_reg(uint8_t addr_7bit, uint8_t reg, uint8_t* pVal)
{
    return read_regs(addr_7bit, reg, pVal, 1);
}

static status_t write_regs(uint8_t addr_7bit, uint8_t reg, uint8_t* data, size_t len)
{
    lpi2c_master_transfer_t xfer = {0};
    status_t res;
    assert(data);

    xfer.slaveAddress = addr_7bit;
    xfer.direction = kLPI2C_Write;
    xfer.subaddress = reg;
    xfer.subaddressSize = 1;
    xfer.data = data;
    xfer.dataSize = len;
    xfer.flags = kLPI2C_TransferDefaultFlag;

    res = LPI2C_MasterTransferBlocking(EDID_I2C, &xfer);
    if (res != kStatus_Success) {
        PRINTF("Failed to write %u bytes to reg %u of %u\r\n", len, reg, addr_7bit);
    }
    return res;
}

static status_t write_reg(uint8_t addr_7bit, uint8_t reg, uint8_t val)
{
    return write_regs(addr_7bit, reg, &val, 1);
}

static void dump_edid_data(void)
{
    char line[100];
    uint8_t* tmp;
    int used;
    PRINTF("EDID DATA:\r\n");
    PRINTF("---------\r\n");
    for (int offset = 0; offset < NUM_EDID_BYTES_TO_READ; offset += 16) {
        used = sprintf(line, "0x%2x   ", offset);
        tmp = &edid_data[offset];
        used = sprintf(line+used, "0x%2x 0x%2x 0x%2x 0x%2x ", tmp[0], tmp[1], tmp[2], tmp[3]);
        tmp += 4;
        used = sprintf(line+used, "0x%2x 0x%2x 0x%2x 0x%2x  ", tmp[0], tmp[1], tmp[2], tmp[3]);
        tmp += 4;
        used = sprintf(line+used, "0x%2x 0x%2x 0x%2x 0x%2x ", tmp[0], tmp[1], tmp[2], tmp[3]);
        tmp += 4;
        used = sprintf(line+used, "0x%2x 0x%2x 0x%2x 0x%2x\r\n", tmp[0], tmp[1], tmp[2], tmp[3]);
        PRINTF(line);
    }
}

#define ADV7511_REG_POWER                       0x41
#define ADV7511_REG_POWER2                      0xd6
#define ADV7511_POWER_POWER_DOWN                0x40
#define ADV7511_REG_POWER2_HPD_SRC_MASK         0xc0
#define ADV7511_REG_POWER2_HPD_SRC_NONE         0xc0

static void power_on_advxxxx(void)
{
    uint8_t tmp = 0;
    read_reg(MAIN_ADDR, ADV7511_REG_POWER, &tmp);
    tmp &= ~ADV7511_POWER_POWER_DOWN;
    tmp |= 0;
    write_reg(MAIN_ADDR, ADV7511_REG_POWER, tmp);

    read_reg(MAIN_ADDR, ADV7511_REG_POWER2, &tmp);
    tmp &= ~ADV7511_REG_POWER2_HPD_SRC_MASK;
    tmp |= ADV7511_REG_POWER2_HPD_SRC_NONE;
    write_reg(MAIN_ADDR, ADV7511_REG_POWER2, tmp);
}

static void process_edid(void)
{
    uint8_t chunk = 64;

    PRINTF("Reading %d bytes of EDID data\r\n", NUM_EDID_BYTES_TO_READ);

    for (int offset = 0; offset < NUM_EDID_BYTES_TO_READ; offset += chunk) {
        status_t res = read_regs(EDID_ADDR, offset, &edid_data[offset], chunk);
        if (res != kStatus_Success) {
            PRINTF("Failed to read 0x%x (%d) bytes at offset 0x%x (%u), error %d\r\n",
                   offset, offset, chunk, chunk);
            return;
        }
    }

    dump_edid_data();
    decode_edid(edid_data, NUM_EDID_BYTES_TO_READ, &decoded);
}


/*!
 * @brief Main function
 */
int main(void)
{
    BOARD_ConfigMPU();
    BOARD_InitMipiPanelPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();
    SystemCoreClockUpdate();

    PRINTF("\r\nEDID Reading example.\r\n");

    lpi2c_master_config_t masterConfig = {0};

    /*
     * masterConfig.debugEnable = false;
     * masterConfig.ignoreAck = false;
     * masterConfig.pinConfig = kLPI2C_2PinOpenDrain;
     * masterConfig.baudRate_Hz = 100000U;
     * masterConfig.busIdleTimeout_ns = 0;
     * masterConfig.pinLowTimeout_ns = 0;
     * masterConfig.sdaGlitchFilterWidth_ns = 0;
     * masterConfig.sclGlitchFilterWidth_ns = 0;
     */
    LPI2C_MasterGetDefaultConfig(&masterConfig);

    /* Change the default baudrate configuration */
    masterConfig.baudRate_Hz = I2C_BAUDRATE;

    LPI2C_MasterInit(EDID_I2C, &masterConfig, LPI2C_CLOCK_FREQUENCY);
    power_on_advxxxx();
    SDK_DelayAtLeastUs(1000*800, SystemCoreClock);
    process_edid();

    PRINTF("DONE\r\n");
    while (1)
    {
    }
}
