/*
 * Copyright 2021 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_debug_console.h"
#include "touch_support.h"
#include "display_support.h"
#include "board.h"

#if USE_FT5406_TOUCH()
  #include "fsl_ft5406_rt.h"
  #include "pca9530.h"
#elif USE_GT911_TOUCH()
  #include "fsl_gt911.h"
#endif

#include "tx_api.h"
#include "gx_api.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

#define TOUCH_STATE_TOUCHED     1
#define TOUCH_STATE_RELEASED    2
#define MIN_DRAG_DELTA          10

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

#if USE_GT911_TOUCH()
static void BOARD_PullMIPIPanelTouchResetPin(bool pullUp);
static void BOARD_ConfigMIPIPanelTouchIntPin(gt911_int_pin_mode_t mode);
#endif

/*******************************************************************************
 * Variables
 ******************************************************************************/

#if USE_FT5406_TOUCH()
static ft5406_rt_handle_t s_touchHandle;
#elif USE_GT911_TOUCH()
static gt911_handle_t s_touchHandle;
static const gt911_config_t s_touchConfig = {
    .I2C_SendFunc     = BOARD_MIPIPanelTouch_I2C_Send,
    .I2C_ReceiveFunc  = BOARD_MIPIPanelTouch_I2C_Receive,
    .pullResetPinFunc = BOARD_PullMIPIPanelTouchResetPin,
    .intPinFunc       = BOARD_ConfigMIPIPanelTouchIntPin,
    .timeDelayMsFunc  = VIDEO_DelayMs,
    .touchPointNum    = 1,
    .i2cAddrMode      = kGT911_I2cAddrMode0,
    .intTrigMode      = kGT911_IntRisingEdge,
};
static int s_touchResolutionX;
static int s_touchResolutionY;
#endif

#if IS_TOUCH_SUPPORTED()
static GX_VALUE last_pos_x = 0;
static GX_VALUE last_pos_y = 0;

TX_THREAD touch_thread;
UCHAR touch_thread_stack[2048];
#endif

/*******************************************************************************
 * Code
 ******************************************************************************/

#if USE_GT911_TOUCH()
static void BOARD_PullMIPIPanelTouchResetPin(bool pullUp)
{
    if (pullUp)
    {
        GPIO_PinWrite(BOARD_MIPI_PANEL_TOUCH_RST_GPIO, BOARD_MIPI_PANEL_TOUCH_RST_PIN, 1);
    }
    else
    {
        GPIO_PinWrite(BOARD_MIPI_PANEL_TOUCH_RST_GPIO, BOARD_MIPI_PANEL_TOUCH_RST_PIN, 0);
    }
}

static void BOARD_ConfigMIPIPanelTouchIntPin(gt911_int_pin_mode_t mode)
{
    if (mode == kGT911_IntPinInput)
    {
        BOARD_MIPI_PANEL_TOUCH_INT_GPIO->GDIR &= ~(1UL << BOARD_MIPI_PANEL_TOUCH_INT_PIN);
    }
    else
    {
        if (mode == kGT911_IntPinPullDown)
        {
            GPIO_PinWrite(BOARD_MIPI_PANEL_TOUCH_INT_GPIO, BOARD_MIPI_PANEL_TOUCH_INT_PIN, 0);
        }
        else
        {
            GPIO_PinWrite(BOARD_MIPI_PANEL_TOUCH_INT_GPIO, BOARD_MIPI_PANEL_TOUCH_INT_PIN, 1);
        }

        BOARD_MIPI_PANEL_TOUCH_INT_GPIO->GDIR |= (1UL << BOARD_MIPI_PANEL_TOUCH_INT_PIN);
    }
}
#endif /* USE_GT911_TOUCH() */

void BOARD_InitTouchPanel(void)
{
#if USE_FT5406_TOUCH()
    // Use PCA9530 to initialize the LPI2C5 bus
    PCA9530_Init();
    if (kStatus_Success != FT5406_RT_Init(&s_touchHandle, LPI2C5))
    {
        PRINTF("Touch IC initialization failed\r\n");
        assert(false);
    }
#elif USE_GT911_TOUCH()
    status_t status;

    const gpio_pin_config_t resetPinConfig = {
        .direction = kGPIO_DigitalOutput, .outputLogic = 0, .interruptMode = kGPIO_NoIntmode};
    GPIO_PinInit(BOARD_MIPI_PANEL_TOUCH_INT_GPIO, BOARD_MIPI_PANEL_TOUCH_INT_PIN, &resetPinConfig);
    GPIO_PinInit(BOARD_MIPI_PANEL_TOUCH_RST_GPIO, BOARD_MIPI_PANEL_TOUCH_RST_PIN, &resetPinConfig);

    // The touch interface on the RK055AHD091 relies on 3.3V being available
    // and if jumper J7 is in position 1-2 on the back of the display then
    // that 3.3V is internally generated from the 5V which is turned on
    // only if the POWER_EN signal is high.
    //
    // There are several solutions to this:
    // 1) Move the J7 jumper to position 2-3 or
    // 2) Initialize the display before initializing the touch interface or
    // 3) Set the POWER_EN signal high
    //
    // We choose option 3) and set the POWER_EN signal high here.
    PCA6416_SetPins(PCA_LCD_BL_PWR);

    status = GT911_Init(&s_touchHandle, &s_touchConfig);

    if (kStatus_Success != status)
    {
        PRINTF("Touch IC initialization failed\r\n");
        assert(false);
    }

    GT911_GetResolution(&s_touchHandle, &s_touchResolutionX, &s_touchResolutionY);
#else
    PRINTF("The chosen display does not have touch support\r\n");
#endif
}

#if IS_TOUCH_SUPPORTED()
static status_t DEMO_ReadTouch(void *handle, int *p_x, int *p_y)
{
#if USE_FT5406_TOUCH()
    status_t status;
    int touch_x = 0;
    int touch_y = 0;
    touch_event_t ev;

    status = FT5406_RT_GetSingleTouch((ft5406_rt_handle_t*)handle, &ev, &touch_x, &touch_y);
    if (status == kStatus_Success)
    {
        if ((ev == kTouch_Down) || (ev == kTouch_Contact)) {
            *p_x = touch_x;
            *p_y = touch_y;
            return kStatus_Fail;
        }
    }
    return kStatus_Fail;

#elif USE_GT911_TOUCH()

    status_t status;
    int touch_x = 0;
    int touch_y = 0;

    status = GT911_GetSingleTouch((gt911_handle_t*)handle, &touch_x, &touch_y);
    if (status == kStatus_Success)
    {
        *p_x = touch_x * DEMO_PANEL_WIDTH / s_touchResolutionX;
        *p_y = touch_y * DEMO_PANEL_HEIGHT / s_touchResolutionY;
        return kStatus_Success;
    }
    else
    {
        return kStatus_Fail;
    }
#endif
}

static VOID gx_send_pen_down_event(GX_VALUE x, GX_VALUE y)
{
    GX_EVENT event;

    event.gx_event_type                                  = GX_EVENT_PEN_DOWN;
    event.gx_event_payload.gx_event_pointdata.gx_point_x = x;
    event.gx_event_payload.gx_event_pointdata.gx_point_y = y;
    event.gx_event_sender                                = 0;
    event.gx_event_target                                = 0;
    event.gx_event_display_handle                        = 0;
    gx_system_event_send(&event);

    last_pos_x = x;
    last_pos_y = y;
}

static VOID gx_send_pen_up_event(VOID)
{
    GX_EVENT event;

    event.gx_event_type                                  = GX_EVENT_PEN_UP;
    event.gx_event_payload.gx_event_pointdata.gx_point_x = last_pos_x;
    event.gx_event_payload.gx_event_pointdata.gx_point_y = last_pos_y;
    event.gx_event_sender                                = 0;
    event.gx_event_target                                = 0;
    event.gx_event_display_handle                        = 0;
    gx_system_event_send(&event);
}

static VOID gx_send_pen_drag_event(GX_VALUE x, GX_VALUE y)
{
    GX_EVENT event;

    event.gx_event_type                                  = GX_EVENT_PEN_DRAG;
    event.gx_event_payload.gx_event_pointdata.gx_point_x = x;
    event.gx_event_payload.gx_event_pointdata.gx_point_y = y;
    event.gx_event_sender                                = 0;
    event.gx_event_target                                = 0;
    event.gx_event_display_handle                        = 0;
    gx_system_event_fold(&event);

    last_pos_x = x;
    last_pos_y = y;
}

static VOID touch_thread_entry(ULONG thread_input)
{
    static int touch_state = TOUCH_STATE_RELEASED;
    status_t status;
    int x, y;
    int x_delta, y_delta;

    (void)thread_input;

    while (1)
    {
        tx_thread_sleep(5);
        status = DEMO_ReadTouch(&s_touchHandle, &x, &y);
        if (status == kStatus_Success)
        {
            if (touch_state == TOUCH_STATE_RELEASED)
            {
                touch_state = TOUCH_STATE_TOUCHED;
                gx_send_pen_down_event(x, y);
            }
            else
            {
                x_delta = abs(x - last_pos_x);
                y_delta = abs(y - last_pos_y);
                if (x_delta > MIN_DRAG_DELTA || y_delta > MIN_DRAG_DELTA)
                {
                    gx_send_pen_drag_event(x, y);
                }
            }
        }
        else
        {
            if (touch_state == TOUCH_STATE_TOUCHED)
            {
                touch_state = TOUCH_STATE_RELEASED;
                gx_send_pen_up_event();
            }
        }
    }
}
#endif /* IS_TOUCH_SUPPORTED() */

void start_touch_thread(void)
{
#if IS_TOUCH_SUPPORTED()
    /* Create the touch driver thread.  */
    tx_thread_create(&touch_thread, "GUIX Touch Thread", touch_thread_entry, 0,
                     touch_thread_stack, sizeof(touch_thread_stack),
                     GX_SYSTEM_THREAD_PRIORITY - 1,GX_SYSTEM_THREAD_PRIORITY - 1,
                     TX_NO_TIME_SLICE, TX_AUTO_START);
#endif
}
