/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2021 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*******************************************************************************
 * Includes
 ******************************************************************************/

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#if defined(FSL_FEATURE_HAS_L1CACHE) || defined(__DCACHE_PRESENT)
#include "fsl_cache.h"
#endif

#include "fsl_caam.h"

#include <string.h>

/*******************************************************************************
 * Definitions
 ******************************************************************************/


/*******************************************************************************
 * Variables
 ******************************************************************************/

#if defined(FSL_FEATURE_HAS_L1CACHE) || defined(__DCACHE_PRESENT)
/* Note: Usually the output data are not cached, because CAAM driver cleans them before scheduling job */
/* and invalidate right after that, but in some cases the computing can take longer time and data could be written */
/* to memory by CAAM after invalidate happen, so it's recommended to invalidate CAAM output data again, */
/* right before their usage, as shown in AES example. Or move them to non-cache memory as shown in SHA example */
/*! @brief CAAM job ring interface 0 in system memory. */
AT_NONCACHEABLE_SECTION(static caam_job_ring_interface_t s_jrif0);
/*! @brief CAAM job ring interface 1 in system memory. */
AT_NONCACHEABLE_SECTION(static caam_job_ring_interface_t s_jrif1);
/*! @brief CAAM job ring interface 2 in system memory. */
AT_NONCACHEABLE_SECTION(static caam_job_ring_interface_t s_jrif2);
/*! @brief CAAM job ring interface 3 in system memory. */
AT_NONCACHEABLE_SECTION(static caam_job_ring_interface_t s_jrif3);
#else
/*! @brief CAAM job ring interface 0 in system memory. */
static caam_job_ring_interface_t s_jrif0;
/*! @brief CAAM job ring interface 1 in system memory. */
static caam_job_ring_interface_t s_jrif1;
/*! @brief CAAM job ring interface 2 in system memory. */
static caam_job_ring_interface_t s_jrif2;
/*! @brief CAAM job ring interface 3 in system memory. */
static caam_job_ring_interface_t s_jrif3;
#endif /* __DCACHE_PRESENT || FSL_FEATURE_HAS_L1CACHE */

/*! @brief 16 bytes key for CBC method: "ultrapassword123". */
static uint8_t s_CbcKey128[] = {0x75, 0x6c, 0x74, 0x72, 0x61, 0x70, 0x61, 0x73,
                                0x73, 0x77, 0x6f, 0x72, 0x64, 0x31, 0x32, 0x33};

/*! @brief 24 bytes key for CBC method: "UltraMegaSecretPassword1". */
static uint8_t s_CbcKey192[] = {0x55, 0x6c, 0x74, 0x72, 0x61, 0x4d, 0x65, 0x67, 0x61, 0x53, 0x65, 0x63,
                                0x72, 0x65, 0x74, 0x50, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64, 0x31};

/*! @brief 32 bytes key for CBC method: "Thispasswordisveryuncommonforher". */
static uint8_t s_CbcKey256[] = {0x54, 0x68, 0x69, 0x73, 0x70, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72,
                                0x64, 0x69, 0x73, 0x76, 0x65, 0x72, 0x79, 0x75, 0x6e, 0x63, 0x6f,
                                0x6d, 0x6d, 0x6f, 0x6e, 0x66, 0x6f, 0x72, 0x68, 0x65, 0x72};

/*!
 * @brief Plaintext for CBC method.
 * 16-byte multiple, last '\0' is not used.
 */
static uint8_t s_CbcPlain[] =
    "Be that word our sign of parting, bird or fiend! I shrieked upstarting"
    "Get thee back into the tempest and the Nights Plutonian shore!"
    "Leave no black plume as a token of that lie thy soul hath spoken!"
    "Leave my loneliness unbroken! quit the bust above my door!"
    "Take thy beak from out my heart, and take thy form from off my door!"
    "Quoth the raven, Nevermore.  ";

/*! @brief Decrypted plaintext from CBC method goes here. */
static uint8_t s_CbcPlainDecrypted[sizeof(s_CbcPlain) - 1U];

/*! @brief Encrypted ciphertext from CBC method goes here. */
static uint8_t s_CbcCipher[sizeof(s_CbcPlain) - 1U];

/*! @brief Expected ciphertext from CBC method using s_CbcKey128 key. */
static const uint8_t s_Cbc128CipherExpected[] = {
    0xeb, 0x69, 0xb5, 0xae, 0x7a, 0xbb, 0xb8, 0xee, 0x4d, 0xe5, 0x28, 0x97, 0xca, 0xab, 0x60, 0x65, 0x63, 0xf9, 0xe8,
    0x4c, 0x7f, 0xda, 0x0a, 0x02, 0x3a, 0x93, 0x16, 0x0d, 0x64, 0x56, 0x5a, 0x86, 0xf2, 0xe8, 0x5b, 0x38, 0x1d, 0x31,
    0xd7, 0x65, 0x7e, 0x8f, 0x8d, 0x53, 0xc5, 0xa6, 0x0c, 0x5d, 0xc5, 0x43, 0x98, 0x3b, 0x49, 0x3a, 0xce, 0x7d, 0xf9,
    0xb5, 0xf7, 0x95, 0x47, 0x89, 0xaf, 0xd8, 0x2f, 0xbd, 0xa4, 0xd8, 0x7f, 0xb9, 0x13, 0x3a, 0xcd, 0x17, 0xc8, 0xc4,
    0xb0, 0x5d, 0xe8, 0xf5, 0x19, 0x39, 0x6a, 0x14, 0x1b, 0x1b, 0x78, 0x5e, 0xe0, 0xd6, 0x67, 0x9a, 0x36, 0x17, 0x9c,
    0x7a, 0x88, 0x26, 0xfd, 0x8f, 0x3d, 0x82, 0xc9, 0xb1, 0x2a, 0x9c, 0xc0, 0xdd, 0xdb, 0x78, 0x61, 0x3b, 0x22, 0x5d,
    0x48, 0x3c, 0xab, 0x10, 0xd3, 0x5d, 0x0d, 0xa1, 0x25, 0x3e, 0x4d, 0xd6, 0x8e, 0xc4, 0x1b, 0x68, 0xbb, 0xa4, 0x2d,
    0x97, 0x2b, 0xd6, 0x23, 0xa0, 0xf2, 0x90, 0x8e, 0x07, 0x75, 0x44, 0xb3, 0xe2, 0x5a, 0x35, 0x38, 0x4c, 0x5d, 0x35,
    0xa9, 0x7c, 0xa3, 0xb6, 0x38, 0xe7, 0xf5, 0x20, 0xdc, 0x0e, 0x6c, 0x7c, 0x4b, 0x4f, 0x93, 0xc1, 0x81, 0x69, 0x02,
    0xb7, 0x66, 0x37, 0x24, 0x0d, 0xb8, 0x9a, 0xa8, 0xd4, 0x42, 0x75, 0x28, 0xe8, 0x33, 0x89, 0x1e, 0x60, 0x82, 0xe9,
    0xf6, 0x45, 0x72, 0x64, 0x65, 0xd2, 0xcd, 0x19, 0xd9, 0x5e, 0xa2, 0x59, 0x31, 0x82, 0x53, 0x20, 0x35, 0x13, 0x76,
    0x7f, 0xeb, 0xc3, 0xbe, 0xfa, 0x4a, 0x10, 0x83, 0x81, 0x0f, 0x24, 0x6d, 0xca, 0x53, 0x07, 0xb9, 0xe0, 0xb9, 0x5d,
    0x91, 0x2d, 0x90, 0x86, 0x5b, 0x9d, 0xaa, 0xcd, 0x28, 0xea, 0x11, 0xfb, 0x83, 0x39, 0x9c, 0xf5, 0x3b, 0xd9, 0xef,
    0x38, 0xc7, 0xa4, 0xad, 0x47, 0xf2, 0x2d, 0xd6, 0x6b, 0x26, 0x28, 0x59, 0xaa, 0x33, 0x01, 0x73, 0xc9, 0x46, 0x97,
    0xa3, 0xe5, 0x11, 0x71, 0x66, 0xef, 0x1f, 0x0b, 0xbc, 0xe7, 0x4f, 0x8c, 0x79, 0xe2, 0x39, 0x14, 0x85, 0xcd, 0xa9,
    0x59, 0xed, 0x78, 0x9d, 0x37, 0xf5, 0x46, 0xfc, 0xa9, 0x8a, 0x16, 0x0a, 0x76, 0x58, 0x6d, 0x59, 0x9e, 0x65, 0xbe,
    0x1b, 0xc2, 0x09, 0xa1, 0xf9, 0x40, 0xab, 0xdb, 0x2e, 0x11, 0x30, 0x29, 0x49, 0x75, 0xf7, 0x74, 0xe1, 0xf3, 0x78,
    0x97, 0x69, 0x2c, 0x6a, 0x0e, 0x0d, 0xbd, 0x72, 0x3d, 0x75, 0xd6, 0x0a, 0x8c, 0xc2, 0x92, 0xd9, 0xb6, 0x46, 0x91,
    0xa7, 0xe4, 0x74, 0x71, 0xf5, 0xb4, 0x21, 0x86, 0x18, 0xa8};

/*! @brief Expected ciphertext from CBC method using s_CbcKey192 key. */
static const uint8_t s_Cbc192CipherExpected[] = {
    0xb5, 0xb8, 0xe5, 0x87, 0x40, 0x71, 0xdf, 0x48, 0x17, 0xf1, 0xe0, 0xa4, 0x92, 0xf1, 0xcf, 0x78, 0xb4, 0xb3, 0x92,
    0x42, 0xd6, 0x3b, 0x23, 0x3c, 0xa7, 0x82, 0xcc, 0x6a, 0xa4, 0xf5, 0x52, 0x8e, 0xdf, 0x02, 0x14, 0x2d, 0x1d, 0xae,
    0x3e, 0x86, 0x87, 0x41, 0x8d, 0xe9, 0x5b, 0x12, 0x38, 0x24, 0x7e, 0x46, 0xa7, 0xb1, 0x5f, 0x8a, 0x8f, 0x69, 0xdc,
    0x56, 0x8f, 0x37, 0x80, 0x53, 0xff, 0x67, 0x67, 0x54, 0xa7, 0x79, 0x2b, 0x7b, 0x66, 0x21, 0x78, 0x80, 0x34, 0x02,
    0x18, 0xd7, 0xc0, 0xef, 0x05, 0xdb, 0x25, 0x4d, 0x42, 0x05, 0xbb, 0x69, 0x35, 0x63, 0xc1, 0x31, 0xe3, 0x47, 0xc2,
    0xde, 0x67, 0xfe, 0x9f, 0x60, 0xf6, 0x6c, 0xb5, 0x41, 0x5e, 0x25, 0xa6, 0xec, 0xfe, 0xb0, 0x3e, 0x87, 0x61, 0x8e,
    0x5c, 0x03, 0x8e, 0x8b, 0x20, 0x74, 0xcd, 0x49, 0xa8, 0x04, 0xb0, 0xca, 0x10, 0xaa, 0x27, 0x5d, 0xe7, 0xfe, 0x90,
    0x3e, 0x50, 0xe4, 0x3e, 0x94, 0x68, 0xd1, 0xcc, 0x54, 0x28, 0xba, 0x2d, 0x2a, 0x88, 0x0d, 0xfa, 0xb2, 0x0a, 0x15,
    0x8d, 0x0a, 0xdc, 0xbc, 0x16, 0xd8, 0xaf, 0x1d, 0xce, 0x9a, 0xfa, 0x90, 0x96, 0x62, 0xbd, 0x11, 0x62, 0x09, 0x80,
    0xfe, 0xbd, 0x6d, 0xca, 0xbc, 0x6a, 0x07, 0xf9, 0x5e, 0x63, 0xe2, 0x6d, 0xfe, 0x7d, 0x88, 0xa2, 0xb6, 0x8e, 0xaf,
    0x1a, 0x80, 0x62, 0x19, 0x4c, 0x68, 0xfc, 0x61, 0x18, 0x58, 0x33, 0x76, 0x20, 0x84, 0x5d, 0xd6, 0x49, 0x97, 0xb7,
    0x79, 0x83, 0xf0, 0x69, 0x2f, 0xce, 0x73, 0x86, 0x5a, 0x6f, 0xfa, 0x96, 0x66, 0x97, 0xf3, 0xa0, 0xb3, 0xed, 0x67,
    0x36, 0x64, 0x08, 0x28, 0x75, 0xb5, 0x58, 0x19, 0x85, 0x01, 0x28, 0x3e, 0xb1, 0x8e, 0x68, 0x4e, 0x9f, 0x95, 0x86,
    0xae, 0xe0, 0x6e, 0x60, 0xbe, 0xa0, 0xfc, 0x5e, 0x8b, 0x5e, 0xe8, 0x96, 0xe9, 0xfa, 0xcb, 0x3d, 0xce, 0x9d, 0x70,
    0xbe, 0xa2, 0x05, 0x52, 0xbb, 0xa2, 0x79, 0xc9, 0xac, 0xf5, 0x91, 0xa2, 0xe4, 0xda, 0xa4, 0x5f, 0x89, 0x75, 0x45,
    0x7b, 0x58, 0xe3, 0xdb, 0x0f, 0xef, 0xd6, 0xa7, 0x88, 0x9c, 0x0d, 0xf3, 0x5b, 0x49, 0xb1, 0x27, 0xe3, 0x81, 0x92,
    0x93, 0x91, 0xaf, 0x27, 0x6b, 0x5a, 0x2e, 0x1a, 0x0c, 0xb6, 0xc5, 0x50, 0xc4, 0xb3, 0xf8, 0xfd, 0x0a, 0xff, 0xc7,
    0x8c, 0x55, 0xde, 0xde, 0x6f, 0x7c, 0xb9, 0xaa, 0x8d, 0x18, 0x17, 0xc5, 0x55, 0x95, 0x59, 0xd0, 0x00, 0x53, 0x63,
    0xaf, 0xe9, 0xf9, 0xde, 0x93, 0xe2, 0xa6, 0x90, 0xe5, 0xa9};

/*! @brief Expected ciphertext from CBC method using s_CbcKey256 key. */
static const uint8_t s_Cbc256CipherExpected[] = {
    0x09, 0x9b, 0xf5, 0xb3, 0xaf, 0x11, 0xa9, 0xd1, 0xa1, 0x81, 0x78, 0x6c, 0x6e, 0x74, 0xf3, 0xb8, 0x70, 0xee, 0x31,
    0x4d, 0x6d, 0x54, 0xab, 0x37, 0xcb, 0xeb, 0x58, 0x6f, 0x09, 0x5f, 0x72, 0xc4, 0x5a, 0xd0, 0x56, 0xc8, 0x3d, 0x93,
    0x45, 0xe2, 0x7e, 0x97, 0xaa, 0xc3, 0xc9, 0xf5, 0xde, 0x74, 0x73, 0x45, 0x35, 0xea, 0x1f, 0x5e, 0x81, 0xbf, 0x9d,
    0xb5, 0xc9, 0x77, 0x77, 0x1c, 0x00, 0xde, 0x67, 0x34, 0xff, 0x62, 0x48, 0x89, 0xd9, 0xbe, 0x92, 0xd4, 0x7e, 0xaf,
    0x9d, 0x8a, 0x65, 0x14, 0x1f, 0x62, 0xaa, 0x0a, 0xe4, 0x37, 0x8e, 0x18, 0x3c, 0x75, 0x5e, 0x38, 0x6a, 0xa9, 0x5d,
    0x26, 0x54, 0x0a, 0xd8, 0xeb, 0x7a, 0x25, 0xa6, 0xd4, 0x18, 0x13, 0x1f, 0x30, 0xfc, 0x37, 0x09, 0x77, 0x90, 0x26,
    0x88, 0x0e, 0x53, 0x67, 0xba, 0xe2, 0xfa, 0x38, 0xb9, 0x74, 0xa9, 0x5b, 0xda, 0x6a, 0xe0, 0xb3, 0x39, 0xed, 0x07,
    0xae, 0xe6, 0x86, 0x44, 0x2d, 0xf2, 0xd8, 0x1f, 0x86, 0x2c, 0xac, 0x01, 0x4c, 0x9b, 0xce, 0x65, 0x6a, 0x8a, 0x3a,
    0xf0, 0xf9, 0xfd, 0x15, 0x65, 0xb6, 0xaf, 0xdc, 0x90, 0xc5, 0x47, 0x96, 0x28, 0xb0, 0x1c, 0x56, 0x2e, 0xc6, 0xdd,
    0x4e, 0x71, 0xd3, 0x73, 0xf5, 0x7c, 0xa6, 0x66, 0x8b, 0x44, 0xaf, 0x53, 0x61, 0x16, 0xe3, 0x41, 0x94, 0xe7, 0x6d,
    0x3d, 0xdb, 0xe1, 0x92, 0x52, 0x39, 0x05, 0x97, 0xf4, 0x41, 0xc8, 0xbe, 0x54, 0xec, 0x9a, 0x52, 0xf1, 0x79, 0x0c,
    0x71, 0x05, 0x14, 0xc8, 0x16, 0x86, 0xdb, 0xa3, 0x8e, 0x1c, 0x41, 0x5b, 0x7a, 0x3b, 0x77, 0xa9, 0x27, 0x7a, 0xde,
    0xcd, 0xaa, 0x86, 0x2e, 0x52, 0x87, 0x54, 0x1c, 0x88, 0x4c, 0xdb, 0x3e, 0xab, 0x48, 0xaa, 0x51, 0x5a, 0xcd, 0xb0,
    0xe7, 0x68, 0x91, 0x33, 0x9e, 0xfd, 0x07, 0x9d, 0xdf, 0x18, 0x51, 0xa5, 0xc0, 0xa6, 0x68, 0xbc, 0xd2, 0x6b, 0x1f,
    0x03, 0xfc, 0xf3, 0x71, 0xed, 0x5b, 0x28, 0x35, 0xa8, 0x56, 0x93, 0x4c, 0xdc, 0x1f, 0xa1, 0x88, 0xe8, 0xbe, 0x08,
    0x48, 0xe8, 0x28, 0x1d, 0x16, 0xb4, 0x1e, 0xeb, 0xca, 0xdd, 0x43, 0x18, 0xfe, 0x49, 0x24, 0xfd, 0x23, 0x83, 0x44,
    0x2b, 0xc3, 0x33, 0x80, 0x62, 0xb9, 0xa6, 0xb8, 0x48, 0x1e, 0x72, 0x52, 0xef, 0xee, 0x56, 0xd8, 0x05, 0x08, 0xad,
    0xc2, 0xe9, 0xb7, 0x46, 0x12, 0xbc, 0xc8, 0x7d, 0xe2, 0x87, 0x9d, 0x57, 0xf7, 0x6f, 0x10, 0x6e, 0x8c, 0x32, 0x8e,
    0x2f, 0x78, 0x20, 0xf0, 0x23, 0x29, 0x54, 0xef, 0x66, 0x8d};

/*! @brief Initialization vector for CBC method: 16 bytes: "mysecretpassword". */
static uint8_t s_CbcIv[CAAM_AES_BLOCK_SIZE] = {0x6d, 0x79, 0x73, 0x65, 0x63, 0x72, 0x65, 0x74,
                                               0x70, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64};

/*! @brief 16 bytes key for GCM method. */
static uint8_t s_GcmKey[16] = {0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
                               0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08};

/*! @brief Plaintext for GCM method. */
static uint8_t s_GcmPlain[] = {0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5, 0xa5, 0x59, 0x09, 0xc5,
                               0xaf, 0xf5, 0x26, 0x9a, 0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
                               0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72, 0x1c, 0x3c, 0x0c, 0x95,
                               0x95, 0x68, 0x09, 0x53, 0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
                               0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57, 0xba, 0x63, 0x7b, 0x39};

/*! @brief Decrypted plaintext from GCM method goes here. */
static uint8_t s_GcmPlainDecrypted[sizeof(s_GcmPlain)];

/*! @brief Expected ciphertext from GCM method. */
static const uint8_t s_GcmCipherExpected[] = {0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24, 0x4b, 0x72, 0x21, 0xb7,
                                              0x84, 0xd0, 0xd4, 0x9c, 0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
                                              0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e, 0x21, 0xd5, 0x14, 0xb2,
                                              0x54, 0x66, 0x93, 0x1c, 0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
                                              0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97, 0x3d, 0x58, 0xe0, 0x91};

/*! @brief Encrypted ciphertext from GCM method goes here. */
static uint8_t s_GcmCipher[sizeof(s_GcmCipherExpected)];

/*! @brief Initialization vector for GCM method. */
static uint8_t s_GcmIv[12] = {0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad, 0xde, 0xca, 0xf8, 0x88};

/*! @brief Additional authenticated data for GCM method. */
static uint8_t s_GcmAad[] = {0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef, 0xfe, 0xed,
                             0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef, 0xab, 0xad, 0xda, 0xd2};

/*! @brief Expected tag from GCM method. */
static const uint8_t s_GcmTagExpected[] = {0x5b, 0xc9, 0x4f, 0xbc, 0x32, 0x21, 0xa5, 0xdb,
                                           0x94, 0xfa, 0xe9, 0x5a, 0xe7, 0x12, 0x1a, 0x47};

/*! @brief Encrypted tag from GCM method goes here. */
static uint8_t s_GcmTag[sizeof(s_GcmTagExpected)];

/*! @brief Plaintext for SHA. */
static uint8_t s_ShaPlain[] =
    "Be that word our sign of parting, bird or fiend! I shrieked upstarting"
    "Get thee back into the tempest and the Nights Plutonian shore!"
    "Leave no black plume as a token of that lie thy soul hath spoken!"
    "Leave my loneliness unbroken! quit the bust above my door!"
    "Take thy beak from out my heart, and take thy form from off my door!"
    "Quoth the raven, Nevermore.  ";

/*! @brief Expected SHA-256 for the message s_ShaPlain. */
static const unsigned char s_ShaExpected[] = {0x63, 0x76, 0xea, 0xcc, 0xc9, 0xa2, 0xc0, 0x43, 0xf4, 0xfb, 0x01,
                                              0x34, 0x69, 0xb3, 0x0c, 0xf5, 0x28, 0x63, 0x5c, 0xfa, 0xa5, 0x65,
                                              0x60, 0xef, 0x59, 0x7b, 0xd9, 0x1c, 0xac, 0xaa, 0x31, 0xf7};

/*! @brief Output buffer for SHA. */
uint8_t AT_NONCACHEABLE_SECTION(sha_output[32U]);
/*******************************************************************************
 * Prototypes
 ******************************************************************************/

static void RunAesCbcExamples(CAAM_Type *base, caam_handle_t *handle);

static void EncryptDecryptCbc(
    CAAM_Type *base, caam_handle_t *handle, const uint8_t *key, size_t keySize, const uint8_t *cipherExpected);

static void RunAesGcmExamples(CAAM_Type *base, caam_handle_t *handle);

static void EncryptDecryptGcm(CAAM_Type *base, caam_handle_t *handle);

static void EncryptDecryptDesCbc(CAAM_Type *base, caam_handle_t *handle);

static void EncryptDecryptDES3Cbc(CAAM_Type *base, caam_handle_t *handle);

/*******************************************************************************
 * Code
 ******************************************************************************/

/*!
 * @brief Executes examples for AES encryption and decryption in CBC mode.
 */
static void RunAesCbcExamples(CAAM_Type *base, caam_handle_t *handle)
{
    EncryptDecryptCbc(base, handle, s_CbcKey128, sizeof(s_CbcKey128), s_Cbc128CipherExpected);
    EncryptDecryptCbc(base, handle, s_CbcKey192, sizeof(s_CbcKey192), s_Cbc192CipherExpected);
    EncryptDecryptCbc(base, handle, s_CbcKey256, sizeof(s_CbcKey256), s_Cbc256CipherExpected);
}

/*!
 * @brief Encrypts and decrypts in AES CBC mode.
 *
 * @param key encryption key
 * @param keySize size of key in bytes
 * @param cipherExpected expected output of encryption
 */
static void EncryptDecryptCbc(
    CAAM_Type *base, caam_handle_t *handle, const uint8_t *key, size_t keySize, const uint8_t *cipherExpected)
{
    status_t status;

    PRINTF("AES CBC: encrypting using %u bit key ", 8U * keySize);

    status = CAAM_AES_EncryptCbc(base, handle, s_CbcPlain, s_CbcCipher, sizeof(s_CbcCipher), s_CbcIv, key, keySize);
    if (status != kStatus_Success)
    {
        PRINTF("- failed to encrypt!\r\n\r\n");
        return;
    }

    /* Executed only if DCACHE present, sizeof(s_CbcCipher) is multiple of 32 */
    DCACHE_InvalidateByRange((uint32_t)s_CbcCipher, sizeof(s_CbcCipher));
    if (memcmp(s_CbcCipher, cipherExpected, sizeof(s_CbcCipher)) == 0)
    {
        PRINTF("done successfully.\r\n");
    }
    else
    {
        PRINTF("- encrypted text mismatch!\r\n\r\n");
        return;
    }

    PRINTF("AES CBC: decrypting back ");

    status =
        CAAM_AES_DecryptCbc(base, handle, s_CbcCipher, s_CbcPlainDecrypted, sizeof(s_CbcCipher), s_CbcIv, key, keySize);
    if (status != kStatus_Success)
    {
        PRINTF("- failed to decrypt!\r\n\r\n");
        return;
    }

    /* Executed only if DCACHE present, sizeof(s_CbcPlainDecrypted) is multiple of 32 */
    DCACHE_InvalidateByRange((uint32_t)s_CbcPlainDecrypted, sizeof(s_CbcPlainDecrypted));
    if (memcmp(s_CbcPlainDecrypted, s_CbcPlain, sizeof(s_CbcPlainDecrypted)) == 0)
    {
        PRINTF("done successfully.\r\n\r\n");
    }
    else
    {
        PRINTF("- decrypted text mismatch!\r\n\r\n");
    }
}

/*!
 * @brief Executes examples for AES encryption and decryption in GCM mode.
 */
static void RunAesGcmExamples(CAAM_Type *base, caam_handle_t *handle)
{
    EncryptDecryptGcm(base, handle);
}

/*!
 * @brief Encrypts and decrypts in AES GCM mode.
 */
static void EncryptDecryptGcm(CAAM_Type *base, caam_handle_t *handle)
{
    status_t status;

    PRINTF("AES GCM: encrypt ");

    status = CAAM_AES_EncryptTagGcm(base, handle, s_GcmPlain, s_GcmCipher, sizeof(s_GcmPlain), s_GcmIv, sizeof(s_GcmIv),
                                    s_GcmAad, sizeof(s_GcmAad), s_GcmKey, sizeof(s_GcmKey), s_GcmTag, sizeof(s_GcmTag));
    if (status != kStatus_Success)
    {
        PRINTF("- failed to encrypt!\r\n\r\n");
        return;
    }

    if (memcmp(s_GcmTag, s_GcmTagExpected, sizeof(s_GcmTagExpected)) != 0)
    {
        PRINTF("- tag mismatch!\r\n\r\n");
        return;
    }

    if (memcmp(s_GcmCipher, s_GcmCipherExpected, sizeof(s_GcmCipherExpected)) != 0)
    {
        PRINTF("- encrypted text mismatch!\r\n\r\n");
        return;
    }

    PRINTF("done successfully.\r\n");

    PRINTF("AES GCM: decrypt ");

    status = CAAM_AES_DecryptTagGcm(base, handle, s_GcmCipher, s_GcmPlainDecrypted, sizeof(s_GcmCipher), s_GcmIv,
                                    sizeof(s_GcmIv), s_GcmAad, sizeof(s_GcmAad), s_GcmKey, sizeof(s_GcmKey), s_GcmTag,
                                    sizeof(s_GcmTag));
    if (status != kStatus_Success)
    {
        PRINTF("- failed to decrypt!\r\n\r\n");
        return;
    }

    if (memcmp(s_GcmPlainDecrypted, s_GcmPlain, sizeof(s_GcmPlain)) != 0)
    {
        PRINTF("- decrypted text mismatch!\r\n\r\n");
        return;
    }

    PRINTF("done successfully.\r\n\r\n");
}

/*!
 * @brief Executes examples for DES and DES3 encryption and decryption in CBC mode.
 */
static void RunDesExamples(CAAM_Type *base, caam_handle_t *handle)
{
    EncryptDecryptDesCbc(base, handle);
    EncryptDecryptDES3Cbc(base, handle);
}

/*!
 * @brief Encrypts and decrypts in DES CBC mode.
 */
static void EncryptDecryptDesCbc(CAAM_Type *base, caam_handle_t *handle)
{
    status_t status = kStatus_Fail;

    uint8_t DesKey[8] = {0x70, 0x61, 0x73, 0x73, 0x77, 0x6F, 0x72, 0x64};
    /*initialization vector: 8 bytes: "mysecret"*/
    uint8_t DesIve[8] = {0x6d, 0x79, 0x73, 0x65, 0x63, 0x72, 0x65, 0x74};

    static const uint8_t des_test_short[]    = "Once upon a midn";
    static const uint8_t des_encrypted_cbc[] = {0xfa, 0xfe, 0xcd, 0xd1, 0x13, 0x6e, 0x68, 0xf2};
    uint8_t cipher[8]                        = {0};
    uint8_t plaintext[8]                     = {0};

    PRINTF("DES CBC: encrypting ");

    status = CAAM_DES_EncryptCbc(base, handle, des_test_short, cipher, sizeof(cipher), DesIve, DesKey);
    if (status != kStatus_Success)
    {
        PRINTF("- failed to encrypt!\r\n\r\n");
        return;
    }

    if (memcmp(cipher, des_encrypted_cbc, sizeof(des_encrypted_cbc)) == 0)
    {
        PRINTF("done successfully.\r\n");
    }
    else
    {
        PRINTF("- encrypted text mismatch!\r\n\r\n");
        return;
    }

    PRINTF("DES CBC: decrypting back ");

    status = CAAM_DES_DecryptCbc(base, handle, cipher, plaintext, sizeof(plaintext), DesIve, DesKey);
    if (status != kStatus_Success)
    {
        PRINTF("- failed to decrypt!\r\n\r\n");
        return;
    }

    if (memcmp(plaintext, des_test_short, sizeof(plaintext)) == 0)
    {
        PRINTF("done successfully.\r\n\r\n");
    }
    else
    {
        PRINTF("- decrypted text mismatch!\r\n\r\n");
    }
}

/*!
 * @brief Encrypts and decrypts in 3DES CBC mode.
 */
static void EncryptDecryptDES3Cbc(CAAM_Type *base, caam_handle_t *handle)
{
    status_t status = kStatus_Fail;

    /*24 bytes key: "verynicepassword12345678"*/
    uint8_t key1[8] = {0x76, 0x65, 0x72, 0x79, 0x6e, 0x69, 0x63, 0x65};
    uint8_t key2[8] = {0x70, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64};
    uint8_t key3[8] = {0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38};
    /*initialization vector: 8 bytes: "mysecret"*/
    uint8_t DesIve[8] = {0x6d, 0x79, 0x73, 0x65, 0x63, 0x72, 0x65, 0x74};

    /*8-byte multiple*/
    static const uint8_t des3_test[] =
        "And the raven, never flitting, still is sitting, still is sitting"
        "On the pallid bust of Pallas just above my chamber door;"
        "And his eyes have all the seeming of a demons that is dreaming,"
        "And the lamp-light oer him streaming throws his shadow on the floor;"
        "And my soul from out that shadow that lies floating on the floor"
        "Shall be lifted - nevermore!";

    static const uint8_t des3_encrypted_cbc[] = {
        0xb9, 0x30, 0x3f, 0xda, 0x4d, 0x82, 0xc5, 0xe7, 0x27, 0x6b, 0x91, 0xf2, 0x84, 0x15, 0xf9, 0x74, 0xd0, 0xe5,
        0x5f, 0x4b, 0x95, 0xe1, 0x35, 0xac, 0x5d, 0xbe, 0xa8, 0x83, 0x6d, 0xd5, 0x71, 0x0d, 0xde, 0x86, 0x71, 0x61,
        0x58, 0x23, 0xe8, 0x4a, 0x3c, 0x99, 0x96, 0x98, 0x00, 0x1a, 0xe7, 0x42, 0xd4, 0x72, 0xf2, 0xf9, 0xc5, 0x50,
        0xa9, 0x7f, 0xe7, 0xab, 0xee, 0x5a, 0x3b, 0x5b, 0x43, 0x67, 0x59, 0x30, 0xec, 0x37, 0xe5, 0x2f, 0xc6, 0x5c,
        0x45, 0x02, 0x55, 0xe6, 0x75, 0x11, 0x7c, 0x9d, 0x05, 0xa9, 0xe2, 0x97, 0xc8, 0xe3, 0x6a, 0x65, 0x76, 0xce,
        0xc3, 0x56, 0xea, 0xd4, 0x30, 0x0d, 0xe4, 0x4c, 0x9a, 0xee, 0x9c, 0x5e, 0x48, 0xa3, 0x85, 0x85, 0x5f, 0x59,
        0x3b, 0x34, 0xee, 0x32, 0x54, 0xd9, 0x91, 0x9f, 0x97, 0x7c, 0xce, 0x1d, 0x9e, 0xe8, 0xb8, 0x86, 0xbc, 0x3a,
        0x1e, 0x52, 0xe1, 0x35, 0xe3, 0x92, 0xa2, 0xa3, 0xa6, 0xbd, 0x3f, 0x66, 0x76, 0xf3, 0x47, 0x37, 0x61, 0xb8,
        0x12, 0x35, 0xa1, 0x7e, 0xb3, 0xd1, 0x79, 0xfd, 0xb3, 0x69, 0xbc, 0x68, 0x4c, 0xd2, 0x24, 0xdc, 0x04, 0xae,
        0x82, 0x0e, 0x54, 0x4b, 0xcf, 0xa8, 0x40, 0xde, 0x72, 0x4c, 0x23, 0xfc, 0xc9, 0x86, 0x78, 0xc3, 0x50, 0xf0,
        0x1d, 0x62, 0x61, 0x7d, 0xbb, 0xc9, 0x83, 0x6f, 0x21, 0x3f, 0x6f, 0x2a, 0xb1, 0x7e, 0x9f, 0x24, 0xee, 0xe7,
        0x5b, 0x1d, 0xca, 0xb5, 0xd0, 0x31, 0xdc, 0xc5, 0x95, 0xcf, 0x8e, 0x5c, 0xf0, 0xbf, 0xb2, 0x15, 0xfb, 0x68,
        0x65, 0xc7, 0x5d, 0x77, 0x17, 0xc2, 0xf2, 0xd7, 0x19, 0x29, 0xcb, 0x49, 0x62, 0xf3, 0xa1, 0xd0, 0x18, 0xc1,
        0xed, 0x77, 0x9f, 0x4d, 0x28, 0xce, 0x66, 0x62, 0x7d, 0x56, 0xe7, 0x11, 0x5f, 0xf1, 0x7a, 0xca, 0xf4, 0x2d,
        0xeb, 0x6d, 0xca, 0x47, 0xf3, 0x39, 0xfc, 0x2e, 0x81, 0x73, 0x70, 0xe1, 0x4c, 0x37, 0xf8, 0xda, 0x91, 0x79,
        0x7c, 0x4b, 0xa0, 0xb4, 0x0e, 0x1e, 0xcf, 0x7b, 0xb1, 0xba, 0x9c, 0xe8, 0x8b, 0xcf, 0xe0, 0x7b, 0x07, 0x13,
        0x17, 0x0d, 0x55, 0xbf, 0x3d, 0x24, 0x6e, 0xfe, 0x61, 0x68, 0x41, 0xaf, 0x23, 0xf4, 0xd6, 0xd7, 0xf3, 0xab,
        0x78, 0x9c, 0xbf, 0xd3, 0xde, 0x37, 0xe1, 0x0c, 0x02, 0x88, 0xa5, 0xd4, 0x43, 0xf9, 0x08, 0x2d, 0x53, 0xce,
        0x2e, 0x43, 0xdc, 0x25, 0x26, 0xe1, 0x39, 0x17, 0x60, 0x10, 0x2d, 0x07, 0xa5, 0x9e, 0x1c, 0xa6, 0x90, 0x46,
        0x68, 0x14};

    uint32_t length        = sizeof(des3_test) - 1U;
    uint8_t cipher[344]    = {0};
    uint8_t plaintext[344] = {0};

    PRINTF("DES3 CBC: encrypting ");

    status = CAAM_DES3_EncryptCbc(base, handle, des3_test, cipher, length, DesIve, key1, key2, key3);
    if (status != kStatus_Success)
    {
        PRINTF("- failed to encrypt!\r\n\r\n");
        return;
    }

    if (memcmp(cipher, des3_encrypted_cbc, sizeof(des3_encrypted_cbc)) == 0)
    {
        PRINTF("done successfully.\r\n");
    }
    else
    {
        PRINTF("- encrypted text mismatch!\r\n\r\n");
        return;
    }

    PRINTF("DES3 CBC: decrypting back ");

    status = CAAM_DES3_DecryptCbc(base, handle, cipher, plaintext, sizeof(plaintext), DesIve, key1, key2, key3);
    if (status != kStatus_Success)
    {
        PRINTF("- failed to decrypt!\r\n\r\n");
        return;
    }

    if (memcmp(plaintext, des3_test, sizeof(plaintext)) == 0)
    {
        PRINTF("done successfully.\r\n\r\n");
    }
    else
    {
        PRINTF("- decrypted text mismatch!\r\n\r\n");
    }
}
/*!
 * @brief Executes example for SHA.
 */
static void RunShaExamples(CAAM_Type *base, caam_handle_t *handle)
{
    status_t status;
    caam_hash_ctx_t ctx;

    PRINTF("SHA:");

    status = CAAM_HASH_Init(base, handle, &ctx, kCAAM_Sha256, NULL, 0u);

    if (status != kStatus_Success)
    {
        PRINTF("- failed to Hash init!\r\n\r\n");
        return;
    }

    int length = sizeof(s_ShaPlain) - 1;

    status = CAAM_HASH_Update(&ctx, s_ShaPlain, length);

    if (status != kStatus_Success)
    {
        PRINTF("- failed to Hash update!\r\n\r\n");
        return;
    }

    status = CAAM_HASH_Finish(&ctx, sha_output, NULL);

    if (status != kStatus_Success)
    {
        PRINTF("- failed to Hash finish!\r\n\r\n");
        return;
    }

    if (memcmp(sha_output, s_ShaExpected, 32u) != 0)
    {
        PRINTF("- failed: unexpected Hash output!\r\n\r\n");
        return;
    }

    PRINTF("done successfully.\r\n\r\n");
}

/*!
 * @brief Executes example for RNG.
 */
static void RunRngExample(CAAM_Type *base, caam_handle_t *handle)
{
#define RNG_EXAMPLE_RANDOM_NUMBERS     (4U)
#define RNG_EXAMPLE_RANDOM_BYTES       (16U)
#define RNG_EXAMPLE_RANDOM_NUMBER_BITS (RNG_EXAMPLE_RANDOM_NUMBERS * 8U * sizeof(uint32_t))

    status_t status = kStatus_Fail;
    uint32_t number;
    uint32_t data[RNG_EXAMPLE_RANDOM_NUMBERS] = {0};

    PRINTF("RNG : ");

    PRINTF("Generate %u-bit random number: ", RNG_EXAMPLE_RANDOM_NUMBER_BITS);
    status = CAAM_RNG_GetRandomData(base, handle, kCAAM_RngStateHandle0, &data, RNG_EXAMPLE_RANDOM_BYTES,
                                    kCAAM_RngDataAny, NULL);

    if (status != kStatus_Success)
    {
        PRINTF("- failed to get random data !\r\n\r\n");
        return;
    }

    /* Print data */
    PRINTF("0x");
    for (number = 0; number < RNG_EXAMPLE_RANDOM_NUMBERS; number++)
    {
        PRINTF("%08X", data[number]);
    }
    PRINTF("\r\n");

    PRINTF("RNG : Random number generated successfully.\r\n\r\n");
}

/*!
 * @brief Main function.
 */
int main(void)
{
    CAAM_Type *base = CAAM;
    caam_handle_t caamHandle;
    caam_config_t caamConfig;

    /* Init hardware */
    BOARD_ConfigMPU();
    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();

    /* Get default configuration. */
    CAAM_GetDefaultConfig(&caamConfig);

    /* setup memory for job ring interfaces. Can be in system memory or CAAM's secure memory.
     * Although this driver example only uses job ring interface 0, example setup for job ring interface 1 is also
     * shown.
     */
    caamConfig.jobRingInterface[0] = &s_jrif0;
    caamConfig.jobRingInterface[1] = &s_jrif1;
    caamConfig.jobRingInterface[2] = &s_jrif2;
    caamConfig.jobRingInterface[3] = &s_jrif3;

    /* Init CAAM driver, including CAAM's internal RNG */
    if (CAAM_Init(base, &caamConfig) != kStatus_Success)
    {
        /* Make sure that RNG is not already instantiated (reset otherwise) */
        PRINTF("- failed to init CAAM&RNG!\r\n\r\n");
    }

    PRINTF("CAAM AES Peripheral Driver Example\r\n\r\n");

    /* in this driver example, requests for CAAM jobs use job ring 0 */
    PRINTF("*CAAM Job Ring 0* :\r\n\r\n");
    caamHandle.jobRing = kCAAM_JobRing0;

    /* Example of SHA */
    RunShaExamples(base, &caamHandle);

    /* Example of AES CBC */
    RunAesCbcExamples(base, &caamHandle);

    /* in this driver example, requests for CAAM jobs use job ring 1 */
    PRINTF("*CAAM Job Ring 1* :\r\n\r\n");
    caamHandle.jobRing = kCAAM_JobRing1;

    /* Example of AES GCM */
    RunAesGcmExamples(base, &caamHandle);

    /* in this driver example, requests for CAAM jobs use job ring 2 */
    PRINTF("*CAAM Job Ring 2* :\r\n\r\n");
    caamHandle.jobRing = kCAAM_JobRing2;

    /* Example of AES CBC */
    RunAesCbcExamples(base, &caamHandle);

    /* in this driver example, requests for CAAM jobs use job ring 3 */
    PRINTF("*CAAM Job Ring 3* :\r\n\r\n");
    caamHandle.jobRing = kCAAM_JobRing3;

    /* Example of AES GCM */
    RunAesGcmExamples(base, &caamHandle);

    /* Example of DES and DES3 */
    RunDesExamples(base, &caamHandle);

    /* Example of RNG */
    RunRngExample(base, &caamHandle);

    /* Deinit RNG */
    caamHandle.jobRing = kCAAM_JobRing0;
    if (CAAM_RNG_Deinit(CAAM, &caamHandle, kCAAM_RngStateHandle0) != kStatus_Success)
    {
        PRINTF("- failed to deinit RNG!\r\n\r\n");
    }

    while (1)
    {
    }
}
