/*
 * Copyright 2021 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "board.h"
#include "fsl_gpio.h"
#include "fsl_mipi_dsi.h"
#include "fsl_rm68191.h"
#include "fsl_rm68200.h"
#include "fsl_hx8394.h"
#include "fsl_adv7535.h"
#include "lcdifv2_support.h"
#include "pca6416.h"
#include "pca9530.h"


/*******************************************************************************
 * Definitions
 ******************************************************************************/

/* Definitions for MIPI. */
#define DEMO_MIPI_DSI          (&g_mipiDsi)
#define DEMO_MIPI_DSI_LANE_NUM 2

/*
 * The DPHY bit clock must be fast enough to send out the pixels, it should be
 * larger than:
 *
 *         (Pixel clock * bit per output pixel) / number of MIPI data lane
 *
 * Here the desired DPHY bit clock multiplied by ( 9 / 8 = 1.125) to ensure
 * it is fast enough.
 */
#define DEMO_MIPI_DPHY_BIT_CLK_ENLARGE(origin) (((origin) / 8) * 9)

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

static void PANEL_PullResetPin(bool pullUp);
static void PANEL_PullPowerPin(bool pullUp);
static status_t PANEL_DSI_Transfer(dsi_transfer_t *xfer);
static void BOARD_InitMipiDsiClock(void);

/*******************************************************************************
 * Variables
 ******************************************************************************/

uint32_t mipiDsiTxEscClkFreq_Hz;
uint32_t mipiDsiDphyBitClkFreq_Hz;
uint32_t mipiDsiDphyRefClkFreq_Hz;
uint32_t mipiDsiDpiClkFreq_Hz;

const MIPI_DSI_Type g_mipiDsi = {
    .host = DSI_HOST,
    .apb  = DSI_HOST_APB_PKT_IF,
    .dpi  = DSI_HOST_DPI_INTFC,
    .dphy = DSI_HOST_DPHY_INTFC,
};

AT_NONCACHEABLE_SECTION_ALIGN(
    uint8_t s_frameBuffer[DEMO_BUFFER_COUNT][DEMO_PANEL_HEIGHT][DEMO_PANEL_WIDTH][DEMO_BUFFER_BYTE_PER_PIXEL],
    DEMO_FB_ALIGN);


#if (USE_MIPI_PANEL == MIPI_PANEL_RK055AHD091)

static mipi_dsi_device_t dsiDevice = {
    .virtualChannel = 0,
    .xferFunc       = PANEL_DSI_Transfer,
};

static const rm68200_resource_t rm68200Resource = {
    .dsiDevice    = &dsiDevice,
    .pullResetPin = PANEL_PullResetPin,
    .pullPowerPin = PANEL_PullPowerPin,
};

static display_handle_t rm68200Handle = {
    .resource = &rm68200Resource,
    .ops      = &rm68200_ops,
};

#elif (USE_MIPI_PANEL == MIPI_PANEL_RK055IQH091)

static mipi_dsi_device_t dsiDevice = {
    .virtualChannel = 0,
    .xferFunc       = PANEL_DSI_Transfer,
};

static const rm68191_resource_t rm68191Resource = {
    .dsiDevice    = &dsiDevice,
    .pullResetPin = PANEL_PullResetPin,
    .pullPowerPin = PANEL_PullPowerPin,
};

static display_handle_t rm68191Handle = {
    .resource = &rm68191Resource,
    .ops      = &rm68191_ops,
};

#elif (USE_MIPI_PANEL == MIPI_PANEL_RK055MHD091)

static mipi_dsi_device_t dsiDevice = {
    .virtualChannel = 0,
    .xferFunc       = PANEL_DSI_Transfer,
};

static const hx8394_resource_t hx8394Resource = {
    .dsiDevice    = &dsiDevice,
    .pullResetPin = PANEL_PullResetPin,
    .pullPowerPin = PANEL_PullPowerPin,
};

static display_handle_t hx8394Handle = {
    .resource = &hx8394Resource,
    .ops      = &hx8394_ops,
};

#elif (USE_MIPI_PANEL == MIPI_PANEL_RK043FN02HC) || (USE_MIPI_PANEL == MIPI_PANEL_KD050WVTPA045)

  /* Will use the Parallel RGB interface instead */

#else

#define  ADV7533_MAIN_I2C_ADDR           ((uint8_t)0x7A >> 1)
#define  ADV7533_CEC_DSI_I2C_ADDR        ((uint8_t)0x78 >> 1)

static adv7535_resource_t adv7535Resource = {
		.i2cAddr = ADV7533_MAIN_I2C_ADDR,
		.i2cAddrCecDsi = ADV7533_CEC_DSI_I2C_ADDR,
	    .pullResetPin = PANEL_PullResetPin,
	    .pullPowerPin = PANEL_PullPowerPin,
};

static display_handle_t adv7535Handle = {
    .resource = &adv7535Resource,
    .ops      = &adv7535_ops,
};
#endif


/*******************************************************************************
 * Code
 ******************************************************************************/

static void PANEL_PullResetPin(bool pullUp)
{
    if (pullUp)
    {
        //GPIO_PinWrite(BOARD_MIPI_PANEL_RST_GPIO, BOARD_MIPI_PANEL_RST_PIN, 1);
        PCA6416_SetPins(PCA_LCD_DISP_MIPI_RST);
    }
    else
    {
        //GPIO_PinWrite(BOARD_MIPI_PANEL_RST_GPIO, BOARD_MIPI_PANEL_RST_PIN, 0);
        PCA6416_ClearPins(PCA_LCD_DISP_MIPI_RST);
    }
}

/* From the schematic, the power pin is pinned to high. */
static void PANEL_PullPowerPin(bool pullUp)
{
    if (pullUp)
    {
        //GPIO_PinWrite(BOARD_MIPI_PANEL_POWER_GPIO, BOARD_MIPI_PANEL_POWER_PIN, 1);
        PCA6416_SetPins(PCA_LCD_BL_PWR);
    }
    else
    {
        //GPIO_PinWrite(BOARD_MIPI_PANEL_POWER_GPIO, BOARD_MIPI_PANEL_POWER_PIN, 0);
        PCA6416_ClearPins(PCA_LCD_BL_PWR);
    }
}

#if !IS_HDMI_PANEL()
status_t PANEL_DSI_Transfer(dsi_transfer_t *xfer)
{
    return DSI_TransferBlocking(DEMO_MIPI_DSI, xfer);
}
#endif

void BOARD_InitLcdifClock(void)
{
    /*
     * The pixel clock is (height + VSW + VFP + VBP) * (width + HSW + HFP + HBP) * frame rate.
     *
     * Use PLL_528 as clock source.
     *
     * For 60Hz frame rate, the RK055IQH091 pixel clock should be 36MHz.
     * the RK055AHD091 pixel clock should be 62MHz.
     */
    const clock_root_config_t lcdifv2ClockConfig = {
        .clockOff = false,
#if IS_HDMI_PANEL()
        .mux      = kCLOCK_LCDIFV2_ClockRoot_MuxVideoPllOut,
        .div      = DEMO_PIXEL_CLOCK_DIV,
#else
        .mux      = 4, /*!< PLL_528. */
  #if (USE_MIPI_PANEL == MIPI_PANEL_RK055AHD091) || (USE_MIPI_PANEL == MIPI_PANEL_RK055MHD091)
        .div = 9,
  #elif (USE_MIPI_PANEL == MIPI_PANEL_RK043FN02HC)
        .div = 59,  
  #elif (USE_MIPI_PANEL == MIPI_PANEL_KD050WVTPA045)
        .div = 16,
  #else
        .div = 15,
  #endif
#endif
    };

#if IS_HDMI_PANEL()
    CLOCK_InitVideoPllWithFreqKHz(DEMO_VIDEO_FREQ_KHZ, false, 0, 0);
#endif

    CLOCK_SetRootClock(kCLOCK_Root_Lcdifv2, &lcdifv2ClockConfig);

    mipiDsiDpiClkFreq_Hz = CLOCK_GetRootClockFreq(kCLOCK_Root_Lcdifv2);
}

static void BOARD_InitMipiDsiClock(void)
{
    uint32_t mipiDsiEscClkFreq_Hz;

    /* RxClkEsc max 60MHz, TxClkEsc 12 to 20MHz. */
    /* RxClkEsc = 528MHz / 11 = 48MHz. */
    /* TxClkEsc = 528MHz / 11 / 4 = 16MHz. */
    const clock_root_config_t mipiEscClockConfig = {
        .clockOff = false,
        .mux      = 1, /*!< PLL_528. */
        .div      = 1,
    };

    CLOCK_SetRootClock(kCLOCK_Root_Mipi_Esc, &mipiEscClockConfig);

    mipiDsiEscClkFreq_Hz = CLOCK_GetRootClockFreq(kCLOCK_Root_Mipi_Esc);

    const clock_group_config_t mipiEscClockGroupConfig = {
        .clockOff = false,
        .resetDiv = 1,
        .div0     = 1, /* TX esc clock. */
    };

    CLOCK_SetGroupConfig(kCLOCK_Group_MipiDsi, &mipiEscClockGroupConfig);

    mipiDsiTxEscClkFreq_Hz = mipiDsiEscClkFreq_Hz / 2;

#if IS_HDMI_PANEL()
    /* DPHY reference clock, use Video PLL. */
    const clock_root_config_t mipiDphyRefClockConfig = {
        .clockOff = false,
        .mux      = kCLOCK_MIPI_REF_ClockRoot_MuxVideoPllOut,
        .div      = DEMO_MIPI_DPY_REF_DIV,
    };

    CLOCK_SetRootClock(kCLOCK_Root_Mipi_Ref, &mipiDphyRefClockConfig);

    mipiDsiDphyRefClkFreq_Hz = CLOCK_GetRootClockFreq(kCLOCK_Root_Mipi_Ref);
#else
    /* DPHY reference clock, use OSC 24MHz clock. */
    const clock_root_config_t mipiDphyRefClockConfig = {
        .clockOff = false,
        .mux      = 1, /*!< OSC_24M. */
        .div      = 1,
    };

    CLOCK_SetRootClock(kCLOCK_Root_Mipi_Ref, &mipiDphyRefClockConfig);

    mipiDsiDphyRefClkFreq_Hz = BOARD_XTAL0_CLK_HZ;
#endif
}

static status_t BOARD_InitLcdPanel(void)
{
    status_t status = kStatus_Success;

    //const gpio_pin_config_t pinConfig = {kGPIO_DigitalOutput, 0, kGPIO_NoIntmode};

    const display_config_t displayConfig = {
        .resolution   = FSL_VIDEO_RESOLUTION(DEMO_PANEL_WIDTH, DEMO_PANEL_HEIGHT),
        .hsw          = DEMO_HSW,
        .hfp          = DEMO_HFP,
        .hbp          = DEMO_HBP,
        .vsw          = DEMO_VSW,
        .vfp          = DEMO_VFP,
        .vbp          = DEMO_VBP,
        .controlFlags = 0,
        .dsiLanes     = DEMO_MIPI_DSI_LANE_NUM,
    };

    //GPIO_PinInit(BOARD_MIPI_PANEL_POWER_GPIO, BOARD_MIPI_PANEL_POWER_PIN, &pinConfig);
    //GPIO_PinInit(BOARD_MIPI_PANEL_BL_GPIO, BOARD_MIPI_PANEL_BL_PIN, &pinConfig);
    //GPIO_PinInit(BOARD_MIPI_PANEL_RST_GPIO, BOARD_MIPI_PANEL_RST_PIN, &pinConfig);
#if (MIPI_PANEL_RK055AHD091 == USE_MIPI_PANEL)
    PCA6416_ClearPins(PCA_LCD_DISP_MIPI_RST | PCA_LCD_BL_PWR);
    PCA9530_ClearPins(PCA9530_LED0);

    status = RM68200_Init(&rm68200Handle, &displayConfig);

    if (status == kStatus_Success)
    {
        //GPIO_PinWrite(BOARD_MIPI_PANEL_BL_GPIO, BOARD_MIPI_PANEL_BL_PIN, 1);
        PCA9530_SetPins(PCA9530_LED0);
    }
#elif (MIPI_PANEL_RK055IQH091 == USE_MIPI_PANEL)
    PCA6416_ClearPins(PCA_LCD_DISP_MIPI_RST | PCA_LCD_BL_PWR);
    PCA9530_ClearPins(PCA9530_LED0);

    status = RM68191_Init(&rm68191Handle, &displayConfig);

    if (status == kStatus_Success)
    {
        //GPIO_PinWrite(BOARD_MIPI_PANEL_BL_GPIO, BOARD_MIPI_PANEL_BL_PIN, 1);
        PCA9530_SetPins(PCA9530_LED0);
    }
#elif (MIPI_PANEL_RK055MHD091 == USE_MIPI_PANEL)
    PCA6416_ClearPins(PCA_LCD_DISP_MIPI_RST | PCA_LCD_BL_PWR);
    PCA9530_ClearPins(PCA9530_LED0);

    status = RM68191_Init(&rm68191Handle, &displayConfig);

    if (status == kStatus_Success)
    {
        //GPIO_PinWrite(BOARD_MIPI_PANEL_BL_GPIO, BOARD_MIPI_PANEL_BL_PIN, 1);
        PCA9530_SetPins(PCA9530_LED0);
    }
#elif (MIPI_PANEL_RK043FN02HC == USE_MIPI_PANEL) || (MIPI_PANEL_KD050WVTPA045 == USE_MIPI_PANEL)
    PCA6416_ClearPins(PCA_LCD_DISP_MIPI_RST | PCA_LCD_BL_PWR);
    PCA9530_ClearPins(PCA9530_LED0);

    /* Power on. */
    PANEL_PullPowerPin(true);
    SDK_DelayAtLeastUs(1000U, SystemCoreClock);

    /* Perform reset. */
    PANEL_PullResetPin(false);
    SDK_DelayAtLeastUs(1000U, SystemCoreClock);
    PANEL_PullResetPin(true);
    SDK_DelayAtLeastUs(1000U, SystemCoreClock);

    PCA9530_SetPins(PCA9530_LED0);
#else
    status = ADV7535_Init(&adv7535Handle, &displayConfig);
#endif

    return status;
}

static void BOARD_SetMipiDsiConfig(void)
{
    dsi_config_t dsiConfig;
    dsi_dphy_config_t dphyConfig;

    const dsi_dpi_config_t dpiConfig = {.pixelPayloadSize = DEMO_PANEL_WIDTH,
                                        .dpiColorCoding   = kDSI_Dpi24Bit,
                                        .pixelPacket      = kDSI_PixelPacket24Bit,
#if IS_HDMI_PANEL()
										.videoMode        = kDSI_DpiNonBurstWithSyncPulse,
                                        .bllpMode         = kDSI_DpiBllpNull,
                                        .polarityFlags    = kDSI_DpiVsyncActiveLow | kDSI_DpiHsyncActiveLow,
#else
                                        .videoMode        = kDSI_DpiBurst,
                                        .bllpMode         = kDSI_DpiBllpLowPower,
                                        .polarityFlags    = kDSI_DpiVsyncActiveLow | kDSI_DpiHsyncActiveLow,
#endif
                                        .hfp              = DEMO_HFP,
                                        .hbp              = DEMO_HBP,
                                        .hsw              = DEMO_HSW,
                                        .vfp              = DEMO_VFP,
                                        .vbp              = DEMO_VBP,

                                        .panelHeight      = DEMO_PANEL_HEIGHT,
                                        .virtualChannel   = 0};

    /*
     * dsiConfig.numLanes = 4;
     * dsiConfig.enableNonContinuousHsClk = false;
     * dsiConfig.autoInsertEoTp = true;
     * dsiConfig.numExtraEoTp = 0;
     * dsiConfig.htxTo_ByteClk = 0;
     * dsiConfig.lrxHostTo_ByteClk = 0;
     * dsiConfig.btaTo_ByteClk = 0;
     */
    DSI_GetDefaultConfig(&dsiConfig);
    dsiConfig.numLanes       = DEMO_MIPI_DSI_LANE_NUM;
#if IS_HDMI_PANEL()
    dsiConfig.autoInsertEoTp = false;
#else
    dsiConfig.autoInsertEoTp = true;
#endif

    /* Init the DSI module. */
    DSI_Init(DEMO_MIPI_DSI, &dsiConfig);

    /* Init DPHY.
     *
     * The DPHY bit clock must be fast enough to send out the pixels, it should be
     * larger than:
     *
     *         (Pixel clock * bit per output pixel) / number of MIPI data lane
     *
     * Here the desired DPHY bit clock multiplied by ( 9 / 8 = 1.125) to ensure
     * it is fast enough.
     *
     * Note that the DSI output pixel is 24bit per pixel.
     */
    mipiDsiDphyBitClkFreq_Hz = mipiDsiDpiClkFreq_Hz * (24 / DEMO_MIPI_DSI_LANE_NUM);

#if !IS_HDMI_PANEL()
    mipiDsiDphyBitClkFreq_Hz = DEMO_MIPI_DPHY_BIT_CLK_ENLARGE(mipiDsiDphyBitClkFreq_Hz);
#endif

    DSI_GetDphyDefaultConfig(&dphyConfig, mipiDsiDphyBitClkFreq_Hz, mipiDsiTxEscClkFreq_Hz);

    mipiDsiDphyBitClkFreq_Hz = DSI_InitDphy(DEMO_MIPI_DSI, &dphyConfig, mipiDsiDphyRefClkFreq_Hz);

    /* Init DPI interface. */
    DSI_SetDpiConfig(DEMO_MIPI_DSI, &dpiConfig, DEMO_MIPI_DSI_LANE_NUM, mipiDsiDpiClkFreq_Hz, mipiDsiDphyBitClkFreq_Hz);
}

status_t BOARD_InitDisplayInterface(void)
{
#if (MIPI_PANEL_RK043FN02HC == USE_MIPI_PANEL) || (MIPI_PANEL_KD050WVTPA045 == USE_MIPI_PANEL)

    /* LCDIF v2 output to Parallel LCDIF. */
    CLOCK_EnableClock(kCLOCK_Video_Mux);
    VIDEO_MUX->VID_MUX_CTRL.SET = VIDEO_MUX_VID_MUX_CTRL_PARA_LCD_SEL_MASK;

#else

    /* LCDIF v2 output to MIPI DSI. */
    CLOCK_EnableClock(kCLOCK_Video_Mux);
    VIDEO_MUX->VID_MUX_CTRL.SET = VIDEO_MUX_VID_MUX_CTRL_MIPI_DSI_SEL_MASK;

    /* 1. Power on and isolation off. */
    PGMC_BPC4->BPC_POWER_CTRL |= (PGMC_BPC_BPC_POWER_CTRL_PSW_ON_SOFT_MASK | PGMC_BPC_BPC_POWER_CTRL_ISO_OFF_SOFT_MASK);

    /* 2. Assert MIPI reset. */
    IOMUXC_GPR->GPR62 &=
        ~(IOMUXC_GPR_GPR62_MIPI_DSI_PCLK_SOFT_RESET_N_MASK | IOMUXC_GPR_GPR62_MIPI_DSI_ESC_SOFT_RESET_N_MASK |
          IOMUXC_GPR_GPR62_MIPI_DSI_BYTE_SOFT_RESET_N_MASK | IOMUXC_GPR_GPR62_MIPI_DSI_DPI_SOFT_RESET_N_MASK);

    /* 3. Setup clock. */
    BOARD_InitMipiDsiClock();

    /* 4. Deassert PCLK and ESC reset. */
    IOMUXC_GPR->GPR62 |=
        (IOMUXC_GPR_GPR62_MIPI_DSI_PCLK_SOFT_RESET_N_MASK | IOMUXC_GPR_GPR62_MIPI_DSI_ESC_SOFT_RESET_N_MASK);

    /* 5. Configures peripheral. */
    BOARD_SetMipiDsiConfig();

    /* 6. Deassert BYTE and DBI reset. */
    IOMUXC_GPR->GPR62 |=
        (IOMUXC_GPR_GPR62_MIPI_DSI_BYTE_SOFT_RESET_N_MASK | IOMUXC_GPR_GPR62_MIPI_DSI_DPI_SOFT_RESET_N_MASK);

#endif

    /* 7. Configure the panel. */
    return BOARD_InitLcdPanel();
}
