/*
 * Copyright 2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*  Standard C Included Files */
#include <stdio.h>
#include <string.h>
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "fsl_debug_console.h"
#include "fsl_lpi2c.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/* Get frequency of lpi2c clock */
#define LPI2C_CLOCK_FREQUENCY (CLOCK_GetFreq(kCLOCK_OscRc48MDiv2))

#define I2C_BAUDRATE               100000U


/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
 * Code
 ******************************************************************************/

static bool probe_addr(LPI2C_Type* bus, uint8_t addr7bit, bool verbose)
{
    uint8_t data = 0;
    status_t res;
    res = LPI2C_MasterStart(bus, addr7bit, kLPI2C_Write);
    if (res == kStatus_Success) {
        res = LPI2C_MasterSend(bus, &data, 1);
        if (res == kStatus_Success) {
            res = LPI2C_MasterStop(bus);
            if (res == kStatus_Success) {
                if (verbose) {
                    PRINTF("0x%x: xx : Found I2C device on 7-bit adress 0x%x\r\n", addr7bit, addr7bit);
                }
                return true;
            } else if (verbose) {
                PRINTF("0x%x: -- : Master Stop FAIL, error=%d\r\n", addr7bit, res);
            }
        } else if (verbose) {
            PRINTF("0x%x: -- : Master Send FAIL, error=%d\r\n", addr7bit, res);
        }
    } else if (verbose) {
        PRINTF("0x%x: -- : Master Started FAIL, error=%d\r\n", addr7bit, res);
    }
    return false;
}

static void probe_bus(LPI2C_Type* bus)
{
    uint8_t start = 0;

    PRINTF("     0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f\r\n");
    for (uint8_t offset = 0; offset < 0x80; offset += 0x10) {
        if (offset == 0) {
            PRINTF("00:         ");
            start = 3;
        } else {
            PRINTF("%x:", offset);
            start = 0;
        }
        for (uint8_t x = start; x <= 0xf; x++) {
            if (probe_addr(bus, offset+x, false)) {
                if ((offset+x)<0x10) {
                    PRINTF(" 0%x", offset+x);
                } else {
                    PRINTF(" %x", offset+x);
                }
            } else {
                PRINTF(" --");
            }
        }
        PRINTF("\r\n");
    }
}


/*!
 * @brief Main function
 */
int main(void)
{
    BOARD_ConfigMPU();
    BOARD_InitMipiPanelPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();
    SystemCoreClockUpdate();

    PRINTF("\r\nLPI2C probe example.\r\n");

    lpi2c_master_config_t masterConfig = {0};

    /*
     * masterConfig.debugEnable = false;
     * masterConfig.ignoreAck = false;
     * masterConfig.pinConfig = kLPI2C_2PinOpenDrain;
     * masterConfig.baudRate_Hz = 100000U;
     * masterConfig.busIdleTimeout_ns = 0;
     * masterConfig.pinLowTimeout_ns = 0;
     * masterConfig.sdaGlitchFilterWidth_ns = 0;
     * masterConfig.sclGlitchFilterWidth_ns = 0;
     */
    LPI2C_MasterGetDefaultConfig(&masterConfig);

    /* Change the default baudrate configuration */
    masterConfig.baudRate_Hz = I2C_BAUDRATE;

    PRINTF("\r\nProbing I2C-A (I2C5)\r\n");
    LPI2C_MasterInit(LPI2C5, &masterConfig, LPI2C_CLOCK_FREQUENCY);
    probe_bus(LPI2C5);

    PRINTF("\r\nProbing I2C-B (I2C6)\r\n");
    LPI2C_MasterInit(LPI2C6, &masterConfig, LPI2C_CLOCK_FREQUENCY);
    probe_bus(LPI2C6);

    PRINTF("\r\nProbing I2C-C (I2C1)\r\n");
    LPI2C_MasterInit(LPI2C1, &masterConfig, LPI2C_CLOCK_FREQUENCY);
    probe_bus(LPI2C1);

    PRINTF("DONE\r\n");
    while (1)
    {
    }
}
