/*
 * Copyright 2019 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */


#include <stdio.h>
#include "demo_config.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "fsl_debug_console.h"
#include "board.h"
#include "fsl_lpi2c.h"
#include "get_sensor_data.h"

/*******************************************************************************
 * Macro Definitions
 ******************************************************************************/

#define MIKROE_IMU_ADDR (0x1EU)

static fxos_handle_t FXOSdrv = {0};

/*******************************************************************************
 * Code
 ******************************************************************************/

status_t SENSOR_Init(void)
{
    int32_t status;
    fxos_config_t config     = {0};

    /* I2C initialize */
    BOARD_Accel_I2C_Init();
    /* Configure the I2C function */
    config.I2C_SendFunc    = BOARD_Accel_I2C_Send;
    config.I2C_ReceiveFunc = BOARD_Accel_I2C_Receive;

    config.slaveAddress = MIKROE_IMU_ADDR;
    /* Initialize accelerometer sensor */
    status = FXOS_Init(&FXOSdrv, &config);
    if (status != kStatus_Success)
    {
    	PRINTF("FXOS sensor initialization failed %d\r\n", status);
        return status;
    }

    return status;
}

status_t SENSOR_UpdateData(int16_t *sensorDiff, int16_t *sensorData)
{
    fxos8700_accelmagdata_t rawDataAccel;
    status_t status = kStatus_Success;

    for (int j = 0; j < PATCH_SIZE; j++)
    {
        status = SENSOR_Run(&rawDataAccel);
        if (status != kStatus_Success)
        {
            return status;
        }

        sensorDiff[j * NUM_CHANELS + 0] = sensorData[j * NUM_CHANELS + 0] - rawDataAccel.accel[0];
        sensorDiff[j * NUM_CHANELS + 1] = sensorData[j * NUM_CHANELS + 1] - rawDataAccel.accel[1];
        sensorDiff[j * NUM_CHANELS + 2] = sensorData[j * NUM_CHANELS + 2] - rawDataAccel.accel[2];

        sensorData[j * NUM_CHANELS + 0] = rawDataAccel.accel[0];
        sensorData[j * NUM_CHANELS + 1] = rawDataAccel.accel[1];
        sensorData[j * NUM_CHANELS + 2] = rawDataAccel.accel[2];

    }
    return status;
}

void SENSOR_PreprocessSample(uint8_t *dstData, const int16_t *srcData, tensor_type_t inputType)
{
    /* Load input data */
    for (int i = 0; i < PATCH_SIZE * NUM_CHANELS; i += 6)
    {
        for (int ch = 0; ch < NUM_CHANELS; ch++)
        {
            //float scaled = (ch == 3 || ch == 4 || ch == 5) ? srcData[i + ch] / 10 : srcData[i + ch];
        	float scaled = srcData[i + ch];
            if (inputType == kTensorType_UINT8)
            {
                dstData[i + ch] = (uint8_t)(MIN(MAX((scaled + 1) * 127.5, 0.0), 255.0));
            }
            else
            {
                ((float *)dstData)[i + ch] = scaled;
            }
        }
    }
}

status_t SENSOR_GetData(uint8_t *dstData, tensor_type_t inputType, bool *skip)
{
    static int16_t s_sensorData[PATCH_SIZE * NUM_CHANELS]     = {0};
    static int16_t s_sensorDataDiff[PATCH_SIZE * NUM_CHANELS] = {0};
    status_t status                                           = kStatus_Success;

    *skip = false;

    status = SENSOR_UpdateData(s_sensorDataDiff, s_sensorData);
    if (status != kStatus_Success)
    {
        return status;
    }

    printf("Data for inference are ready");

    SENSOR_PreprocessSample(dstData, s_sensorDataDiff, inputType);

    return status;
}


status_t SENSOR_Run(fxos8700_accelmagdata_t *rawDataAccel){

  status_t status = kStatus_Success;
  fxos_data_t sensorData   = {0};


  status = FXOS_ReadSensorData(&FXOSdrv, &sensorData);
  if (status != kStatus_Success)
  {
	  /*PRINTF("FXOS bad data\r\n");*/
	  return status;
  }

  rawDataAccel->accel[0] = ((int16_t)sensorData.accelXMSB << 8) | sensorData.accelXLSB;
  rawDataAccel->accel[0] /= 4;
  rawDataAccel->accel[1] = ((int16_t)sensorData.accelYMSB << 8) | sensorData.accelYLSB;
  rawDataAccel->accel[1] /= 4;
  rawDataAccel->accel[2] = ((int16_t)sensorData.accelZMSB << 8) | sensorData.accelZLSB;
  rawDataAccel->accel[2] /= 4;
  rawDataAccel->mag[0] = ((int16_t)sensorData.magXMSB << 8) | sensorData.magXLSB;
  rawDataAccel->mag[1] = ((int16_t)sensorData.magYMSB << 8) | sensorData.magYLSB;
  rawDataAccel->mag[2] = ((int16_t)sensorData.magZMSB << 8) | sensorData.magZMSB;


  return status;
}

status_t SENSOR_CollectData(void)
{
    fxos8700_accelmagdata_t rawDataAccel;
    status_t status = kStatus_Success;

    status = SENSOR_Init();
    if (status != kStatus_Success)
    {
        return status;
    }

    printf("\rtime,ax,ay,az\r");

    for (int sample_num = 0; sample_num < SAMPLE_NUM; ++sample_num)
    {
        status = SENSOR_Run(&rawDataAccel);
        if (status != kStatus_Success)
        {
            return status;
        }

        printf("%s%d, %d, %d, %d", EOL, sample_num,
        		rawDataAccel.accel[0], rawDataAccel.accel[1], rawDataAccel.accel[2]);
    }
    printf("\r\nAll samples are collected.\r\n");

    return status;
}
