/*
 * Copyright 2021 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <iostream>
#include <algorithm>

#include "board.h"
#include "peripherals.h"
#include "pin_mux.h"
#include "demo_config.h"
#include "get_sensor_data.h"
#include "output_postproc.h"
#include "timer.h"

#define CHECK_STATUS(STATUS, FUNCTION, MESSAGE)                  \
    (STATUS) = (FUNCTION);                                       \
    if ((STATUS) != 0)                                           \
    {                                                            \
        printf(MESSAGE EOL "Error code: %d" EOL, (int)(STATUS)); \
    }                                                            \
    do                                                           \
    {                                                            \
        ;                                                        \
    } while (STATUS)


int main(void)
{
    /* Init board hardware. */
    BOARD_InitBootPins();
    BOARD_InitBootClocks();
    BOARD_I2C_ConfigurePins();
    BOARD_InitBootPeripherals();

#ifndef BOARD_INIT_DEBUG_CONSOLE_PERIPHERAL
    /* Init FSL debug console. */
    BOARD_InitDebugConsole();
#endif
    TIMER_Init();


    bool skipOutput;
    status_t status;
#if DATA_COLLECT == true
    CHECK_STATUS(status, SENSOR_CollectData(), "Failed retrieving sensors data");
#endif

    printf("Anomaly Detection example using a TensorFlow Lite model.");
    printf("Detection threshold: %d.%d", (int)DETECTION_TRESHOLD, ( ((int)DETECTION_TRESHOLD * 100) % 100));


    CHECK_STATUS(status, MODEL_Init(), "Inference failed.");

    tensor_dims_t inputDims;
    tensor_type_t inputType;
    uint8_t *inputData = MODEL_GetInputTensorData(&inputDims, &inputType);

    tensor_dims_t outputDims;
    tensor_type_t outputType;
    uint8_t *outputData = MODEL_GetOutputTensorData(&outputDims, &outputType);

#ifdef MICRO_BUFF
    /* Buffer for storing data from input tensor. */
    uint8_t inputDataBuffer[PATCH_SIZE * NUM_CHANELS * 4];
    std::fill_n(inputDataBuffer, PATCH_SIZE * NUM_CHANELS * 4, 0);
#endif

    CHECK_STATUS(status, SENSOR_Init(), "initialization of sensor failed");

    while (1)
    {
        /* Expected tensor dimensions: [batches, height, width, channels] */
        CHECK_STATUS(status, SENSOR_GetData(inputData, inputType, &skipOutput), "Failed retrieving sensors data");

        auto startTime = TIMER_GetTimeInUS();
#ifdef MICRO_BUFF
        /* Input tensor share memory with output tensor.   */
        /* Data must be copied before inference is called. */
        std::copy_n(inputData, PATCH_SIZE * NUM_CHANELS * 4, inputDataBuffer);
        inputData = inputDataBuffer;
#endif

        MODEL_RunInference();
        auto endTime = TIMER_GetTimeInUS();

        MODEL_ProcessOutput(outputData, inputData, &outputDims, outputType, endTime - startTime, &skipOutput);
    }
}
