/* buf.c - Buffer management */

/*
 * Copyright 2021 NXP
 * Copyright (c) 2015-2019 Intel Corporation
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#define LOG_MODULE_NAME net_buf
#define LOG_LEVEL CONFIG_NET_BUF_LOG_LEVEL

#include <stdio.h>
#include <errno/errno.h>
#include <stddef.h>
#include <string.h>
#include <sys/byteorder.h>

#include <porting.h>

#include <net/buf.h>

#define LOG_ENABLE IS_ENABLED(CONFIG_NET_BUF_LOG)
#define LOG_MODULE_NAME net_buf
#include "fsl_component_log.h"
LOG_MODULE_DEFINE(LOG_MODULE_NAME, kLOG_LevelTrace);

#if (defined(CONFIG_NET_BUF_LOG) && (CONFIG_NET_BUF_LOG > 0U))
#define NET_BUF_DBG(fmt, ...) LOG_DBG(fmt, ##__VA_ARGS__)
#define NET_BUF_ERR(fmt, ...) LOG_ERR(fmt, ##__VA_ARGS__)
#define NET_BUF_WARN(fmt, ...) LOG_WRN(fmt, ##__VA_ARGS__)
#define NET_BUF_INFO(fmt, ...) LOG_INF(fmt, ##__VA_ARGS__)
#else
#define NET_BUF_DBG(fmt, ...)
#define NET_BUF_ERR(fmt, ...)
#define NET_BUF_WARN(fmt, ...)
#define NET_BUF_INFO(fmt, ...)
#endif /* CONFIG_NET_BUF_LOG */

#define NET_BUF_ASSERT(cond, ...) __ASSERT(cond, "" __VA_ARGS__)

#if CONFIG_NET_BUF_WARN_ALLOC_INTERVAL > 0
#define WARN_ALLOC_INTERVAL BT_SECONDS(CONFIG_NET_BUF_WARN_ALLOC_INTERVAL)
#else
#define WARN_ALLOC_INTERVAL osaWaitForever_c
#endif

/* Linker-defined symbol bound to the static pool structs */
#if defined(__CC_ARM) || defined(__ARMCC_VERSION)
extern struct net_buf_pool Image$$RW_net_buf_pool$$Base[];
static struct net_buf_pool * _net_buf_pool_list = &Image$$RW_net_buf_pool$$Base[0];
#elif defined(__ICCARM__)
#pragma section ="._net_buf_pool"
extern struct net_buf_pool* _net_buf_pool_list = (struct net_buf_pool*)__section_begin("._net_buf_pool");
#elif defined(__GNUC__)
extern struct net_buf_pool _net_buf_pool_list[];
#else
#endif

struct net_buf_pool *net_buf_pool_get(int id)
{
	return &_net_buf_pool_list[id];
}

static int pool_id(struct net_buf_pool *pool)
{
	return pool - _net_buf_pool_list;
}

int net_buf_id(struct net_buf *buf)
{
	struct net_buf_pool *pool = net_buf_pool_get(buf->pool_id);

	return buf - pool->__bufs;
}

static inline struct net_buf *pool_get_uninit(struct net_buf_pool *pool,
					      uint16_t uninit_count)
{
	struct net_buf *buf;

	buf = &pool->__bufs[pool->buf_count - uninit_count];

	buf->pool_id = pool_id(pool);

	return buf;
}

void net_buf_reset(struct net_buf *buf)
{
	__ASSERT_NO_MSG(buf->flags == 0U);
	__ASSERT_NO_MSG(buf->frags == NULL);

	net_buf_simple_reset(&buf->b);
}

static uint8_t *generic_data_ref(struct net_buf *buf, uint8_t *data)
{
	uint8_t *ref_count;

	ref_count = data - 1;
	(*ref_count)++;

	return data;
}

static uint8_t *mem_pool_data_alloc(struct net_buf *buf, size_t *size,
				 k_timeout_t timeout)
{
#if 0
	struct net_buf_pool *buf_pool = net_buf_pool_get(buf->pool_id);
	struct k_mem_pool *pool = buf_pool->alloc->alloc_data;
	struct k_mem_block block;
	uint8_t *ref_count;

	/* Reserve extra space for k_mem_block_id and ref-count (uint8_t) */
	if (k_mem_pool_alloc(pool, &block,
			     sizeof(struct k_mem_block_id) + 1 + *size,
			     timeout)) {
		return NULL;
	}

	/* save the block descriptor info at the start of the actual block */
	memcpy(block.data, &block.id, sizeof(block.id));

	ref_count = (uint8_t *)block.data + sizeof(block.id);
	*ref_count = 1U;

	/* Return pointer to the byte following the ref count */
	return ref_count + 1;
#else
	return NULL;
#endif
}

static void mem_pool_data_unref(struct net_buf *buf, uint8_t *data)
{
#if 0
	struct k_mem_block_id id;
	uint8_t *ref_count;

	ref_count = data - 1;
	if (--(*ref_count)) {
		return;
	}

	/* Need to copy to local variable due to alignment */
	memcpy(&id, ref_count - sizeof(id), sizeof(id));
	k_mem_pool_free_id(&id);
#endif
}

const struct net_buf_data_cb net_buf_var_cb = {
	.alloc = mem_pool_data_alloc,
	.ref   = generic_data_ref,
	.unref = mem_pool_data_unref,
};

static uint8_t *fixed_data_alloc(struct net_buf *buf, size_t *size,
			      k_timeout_t timeout)
{
	struct net_buf_pool *pool = net_buf_pool_get(buf->pool_id);
	const struct net_buf_pool_fixed *fixed = (const struct net_buf_pool_fixed *)pool->alloc->alloc_data;

	*size = MIN(fixed->data_size, *size);

	return fixed->data_pool + fixed->data_size * net_buf_id(buf);
}

static void fixed_data_unref(struct net_buf *buf, uint8_t *data)
{
	/* Nothing needed for fixed-size data pools */
}

const struct net_buf_data_cb net_buf_fixed_cb = {
	.alloc = fixed_data_alloc,
	.unref = fixed_data_unref,
};

#if (CONFIG_HEAP_MEM_POOL_SIZE > 0)

static uint8_t *heap_data_alloc(struct net_buf *buf, size_t *size,
			     k_timeout_t timeout)
{
	uint8_t *ref_count;

	ref_count = k_malloc(1 + *size);
	if (!ref_count) {
		return NULL;
	}

	*ref_count = 1U;

	return ref_count + 1;
}

static void heap_data_unref(struct net_buf *buf, uint8_t *data)
{
	uint8_t *ref_count;

	ref_count = data - 1;
	if (--(*ref_count)) {
		return;
	}

	k_free(ref_count);
}

static const struct net_buf_data_cb net_buf_heap_cb = {
	.alloc = heap_data_alloc,
	.ref   = generic_data_ref,
	.unref = heap_data_unref,
};

const struct net_buf_data_alloc net_buf_heap_alloc = {
	.cb = &net_buf_heap_cb,
};

#endif /* CONFIG_HEAP_MEM_POOL_SIZE > 0 */

static uint8_t *data_alloc(struct net_buf *buf, size_t *size, k_timeout_t timeout)
{
	struct net_buf_pool *pool = net_buf_pool_get(buf->pool_id);

	return pool->alloc->cb->alloc(buf, size, timeout);
}

static uint8_t *data_ref(struct net_buf *buf, uint8_t *data)
{
	struct net_buf_pool *pool = net_buf_pool_get(buf->pool_id);

	return pool->alloc->cb->ref(buf, data);
}

static void data_unref(struct net_buf *buf, uint8_t *data)
{
	struct net_buf_pool *pool = net_buf_pool_get(buf->pool_id);

	if (buf->flags & NET_BUF_EXTERNAL_DATA) {
		return;
	}

	pool->alloc->cb->unref(buf, data);
}

#if (defined(CONFIG_NET_BUF_LOG) && (CONFIG_NET_BUF_LOG > 0U))
struct net_buf *net_buf_alloc_len_debug(struct net_buf_pool *pool, size_t size,
					k_timeout_t timeout, const char *func,
					int line)
#else
struct net_buf *net_buf_alloc_len(struct net_buf_pool *pool, size_t size,
				  k_timeout_t timeout)
#endif
{
	uint32_t alloc_start = OSA_TimeGetMsec();
	struct net_buf *buf;
	unsigned int key;
	osa_status_t ret;

	__ASSERT_NO_MSG(pool);

	NET_BUF_DBG("%s():%d: pool %p size %lu timeout %d", func, line, pool,
		    size, timeout);

	if (NULL == pool->free)
	{
		ret = OSA_MsgQCreate((osa_msgq_handle_t)pool->freeHandle, pool->buf_count, sizeof(void *));
		__ASSERT_NO_MSG(KOSA_StatusSuccess == ret);
        if (KOSA_StatusSuccess == ret)
        {
            pool->free = (osa_msgq_handle_t)pool->freeHandle;
        }
	}

	/* We need to lock interrupts temporarily to prevent race conditions
	 * when accessing pool->uninit_count.
	 */
	key = DisableGlobalIRQ();

	/* If there are uninitialized buffers we're guaranteed to succeed
	 * with the allocation one way or another.
	 */
	if (pool->uninit_count) {
		uint16_t uninit_count;

		/* If this is not the first access to the pool, we can
		 * be opportunistic and try to fetch a previously used
		 * buffer from the LIFO with K_NO_WAIT.
		 */
		if (pool->uninit_count < pool->buf_count) {
            osa_status_t ret;

            ret = OSA_MsgQGet(pool->free, &buf, osaWaitNone_c);
			if (KOSA_StatusSuccess == ret) {
				EnableGlobalIRQ(key);
				goto success;
			}
		}

		uninit_count = pool->uninit_count--;
		EnableGlobalIRQ(key);

		buf = pool_get_uninit(pool, uninit_count);
		goto success;
	}

	EnableGlobalIRQ(key);

#if (defined(CONFIG_NET_BUF_LOG) && (CONFIG_NET_BUF_LOG_LEVEL >= LOG_LEVEL_WRN))
	if (timeout == osaWaitForever_c) {
		uint32_t ref = OSA_TimeGetMsec();
		ret = OSA_MsgQGet(pool->free, &buf, osaWaitNone_c);
		while (KOSA_StatusSuccess != ret) {
#if (defined(CONFIG_NET_BUF_POOL_USAGE) && (CONFIG_NET_BUF_POOL_USAGE > 0U))
			NET_BUF_WARN("%s():%d: Pool %s low on buffers.",
				     func, line, pool->name);
#else
			NET_BUF_WARN("%s():%d: Pool %p low on buffers.",
				     func, line, pool);
#endif
			ret = OSA_MsgQGet(pool->free, &buf, WARN_ALLOC_INTERVAL);
#if (defined(CONFIG_NET_BUF_POOL_USAGE) && (CONFIG_NET_BUF_POOL_USAGE > 0U))
			NET_BUF_WARN("%s():%d: Pool %s blocked for %u secs",
				     func, line, pool->name,
				     (OSA_TimeGetMsec() - ref) / MSEC_PER_SEC);
#else
			NET_BUF_WARN("%s():%d: Pool %p blocked for %u secs",
				     func, line, pool,
				     (OSA_TimeGetMsec() - ref) / MSEC_PER_SEC);
#endif
		}
	} else {
		ret = OSA_MsgQGet(pool->free, &buf, timeout);
	}
#else
	ret = OSA_MsgQGet(pool->free, &buf, timeout);
#endif
	if (KOSA_StatusSuccess != ret) {
		NET_BUF_ERR("%s():%d: Failed to get free buffer", func, line);
		return NULL;
	}

success:
	NET_BUF_DBG("allocated buf %p", buf);

	if (size) {
		size_t req_size = size;

		if (timeout != osaWaitNone_c && timeout != osaWaitForever_c) {
			uint32_t diff = OSA_TimeGetMsec() - alloc_start;

			timeout -= MIN(timeout, diff);
		}

		buf->__buf = data_alloc(buf, &size, timeout);
		if (!buf->__buf) {
			NET_BUF_ERR("%s():%d: Failed to allocate data",
				    func, line);
			net_buf_destroy(buf);
			return NULL;
		}

		NET_BUF_ASSERT(req_size <= size);
        (void)req_size;
	} else {
		buf->__buf = NULL;
	}

	buf->ref   = 1U;
	buf->flags = 0U;
	buf->frags = NULL;
	buf->size  = size;
	net_buf_reset(buf);

#if (defined(CONFIG_NET_BUF_POOL_USAGE) && (CONFIG_NET_BUF_POOL_USAGE > 0U))
	pool->avail_count--;
	__ASSERT_NO_MSG(pool->avail_count >= 0);
    NET_BUF_WARN("%s():%d: Pool %p available count %d",
             func, line, pool, pool->avail_count);
#endif
	return buf;
}

#if (defined(CONFIG_NET_BUF_LOG) && (CONFIG_NET_BUF_LOG > 0U))
struct net_buf *net_buf_alloc_fixed_debug(struct net_buf_pool *pool,
					  k_timeout_t timeout, const char *func,
					  int line)
{
	const struct net_buf_pool_fixed *fixed = pool->alloc->alloc_data;

	return net_buf_alloc_len_debug(pool, fixed->data_size, timeout, func,
				       line);
}
#else
struct net_buf *net_buf_alloc_fixed(struct net_buf_pool *pool,
				    k_timeout_t timeout)
{
	const struct net_buf_pool_fixed *fixed = (const struct net_buf_pool_fixed *)pool->alloc->alloc_data;

	return net_buf_alloc_len(pool, fixed->data_size, timeout);
}
#endif

#if (defined(CONFIG_NET_BUF_LOG) && (CONFIG_NET_BUF_LOG > 0U))
struct net_buf *net_buf_alloc_with_data_debug(struct net_buf_pool *pool,
					      void *data, size_t size,
					      k_timeout_t timeout,
					      const char *func, int line)
#else
struct net_buf *net_buf_alloc_with_data(struct net_buf_pool *pool,
					void *data, size_t size,
					k_timeout_t timeout)
#endif
{
	struct net_buf *buf;

#if (defined(CONFIG_NET_BUF_LOG) && (CONFIG_NET_BUF_LOG > 0U))
	buf = net_buf_alloc_len_debug(pool, 0, timeout, func, line);
#else
	buf = net_buf_alloc_len(pool, 0, timeout);
#endif
	if (!buf) {
		return NULL;
	}

	net_buf_simple_init_with_data(&buf->b, data, size);
	buf->flags = NET_BUF_EXTERNAL_DATA;

	return buf;
}

#if (defined(CONFIG_NET_BUF_LOG) && (CONFIG_NET_BUF_LOG > 0U))
struct net_buf *net_buf_get_debug(osa_msgq_handle_t fifo, k_timeout_t timeout,
				  const char *func, int line)
#else
struct net_buf *net_buf_get(osa_msgq_handle_t fifo, k_timeout_t timeout)
#endif
{
	struct net_buf *buf, *frag;
	osa_status_t ret;

	NET_BUF_DBG("%s():%d: fifo %p timeout %d", func, line, fifo, timeout);

	ret = OSA_MsgQGet(fifo, &buf, timeout);
	if (KOSA_StatusSuccess != ret) {
		return NULL;
	}

	NET_BUF_DBG("%s():%d: buf %p fifo %p", func, line, buf, fifo);

	/* Get any fragments belonging to this buffer */
	for (frag = buf; (frag->flags & NET_BUF_FRAGS); frag = frag->frags) {
		ret = OSA_MsgQGet(fifo, &frag->frags, osaWaitNone_c);
		__ASSERT_NO_MSG(KOSA_StatusSuccess == ret);

		/* The fragments flag is only for FIFO-internal usage */
		frag->flags &= ~NET_BUF_FRAGS;
	}

	/* Mark the end of the fragment list */
	frag->frags = NULL;

	return buf;
}

void net_buf_simple_init_with_data(struct net_buf_simple *buf,
				   void *data, size_t size)
{
	buf->__buf = (uint8_t *)data;
	buf->data  = (uint8_t *)data;
	buf->size  = size;
	buf->len   = size;
}

void net_buf_simple_reserve(struct net_buf_simple *buf, size_t reserve)
{
	__ASSERT_NO_MSG(buf);
	__ASSERT_NO_MSG(buf->len == 0U);
	NET_BUF_DBG("buf %p reserve %lu", buf, reserve);

	buf->data = buf->__buf + reserve;
}

#if 0
void net_buf_slist_put(sys_slist_t *list, struct net_buf *buf)
{
	struct net_buf *tail;
	unsigned int key;

	__ASSERT_NO_MSG(list);
	__ASSERT_NO_MSG(buf);

	for (tail = buf; tail->frags; tail = tail->frags) {
		tail->flags |= NET_BUF_FRAGS;
	}

	key = DisableGlobalIRQ();
	sys_slist_append_list(list, &buf->node, &tail->node);
	EnableGlobalIRQ(key);
}

struct net_buf *net_buf_slist_get(sys_slist_t *list)
{
	struct net_buf *buf, *frag;
	unsigned int key;

	__ASSERT_NO_MSG(list);

	key = DisableGlobalIRQ();
	buf = (void *)bt_list_get(list);
	EnableGlobalIRQ(key);

	if (!buf) {
		return NULL;
	}

	/* Get any fragments belonging to this buffer */
	for (frag = buf; (frag->flags & NET_BUF_FRAGS); frag = frag->frags) {
		key = DisableGlobalIRQ();
		frag->frags = (void *)bt_list_get(list);
		EnableGlobalIRQ(key);

		__ASSERT_NO_MSG(frag->frags);

		/* The fragments flag is only for list-internal usage */
		frag->flags &= ~NET_BUF_FRAGS;
	}

	/* Mark the end of the fragment list */
	frag->frags = NULL;

	return buf;
}
#endif

void net_buf_put(osa_msgq_handle_t fifo, struct net_buf *buf)
{
	struct net_buf *tail;
	osa_status_t ret;

	__ASSERT_NO_MSG(fifo);
	__ASSERT_NO_MSG(buf);

	for (tail = buf; tail->frags; tail = tail->frags) {
		tail->flags |= NET_BUF_FRAGS;
	}

	ret = OSA_MsgQPut(fifo, &buf);
	__ASSERT_NO_MSG(KOSA_StatusSuccess == ret);
    (void)ret;
}

#if (defined(CONFIG_NET_BUF_LOG) && (CONFIG_NET_BUF_LOG > 0U))
void net_buf_unref_debug(struct net_buf *buf, const char *func, int line)
#else
void net_buf_unref(struct net_buf *buf)
#endif
{
	__ASSERT_NO_MSG(buf);

	while (buf) {
		struct net_buf *frags = buf->frags;
		struct net_buf_pool *pool;

#if (defined(CONFIG_NET_BUF_LOG) && (CONFIG_NET_BUF_LOG > 0U))
		if (!buf->ref) {
			NET_BUF_ERR("%s():%d: buf %p double free", func, line,
				    buf);
			return;
		}
#endif
		NET_BUF_DBG("buf %p ref %u pool_id %u frags %p", buf, buf->ref,
			    buf->pool_id, buf->frags);

		if (--buf->ref > 0) {
			return;
		}

		if (buf->__buf) {
			data_unref(buf, buf->__buf);
			buf->__buf = NULL;
		}

		buf->data = NULL;
		buf->frags = NULL;

		pool = net_buf_pool_get(buf->pool_id);

#if (defined(CONFIG_NET_BUF_POOL_USAGE) && (CONFIG_NET_BUF_POOL_USAGE > 0U))
		pool->avail_count++;
		__ASSERT_NO_MSG(pool->avail_count <= pool->buf_count);
        NET_BUF_WARN("%s():%d: Pool %p available count %d",
                 func, line, pool, pool->avail_count);
#endif

		if (pool->destroy) {
			pool->destroy(buf);
		} else {
			net_buf_destroy(buf);
		}

		buf = frags;
	}
}

struct net_buf *net_buf_ref(struct net_buf *buf)
{
	__ASSERT_NO_MSG(buf);

	NET_BUF_DBG("buf %p (old) ref %u pool_id %u",
		    buf, buf->ref, buf->pool_id);
	buf->ref++;
	return buf;
}

struct net_buf *net_buf_clone(struct net_buf *buf, k_timeout_t timeout)
{
	uint32_t alloc_start = OSA_TimeGetMsec();
	struct net_buf_pool *pool;
	struct net_buf *clone;

	__ASSERT_NO_MSG(buf);

	pool = net_buf_pool_get(buf->pool_id);

	clone = net_buf_alloc_len(pool, 0, timeout);
	if (!clone) {
		return NULL;
	}

	/* If the pool supports data referencing use that. Otherwise
	 * we need to allocate new data and make a copy.
	 */
	if (pool->alloc->cb->ref && !(buf->flags & NET_BUF_EXTERNAL_DATA)) {
		clone->__buf = data_ref(buf, buf->__buf);
		clone->data = buf->data;
		clone->len = buf->len;
		clone->size = buf->size;
	} else {
		size_t size = buf->size;

		if (timeout != osaWaitNone_c && timeout != osaWaitForever_c) {
			uint32_t diff = OSA_TimeGetMsec() - alloc_start;

			timeout -= MIN(timeout, diff);
		}

		clone->__buf = data_alloc(clone, &size, timeout);
		if (!clone->__buf || size < buf->size) {
			net_buf_destroy(clone);
			return NULL;
		}

		clone->size = size;
		clone->data = clone->__buf + net_buf_headroom(buf);
		net_buf_add_mem(clone, buf->data, buf->len);
	}

	return clone;
}

struct net_buf *net_buf_frag_last(struct net_buf *buf)
{
	__ASSERT_NO_MSG(buf);

	while (buf->frags) {
		buf = buf->frags;
	}

	return buf;
}

void net_buf_frag_insert(struct net_buf *parent, struct net_buf *frag)
{
	__ASSERT_NO_MSG(parent);
	__ASSERT_NO_MSG(frag);

	if (parent->frags) {
		net_buf_frag_last(frag)->frags = parent->frags;
	}
	/* Take ownership of the fragment reference */
	parent->frags = frag;
}

struct net_buf *net_buf_frag_add(struct net_buf *head, struct net_buf *frag)
{
	__ASSERT_NO_MSG(frag);

	if (!head) {
		return net_buf_ref(frag);
	}

	net_buf_frag_insert(net_buf_frag_last(head), frag);

	return head;
}

#if (defined(CONFIG_NET_BUF_LOG) && (CONFIG_NET_BUF_LOG > 0U))
struct net_buf *net_buf_frag_del_debug(struct net_buf *parent,
				       struct net_buf *frag,
				       const char *func, int line)
#else
struct net_buf *net_buf_frag_del(struct net_buf *parent, struct net_buf *frag)
#endif
{
	struct net_buf *next_frag;

	__ASSERT_NO_MSG(frag);

	if (parent) {
		__ASSERT_NO_MSG(parent->frags);
		__ASSERT_NO_MSG(parent->frags == frag);
		parent->frags = frag->frags;
	}

	next_frag = frag->frags;

	frag->frags = NULL;

#if (defined(CONFIG_NET_BUF_LOG) && (CONFIG_NET_BUF_LOG > 0U))
	net_buf_unref_debug(frag, func, line);
#else
	net_buf_unref(frag);
#endif

	return next_frag;
}

size_t net_buf_linearize(void *dst, size_t dst_len, struct net_buf *src,
			 size_t offset, size_t len)
{
	struct net_buf *frag;
	size_t to_copy;
	size_t copied;

	len = MIN(len, dst_len);

	frag = src;

	/* find the right fragment to start copying from */
	while (frag && offset >= frag->len) {
		offset -= frag->len;
		frag = frag->frags;
	}

	/* traverse the fragment chain until len bytes are copied */
	copied = 0;
	while (frag && len > 0) {
		to_copy = MIN(len, frag->len - offset);
		memcpy((uint8_t *)dst + copied, frag->data + offset, to_copy);

		copied += to_copy;

		/* to_copy is always <= len */
		len -= to_copy;
		frag = frag->frags;

		/* after the first iteration, this value will be 0 */
		offset = 0;
	}

	return copied;
}

/* This helper routine will append multiple bytes, if there is no place for
 * the data in current fragment then create new fragment and add it to
 * the buffer. It assumes that the buffer has at least one fragment.
 */
size_t net_buf_append_bytes(struct net_buf *buf, size_t len,
			    const void *value, k_timeout_t timeout,
			    net_buf_allocator_cb allocate_cb, void *user_data)
{
	struct net_buf *frag = net_buf_frag_last(buf);
	size_t added_len = 0;
	const uint8_t *value8 = (const uint8_t *)value;

	do {
		uint16_t count = MIN(len, net_buf_tailroom(frag));

		net_buf_add_mem(frag, value8, count);
		len -= count;
		added_len += count;
		value8 += count;

		if (len == 0) {
			return added_len;
		}

		if (allocate_cb) {
			frag = allocate_cb(timeout, user_data);
		} else {
			struct net_buf_pool *pool;

			/* Allocate from the original pool if no callback has
			 * been provided.
			 */
			pool = net_buf_pool_get(buf->pool_id);
			frag = net_buf_alloc_len(pool, len, timeout);
		}

		if (!frag) {
			return added_len;
		}

		net_buf_frag_add(buf, frag);
	} while (1);

#if 0
	/* Unreachable */
	return 0;
#endif
}

#if (defined(CONFIG_NET_BUF_SIMPLE_LOG) && (CONFIG_NET_BUF_SIMPLE_LOG > 0U))
#define NET_BUF_SIMPLE_DBG(fmt, ...) NET_BUF_DBG(fmt, ##__VA_ARGS__)
#define NET_BUF_SIMPLE_ERR(fmt, ...) NET_BUF_ERR(fmt, ##__VA_ARGS__)
#define NET_BUF_SIMPLE_WARN(fmt, ...) NET_BUF_WARN(fmt, ##__VA_ARGS__)
#define NET_BUF_SIMPLE_INFO(fmt, ...) NET_BUF_INFO(fmt, ##__VA_ARGS__)
#else
#define NET_BUF_SIMPLE_DBG(fmt, ...)
#define NET_BUF_SIMPLE_ERR(fmt, ...)
#define NET_BUF_SIMPLE_WARN(fmt, ...)
#define NET_BUF_SIMPLE_INFO(fmt, ...)
#endif /* CONFIG_NET_BUF_SIMPLE_LOG */

void net_buf_simple_clone(const struct net_buf_simple *original,
			  struct net_buf_simple *clone)
{
	memcpy(clone, original, sizeof(struct net_buf_simple));
}

void *net_buf_simple_add(struct net_buf_simple *buf, size_t len)
{
	uint8_t *tail = net_buf_simple_tail(buf);

	NET_BUF_SIMPLE_DBG("buf %p len %lu", buf, len);

	__ASSERT_NO_MSG(net_buf_simple_tailroom(buf) >= len);

	buf->len += len;
	return tail;
}

void *net_buf_simple_add_mem(struct net_buf_simple *buf, const void *mem,
			     size_t len)
{
	NET_BUF_SIMPLE_DBG("buf %p len %lu", buf, len);

	return memcpy(net_buf_simple_add(buf, len), mem, len);
}

uint8_t *net_buf_simple_add_u8(struct net_buf_simple *buf, uint8_t val)
{
	uint8_t *u8;

	NET_BUF_SIMPLE_DBG("buf %p val 0x%02x", buf, val);

	u8 = (uint8_t *)net_buf_simple_add(buf, 1);
	*u8 = val;

	return u8;
}

void net_buf_simple_add_le16(struct net_buf_simple *buf, uint16_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %u", buf, val);

	sys_put_le16(val, (uint8_t *)net_buf_simple_add(buf, sizeof(val)));
}

void net_buf_simple_add_be16(struct net_buf_simple *buf, uint16_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %u", buf, val);

	sys_put_be16(val, (uint8_t *)net_buf_simple_add(buf, sizeof(val)));
}

void net_buf_simple_add_le24(struct net_buf_simple *buf, uint32_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %u", buf, val);

	sys_put_le24(val, (uint8_t *)net_buf_simple_add(buf, 3));
}

void net_buf_simple_add_be24(struct net_buf_simple *buf, uint32_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %u", buf, val);

	sys_put_be24(val, (uint8_t *)net_buf_simple_add(buf, 3));
}

void net_buf_simple_add_le32(struct net_buf_simple *buf, uint32_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %u", buf, val);

	sys_put_le32(val, (uint8_t *)net_buf_simple_add(buf, sizeof(val)));
}

void net_buf_simple_add_be32(struct net_buf_simple *buf, uint32_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %u", buf, val);

	sys_put_be32(val, (uint8_t *)net_buf_simple_add(buf, sizeof(val)));
}

void net_buf_simple_add_le48(struct net_buf_simple *buf, uint64_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %" PRIu64, buf, val);

	sys_put_le48(val, (uint8_t *)net_buf_simple_add(buf, 6));
}

void net_buf_simple_add_be48(struct net_buf_simple *buf, uint64_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %" PRIu64, buf, val);

	sys_put_be48(val, (uint8_t *)net_buf_simple_add(buf, 6));
}

void net_buf_simple_add_le64(struct net_buf_simple *buf, uint64_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %" PRIu64, buf, val);

	sys_put_le64(val, (uint8_t *)net_buf_simple_add(buf, sizeof(val)));
}

void net_buf_simple_add_be64(struct net_buf_simple *buf, uint64_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %" PRIu64, buf, val);

	sys_put_be64(val, (uint8_t *)net_buf_simple_add(buf, sizeof(val)));
}

void *net_buf_simple_remove_mem(struct net_buf_simple *buf, size_t len)
{
	NET_BUF_SIMPLE_DBG("buf %p len %zu", buf, len);

	__ASSERT_NO_MSG(buf->len >= len);

	buf->len -= len;
	return buf->data + buf->len;
}

uint8_t net_buf_simple_remove_u8(struct net_buf_simple *buf)
{
	uint8_t val;
	void *ptr;

	ptr = net_buf_simple_remove_mem(buf, sizeof(val));
	val = *(uint8_t *)ptr;

	return val;
}

uint16_t net_buf_simple_remove_le16(struct net_buf_simple *buf)
{
	uint16_t val;
	void *ptr;
	STRUCT_PACKED_PRE
	struct __temp_struct{
		uint16_t __v;
	} STRUCT_PACKED_POST;
	struct __temp_struct *__p;

	ptr = net_buf_simple_remove_mem(buf, sizeof(val));

    __p = (struct __temp_struct *) ((uint16_t *)ptr);
	val = __p->__v;

	return sys_le16_to_cpu(val);
}

uint16_t net_buf_simple_remove_be16(struct net_buf_simple *buf)
{
	uint16_t val;
	void *ptr;
    STRUCT_PACKED_PRE
	struct __temp_struct{
		uint16_t __v;
	} STRUCT_PACKED_POST;
	struct __temp_struct *__p;

	ptr = net_buf_simple_remove_mem(buf, sizeof(val));

    __p = (struct __temp_struct *) ((uint16_t *)ptr);
	val = __p->__v;

	return sys_be16_to_cpu(val);
}

uint32_t net_buf_simple_remove_le24(struct net_buf_simple *buf)
{
	STRUCT_PACKED_PRE
    struct uint24 {
		uint32_t u24 : 24;
	} STRUCT_PACKED_POST;
    struct uint24 val;
	void *ptr;
    STRUCT_PACKED_PRE
	struct __temp_struct{
		struct uint24 __v;
	} STRUCT_PACKED_POST;
	struct __temp_struct *__p;

	ptr = net_buf_simple_remove_mem(buf, sizeof(val));

    __p = (struct __temp_struct *) ((struct uint24 *)ptr);
	val = __p->__v;

	return sys_le24_to_cpu(val.u24);
}

uint32_t net_buf_simple_remove_be24(struct net_buf_simple *buf)
{
	STRUCT_PACKED_PRE
    struct uint24 {
		uint32_t u24 : 24;
	} STRUCT_PACKED_POST;
    struct uint24 val;
	void *ptr;
    STRUCT_PACKED_PRE
	struct __temp_struct{
		struct uint24 __v;
	} STRUCT_PACKED_POST;
	struct __temp_struct *__p;

	ptr = net_buf_simple_remove_mem(buf, sizeof(val));

    __p = (struct __temp_struct *) ((struct uint24 *)ptr);
	val = __p->__v;

	return sys_be24_to_cpu(val.u24);
}

uint32_t net_buf_simple_remove_le32(struct net_buf_simple *buf)
{
	uint32_t val;
	void *ptr;
    STRUCT_PACKED_PRE
	struct __temp_struct{
		uint32_t __v;
	} STRUCT_PACKED_POST;
	struct __temp_struct *__p;

	ptr = net_buf_simple_remove_mem(buf, sizeof(val));

    __p = (struct __temp_struct *) ((uint32_t *)ptr);
	val = __p->__v;

	return sys_le32_to_cpu(val);
}

uint32_t net_buf_simple_remove_be32(struct net_buf_simple *buf)
{
	uint32_t val;
	void *ptr;
    STRUCT_PACKED_PRE
	struct __temp_struct{
		uint32_t __v;
	} STRUCT_PACKED_POST;
	struct __temp_struct *__p;

	ptr = net_buf_simple_remove_mem(buf, sizeof(val));

    __p = (struct __temp_struct *) ((uint32_t *)ptr);
	val = __p->__v;

	return sys_be32_to_cpu(val);
}

uint64_t net_buf_simple_remove_le48(struct net_buf_simple *buf)
{
    STRUCT_PACKED_PRE
    struct uint48 {
		uint64_t u48 : 48;
	} STRUCT_PACKED_POST;
    struct uint48 val;
	void *ptr;
    STRUCT_PACKED_PRE
	struct __temp_struct{
		struct uint48 __v;
	} STRUCT_PACKED_POST;
	struct __temp_struct *__p;

	ptr = net_buf_simple_remove_mem(buf, sizeof(val));

    __p = (struct __temp_struct *) ((struct uint48 *)ptr);
	val = __p->__v;

	return sys_le48_to_cpu(val.u48);
}

uint64_t net_buf_simple_remove_be48(struct net_buf_simple *buf)
{
    STRUCT_PACKED_PRE
    struct uint48 {
		uint64_t u48 : 48;
	} STRUCT_PACKED_POST;
    struct uint48 val;
	void *ptr;
    STRUCT_PACKED_PRE
	struct __temp_struct{
		struct uint48 __v;
	} STRUCT_PACKED_POST;
	struct __temp_struct *__p;

	ptr = net_buf_simple_remove_mem(buf, sizeof(val));

    __p = (struct __temp_struct *) ((struct uint48 *)ptr);
	val = __p->__v;

	return sys_be48_to_cpu(val.u48);
}

uint64_t net_buf_simple_remove_le64(struct net_buf_simple *buf)
{
	uint64_t val;
	void *ptr;
    STRUCT_PACKED_PRE
	struct __temp_struct{
		uint64_t __v;
	} STRUCT_PACKED_POST;
	struct __temp_struct *__p;

	ptr = net_buf_simple_remove_mem(buf, sizeof(val));

    __p = (struct __temp_struct *) ((uint64_t *)ptr);
	val = __p->__v;

	return sys_le64_to_cpu(val);
}

uint64_t net_buf_simple_remove_be64(struct net_buf_simple *buf)
{
	uint64_t val;
	void *ptr;
    STRUCT_PACKED_PRE
	struct __temp_struct{
		uint64_t __v;
	} STRUCT_PACKED_POST;
	struct __temp_struct *__p;

	ptr = net_buf_simple_remove_mem(buf, sizeof(val));

    __p = (struct __temp_struct *) ((uint64_t *)ptr);
	val = __p->__v;

	return sys_be64_to_cpu(val);
}

void *net_buf_simple_push(struct net_buf_simple *buf, size_t len)
{
	NET_BUF_SIMPLE_DBG("buf %p len %lu", buf, len);

	__ASSERT_NO_MSG(net_buf_simple_headroom(buf) >= len);

	buf->data -= len;
	buf->len += len;
	return buf->data;
}

void *net_buf_simple_push_mem(struct net_buf_simple *buf, const void *mem,
			      size_t len)
{
	NET_BUF_SIMPLE_DBG("buf %p len %zu", buf, len);

	return memcpy(net_buf_simple_push(buf, len), mem, len);
}

void net_buf_simple_push_le16(struct net_buf_simple *buf, uint16_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %u", buf, val);

	sys_put_le16(val, (uint8_t *)net_buf_simple_push(buf, sizeof(val)));
}

void net_buf_simple_push_be16(struct net_buf_simple *buf, uint16_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %u", buf, val);

	sys_put_be16(val, (uint8_t *)net_buf_simple_push(buf, sizeof(val)));
}

void net_buf_simple_push_u8(struct net_buf_simple *buf, uint8_t val)
{
	uint8_t *data = (uint8_t *)net_buf_simple_push(buf, 1);

	*data = val;
}

void net_buf_simple_push_le24(struct net_buf_simple *buf, uint32_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %u", buf, val);

	sys_put_le24(val, (uint8_t *)net_buf_simple_push(buf, 3));
}

void net_buf_simple_push_be24(struct net_buf_simple *buf, uint32_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %u", buf, val);

	sys_put_be24(val, (uint8_t *)net_buf_simple_push(buf, 3));
}

void net_buf_simple_push_le32(struct net_buf_simple *buf, uint32_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %u", buf, val);

	sys_put_le32(val, (uint8_t *)net_buf_simple_push(buf, sizeof(val)));
}

void net_buf_simple_push_be32(struct net_buf_simple *buf, uint32_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %u", buf, val);

	sys_put_be32(val, (uint8_t *)net_buf_simple_push(buf, sizeof(val)));
}

void net_buf_simple_push_le48(struct net_buf_simple *buf, uint64_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %" PRIu64, buf, val);

	sys_put_le48(val, (uint8_t *)net_buf_simple_push(buf, 6));
}

void net_buf_simple_push_be48(struct net_buf_simple *buf, uint64_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %" PRIu64, buf, val);

	sys_put_be48(val, (uint8_t *)net_buf_simple_push(buf, 6));
}

void net_buf_simple_push_le64(struct net_buf_simple *buf, uint64_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %" PRIu64, buf, val);

	sys_put_le64(val, (uint8_t *)net_buf_simple_push(buf, sizeof(val)));
}

void net_buf_simple_push_be64(struct net_buf_simple *buf, uint64_t val)
{
	NET_BUF_SIMPLE_DBG("buf %p val %" PRIu64, buf, val);

	sys_put_be64(val, (uint8_t *)net_buf_simple_push(buf, sizeof(val)));
}

void *net_buf_simple_pull(struct net_buf_simple *buf, size_t len)
{
	NET_BUF_SIMPLE_DBG("buf %p len %lu", buf, len);

	__ASSERT_NO_MSG(buf->len >= len);

	buf->len -= len;
	return buf->data += len;
}

void *net_buf_simple_pull_mem(struct net_buf_simple *buf, size_t len)
{
	void *data = buf->data;

	NET_BUF_SIMPLE_DBG("buf %p len %lu", buf, len);

	__ASSERT_NO_MSG(buf->len >= len);

	buf->len -= len;
	buf->data += len;

	return data;
}

uint8_t net_buf_simple_pull_u8(struct net_buf_simple *buf)
{
	uint8_t val;

	val = buf->data[0];
	net_buf_simple_pull(buf, 1);

	return val;
}

uint16_t net_buf_simple_pull_le16(struct net_buf_simple *buf)
{
	uint16_t val;
	STRUCT_PACKED_PRE
	struct __temp_struct{
		uint16_t __v;
	} STRUCT_PACKED_POST;

	struct __temp_struct *__p = (struct __temp_struct *) ((uint16_t *)buf->data);
	val = __p->__v;
	net_buf_simple_pull(buf, sizeof(val));

	return sys_le16_to_cpu(val);
}

uint16_t net_buf_simple_pull_be16(struct net_buf_simple *buf)
{
	uint16_t val;
	STRUCT_PACKED_PRE
	struct __temp_struct{
		uint16_t __v;
	} STRUCT_PACKED_POST;

	struct __temp_struct *__p = (struct __temp_struct *) ((uint16_t *)buf->data);
	val = __p->__v;
	net_buf_simple_pull(buf, sizeof(val));

	return sys_be16_to_cpu(val);
}

uint32_t net_buf_simple_pull_le24(struct net_buf_simple *buf)
{
	STRUCT_PACKED_PRE
	struct uint24 {
		uint32_t u24:24;
	} STRUCT_PACKED_POST;
	struct uint24 val;
	STRUCT_PACKED_PRE
	struct __temp_struct{
		struct uint24 __v;
	} STRUCT_PACKED_POST;

	struct __temp_struct *__p = (struct __temp_struct *) ((struct uint24 *)buf->data);
	val = __p->__v;
	net_buf_simple_pull(buf, sizeof(val));

	return sys_le24_to_cpu(val.u24);
}

uint32_t net_buf_simple_pull_be24(struct net_buf_simple *buf)
{
	STRUCT_PACKED_PRE
	struct uint24 {
		uint32_t u24:24;
	} STRUCT_PACKED_POST;
	struct uint24 val;
	STRUCT_PACKED_PRE
	struct __temp_struct{
		struct uint24 __v;
	} STRUCT_PACKED_POST;

	struct __temp_struct *__p = (struct __temp_struct *) ((struct uint24 *)buf->data);
	val = __p->__v;
	net_buf_simple_pull(buf, sizeof(val));

	return sys_be24_to_cpu(val.u24);
}

uint32_t net_buf_simple_pull_le32(struct net_buf_simple *buf)
{
	uint32_t val;
	STRUCT_PACKED_PRE
	struct __temp_struct{
		uint32_t __v;
	} STRUCT_PACKED_POST;

	struct __temp_struct *__p = (struct __temp_struct *) ((uint32_t *)buf->data);
	val = __p->__v;
	net_buf_simple_pull(buf, sizeof(val));

	return sys_le32_to_cpu(val);
}

uint32_t net_buf_simple_pull_be32(struct net_buf_simple *buf)
{
	uint32_t val;
	STRUCT_PACKED_PRE
	struct __temp_struct{
		uint32_t __v;
	} STRUCT_PACKED_POST;

	struct __temp_struct *__p = (struct __temp_struct *) ((uint32_t *)buf->data);
	val = __p->__v;
	net_buf_simple_pull(buf, sizeof(val));

	return sys_be32_to_cpu(val);
}

uint64_t net_buf_simple_pull_le48(struct net_buf_simple *buf)
{
	STRUCT_PACKED_PRE
	struct uint48 {
		uint64_t u48:48;
	} STRUCT_PACKED_POST;
	struct uint48 val;
	STRUCT_PACKED_PRE
	struct __temp_struct{
		struct uint48 __v;
	} STRUCT_PACKED_POST;

	struct __temp_struct *__p = (struct __temp_struct *) ((struct uint48 *)buf->data);
	val = __p->__v;
	net_buf_simple_pull(buf, sizeof(val));

	return sys_le48_to_cpu(val.u48);
}

uint64_t net_buf_simple_pull_be48(struct net_buf_simple *buf)
{
	STRUCT_PACKED_PRE
	struct uint48 {
		uint64_t u48:48;
	} STRUCT_PACKED_POST;
	struct uint48 val;
	STRUCT_PACKED_PRE
	struct __temp_struct{
		struct uint48 __v;
	} STRUCT_PACKED_POST;

	struct __temp_struct *__p = (struct __temp_struct *) ((struct uint48 *)buf->data);
	val = __p->__v;
	net_buf_simple_pull(buf, sizeof(val));

	return sys_be48_to_cpu(val.u48);
}

uint64_t net_buf_simple_pull_le64(struct net_buf_simple *buf)
{
	uint64_t val;
	STRUCT_PACKED_PRE
	struct __temp_struct{
		uint64_t __v;
	} STRUCT_PACKED_POST;

	struct __temp_struct *__p = (struct __temp_struct *) ((uint64_t *)buf->data);
	val = __p->__v;
	net_buf_simple_pull(buf, sizeof(val));

	return sys_le64_to_cpu(val);
}

uint64_t net_buf_simple_pull_be64(struct net_buf_simple *buf)
{
	uint64_t val;
	STRUCT_PACKED_PRE
	struct __temp_struct{
		uint64_t __v;
	} STRUCT_PACKED_POST;

	struct __temp_struct *__p = (struct __temp_struct *) ((uint64_t *)buf->data);
	val = __p->__v;
	net_buf_simple_pull(buf, sizeof(val));

	return sys_be64_to_cpu(val);
}

size_t net_buf_simple_headroom(struct net_buf_simple *buf)
{
	return buf->data - buf->__buf;
}

size_t net_buf_simple_tailroom(struct net_buf_simple *buf)
{
	return buf->size - net_buf_simple_headroom(buf) - buf->len;
}

uint16_t net_buf_simple_max_len(struct net_buf_simple *buf)
{
	return buf->size - net_buf_simple_headroom(buf);
}
