/*
 * Copyright 2020-2021 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef _DISPLAY_SUPPORT_H_
#define _DISPLAY_SUPPORT_H_

#include "fsl_dc_fb.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define DEMO_PANEL_RK055AHD091        1   /* 720 * 1280 */
#define DEMO_PANEL_RK055IQH091        2   /* 540 * 960 */
#define DEMO_PANEL_RK043FN02HC        3   /* 480 * 272 */
#define DEMO_PANEL_KD050WVTPA045      4   /* 800 * 480 */
#define DEMO_PANEL_HDMI_720X480_60   10
#define DEMO_PANEL_HDMI_720X576_50   11
#define DEMO_PANEL_HDMI_800X480_60   12
#define DEMO_PANEL_HDMI_1024X768_60  13
#define DEMO_PANEL_HDMI_1280X720_60  14

#define DEMO_DISPLAY_CONTROLLER_ELCDIF  0
#define DEMO_DISPLAY_CONTROLLER_LCDIFV2 1

#ifndef DEMO_PANEL
  #if defined(CPU_MIMXRT1052DVL6B) || defined(CPU_MIMXRT1062DVL6A) || defined(CPU_MIMXRT1064DVL6A)

    #define DEMO_PANEL                 DEMO_PANEL_RK043FN02HC

    #define APP_PXP_OUT_FORMAT         kPXP_OutputPixelFormatRGB565
    #define APP_CSC1_MODE              kPXP_Csc1YUV2RGB
    #define APP_CSC1_MODE_ENABLE       false
    #define APP_ROTATE_DISPLAY         kPXP_Rotate180
    #define APP_ROTATE_DISPLAY_NUM     180
    #define APP_FLIP_DISPLAY           kPXP_FlipDisable

    /* Frame buffer data alignment, for better performance, the LCDIF frame buffer should be 64B align. */
    #define DEMO_FRAME_BUFFER_ALIGN    64
    #define EIQ_DEFAULT_CAPTURE_RATE   50

  #elif defined( CPU_MIMXRT1176DVMAA_cm7 ) || defined( CPU_MIMXRT1166DVM6A_cm7 )

    #define DEMO_PANEL                 DEMO_PANEL_RK043FN02HC

    #define APP_PXP_OUT_FORMAT         kPXP_OutputPixelFormatRGB565
    #define APP_CSC1_MODE              kPXP_Csc1YCbCr2RGB
    #define APP_CSC1_MODE_ENABLE       false
    #define APP_ROTATE_DISPLAY         kPXP_Rotate0//kPXP_Rotate90
    #define APP_ROTATE_DISPLAY_NUM     0//90
    #define APP_FLIP_DISPLAY           kPXP_FlipDisable//kPXP_FlipBoth

    /* Frame buffer data alignment, for better performance, the LCD frame buffer should be 32B align. */
    #define DEMO_FRAME_BUFFER_ALIGN    32
    #define EIQ_DEFAULT_CAPTURE_RATE   25

  #endif
#endif // CPU_MIMXRT1176DVMAA

#ifndef DEMO_DISPLAY_CONTROLLER
  #if ((DEMO_PANEL_RK043FN02HC == DEMO_PANEL) || (DEMO_PANEL_KD050WVTPA045 == DEMO_PANEL))
    /* These Parallel RGB displays can only use the ELCDIF interface. */
    #define DEMO_DISPLAY_CONTROLLER DEMO_DISPLAY_CONTROLLER_ELCDIF
  #else
    /* Use LCDIFV2 by default, could use ELCDIF by changing this macro. */
    #define DEMO_DISPLAY_CONTROLLER DEMO_DISPLAY_CONTROLLER_LCDIFV2
  #endif
#endif

#if (DEMO_PANEL == DEMO_PANEL_RK055AHD091)

  #define DEMO_PANEL_WIDTH  (720)
  #define DEMO_PANEL_HEIGHT (1280)
  #define DEMO_PANEL_NAME   "RK055AHD091"

  #define DEMO_HSW 8
  #define DEMO_HFP 32
  #define DEMO_HBP 32
  #define DEMO_VSW 2
  #define DEMO_VFP 16
  #define DEMO_VBP 14

#elif (DEMO_PANEL == DEMO_PANEL_RK055IQH091)

  #define DEMO_PANEL_WIDTH  (540)
  #define DEMO_PANEL_HEIGHT (960)
  #define DEMO_PANEL_NAME   "RK055IQH091"

  #define DEMO_HSW 2
  #define DEMO_HFP 32
  #define DEMO_HBP 30
  #define DEMO_VSW 2
  #define DEMO_VFP 16
  #define DEMO_VBP 14

#elif (DEMO_PANEL == DEMO_PANEL_RK043FN02HC)

  #define DEMO_PANEL_WIDTH  (480)
  #define DEMO_PANEL_HEIGHT (272)
  #define DEMO_PANEL_NAME   "RK043FN02HC"

  #define DEMO_HSW 2
  #define DEMO_HFP 32
  #define DEMO_HBP 30
  #define DEMO_VSW 2
  #define DEMO_VFP 16
  #define DEMO_VBP 14

#elif (DEMO_PANEL_KD050WVTPA045 == DEMO_PANEL)

  #define DEMO_PANEL_WIDTH  (800)
  #define DEMO_PANEL_HEIGHT (480)
  #define DEMO_PANEL_NAME   "KD050WVTPA045"

  #define DEMO_HSW 48
  #define DEMO_HFP 40
  #define DEMO_HBP 40
  #define DEMO_VSW 3
  #define DEMO_VFP 141
  #define DEMO_VBP 29

#elif (DEMO_PANEL_HDMI_720X480_60 == DEMO_PANEL)

  #define DEMO_PANEL_WIDTH  (720)
  #define DEMO_PANEL_HEIGHT (480)
  #define DEMO_PANEL_NAME   "HDMI_720X480_60"

  #define DEMO_HSW 62
  #define DEMO_HBP 60
  #define DEMO_HFP 16
  #define DEMO_VSW 6
  #define DEMO_VBP 30
  #define DEMO_VFP 9

  #define DEMO_VIDEO_FREQ_KHZ    594000
  #define DEMO_PIXEL_CLOCK_DIV   22
  #define DEMO_MIPI_DPY_REF_DIV  22

#elif (DEMO_PANEL_HDMI_720X576_50 == DEMO_PANEL)

  #define DEMO_PANEL_WIDTH  (720)
  #define DEMO_PANEL_HEIGHT (576)
  #define DEMO_PANEL_NAME   "HDMI_720X576_50"

  #define DEMO_HSW 64
  #define DEMO_HBP 68
  #define DEMO_HFP 12
  #define DEMO_VSW 5
  #define DEMO_VBP 39
  #define DEMO_VFP 5

  #define DEMO_VIDEO_FREQ_KHZ    594000
  #define DEMO_PIXEL_CLOCK_DIV   22
  #define DEMO_MIPI_DPY_REF_DIV  22

#elif (DEMO_PANEL_HDMI_800X480_60 == DEMO_PANEL)

  #define DEMO_PANEL_WIDTH  (800)
  #define DEMO_PANEL_HEIGHT (480)
  #define DEMO_PANEL_NAME   "HDMI_800X480_60"

  #define DEMO_HSW 20
  #define DEMO_HFP 70
  #define DEMO_HBP 26
  #define DEMO_VSW 2
  #define DEMO_VFP 7
  #define DEMO_VBP 21

  #define DEMO_VIDEO_FREQ_KHZ    594000
  #define DEMO_PIXEL_CLOCK_DIV   22
  #define DEMO_MIPI_DPY_REF_DIV  22

#elif (DEMO_PANEL_HDMI_1024X768_60 == DEMO_PANEL)

  #define DEMO_PANEL_WIDTH  (1024)
  #define DEMO_PANEL_HEIGHT (768)
  #define DEMO_PANEL_NAME   "HDMI_1024X768_60"

  #define DEMO_HSW 136
  #define DEMO_HBP 160
  #define DEMO_HFP 24
  #define DEMO_VSW 6
  #define DEMO_VBP 29
  #define DEMO_VFP 3

  #define DEMO_VIDEO_FREQ_KHZ    260000
  #define DEMO_PIXEL_CLOCK_DIV   4
  #define DEMO_MIPI_DPY_REF_DIV  10

#elif (DEMO_PANEL_HDMI_1280X720_60 == DEMO_PANEL)

  #define DEMO_PANEL_WIDTH  (1280)
  #define DEMO_PANEL_HEIGHT (720)
  #define DEMO_PANEL_NAME   "HDMI_1280X720_60"

  #define DEMO_HSW 40
  #define DEMO_HBP 220
  #define DEMO_HFP 110
  #define DEMO_VSW 5
  #define DEMO_VBP 20
  #define DEMO_VFP 5

  #define DEMO_VIDEO_FREQ_KHZ    594000
  #define DEMO_PIXEL_CLOCK_DIV   8
  #define DEMO_MIPI_DPY_REF_DIV  22

#else
  #error Must set DEMO_PANEL to one of the valid DEMO_PANEL_* values
#endif

#define IS_HDMI_PANEL()         (DEMO_PANEL >= 10)
#define USE_FT5406_TOUCH()      (DEMO_PANEL_RK043FN02HC == DEMO_PANEL)
#define USE_GT911_TOUCH()       ((DEMO_PANEL_RK055AHD091 == DEMO_PANEL) || (DEMO_PANEL_RK055IQH091 == DEMO_PANEL) || (DEMO_PANEL_RK055MHD091 == DEMO_PANEL))
#define IS_TOUCH_SUPPORTED()    (USE_FT5406_TOUCH() || USE_GT911_TOUCH())
#define IS_PARALLEL_RGB_PANEL() ((DEMO_PANEL_RK043FN02HC == DEMO_PANEL) || (DEMO_PANEL_KD050WVTPA045 == DEMO_PANEL))

#if (IS_PARALLEL_RGB_PANEL() && (DEMO_DISPLAY_CONTROLLER == DEMO_DISPLAY_CONTROLLER_LCDIFV2))
  #error This combination of a Parallel RGB display and the LCDIFv2 interface is not \
         supported on the Embedded Artists uCOM Carrier Board. Switch to using the \
         eLCDIF interface instead. If you develop your own carrier board and add \
         support for 24-bit Parallel RGB then you can comment out this error handling.
#endif

#define DEMO_BUFFER_FIXED_ADDRESS 0

#if DEMO_BUFFER_FIXED_ADDRESS
//TODO check other devices
#define DEMO_BUFFER0_ADDR 0x80000000
#define DEMO_BUFFER1_ADDR 0x80200000
#endif
/* Definitions for the frame buffer. */
#define DEMO_LCD_BUFFER_COUNT 2
#define DEMO_BUFFER_PIXEL_FORMAT   kVIDEO_PixelFormatRGB565
#define DEMO_LCD_BUFFER_BPP 2

#define DEMO_BUFFER_WIDTH DEMO_PANEL_WIDTH
#define DEMO_BUFFER_HEIGHT DEMO_PANEL_HEIGHT

#define DEMO_BUFFER_STRIDE_BYTE (DEMO_BUFFER_WIDTH * DEMO_LCD_BUFFER_BPP)
/* There is not frame buffer aligned requirement, consider the 64-bit AXI data
 * bus width and 32-byte cache line size, the frame buffer alignment is set to
 * 32 byte.
 */

/* Where the frame buffer is shown in the screen. */
#define DEMO_BUFFER_START_X 0U
#define DEMO_BUFFER_START_Y 0U

/*******************************************************************************
 * Variables
 ******************************************************************************/

/*! @brief Display controller. */
extern const dc_fb_t g_dc;

/*******************************************************************************
 * API
 ******************************************************************************/
#if defined(__cplusplus)
extern "C" {
#endif /* __cplusplus*/

/*!
 * @brief Initializes display controler.
 * 
 * @return status code
 */
status_t BOARD_PrepareDisplayController(void);

/*!
 * @brief Resets display controler.
 */
void BOARD_ResetDisplayMix(void);

#if defined(__cplusplus)
}
#endif /* __cplusplus */

#endif /* _DISPLAY_SUPPORT_H_ */
